/*	68020/68010 debugger	 */

#include "../h/types.h"

#include "reg.h"
#include "pte.h"
#include "machparam.h"
#include "psl.h"

#include "param.h"
#include "systm.h"
#include "dir.h"
#include "user.h"
#include "kernel.h"
#include "map.h"
#include "vm.h"
#include "proc.h"
#include "buf.h"
#include "reboot.h"
#include "conf.h"
#include "vnode.h"
#include "../ufs/inode.h"
#include "file.h"
#include "text.h"
#include "clist.h"
#include "callout.h"
#include "cmap.h"
#include "mbuf.h"
#include "msgbuf.h"
#include <syslog.h>
#include "quota.h"
#include "vmmac.h"
#include "debug.h"

#include "frame.h"
#include "trap.h"
#include "board.h"
#include "../is68kdev/qbvar.h"
#ifdef	GWS
#include "../is68kdev/gpreg.h"
#endif	GWS

int		enable_debugger;

#define	KVSYM_SIZE	100
int		kvsym_size;
struct debug_nl	kvsymtab[KVSYM_SIZE];


debug_input(c)
	register char c;
{
#if	DEBUGGER
	if (((c &= 0x7f) == MONITOR) && enable_debugger) {
  		trap15();
		return(1);
	} else
#endif	DEBUGGER
		return(0);
}

#if	DEBUGGER
/* these variables are defined by dbg_excpt in locore.s */
struct ex_frame {
	u_short	sr;
	u_int	pc;
	u_short	fmt_vec;
	u_short     sp_status;
}	*dbg_ex_frame;

u_int	dbg_reg_sav[16],		/* register save area */
	dbg_usp_sav,
	dbg_ssp_sav,
	dbg_nmi_sav;

int	dbg_step_over,			/* stepping over brkpt flag  */
	dbg_indebug,			/* in debugger flag */
	dbg_format = LONG;		/* display format -- BYTE, WORD, LONG */
		
struct brk_tbl_struct {			/* break point table */
	u_short	*adr;
	u_short	instr;
}	dbg_brk_tbl[NUM_BRKPTS];

u_char  dbg_buf[80];
char	dbg_huh_msg[] = "?";
char	*dbg_ex_msg[] = {
	 "...", "...", "BUS", "ADR", "ILN", "ZDV", "CHK", "TRV",
	 "PRV", "TRS", "AEM", "...", "...", "FEM", "FMT", "UIV",
	 "...", "",    "...", "...", "...", "...", "...", "...",
	 "SPU", "AV1", "AV2", "AV3", "AV4", "AV5", "AV6", "AV7",
	 "TR0", "TR1", "TR2", "TR3", "TR4", "TR5", "TR6", "TR7",
	 "TR8", "TR9", "T10", "T11", "T12", "T13", "T14", "T15"
};

struct debug_nl	symtab[NDBG_SYM] = { 0, 0 };	/* symbol table stuff */
char		strtab[NDBG_STR] = 0;
struct debug_nl	*ssymp;
int		sym_size = NDBG_SYM;
int		str_size = NDBG_STR;
int		sym_magic = DBG_MAGIC;

long		dot;
extern int	codeonly;

dbg()
{
	register struct ex_frame	*exf = dbg_ex_frame;
	register int	brk_num;
	int	res;
	
	if (dbg_step_over && (exf->sr & PSL_T)) {
		dbg_step_over = 0;
		exf->sr &= ~PSL_T;
		dbg_replace_brkpts();
		return;
	}
	if (dbg_indebug) {
		if (dbg_find_brkpt(exf->pc) != -1) {
			dbg_step_over = 1;
			exf->sr |= PSL_T;
		} else
			dbg_replace_brkpts();
		return;
	} else
		dbg_indebug = 1;

#ifdef	GWS
	vtctop_aged();		/* make the console comes to the top */
#endif	GWS
	if ((brk_num = dbg_find_brkpt(exf->pc - 2)) != -1 && 
	    !(exf->sr & PSL_T)) {
		exf->pc -= 2;
		printf("\nBreakpoint #%d", brk_num);
	} else if ((exf->sr & PSL_T) == 0)
		printf("\n%s", dbg_ex_msg[(exf->fmt_vec & ESF_VECMSK)>>2]);

	printf(": %s: PID %d:",u.u_comm, u.u_procp->p_pid);
	if (exf->sr & PSL_T)
		exf->sr &= ~PSL_T;
	else
		printf(": PC=%a SR=%4x\n",exf->pc,exf->sr);

	dbg_remove_brkpts();
	dot = exf->pc;
	dbg_dis_print();

	while (1) {
		while (cngetc() != -1)		/* flush uart buf */
			;
		printf("d:");			/* display prompt */
		dbg_get_line(dbg_buf);			/* get command line */
		switch (dbg_buf[0]) {
		    default:
			printf("\
-- Standard Functions:\n\
	?		Display Breakpoints\n\
	A adr		Disassemble\n\
	!		Continue to disassemble\n\
	D adr		Dump Memory\n\
	E adr		Examine/Change Memory\n\
	W adr		Write Memory\n\
	F B|W|L		Set Display Format\n\
	G adr		Go\n\
	I adr		Insert Breakpoint\n\
	K *|adr		Delete Breakpoint(s)\n\
	R adr		Display/Set Register(s)\n\
	S		Single Step\n\
	<esc>		Single Step\n\
	B		Reboot\n\
	X		Dump system\n");
			printf("\
-- Extended Functions:\n\
	C		Check Cache Consistancy\n\
	c		Print cmap\n\
	s		Print speeping procs\n\
	^		Print stack backtrace\n\
	m<ctx>		Print mmu\n\
	b		Print buffer use\n\
	g		Toggle GIP\n");

/*-- Extended Functions: */
#ifdef	M68025
  		    case 'C':	if (dbg_buf[1] == 'C')
					cache_chk();
				else if (dbg_buf[1] == 'F')
  		    			cache_flush();
				else if (dbg_buf[1] == 'P')
  		    			cache_print();
				break;
#endif	M68025
  		    case 'c':	prcmap();			break;
  		    case 's':	sleeptrace();			break;
  		    case '^':	stacktrace();			break;
#ifdef	M68010
  		    case 'm':	prmmu( dbg_buf[1] - '0');	break;
#endif	M68010
  		    case 'b':	bufertrace();			break;
#ifdef	VBUS
#ifdef	GWS
  		    case 'g':	gptype ^= GPHASGIP;
				printf("GIP %s\n",
					(gptype&GPHASGIP)?"on":"off"); 
				break;
#endif	GWS
#endif	VBUS

/*-- Standard Functions: */
		    case NULL:					break;
		    case '?':	dbg_dsp_brkpt();		break;
		    case 'A':	dbg_disasm(dbg_buf + 1);	break;
		    case '!':	dbg_disasm(0);			break;
		    case 'B':	panic("reboot");		break;
		    case 'X':	dumpsys();			break;
		    case 'D':	dbg_dump();			break;
		    case 'E':	dbg_examine(0);			break;
		    case 'W':	dbg_examine(1);			break;
		    case 'F':	dbg_set_format();		break;
		    case 'I':	dbg_insert_brkpt(exf);		break;
		    case 'K':	dbg_rm_brkpt();			break;
		    case 'R':	dbg_dsp_regs(exf);		break;

		    case 'G':
		    case MONITOR:
			if (dbg_buf[1] != NULL) {
				if (dbg_get_adr(&dbg_buf[1], &res))
					break;
				exf->pc = (u_int) res;
			}

			if (dbg_find_brkpt(exf->pc) != -1) {
				dbg_step_over = 1;
				exf->sr |= PSL_T;
			} else
				dbg_replace_brkpts();
			dbg_indebug = 0;
			return;

		    case 'S':	
			exf->sr |= PSL_T;	
			dbg_indebug = 0;
			return;
		}
	}
}

dbg_set_format() 
{
	switch (dbg_buf[1]) {
            case 'B':	dbg_format = BYTE;	printf("BYTE\n");	break;
	    case 'W':	dbg_format = WORD;	printf("WORD\n");	break;
	    case 'L':	dbg_format = LONG;	printf("LONG\n");	break;
	}
}

dbg_dump() 
{
	static u_char *dump_sav = 0;
	register int i, j;
	int val;

	if (dbg_buf[1] != NULL) {
		if (dbg_get_adr(&dbg_buf[1], &dump_sav))
			return 1;
	}
	    
	/* if format word | long keep it even */
	if (!(dbg_format & 0x1) && (((u_char) dump_sav) & 0x1))
		dump_sav -= 1;
	
	for (i = 0; i < NUM_LINES; i++) {
	    printf("%a: ", dump_sav);
	    for (j = 0; j < NUM_BYTES; j += dbg_format) {
		if (dbg_readmem(dump_sav, dbg_format, &val))
			return 1;
		switch (dbg_format) {
		    case BYTE:
			printf("%2x", (u_char)val);
			break;
		    case WORD:
			printf("%4x", (u_short)val);
			break;
		    case LONG:
			printf("%8x", (u_long)val);
			break;
		}
		dump_sav += dbg_format;
		printf(" ");
	    }
	    dump_sav -= NUM_BYTES;
	    printf("|");
	    for (j = 0; j < NUM_BYTES; j++) {
		    if (dbg_readmem(dump_sav++, BYTE, &val))
			return 1;
		    if ((val > 0x1f) && (val < 0x7f))
			    printf("%c",val);
		    else
			    printf(" ");
	    }
	    printf("|\n");
	}
	return 0;
}

dbg_examine(write_only) 
{
	static u_int	exam_sav = 0;
	register u_int	c;
	u_int 		val;

	if (dbg_buf[1] == NULL)
		exam_sav += dbg_format;
	else {
		if (dbg_get_adr(&dbg_buf[1], &exam_sav))
			return 1;
	}
	while (1) {
		printf("%a = ", exam_sav);
		if (!write_only) {
			if (dbg_readmem(exam_sav, dbg_format, &val))
				return 1;
			switch (dbg_format) {
			    case BYTE:
				printf("%2x", (u_char)val);
				break;
			    case WORD:
				printf("%4x", (u_short)val);
				break;
			    case LONG:
				printf("%a", (u_int)val);
				break;
			}
			printf(": ");
		}
		while (cngetc() != -1)		/* flush */
			;
		while ((c = cngetc()) == -1)
			;
	    	dbg_buf[0] =  c & 0x7f;
		switch (dbg_buf[0]) {
		    case NULL:
		    case CR:		
			printf("\n");
	        	return 0;

		    case '+':
		    case LF:		
			printf("+");
			exam_sav += dbg_format;
			printf("\n");
			break;

		    case '-':
			printf("-");
	        	exam_sav -= dbg_format;
			printf("\n");
			break;

		    case '@':
			printf("@");
			if (dbg_readmem(exam_sav, LONG, &exam_sav))
				return 1;
			printf("\n");
			break;

		    default:
			printf("%c",dbg_buf[0]);
			dbg_get_line(&dbg_buf[1]);
			if (dbg_get_adr(dbg_buf, &val))
				return 1;
			if (dbg_writemem(exam_sav, val, dbg_format))
				return 1;
			break;
		}
	}
}

dbg_dsp_regs(exf) 
	register struct ex_frame	*exf;
{
	register int i;
	u_int value;

	if (dbg_buf[1] != NULL) {
	    if (dbg_get_adr(&dbg_buf[3], &value))
		return 1;
	    switch (dbg_buf[1]) {
		case 'P':	exf->pc = value;			break;
		case 'S':	exf->sr = (u_short) value;		break;
		case 'D':	dbg_reg_sav[dbg_buf[2] - '0'] = value;	break;
		case 'A':	dbg_reg_sav[dbg_buf[2] - '0'+8] = value;break;
		default:	printf("%s", dbg_huh_msg);		return 0;
	    }
	}
	
	printf("    ");					/* header */
	for (i = 0; i < 8; i++)
		printf("       %d ",i);
	printf("\nD   ");				/* data regs */
	for (i = 0; i < 8; i++) 
		printf("%8x ", dbg_reg_sav[i]);
	printf("\n");
	for (i = 0; i < 8; i++) 
		printf("A%d: 	%a\n", i, dbg_reg_sav[8+i]);
	printf("PC:	%a 	SR:     %4x ", exf->pc, exf->sr);
	if (exf->sr & PSL_S)
		printf("	USP: %8x \n", dbg_usp_sav);
	else
		printf("	SSP: %a \n", dbg_ssp_sav);
	return 0;
}

dbg_dsp_brkpt() 
{
	register struct brk_tbl_struct	*tblptr = dbg_brk_tbl;
	register int i;

	for (i = 0; i < NUM_BRKPTS; i++, tblptr++)
		if (tblptr->adr != 0)
			printf("%d at %a\n", i, tblptr->adr);
	return 0;
}

dbg_insert_brkpt(exf)
	register struct ex_frame	*exf;
{
	register int i;
	register struct brk_tbl_struct * tblptr = dbg_brk_tbl;
	int val;

	for (i = 0; i < NUM_BRKPTS; i++, tblptr++)
		if (tblptr->adr == 0)
			break;
	if (i >= NUM_BRKPTS) {
		printf("table is full\n");
		return 1;
	}
	if (dbg_buf[1] != NULL) {
		if (dbg_get_expr(&dbg_buf[1], &val))
			return 1;
		if (dbg_find_brkpt(val) != -1)
			return 1;
		tblptr->adr = (u_short *)val;
	} else if (dbg_find_brkpt(exf->pc) != -1)
		return 1;
	else
		tblptr->adr = (u_short *) exf->pc;
	return 0;
}

dbg_rm_brkpt() 
{
	register int i;
	int val;

	if (dbg_buf[1] == '*')
		for (i = 0; i < NUM_BRKPTS; i++)
			dbg_brk_tbl[i].adr = 0;
	else if (gethex(&dbg_buf[1], &val) == 0 && val < NUM_BRKPTS)
		dbg_brk_tbl[val].adr = 0;
	else
		return 1;
	return 0;
}

dbg_replace_brkpts() 
{
	register int i;
	register struct brk_tbl_struct * tblptr = dbg_brk_tbl;

	for (i = 0; i < NUM_BRKPTS; i++, tblptr++)
		if (tblptr->adr != 0) {
			tblptr->instr = *tblptr->adr;
			dbg_insert_word(tblptr->adr, BRKPT);
		}
	return 0;
}

dbg_find_brkpt(address)
	register u_short * address;
{
	register struct brk_tbl_struct * tblptr = dbg_brk_tbl;
	register int i;

	for (i = 0; i < NUM_BRKPTS; i++, tblptr++)
		if (tblptr->adr == address)
			return(i);
	return(-1);
}

dbg_remove_brkpts() 
{
	register struct brk_tbl_struct	*tblptr = dbg_brk_tbl;
	register int	i;

	for (i = 0; i < NUM_BRKPTS; i++, tblptr++)
	        if (tblptr->adr != 0)
			dbg_insert_word(tblptr->adr, tblptr->instr);
	return 0;
}

dbg_get_adr(str, vaddr)
	register char	*str;
	register int	*vaddr;
{
	switch (*str) {
	    case NULL:
		return 1;
		break;

	    case 'R':
		if (*(str + 1) == 'D')
			*vaddr = dbg_reg_sav[(*(str + 2)) - '0'];
		else if (*(str + 1) == 'A')
			*vaddr = dbg_reg_sav[(*(str + 2)) - '0' + 8];
		else
			return 1;
		break;

	    case '@':
		if (dbg_get_expr(str+1, vaddr))
			return 1;
		if (dbg_readmem(*vaddr, LONG, vaddr))
			return 1;
		break;

	    default:
		if (dbg_get_expr(str, vaddr))
			return 1;
		break;
	}
	return 0;
}

dbg_get_line(dbg_bufptr)
	register u_char *dbg_bufptr;
{   
	static u_char temp[80];
	register u_char *str =temp;
	register int stop = 0;
	register int c;
	
	for (;;) {
	    while ((c = cngetc()) == -1)
		;
	    c &= 0x7f;
	    switch (c) {
	      case MONITOR:
		*str++ = c;

	      case CR:
	      case LF:	    
		printf("\n");
		*str = NULL;
		str = temp;
		for (; *str != NULL; str++)
			 if (!iswhitespace(*str))
				 *dbg_bufptr++ = *str;
		*dbg_bufptr = NULL;
		return;

	      case '!':
		printf("!\n");
		*dbg_bufptr++ = c;
		*dbg_bufptr = NULL;
		return;

	      case ESC:
		*dbg_bufptr++ = 'S';
		*dbg_bufptr = NULL;
		return;

	      case DEL:
	      case BS:
		if (stop > 0) {
			str--;
			stop--;
			printf("\b \b");
		}
		break;

	      case NAK:
		while (stop > 0) {
			str--;
			stop--;
			printf("\b \b");
		}
		break;

	      default :
		if (c > 0x19)
			printf("%c",c);
		*str++ = c;
		stop++;
		break;
	    }
	}
}

/*
 *	<expr> ::= <physaddr> | <physaddr> <binop> <expr>
 *	<physaddr> ::= <symbol> | <hex_const>
 *	<symbol> ::= <string> | '_'<string>
 *	<string> ::= <char> | <char> <string> | NULL
 *	<binop> ::= <'+' | '-' | ... >
 *	<hex_const> ::= <digit> | <digit> <hex_const> | NULL
 *	<char> ::= <alpha_char> | <digit>
 *	<alpha_char> ::= <a..z> | <A..Z> |
 *	<digit> ::= <0..9> | <a..f> | <A..F>
 */

dbg_isdelim(c)
	char c;
{
	return (isnull(c) | iswhitespace(c) | isbinop(c));
}

dbg_gettok(str,tokp)
	char **str;
	register char *tokp;
{
	while (iswhitespace(**str))
		(*str)++;
	if (dbg_isdelim(**str)) {
		*tokp++ = **str;
		(*str)++;
	} else {
		do {
			*tokp++ = **str;
			(*str)++;
		} while (!(dbg_isdelim(**str)));
	}
	*tokp = 0;
}

dbg_binop(val1, op, val2, vaddr)
	int val1;
	char op;
	int val2;
	int *vaddr;
{
	switch (op) {
	   case '+':	*vaddr = (val1 + val2);	break;
	   case '-':	*vaddr = (val1 - val2);	break;
	   case '*':	*vaddr = (val1 * val2);	break;
	   case '/':	*vaddr = (val1 / val2);	break;
	   case '%':	*vaddr = (val1 % val2);	break;
	   case '&':	*vaddr = (val1 & val2);	break;
	   case '|':	*vaddr = (val1 | val2);	break;
	   default:	return 1;
	}
	return 0;
}


dbg_get_physaddr(str, vaddr)
	char *str;
	int *vaddr;
{
	if (issymbolic(*str))
		return dbg_map_sym(str, vaddr);
	else
		return gethex(str, vaddr);
}

dbg_get_expr(str, vaddr)
	char *str;
	int *vaddr;
{
	char *bufptr = str;
	char token[MAXTOK];
	u_int operand;

	dbg_gettok(&bufptr, token);
	if (dbg_get_physaddr(token, vaddr))
		return 1;
	dbg_gettok(&bufptr, token);
	if (isnull(token[0]))
		return 0;
	if (!isbinop(token[0]))
		return 1;
	if (dbg_get_expr(bufptr, &operand))
		return 1;
	if (dbg_binop(*vaddr, token[0], operand, vaddr))
		return 1;
	return 0;
}

dbg_map_sym(buf, vaddr)
	char *buf;
	int *vaddr;
{
	register int i;
	char mybuf[MAXTOK+1];

	if (*buf == '_')
		strcpy(mybuf,buf);
	else {
		*mybuf = '_';
		strcpy(mybuf+1,buf);
	}
	for (i = 0, ssymp = symtab; i < sym_size; i++, ssymp++)
		if (strcmp(mybuf,ssymp->name) == 0) {
			*vaddr = ssymp->value;
			return 0;
		}
	for (i = 0, ssymp = kvsymtab; i < kvsym_size; i++, ssymp++)
		if (strcmp(mybuf,ssymp->name) == 0) {
			*vaddr = ssymp->value;
			return 0;
		}
	printf(" invalid symbol\n");
	return 1;
}

dbg_addkvsymbol(name, value)
	char *name;
{
	kvsymtab[kvsym_size].name = name;
	kvsymtab[kvsym_size].value = value;
	if (kvsym_size++ > KVSYM_SIZE)
		panic("kvsymsize");
}

unsigned long
dbg_get_kvnearest(value, nearest, lo)
	unsigned long value;
	char *nearest;
{
	register int klo, khi, i;

	klo = 0;
	khi = kvsym_size - 1;
	while ((khi - klo) > 1) {
		i = klo + (khi - klo)/2;
		if (kvsymtab[i].value == value)
			khi = klo = i;
		else if (kvsymtab[i].value < value)
			klo = i;
		else
			khi = i;
	}
	if (kvsymtab[khi].value <= value)
		klo = khi;
	if ((value - kvsymtab[klo].value) < (value - symtab[lo].value)) {
		strcpy(nearest,kvsymtab[klo].name);
		return (value - kvsymtab[klo].value);
	} else {
		strcpy(nearest,symtab[lo].name);
		return (value - symtab[lo].value);
	}
}


u_int
dbg_get_nearest(value,nearest)
	u_int value;
	char *nearest;
{
	register int lo, hi, i;

	lo = 0;
	hi = sym_size - 1;
	while ((hi - lo) > 1) {
		i = lo + (hi - lo)/2;
		if (symtab[i].value == value)
			lo = hi = i;
		else if (symtab[i].value < value)
			lo = i;
		else
			hi = i;
	}
	if (symtab[hi].value <= value)	/* boundary: value > initial high */
		lo = hi;
	return dbg_get_kvnearest(value, nearest, lo);
}

dbg_disasm(buf)
	char *buf;
{
	register i = NUM_LINES;
	int j;

	codeonly++;

	if (buf) {
		if (dbg_get_expr(buf, &j))
			return 1;
		dot = j;
	}
	while (i--)
		dbg_dis_print();
	return 0;
}

static int count;

static union {
	u_int val;
	u_char chars[4];
} dis_buf;

int
getnextbyte()
{
	if ((count % sizeof(u_int)) == 0)
		(void) dbg_readmem(dot, LONG, &dis_buf.val);
	return dis_buf.chars[count++ % sizeof(u_int)];
}

dbg_dis_print()
{
	if (dbg_readmem(dot, LONG, &dis_buf.val))
		return 1;
	count = 2;
	printf("%a:	",dot);
	dot += 2;
	das_print_ins(*(u_short *)&dis_buf.val, 1);
	printf("\n");
	return 0;
}

dbg_readmem(kaddr, size, vaddr)
	caddr_t kaddr;
	int size;
	int *vaddr;
{
	DELAY(8000);
	if (
#ifdef	M68020
	    ((((int)kaddr) & MEMR_MASK) <= USRV_BASE) &&
#endif	M68020
	    (kernacc(kaddr, size, B_READ) == 0)) {
		printf("invalid kernel access: %x\n", kaddr);
		return 1;
	}
	*vaddr = 0;
	switch (size) {
	    case BYTE:
		*vaddr = *((u_char *)kaddr);
		break;
	    case WORD:
		*vaddr = *((u_short *)kaddr);
		break;
	    case LONG:
		*vaddr = *((u_int *)kaddr);
		break;
	}
	return 0;
}

dbg_writemem(kaddr, value, size)
	caddr_t kaddr;
	int value;
	int size;
{
	if (
#ifdef	M68020
	    ((((int)kaddr) & MEMR_MASK) <= USRV_BASE) &&
#endif	M68020
	    (kernacc(kaddr, size, B_WRITE) == 0)) {
		printf("invalid kernel access: %x\n", kaddr);
		return 1;
	}
	switch (size) {
	    case BYTE:
		*((u_char *)kaddr) = value;
		break;
	    case WORD:
		*((u_short *)kaddr) = value;
		break;
	    case LONG:
		*((u_int *)kaddr) = value;
		break;
	}
	return 0;
}

char *
psymbol(val, non_das)
	unsigned int val;
	int non_das;
{
	register u_int offset;
	static char buf[MAXTOK];
	
	if ((sym_magic != NEWDBG_MAGIC) || 
	    ((offset = dbg_get_nearest(val,buf)) < 0) ||
	    (offset > 0x100000)) {
		if (non_das)
			sprintf(buf, "%8x", val);
		else
			LPrintf("%8x",val);
	} else {
		if (offset)
			sprintf(buf+strlen(buf),"+%x", offset);
		if (non_das == 0)
			LPrintf("%s",buf);
	}
	return(buf);
}

preloc(loc, val)
	register long loc;
	int val;
{
	psymbol(val,0);
}

flushbuf() {}

gethex(cp, vaddr)
	register char *cp;
	register int *vaddr;
{
	register int c, n;

	n = 0;
	while (c = *cp++) {
		if (c >= '0' && c <= '9')
			c = (c - '0');
		else if (c >= 'a' && c <= 'f')
			c = (c - 'a') + 10;
		else if (c >= 'A' && c <= 'F')
			c = (c - 'A') + 10;
		else if (c == ' ' || c == '\r' || c == '\n')
			break;
		else {
			printf(" invalid hex value\n");
			return 1;
		}
		n = (n<<4) + c;
	}
	*vaddr = n;
	return 0;
}

extern char	*prf_bufp;		/* see subr_prf.c!! */
extern char	*linep;			/* see das.c !! */
#define	TOBUF	0x10			/* KLUDGE */

LPrintf (fmt, args)
	char *fmt;
{
	prf_bufp = linep;
	prf(fmt, &args, TOBUF, (struct tty *)0);
	*prf_bufp++ = '\0';
	linep = prf_bufp - 1;
}

/*
 *	given kernel virtual addr and value, change prot
 *	bits on page in which kaddr resides, stuff the value in kaddr
 *	and restore prot bits
 */
dbg_insert_word(kaddr, value)
	caddr_t kaddr;
	u_short value;
{ 
#ifdef	M68020
	struct pte oldpte;
	register struct pte *pte;
	register int page;
	register int s;
	
	page = btop(svtop(kaddr));
	if (page & ~PG_PFNUM)		/* offset is too big (or negative) */
		return -1;
	pte = &Sysmap[btop(kaddr - SYSV_BASE)];
	oldpte = *pte;
	s = spl7();
	*(int *)pte &= ~PG_PROT;		/* change bits */
	*(int *)pte |= PG_KW;
	distcl(pte);
	refbits();
	newptes(pte, clbase(btop(kaddr - SYSV_BASE)), CLSIZE); 
	*(u_short *)kaddr = value;		/* insert new value */
	*pte = oldpte;				/* restore pte */
	distcl(pte);
	refbits();
	newptes(pte, clbase(btop(kaddr - SYSV_BASE)), CLSIZE);
	splx(s);
#else	M68020
	*(u_short *)kaddr = value;		/* insert new value */
#endif	M68020
	return 0;
}
#endif	DEBUGGER
