/*
 * Copyright (c) 1982,1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)autoconf.c	6.22 (Berkeley) 3/20/86
 */

/*
 * Setup the system to run on the current machine: configure() is called at boot
 * time and initializes the bus devices. Available devices are determined (from
 * possibilities mentioned in ioconf.c), and the drivers are initialized.
 */

#include "pte.h"
#include "psl.h"

#include "param.h"
#include "user.h"
#include "systm.h"
#include "kernel.h"
#include "map.h"
#include "buf.h"
#include "dk.h"
#include "vm.h"
#include "conf.h"
#include "dkbad.h"

#include "scb.h"
#include "../is68kdev/qbvar.h"

/*
 * The following several variables are related to the configuration process, and
 * are used in initializing the machine.
 */
u_short	*caddr;		/* address being probed */
int	dkn;		/* number of iostat dk numbers assigned so far */
int	howto;
int	devtype;

extern char *bdevname[];	/* see ../conf/param.c */
extern int nbdevnames;		/* see ../conf/param.c */
extern char *clu_name_list[];	/* see ../conf/param.c */

/*
 * Determine controller, device, and memory configuration for a machine
 */
configure()
{
	printf("\n");
	sioinit();
	qbconfigure();
	setroot();

	/*
	 * Configure swap area and related system parameter based on device(s) 
	 * used.
	 */
	swapconf();
	printf("\n\n");
#ifndef	M68020
	prcpuspeed();
	memenable();
#endif	M68020
}

/*
 * Configure controllers and devices on the bus.
 */
qbconfigure()
{
	register struct qb_device 	*qi;
	register struct qb_ctlr 	*qm;
	register struct qb_driver 	*udp;
	register struct ivecvec 	*iv;
	u_short 			**ap;
	u_short				*reg,
					*addr;
	int 				i;
	int 				s;
	int				*ip;

	/*
	 * Allow any pending interrupts to be serviced as STRAY
	 */
	s = spl0();
	DELAY(4000);
	splx(s);

	/*
	 * Check each bus controller. For each one which is potentially there,
	 * see if it is really there, and if it is record it and then go looking
	 * for slaves.
	 */
	for (qm = qbcinit; udp = qm->qm_driver; qm++) {
	    if ((int)udp == -1)
		continue;
	    if (vbnum > 0  &&  !name_in_list(udp->ud_dname, clu_name_list))
		continue;
	    addr = qm->qm_addr;
	    /*
	     * Use the particular address specified first, or if it is given as
	     * "0", or there is no device at that address, try all the standard
	     * addresses in the driver until we find it.
	     */
	    for (ap = udp->ud_addr; addr || (addr = *ap++); addr = NULL) {
		reg = addr;
		/*
		 * if the address is already in use, or we get a violation 
		 * trying to touch registers then continue.
		 */
		if (((int)reg != -1) && (!iosavail((caddr_t)reg, 1) || 
		    (badaddr((caddr_t)reg, 2) && badaddr((caddr_t)reg, 1) &&
		    badaddr((caddr_t)reg, -2) && badaddr((caddr_t)reg, -1))))
			continue;
		cvec = NSCBVEC;
		clevmax = clev = -1;
		caddr = reg;
		/*
		 * Cause an interrupt on the device, asynchronously redefining 
		 * 'cvec' to be the controllers interrupt vector. Probe returns 
		 * the size of the controllers registers. Interupts must be 
		 * enabled during the probe.
		 */
		s = spl0();
		i = (*udp->ud_probe)(reg, qm->qm_ctlr, qm);
		splx(s);
		caddr = 0;
		if (i == 0  || (((int)reg != -1) && !iosavail((caddr_t)reg, i)))
			continue;
		printf("   %s%d	at ",udp->ud_mname,qm->qm_ctlr);
		if ((int)addr == -1)
		    printf("*** no address 		");
#ifdef	VBUS
		else if ((addr >= (u_short *)vme_short) && 
		    (addr <= (u_short *)&vme_short[0xFFFE]) )
			printf("shortio 0x%4x	 	",
				(char *)addr - vme_short);
#endif	VBUS
		else
#ifdef	QBUS
		    printf("address 0x%x/0%o 	", svtop(addr), svtop(addr));
#else	QBUS
		    printf("address 0x%8x	 	", svtop(addr));
#endif	QBUS
		if ((int)addr != -1)
		    iosused((caddr_t)reg, i);

		switch (cvec) {
		    case 0:
			printf("*** no vector\n");
			qm->qm_psl = PSL_IPL6;
			goto novec;

		    case NSCBVEC:
			printf("*** didn't interrupt\n");
			continue;
		}
#ifdef	QBUS
		printf("vector 0x%x/0%o  ", cvec, cvec);
#else	QBUS
		printf("vector 0x%x  ", cvec);
#endif	QBUS

#ifndef	M68020
		if (cvec < SCBDEV) {
			printf("*** interrupt vector too low\n");
			continue;
		}
#endif	M68020
		/* fill in the interrupt mask psl for the controller */
		if (clev != -1) {
			printf("level %d  ", clev 
#ifdef	QBUS
							+ 3
#endif	QBUS
								);
			if ((clevmax != -1) && (clev > clevmax)) {
				printf("*** max is %d!\n", clevmax
#ifdef	QBUS
								    + 3
#endif	QBUS
									);
				continue;
			}
			qm->qm_psl = PSL_CURMOD | (clev << PSL_IPL_SHIFT);
		} else
			qm->qm_psl = PSL_IPL6;
		/*
		 * fill in the interrupt vector slots for the controller,
		 * checking for devices interrupting at slots already used.
		 */
		for (iv = qm->qm_intr; iv->iv_intr; iv++) {
			if (scb.scb_vec.scb_blk[cvec + iv->iv_offs] != Xstray)
				break;
			scb.scb_vec.scb_blk[cvec + iv->iv_offs] = iv->iv_intr;
		}
		if (iv->iv_intr) {
			printf("*** vector in use\n");
			continue;
		}
		printf("\n");
		/*
		 * the controller is there at addr 'reg'. If indicated, fill in 
		 * the drivers back pointer to the qb_ctrl structure.
		 */
    novec:	qm->qm_alive = 1;
		qm->qm_addr = reg;
		if (udp->ud_minfo)
			udp->ud_minfo[qm->qm_ctlr] = qm;

		/*
		 * look through all the devices on the machine for ones that are
		 * controlled by the current controller, and are not yet alive.
		 */
		for (qi = qbdinit; qi->qi_driver; qi++) {
		    if (qi->qi_driver != udp || qi->qi_alive ||
		    	qi->qi_ctlr != qm->qm_ctlr && qi->qi_ctlr != '?' )
			    continue;
		    /*
		     * If the slave device is present then logically attach the 
		     * controller to the device.
		     */
		    if ((*udp->ud_slave)(qi, reg)) {
			qi->qi_alive = 1;
			qi->qi_ctlr = qm->qm_ctlr;
			if (qi->qi_dk && dkn < DK_NDRIVE)
			    qi->qi_dk = dkn++;
			else
			    qi->qi_dk = -1;
			qi->qi_mi = qm;
			udp->ud_dinfo[qi->qi_unit] = qi;
			printf("	%s%d	at %s%d	slave %d	",
				udp->ud_dname,qi->qi_unit,udp->ud_mname,
				qm->qm_ctlr,qi->qi_slave);
			(*udp->ud_attach)(qi);
			printf("\n");
		    }
		}
		break;
	    }
	}

	/*
	 * From now on all stray vectors will produce 'stray' messages in trap.
	 * Move the configured scb into the vector table area.
	 */
	cvec = 0;
	for (i = 2 ; i < NSCBVEC ; i++)
		((int *)0)[i] = (int)scb.scb_vec.scb_blk[i];
}

freevec()
{
	int i;

	for (i = SCBDEV ; i < NSCBVEC ; i++)
		if ((i%4) == 0 && scb.scb_vec.scb_blk[i] == Xstray)
			return (i);
}

/*
 * The following structure is to detect overlapping io address space registers
 */
#define	NCTRLRS	40
struct iospace {
	char	*start;		/* address of first register of device */
	char	*end;		/* address just past last register of device */
} iosp[NCTRLRS];

iosused(p, n)
	char *p;
	int n;
{
	register struct iospace *sp;

	for (sp = iosp;  sp < &iosp[NCTRLRS];  sp += 1)
		if (sp->start == 0) {
			sp->start = p;
			sp->end = p + n;
			return;
		}
	panic("autoconf: too many controllers");	
}

iosavail(p, n)
	register char *p;
	int n;
{
	register struct iospace *sp;
	register char *q = p + n;

	for (sp = iosp;  sp < &iosp[NCTRLRS];  sp += 1)
		if (sp->start  &&  p < sp->end  &&  q > sp->start)
			return 0;
	return 1;
}

int
name_in_list(name, namelist)
	register char *name;
	register char *namelist[];
{
	if (namelist)
		while (*namelist)
			if (strcmp(name, *namelist++) == 0)
				return 1;
	return 0;
}

/*
 * If the root device was not specified in the configuration process,
 * use the "devtype" paramater from boot to determine where the root and
 * swap are located.
 */
setroot()
{
	if (rootdev == 0)
		rootdev = devtype;
	if (dumpdev == 0)
		dumpdev = makedev(major(rootdev), (minor(rootdev)&~7) | 1);
	if (argdev == 0)
		argdev = dumpdev;
	if (swdevt[0].sw_dev == 0)
		swdevt[0].sw_dev = dumpdev;
	printf("\n   ");
	if (diskless) {
		printf("Host: %s	Server: %s", hostname, servername);
		if (gatewaynamelen)
			printf("	Gateway: %s", gatewayname);
	} else {
	    if (major(rootdev) == 0)
		panic("no root device specified");
	    printf("root on %s%d%c", bdevname[major(rootdev)], 
		minor(rootdev)>>3, 'a'+(minor(rootdev)&7));
	    printf(", dump on %s%d%c", bdevname[major(dumpdev)], 
		minor(dumpdev)>>3, 'a'+(minor(dumpdev)&7));
	    printf(", args on %s%d%c", bdevname[major(argdev)], 
		minor(argdev)>>3, 'a'+(minor(argdev)&7));
	}
}

/*
 * Configure swap space and related parameters.
 */
swapconf()
{
	register struct swdevt *swp;
	register int nblks;

	if (!diskless)
		printf(", swap on");
	for (swp = swdevt; swp->sw_dev; swp++) {
		if (bdevsw[major(swp->sw_dev)].d_psize) {
			nblks =
			  (*bdevsw[major(swp->sw_dev)].d_psize)(swp->sw_dev);
			if (swp->sw_nblks == 0 || swp->sw_nblks > nblks)
			    swp->sw_nblks = nblks;
			swp->sw_vp = bdevvp(swp->sw_dev);
		}
		if (!diskless)
		    printf(" %s%d%c", bdevname[major(swp->sw_dev)], 
			minor(swp->sw_dev)>>3, 'a'+(minor(swp->sw_dev)&7));
	}
	if (dumplo == 0 && bdevsw[major(dumpdev)].d_psize)
		dumplo = (*bdevsw[major(dumpdev)].d_psize)(dumpdev) - 
				ctod(physmem);
	if (dumplo < 0)
		dumplo = 0;
}

/*
 * dump the system
 */
struct buf dumpbuf;
int	dumpmag = 0x8fca0101;	/* magic number for savecore */
int	dumpsize = 0;		/* also for savecore */
int	dumpok = 1;		/* KLUDGE: move to param.c */

/*
 * Doadump comes here after turning off memory management and
 * getting on the dump stack, either when called above, or by
 * the auto-restart code.
 */
dumpsys()
{
	if (dumpok == 0)
		return;
#ifdef notdef
	if ((minor(dumpdev)&07) != 1)
		return;
#endif
	dumpsize = physmem;
	printf("dumping to %s%d%c, offset %d\n", bdevname[major(dumpdev)], 
		minor(dumpdev)>>3, 'a'+(minor(dumpdev)&7), dumplo);
	DELAY(0x100000);
	printf("dump: ");
	if (cngetc() != -1) {
		printf("aborted\n");
		return;
	}
	switch ((*bdevsw[major(dumpdev)].d_dump)(dumpdev)) {
	    case ENODEV:
		printf("device does not support dump\n");
		break;
	    case ENXIO:
		printf("device bad\n");
		break;
	    case EFAULT:
		printf("device not ready\n");
		break;
	    case EINVAL:
		printf("area too small\n");
		break;
	    case EIO:
		printf("i/o error\n");
		break;
	    default:
		printf("succeeded\n");
		break;
	}
}

/*
 * Calculate standard disk partition sizes.
 */
#define	NPARTITIONS	8
#define	PART(x)		(x - 'a')

/* Default partition sizes, where they exist.  */
#define	NDEFAULTS	5
#define	SMALLDISK	3
static int	defpart[NDEFAULTS][NPARTITIONS] = {
   { 15884, 66880, 0, 15884, 307200, 0, 0, 291346 },	/* ~ 356+ Mbytes */
   { 15884, 33440, 0, 15884, 55936, 0, 0, 291346 },	/* ~ 206-355 Mbytes */
   { 15884, 33440, 0, 15884, 55936, 0, 0, 0 },		/* ~ 61-205 Mbytes */
   { 15884, 10032, 0, 15884, 0, 0, 0, 0 },		/* ~ 20-60 Mbytes */
   { 0, 0, 0, 0, 0, 0, 0, 0 },				/* < 20 Mbytes */
};

/*
 * Each array defines a layout for a disk; that is, the collection of partitions
 * totally covers the physical space on a disk.
 */
#define	NLAYOUTS	3
static char	layouts[NLAYOUTS][NPARTITIONS] = {
   { 'a', 'b', 'h', 'g' },
   { 'a', 'b', 'h', 'd', 'e', 'f' },
   { 'c' },
};

/*
 * Each disk has some space reserved for a bad sector forwarding table.  DEC 
 * standard 144 uses the first 5 even numbered sectors in the last track of the
 * last cylinder for replicated storage of the bad sector table; another 
 * MAXBADBLK sectors past this is needed as a pool of replacement sectors.
 */
struct size {
	daddr_t	nblocks;
	int	cyloff;
};

diskpart(size, nspt, ntpc, ncpd)
	register struct size	*size;
{
	register int curcyl, def, part, layout;
	int 	threshhold, 
		numcyls[NPARTITIONS], 
		numblks[NPARTITIONS], 
		startcyl[NPARTITIONS];
	char 	*lp;
	int	badsecttable = MAXBADBLK;	/* # sectors */
	int	xxx;
	register struct size	*ssize = size;
	int	nspc = nspt * ntpc;

	/*
	 * Bad sector table contains one track for the replicated copies of the
	 * table and enough full tracks preceding the last track to hold the 
	 * pool of free blocks to which bad sectors are mapped.
	 */
	badsecttable = nspt + roundup(badsecttable, nspt);
	threshhold = howmany(nspc, badsecttable);

	/* 
	 * Figure out if disk is large enough for expanded swap area and 'd', 
	 * 'e', and 'f' partitions.
	 */
	for (def = 0; def < NDEFAULTS; def++) {
		curcyl = 0;
		for (part = PART('a'); part < NPARTITIONS; part++)
			curcyl += howmany(defpart[def][part], nspc);
		if (curcyl < ncpd - threshhold)
			break;
	}

	if (def >= NDEFAULTS)
		return (-1);

	/*
	 * Calculate number of cylinders allocated to each disk partition. We 
	 * may waste a bit of space here, but it's in the interest of 
	 * compatibility (for mixed disk systems).
	 */
	for (curcyl = 0, part = PART('a'); part < NPARTITIONS; part++) {
		numcyls[part] = 0;
		numblks[part] = 0;
		if (defpart[def][part] != 0) {
			numcyls[part] = howmany(defpart[def][part], nspc);
			numblks[part] = defpart[def][part];
			curcyl += numcyls[part];
		}
	}
	numcyls[PART('f')] = ncpd - curcyl;
	numblks[PART('f')] = numcyls[PART('f')] * nspc - badsecttable;
	numcyls[PART('g')] =
		numcyls[PART('d')] + numcyls[PART('e')] + numcyls[PART('f')];
	numblks[PART('g')] = numcyls[PART('g')] * nspc - badsecttable;
	numcyls[PART('c')] = ncpd;
	numblks[PART('c')] = numcyls[PART('c')] * nspc;

	/*
	 * Calculate starting cylinder number for each partition. Note the 'h' 
	 * partition is physically located before the 'g' or 'd' partition.  
	 * This is reflected in the layout arrays defined above.
	 */
	for (layout = 0; layout < NLAYOUTS; layout++) {
		curcyl = 0;
		for (lp = layouts[layout]; *lp != 0; lp++) {
			if (numcyls[PART(*lp)])
				startcyl[PART(*lp)] = curcyl;
			else
				startcyl[PART(*lp)] = 0;
			curcyl += numcyls[PART(*lp)];
		}
	}

	if (defpart[def][PART('a')] == 0) {
	    xxx = numblks[PART('f')];
	    part = PART('a');

	    if (xxx >= defpart[SMALLDISK][part]) {
		startcyl[part] = 0;
		numcyls[part]=howmany( defpart[SMALLDISK][part], nspc);
		numblks[PART('a')] = defpart[SMALLDISK][part];
		xxx -= numcyls[part] * nspc;
		part = PART('b');
		if (xxx >= defpart[SMALLDISK][part]) {
		    startcyl[part] = startcyl[PART('a')]+numcyls[PART('a')];
		    numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		    numblks[part] = defpart[SMALLDISK][part];
		    xxx -= numcyls[part] * nspc;
		    part = PART('g');
		    if (xxx >= 0) {
		    	startcyl[part] = startcyl[PART('b')]+numcyls[PART('b')];
		    	numcyls[part] = howmany( xxx, nspc);
		    	numblks[part] = xxx;
		    }
	        } else {
		    startcyl[part] = startcyl[PART('a')]+numcyls[PART('a')];
		    numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		    numblks[part] = defpart[SMALLDISK][part];
		}
	    } else {
		startcyl[part] = 0;
		numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		numblks[part] = defpart[SMALLDISK][part];
	    }
	}

	for (part = PART('a'); part < NPARTITIONS; part++, size++) {
		size->nblocks = numblks[part];
		size->cyloff = startcyl[part];
	}
	printpart(ssize, nspc);
}

printpart(isize, nspc)
	register struct size	*isize;
{
	int part, ppart;
	int ip, overlap = 0;
	register struct size	*size = isize;
	register struct size	*tsize;

	ppart = 0;
	for (part = PART('a'); part < NPARTITIONS; part++, size++) {
		if (size->nblocks == 0)
			continue;
		if ((ppart % 4) == 0)
			printf("\n		");
		printf("%c",'a'+part);
		ppart++;
		overlap = 0;

		for (tsize=isize, ip=PART('a'); ip < NPARTITIONS; ip++, tsize++)
			if (ip != part && tsize->nblocks != 0 &&
			    tsize->cyloff >= size->cyloff && 
			    tsize->cyloff*nspc+tsize->nblocks <=
			    size->cyloff*nspc+size->nblocks) {
				if (!overlap) printf(":");
				printf("%c",'a'+ip);
				overlap++;
			}
		printf(":%d", size->nblocks);
		if ((ppart)%4 != 0) {
			printf("	");
			if (size->nblocks < 100000)
				printf("	");
		}
	}
}
