/**************************************************************************
* laserv - VxWorks side of the laser light show
* 
* Ron Wilder -- Integrated Solutions
* 14 Aug 89 -- date of birth
*
* Current RCS information:
* $Header: laserv.c,v 1.4 89/09/07 11:08:56 laser Locked $
* 
* This module is the real-time half of the laser light show demo.
* After first establishing a socket to the XWindows (host) via ethernet,
* the program goes into a loop receiving commands and new images, translating
* the images into a format suitable for the D/A board, and then displaying the
* images on the wall or whatever the laser is pointed towards.
*
* In addition, the program provides additional "image processing" commands 
* such as invert X or Y, shrink/zoom X or Y, rotate about center, etc.
***************************************************************************/

#include "/vx/h/vxWorks.h"
#include "/vx/h/socket.h"
#include "/vx/h/in.h"
#include "/vx/h/memLib.h"
#include "/vx/h/semLib.h"
#include <math.h>
#include "laser.h"		/* laser light show definitions */
#include "dac.h" 		/* D to A Board structures and definitions */

/* define the factors used to translate the window image into DAC values */

/*translation scale factor X-axis*/
#define XLATE_FACTOR_X MAX_DAC/MAX_XWINDOW_X 

/* if you wish to retain same aspect ratio as in window, set the Y scale
 * factor to the same value as the X scale factor. */
#define XLATE_FACTOR_Y XLATE_FACTOR_X	/* same aspect ratio */

/* If you wish the image to be squared off, use the following #define.
/*#define XLATE_FACTOR_Y MAX_DAC/MAX_XWINDOW_Y */ 

/* compile defines */
#define ORIGINAL TRUE 	/* non semaphore'd code */
/* globals */

boolean connected_to_host;
int		clientSock;	/* socket */
ENET_CMD_PACKET		cmdbuf,*pcmdbuf;	/*command buffer size*/
ENET_IMAGE_PACKET	imagebuf,*pimagebuf; /*laser image packet size*/
LASER_IMAGE 		laserimage, *plaserimage;	/* laser image structure */

boolean 	laser_powered = FALSE;			/* beam on boolean */
boolean		dac_int_running = FALSE;		/* interrupts started boolean */
int	LED = 1;

/* semaphore to access laser_image display buffer */

SEM_ID	LASERSEM;

/* functions within this module for laser light demo program */
int laser();
int	execute_host_commands();
int xlate_to_dac_range();
int image_draw();	
int init_dac();
int	off();
int on();
int laser_off();
int laser_on();
int	dac_Update();
int	set_draw_rate();
int start_dac_int();
int stop_dac_int();
int conUpdate();
int disUpdate();
int spiralwave();
int onoff();
int connect_to_laser_host();

/* external functions for laser light demo program */
/*extern int swave(); */					/* file: dac.c */
/*extern int connect_to_laser_host();*/		/* file: connect.c */
/*extern int image_draw();*/					/* file: dac.c */

/*******************************************************************************
*
* laser - The real-time half of the laser light show demo
*
* This function will establish a connection to the laser host system via
* ethernet (or other internet interface), receive commands and images from
* the host, convert the images into a format that the laser controller can
* use, turn the laser beam on and off, and display the image. Additional 
* functions include image modification (such as inversion, rotation, shrink,
* zoom) as well as internal pattern generation (sine wave lissajous, spirals,
* and other canned patterns). -- note: rotation and canned patterns
* are not yet implemented
*
* general flow:
* - open socket to laser host 
* - start a loop which will:
* - wait for a command from the laser host
* - when a command comes in,
*   - ack the host with cmd_ack   ----- not yet implemented
*   - decode command and subcommand if necessary
*   - if command is an 'action' command then
*     - perform action on current image
*   - if command is a new image then
*     - receive second packet with image
*	  - ack the host with image_ack  ----- not yet implemented
*     - set default variables in laser image structure
*     - translate image into laser format if necessary (based on flag 
*       inside of image structure
*   - loop back to wait for a command 
*
*	 optional extra: if time permits, put in an "I'm alive" over e-net
*/

STATUS
laser()
{
	int n;
	static char	hostName[]=HOSTNAME;

	printf("starting laser program\n");

	/* open socket to laser host */
	if (connect_to_laser_host(hostName,&clientSock) == ERROR) {
		printf("Aborting laser light show demo program due to a connection\n");
		printf("problem with the laser host machine. The problem could be \n");
		printf("hardware (cabling, delni, nw board, host machine) or \n");
		printf("software (wrong host name, host machine not running laser\n");
		printf("program).  G O O D    L U C K ! ! !\n\n");
		laser_off();
		return(ERROR);
	}

	printf("We\'re connected to laser host \'%s\' on socket %d\n",
		hostName,clientSock);
	connected_to_host = TRUE;

	LASERSEM = semCreate();

	/* start receiving commands and images from host */
	/* then execute them */
	if (execute_host_commands() == ERROR) {
		printf("Ending laser light show demo program due to error\n");
		printf("receiving or processing a host command.\n");
	}	

	stop_dac_int();		/* disable dac interrupts */
	laser_off();
	printf("Bye...\n");
	close(clientSock); 
	return (OK); 
}

/************************************************************************
 * execute_host_commands() - receive and execute laser host commands
 * 
 * Ron Wilder - Integrated Solutions Inc
 * 21 Aug 89 - Date of Birth
 *
 */

STATUS
execute_host_commands() 

{
	register int    n, nleft;
	u_short theta;
	double xtheta,ytheta, radius;
#ifndef ORIGINAL
	int nbytes;
#endif

	/* initialize buffers to zero */
	bfill(&cmdbuf,ENET_CMD_PKT_SIZE,0);
	bfill(&imagebuf,ENET_IMAGE_PKT_SIZE,0);
	bfill(&laserimage,LASER_IMAGE_SIZE,0);

	while (connected_to_host) {
		/* receive a command from the host */
	
		/*printf("The command buffer size is: %d\n",sizeof(cmdbuf)); */
		pcmdbuf = &cmdbuf;

		nleft = sizeof (cmdbuf);
		while(nleft > 0) {
			if ((n = recv(clientSock, pcmdbuf, nleft, 0)) == 0) { 
				/* nack the laser host */
				/* ack_nack_host(CMD_NACK); */
	
				/* connection to server has been broken */
	
				/****** need better error recovery...
				 * perhaps just ignore command and continue */
				printf("Packet error while receiving host command!\n");
				printf("clientSock: %d; pcmdbuf:%d; nleft:%d\n",
					clientSock,pcmdbuf,nleft);
				close(clientSock);
				return(ERROR);
			}
			nleft -= n;
			pcmdbuf +=n;
			/*printf("# bytes received in this command packet: %d\n",n); */
		}
	
		/* print out structure */
		/*printf("After receipt of command...\n"); */
		printf("command:%d; subcmd1: %d; subcmd2: %d\n\n",
			cmdbuf.command, cmdbuf.subcmd1, cmdbuf.subcmd2);
			
		/* we have a complete command from the host */
		/* ack the host */
		/* ack_nack_host(CMD_ACK); */
	
		/* we have the command, now decode it */
		switch(cmdbuf.command){
		
			case LASER_OFF:{
				laser_off();
				break;
   			}
	
			case LASER_ON:{
				laser_on();
				break;
			}
	
			case INVERT_IMAGE_X:{
				printf("Inverting the X axis...\n");

				/* need to get actual output buffer pointer) */
				plaserimage = &laserimage;
				for (n = 0; n <plaserimage->image_length; n++){
					if(plaserimage->use_full_scale) {
						plaserimage->full_scale_image[n][XPOS] =
							MAX_DAC - plaserimage->full_scale_image[n][XPOS];
					}
					else {
						plaserimage->actual_image[n][XPOS] =
							MAX_DAC - plaserimage->actual_image[n][XPOS];
					}
				}

				/* trigger output routine to start drawing this pattern */
				/* for now, call a routine in dac.c to draw pattern */
				/*image_draw(); */
				break;
			}
	
			case INVERT_IMAGE_Y:{
				printf("Inverting the Y axis...\n");

				/* need to get actual output buffer pointer) */
				plaserimage = &laserimage;

				for (n = 0; n <plaserimage->image_length; n++){

					if(plaserimage->use_full_scale) {
						plaserimage->full_scale_image[n][YPOS] =
							MAX_DAC - plaserimage->full_scale_image[n][YPOS]; 
					}
					else {
						plaserimage->actual_image[n][YPOS] =
							MAX_DAC - plaserimage->actual_image[n][YPOS]; 
					}
				}

				/* trigger output routine to start drawing this pattern */
				/* for now, call a routine in dac.c to draw pattern */
				/*image_draw(); */
				break;
			}
	
			case INVERT_IMAGE_XY:{
				printf("Inverting the X & Y axes...\n");
				/* need to get actual output buffer pointer) */
				/* for now use global, later use local image and then
				 * update current image through semaphore */
				plaserimage = &laserimage;
				for (n = 0; n <plaserimage->image_length; n++){

					if(plaserimage->use_full_scale) {
						plaserimage->full_scale_image[n][XPOS] =
							MAX_DAC - plaserimage->full_scale_image[n][XPOS];
						plaserimage->full_scale_image[n][YPOS] =
							MAX_DAC - plaserimage->full_scale_image[n][YPOS]; 
					}
					else {
						plaserimage->actual_image[n][XPOS] =
							MAX_DAC - plaserimage->actual_image[n][XPOS];
						plaserimage->actual_image[n][YPOS] =
							MAX_DAC - plaserimage->actual_image[n][YPOS]; 
					}
				}
				/* trigger output routine to start drawing this pattern */
				/* for now, call a routine in dac.c to draw pattern */
				/*image_draw(); */
 				break;
			}

			/* note that this will not save any inversions */	
			case GO_FULL_SCALE: {
				printf("Going back to full scale...\n");
				
				/* set flag that says to display full scale image */
				/* for now use global, later use update semaphore to */
				/* change the current image */
				plaserimage = &laserimage;
				plaserimage->use_full_scale = TRUE;

				/* trigger output routine to start drawing this pattern */
				/* for now, call a routine in dac.c to draw pattern */
				/*image_draw(); */
				break;
			}

			case NEW_DRAW_RATE:{

				/* need to get actual output buffer pointer) */
				/* use semaphore to update image */
				plaserimage = &laserimage;
				plaserimage->draw_rate = cmdbuf.subcmd1;
				set_draw_rate();
				break;
			}

			case SCALE_IMAGE:{
				printf("New X scale: %d%%;  Y scale: %d%%\n",
					cmdbuf.subcmd1, cmdbuf.subcmd2);
				/* note: parameters from host are short integers and
					therefore logic is reversed: ie. divide full scale
					by the value received */
				/* NEW! above comment is the old way... the new way
				   is to divide the value received from the host by
				   MAX gain to get the actual gain factor (zero to one).

				/* need to get actual output buffer pointer) */
				/* and copy into local image buffer */
				/* change all refs from plaserimage to plocal laser image */
				plaserimage = &laserimage;
				for (n = 0; n <plaserimage->image_length; n++){
					
					plaserimage->actual_image[n][XPOS] = (short)
					(((plaserimage->full_scale_image[n][XPOS] - DAC_ZERO) *
						((float)cmdbuf.subcmd1 /MAX_GAIN )) + DAC_ZERO);

					plaserimage->actual_image[n][YPOS] = (short)
					(((plaserimage->full_scale_image[n][YPOS] - DAC_ZERO) *
						((float)cmdbuf.subcmd2 / MAX_GAIN)) + DAC_ZERO);

					/* old way
					plaserimage->actual_image[n][XPOS] = (short)
					(((plaserimage->full_scale_image[n][XPOS] - DAC_ZERO)
						/ cmdbuf.subcmd1) + DAC_ZERO);

					plaserimage->actual_image[n][YPOS] = (short)
					(((plaserimage->full_scale_image[n][YPOS] - DAC_ZERO)
						/ cmdbuf.subcmd2) + DAC_ZERO);
					*/
				}

				/* modify defaults in output struc */
				plaserimage->x_scale_factor = (float)(cmdbuf.subcmd1/MAX_GAIN);
				plaserimage->y_scale_factor = (float)(cmdbuf.subcmd2/MAX_GAIN);
				/* old way
				plaserimage->x_scale_factor = (1.0 / cmdbuf.subcmd1);
				plaserimage->y_scale_factor = (1.0 / cmdbuf.subcmd2);
				*/
				plaserimage->use_full_scale = FALSE;

				/* trigger output routine to start drawing this pattern */
				/* for now, call a routine in dac.c to draw pattern */
				/*image_draw(); */
				/* use semaphore to update image */
				/* swap pointers of local image buffer with current image */
				break;
			}
	
			case ROTATE_IMAGE:{
				printf("Rotating the image...\n");
				printf("degrees per increment: %d\n",
					cmdbuf.subcmd1);
				/*rotate_image(ROTATE_XY,cmdbuf.subcmd1);*/
				
				/* need to get actual output buffer pointer) */
				/* use semaphore to update image */
				plaserimage = &laserimage;

				/* set rotation rate in laser output structure */
				plaserimage->rotate_rate = cmdbuf.subcmd1;
				
				/* the laser rotation interrupt should be looking at 
				the rotate_rate variable in the current display image
				and when it sees that rotate_rate is not zero, it 
				should do the following:
					1. lock IMAGE_CHANGE_SEM semaphore which locks out
					   any new images being transfered to the laser 
					   display buffer.
					2. copy the output buffer to a local static buffer
					3. rotate the local buffer based on rotate_rate (angle)
					4. wait on a semaphore by the image_draw routine
					   that says it is at the top of an image then lock
					   it (grab TOP_OF_IMAGE_SEM semaphore)
					5. swap pointers so that the image_draw routine
					   now uses the rotated image
					6. let go of the TOP_OF_IMAGE_SEM semaphore to 
					   so that the image draw can continue
					7. release IMAGE_CHANGE_SEM so that new images
					   can be transfered to the image_draw interrupt

				Note that the rotate rate interrupt rate should never be
				faster than the draw_rate * image_length else strange
				things would be displayed...
				*/
				break;
			}
	
			case LISSAJOUS_PATTERN: {
				printf("Displaying a Lissajous sine wave pattern.\n");
				printf("X:Y ratio is %d:%d\n",
					cmdbuf.subcmd1, cmdbuf.subcmd2);

				/* need to get actual output buffer pointer) */
				/* set up a local buffer then swap pointers to current img*/
				/* when done with pattern gen. */
				/* use semaphore to update image */
				plaserimage = &laserimage;

				radius = (double)(short)(MAX_DAC / 2);  

				for (theta = 0; theta<360; theta++){

					/* convert angles to radians & factor in lissajous ratio */
					xtheta = theta * cmdbuf.subcmd1 * RAD_PER_DEG;
					ytheta = theta * cmdbuf.subcmd2 * RAD_PER_DEG;

					n = theta;		
					plaserimage->full_scale_image[n][XPOS] =
						(u_short)(DAC_ZERO + (radius * cos(xtheta))); 
					plaserimage->full_scale_image[n][YPOS] =
						(u_short)(DAC_ZERO + (radius * sin(ytheta))); 
				}

				/* set defaults in output struc */
				plaserimage->image_length = 360;
				plaserimage->x_scale_factor = 1.0;
				plaserimage->y_scale_factor = 1.0;
				plaserimage->draw_rate = 2;
				plaserimage->rotate_rate = 0;
				plaserimage->use_full_scale = TRUE;

				/* trigger output routine to start drawing this pattern */
				/* for now, call a routine in dac.c to draw pattern */
				start_dac_int();
				laser_on();
				break;
			}
				
			/* this routine is not interrupt driven!!! beware ... it is only
			 * a test */
			case SPIRAL_PATTERN: {
				printf("Displaying a spiral pattern.\n");
				printf("Number of loops is %d\n",cmdbuf.subcmd1);
				if (cmdbuf.subcmd1 < 0) 
					printf("The direction is clockwise.\n");
				else 
					printf("The direction is counter-clockwise.\n");
				spiralwave();   /* canned for now */
				break;
			}
				
			case END_PROGRAM:{
				if ((cmdbuf.subcmd1 == 99) & (cmdbuf.subcmd2 == 99)) {
					printf("Received command to end program.\n");
					connected_to_host = FALSE;
				}
				else printf("Received bad END_PROGRAM signal.\n");
				break;
			}
	
			case NEW_IMAGE: {
				printf("Receiving a new image packet...\n");
				/* receive image packet */
				pimagebuf = &imagebuf;
				nleft = cmdbuf.subcmd1;
				printf("\nExpecting %d bytes from host.\n",nleft);
	
				while(nleft > 0) {
					if ((n = recv(clientSock, pimagebuf, nleft, 0)) == 0) {
						/* nack the laser host */
						/* ack_nack_host(IMAGE_NACK); */
						/* connection to server has been broken */
						printf("Bad receive of new image! \n");
						close(clientSock);
						return(ERROR);
					}
					nleft -= n;
					pimagebuf +=n;
					printf("# bytes in this image packet: %d\n",n);
				}
				
				printf("New image of %d bytes received.\n",
						(pimagebuf - &imagebuf));
				/* we have a new complete image from the host */
				/* ack the laser host */
				/* ack_nack_host(IMAGE_ACK); */
				/* check to see if it needs to be translated */
				/* first set pointer to beginning of structure */
				pimagebuf = &imagebuf;	/* set pointer to image buffer*/

				/* get free output buffer (with mem management)*/
				/* for now just use one buffer */
				plaserimage = &laserimage; /* set pointer to image out buf */

				/* get length of image to move or translate */
				plaserimage->image_length = pimagebuf->image_length;

				/* straight conversion or translate decision point */

				if (imagebuf.converted_format) {
					printf("Image received in laser format.\n");

					/* copy imagebuf into laser output buffer */
					printf("Copying image into laser out buf.\n");

#ifndef ORIGINAL
					nbytes = pimagebuf->image_length;
					semTake (LASERSEM);
					bcopy (pimagebuf,plaserimage,nbytes);
					semGive (LASERSEM);
#endif
#ifdef ORIGINAL
					for (n = 0; n < plaserimage->image_length ; n++) {
						plaserimage->full_scale_image[n][XPOS] =
							pimagebuf->full_scale_image[n][XPOS];
						plaserimage->full_scale_image[n][YPOS] =
							pimagebuf->full_scale_image[n][YPOS];
					}

#endif ORIGINAL
				}
				else { /* convert the image to laser format */
					/*convert_image_to_laser_format; */
					printf("Converting to laser format.\n");
					xlate_to_dac_range(pimagebuf,plaserimage);
				}	/* end of convert clause */

				/* set defaults in output struc */
				printf("Finishing up structure conversion...\n");
				plaserimage->x_scale_factor = 1.0;
				plaserimage->y_scale_factor = 1.0;
				plaserimage->draw_rate = pimagebuf->draw_rate;
				plaserimage->rotate_rate = pimagebuf->rotate_rate;
				plaserimage->use_full_scale = TRUE;

				/* signal laser that new image is ready to swap into */
				/*	semaphorexxx */
				start_dac_int();
				set_draw_rate();
				laser_on();
				break;
			} /* end of NEW_IMAGE case */
	
			default: {
				printf("Received a bad command: \'%d\'\n",cmdbuf.command);
				printf("No action taken. Waiting for next command.\n");
			}

		} /* end of command decoder */
	} /* end of infinite loop */

	printf("Exiting command decoder\n");
	return (OK); 

	/* need routines for:
	 * im_alive -- use special socket for this
	 * ack_nack_host() -- use special socket for this
	 */

}

/**************************************************************************
 * xlate_to_dac_range - function which translates the image from XWindows 
 * 						format into a range of values for the DAC board
 *						Special attention is payed to correct centering of
 *						the image.
 * Ron Wilder -- Integrated Solutions
 * 22 Aug 89  -- Date of birth
 *
 * Inputs: pointers to the input image structure and the output structure.
 */

STATUS
xlate_to_dac_range(pimagebuf,plaserbuf)
ENET_IMAGE_PACKET 	*pimagebuf;
LASER_IMAGE 		*plaserbuf;
{
	int n;
	for (n = 0; n < pimagebuf->image_length ; n++) {
		plaserbuf->full_scale_image[n][XPOS] =
			(pimagebuf->full_scale_image[n][XPOS] * XLATE_FACTOR_X);

		plaserbuf->full_scale_image[n][YPOS] =
			(pimagebuf->full_scale_image[n][YPOS] * XLATE_FACTOR_Y);

	}
	printf("Translated %d points in new image.\n",n);
	return(OK);
}


/**********************************************************************
 * image_draw  -- takes image from laser host and sends to dac
 */
#ifdef RON
LOCAL image_draw()

{
	int i,j,n;

	L1X = Straight_through;
	laser_on();
	printf("image length is %d\n",plaserimage->image_length);
	printf("xscale is %5.2f\n",plaserimage->x_scale_factor);
	printf("yscale is %5.2f\n",plaserimage->y_scale_factor);
	printf("draw rate is %d\n",plaserimage->draw_rate);

	/* check which image to draw (full scale or scaled) */
	if (plaserimage ->use_full_scale) {
		printf("Drawing full scale image\n");

		for (i = 0; i < 50; i++) {  /* for now fixed length time */
			for (n = 0; n < plaserimage->image_length;n++) { 
			/*	printf("dac: x: %d   y: %d\n",
					plaserimage->full_scale_image[n][XPOS],
					plaserimage->full_scale_image[n][YPOS]);
		*/
				L1X = plaserimage->full_scale_image[n][XPOS];
				L1Y = plaserimage->full_scale_image[n][YPOS];

				for(j=0; j<(200 * plaserimage -> draw_rate); j++);
			}
		}
	}
	else {
		printf("Drawing actual image (scaled)\n");
		for (i = 0; i < 50; i++) /* for now fixed length time */
			for (n = 0; n< plaserimage->image_length;n++) {
				L1X = plaserimage->actual_image[n][XPOS];
				L1Y = plaserimage->actual_image[n][YPOS];
				for(j=0; j<(200 * plaserimage -> draw_rate); j++);
			}
	}
	laser_off();
	return(OK);
}
#endif RON

	/* W A R N I N G ! ! !  THESE COMMENTS ARE NOT IMPLEMENTED IN CODE YET! */
	/*                      THE ALGORITHM MAY ALSO HAVE CHANGED! */			
	/* image_update routine *
	/* 1. check TOP_OF_IMAGE_SEM to make sure we can update image
	   2. grab semaphore
	   3. swap pointers to image buffer and indicate (set a flag) which
	      says that a new image has been loaded (should be in image 
          structure)
	   4. unlock TOP_OF_IMAGE_SEM
	*/

	/* some info on laser_draw interrupt */
	/* 1. when at top of image, lock TOP_OF_IMAGE_SEM semaphore to make 
		  sure that nobody changes image in middle of loop
	   1a. if new image then clear flag.....
	   2. lock it when not at top of image
	   3. draw next dot (read current_pos from structure)
	   4. when at end of image, unlock TOP_OF_IMAGE_SEM for image update 
	   5. return from int
   */
				/* the laser rotation interrupt should be looking at 
				the rotate_rate variable in the current display image
				and when it sees that rotate_rate is not zero, it 
				should do the following:
ADD flag check for new image so rotate has up to date image in buffer.
					1. lock IMAGE_CHANGE_SEM semaphore which locks out
					   any new images being transfered to the laser 
					   display buffer.
					2. copy the output buffer to a local static buffer
					3. rotate the local buffer based on rotate_rate (angle)
					4. wait on a semaphore by the image_draw routine
					   that says it is at the top of an image then lock
					   it (grab TOP_OF_IMAGE_SEM semaphore)
					5. swap pointers so that the image_draw routine
					   now uses the rotated image
					6. let go of the TOP_OF_IMAGE_SEM semaphore to 
					   so that the image draw can continue
					7. release IMAGE_CHANGE_SEM so that new images
					   can be transfered to the image_draw interrupt

				Note that the rotate rate interrupt rate should never be
				faster than the draw_rate * image_length else strange
				things would be displayed...
				*/

int	delayedMode;


#define chan1		(LASERDAC->dac_1)
#define chan2   	(LASERDAC->dac_2)
#define chan3   	(LASERDAC->dac_3)
#define chan4   	(LASERDAC->dac_4)
#define chan5   	(LASERDAC->dac_5)
#define chan6   	(LASERDAC->dac_6)
#define chan7   	(LASERDAC->dac_7)
#define chan8   	(LASERDAC->dac_8)

/*  init_dac routine intializes the DAC I/O ports   */
#ifdef RON	         
init_dac()

{

	int delayedMode = 0;
	int i,retval;	


	if (vxMemProbe(chan1, READ, 2, &retval) != OK)  {
               printf ("\nDAC board NOT present at addr %x\n", &chan1);
	       return (ERROR);
            }
        printf ("\nDAC board present at addr %x\n", &chan1);
	
	
	if (delayedMode) {
		printf ("\nDAC configured for Controlled Update Mode \n");
	} else
		{ 
		chan1 = 0xc100;
		printf ("\nDAC configured for Direct Update Mode \n");
		}
}

#endif RON

/* laser_off laser with shorter title   ie. off */
off(){laser_off();}

/* laser on with shorter title   ie. on */
on(){laser_on();}

/* stop laser */
laser_off()
{
	printf("Turning the laser OFF \n");
	L1X = Straight_through;
	L1X = DAC_ZERO;
	L1Y = DAC_ZERO;
	L1ON_OFF = BEAM_OFF;
	laser_powered = FALSE;
}

/* laser on */
laser_on()
{
	printf("Turning the laser ON \n");
	L1X = Straight_through;
	L1ON_OFF = BEAM_ON;
	L1X = DAC_ZERO;
	L1Y = DAC_ZERO;
	laser_powered = TRUE;
}


LOCAL VOID dac_Update() 

{
	static int oldLevel;
	static short crow;

	/*logMsg("*"); */
	if (laser_powered) {
		chan1 = Straight_through;
		/*logMsg("#"); */
		plaserimage->current_pos++;
		if (plaserimage->current_pos >= plaserimage->image_length) {
 			plaserimage->current_pos = 0; 	
		} 
		crow = plaserimage->current_pos;
		/*oldLevel = intLock();  */
		chan1 = Straight_through;
		if(plaserimage->use_full_scale) {
			L1X = plaserimage->full_scale_image[crow][XPOS];	
			L1Y = plaserimage->full_scale_image[crow][YPOS];
		}
		else {
			L1X = plaserimage->actual_image[crow][XPOS];	
			L1Y = plaserimage->actual_image[crow][YPOS];
		}
		/*logMsg ("X:%d Y:%d",
			plaserimage->full_scale_image[crow][XPOS],	
		    plaserimage->full_scale_image[crow][YPOS]); */
		/*intUnlock (oldLevel);*/ 
	}
/*	logMsg ("out of updateDAC\n"); */
}


/***********************************************************
 * set_draw_rate -- sets the dac update rate
 ***********************************************************/
STATUS set_draw_rate()
{
	int	dots_per_sec;

	/* eventually, put in something that can slow beam down below 30Hz */
	dots_per_sec = (plaserimage->draw_rate); 

	/* hard limit for 68225 auxclk */
	if (dots_per_sec < MIN_DRAW_RATE) dots_per_sec = MIN_DRAW_RATE;

 	/* try to limit freq*/
	if (dots_per_sec > MAX_DRAW_RATE) dots_per_sec = MAX_DRAW_RATE;

	sysAuxClkRateSet(dots_per_sec);
	printf("Changed draw rate to %d\n",plaserimage->draw_rate);
}



/**********************************************************/
/* short title for dac interrupt startup routine */
STATUS start_dac_int()
{
	if (!dac_int_running) {
		conUpdate();
		printf("Starting DAC interrupts.\n");
		dac_int_running = TRUE;
	}
}

/**********************************************************/
/* short title for dac interrupt stop routine */
STATUS stop_dac_int()
{
	if (dac_int_running) {
		disUpdate();
		printf("Disabling DAC interrupts\n");
		dac_int_running = FALSE;
	}
}


/**********************************************************/
/* routine to connect dacWrite to auxClk */
/* this routine will be periodically invoked at a varied laser
		draw rates */

STATUS conUpdate() 
{
	set_draw_rate();

	/*logMsg("connecting dac_Update\n");*/	

	if (sysAuxClkConnect(dac_Update,0) != OK)
	{
		printf ("Cannot connect to Aux Clk.\n");
		return (ERROR);
	}

	if (sysAuxClkEnable() != OK) {
		printf (" Error: Cannot enable sysAuxClk\n");
		return (ERROR);
	}

}


STATUS disUpdate()
{

	sysAuxClkDisable();
}

spiralwave()
{
	int i ;
	double ftheta,xtheta, ytheta, radius;
	double sinval, cosval;
	u_short theta;

	L1X = Straight_through;
	laser_on();
	for (i = 0 ; i < 10; i++) {
		for (theta = 1; theta < 10 * 360; theta+=4) {
			ftheta = ((double)  theta * .01745);     /* degrees to radians */
			radius = 0x7ff * ((double) theta / (10 * 360.0)); 

			sincos(ftheta,&sinval,&cosval);
			L1X = (u_short)(0x800 + (radius * cosval)); 
			L1Y = (u_short)(0x800 + (radius * sinval)); 
		}
	}
	laser_off();
}

/* turns the laser beam on and then off */
onoff()
{

	int i;
	laser_on();
	taskDelay(60);
	laser_off();
	taskDelay(60);
}

/*******************************************************************************
*
* connect_to_laser_host - establish connection to laser host over e-net
*
* This function will open up a socket to the laser light demo's host over 
* an internet link. Using ethernet for now.
*
*/

STATUS
connect_to_laser_host(hostName,pclientSock)
char	*hostName;			/* name of laser host */
int	*pclientSock;			/* socket fd */
{

	int i;					/* loop counter */

	/* sockaddr_in: (file: in.h) is internet style socket addr structure */
	struct sockaddr_in serverAddr;	/* server's address */
	struct sockaddr_in clientAddr;	/* client's address */
	
	/* Open the socket. Use ARPA Internet address format and stream
	 * sockets. Format described in "socket.h". */

	/* open a socket and get a socket descriptor (clientSock)*/
	if ((*pclientSock = socket(AF_INET, SOCK_STREAM, 0)) == ERROR) {
		printf("connect_to_laser_host: Couldn't get a socket descriptor.\n");
		printf("Try rebooting this computer, and restarting this program.\n");
		return(ERROR);
	}

	/* Zero out the socket structures (sock_addr). (Do before socket call.) */
	bfill(&serverAddr, sizeof(serverAddr), 0);
	bfill(&clientAddr, sizeof(clientAddr), 0);
	
	/* Plug server's socket info into the socket structure */
	serverAddr.sin_family = AF_INET;
	serverAddr.sin_port = SERVER_NUM;  /* socket port */

	/* get server's Internet address by looking up hostname in host table*/
	if ((serverAddr.sin_addr.s_addr = hostGetByName(hostName)) == ERROR) {
		printf("\n*********************************************************\n");
		printf("Couldn't find host \'%s\' in host table.\n",
		hostName);
		printf("Make sure that the laser host name is \'%s\'.  If not\n",
			hostName);
		printf("then change the HOSTNAME define in laser.h to the\n");
		printf("appropriate name and recompile the program.\n");
		printf("*********************************************************\n");
		close(*pclientSock);  
		return(ERROR);
	}
	
	/* Establish socket connection with host */

	if (connect(*pclientSock, (SOCKADDR *) & serverAddr, sizeof(serverAddr))
	    == ERROR) {
		printf("***********************************************************\n");
		printf("Couldn't establish a connection with the laser host machine\n");
		printf("\'%s\'. Make sure the host is running the laser program.\n",
			hostName);
		printf("***********************************************************\n");
		close(*pclientSock); 
		return(ERROR);
	}
	return(OK);
/*
	for (i = 0; i<100;i++) {
		if (connect(*pclientSock, (SOCKADDR *) & serverAddr, sizeof(serverAddr))
	    	== OK) {
			return(OK);
		}
		else {
			printf("\nTrying to connect to host: attempt #%d\r",i);
			taskDelay(2*SEC);
		}
	}

	printf("\nCouldn't establish a connection with the host machine for\n");
	printf("the laser light show.\n");
	printf("Closing the socket.\n");
	close(*pclientSock); 
	return(ERROR);
*/	
  
}
