/*******************************************************************
 * laserevent.c -- Event handlers for the laser light show demo
 * 
 * Ron Wilder -- Integrated Solutions Inc.
 * 1 Sep 1989 -- Date of Birth
 *
 * $Header: laserevent.c,v 1.4 89/09/07 11:08:47 laser Locked $
 *
 * This module contains all of the event handlers required for the 
 * laser light show demo program. 
 * Events include: scroll bar motion, button pressing, and drawing
 * Some other module are also in here that are not event handlers
 * such as the command to clear the drawing area and put up crosshair
 *******************************************************************/ 

#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/AsciiText.h>
#include <X11/Cardinals.h>
#include <stdio.h>
#include <math.h>
#include "laserxt.h"
#include "laser.h"

#define ERROR -1

/* Misc Globals */
extern int serverSock;							/* server socket */
extern ENET_CMD_PACKET		cmdbuf;				/* command packet */
extern ENET_IMAGE_PACKET	imagebuf;			/* image packet */
extern float current_draw_rate;					/* laser motion speed */
extern float current_X_gain;					/* current gain factor */
extern float current_Y_gain;					/* current gain factor */
extern boolean user_interface_test;				/* bypass laser -- test X only*/
extern boolean helpflag;						/* help mode on if TRUE */
extern boolean XY_ganged;						/* gang gain pots if TRUE*/
extern boolean laser_on_flg;					/* laser beam on if TRUE*/
/*extern int laser_button;*/					/* holds button number */
extern Widget draw;								/* drawing area widget */
extern Widget msg_area;							/* area to put messages */
extern Widget draw_rate_scroll;					/* Draw rate scroll bar widget*/
extern Widget X_gain_scroll;					/* X gain scrollbar widget */
extern Widget Y_gain_scroll;					/* Y gain scrollbar widget */
extern Widget button_labels[];
extern BUTTON button[];
extern char *fname[];
extern FILE *f1, *open();
extern char *pword[];
extern char *correct_password[];
extern line_segment_data data;
extern XGCValues values;

/*extern unsigned short current_rot_rate;*/		/* current rotation rate */
/*extern unsigned short current_position;*/		/* current image index */

extern int send_cmd_to_laser();
extern int clear_image_buf();

extern char message[];
extern int leftMargin;
extern char msgbuf[];

/***************************************************************************/
/********************************** put_msg ********************************/
/***************************************************************************/

void
put_msg(msg)
	char *msg[];

{
	int n;
	Arg pargs[1];

	/*for (n = 0; n< strlen(message) ; n++) message[n] = '\0'; */

	/*for (n = 0; n< MAX_MSG_LENGTH ; n++) message[n] = ' '; */

	/* kludge for now */
	for (n = 0; n< 80 ; n++) message[n] = ' '; 
	
	sprintf(message, "%s", msg );
	message[strlen(msg)] = ' ';
	leftMargin = (leftMargin == 2) ? 3 : 2;
	XtSetArg(pargs[0],XtNleftMargin, leftMargin);
	XtSetValues(msg_area,pargs,ONE);
}

/***************************************************************************/
/**************************** draw_cross_hair ******************************/
/***************************************************************************/
void
draw_cross_hair()
{
	
	/* clear window first */
	XClearWindow(XtDisplay(draw), XtWindow(draw));

	/* set line to dashes */
	values.line_style = LineOnOffDash;
	values.function = GXclear;
	data.gc = XtGetGC(draw, GCForeground | GCBackground |
				GCFunction | GCLineStyle, &values);

	/* draw cross hair */
	XDrawLine (XtDisplay(draw), XtWindow(draw),data.gc,
		CENTER_OF_XWINDOW_X, MIN_XWINDOW_Y,
		CENTER_OF_XWINDOW_X, MAX_XWINDOW_Y);
	
	XDrawLine (XtDisplay(draw), XtWindow(draw),data.gc,
		MIN_XWINDOW_X, CENTER_OF_XWINDOW_Y,
		MAX_XWINDOW_X, CENTER_OF_XWINDOW_Y);
	
	/* set line back to solid */
	values.line_style = LineSolid;
	values.function = GXclear;
	data.gc = XtGetGC(draw, GCForeground | GCBackground |
				GCFunction | GCLineStyle, &values);
}

/******************************************************************
 * start_image_capture -- event handler
 * handles image start when mouse button is pressed inside draw
 * window.  
 *******************************************************************/


void start_image_capture(w, dataxx, event)
Widget	w;
line_segment_data	*dataxx;
XEvent			*event;
{

	/* temporary for now ... put in a crosshair */

	if (helpflag == TRUE) {

		put_msg("Pressing this button in the drawing area starts a new image.");

	}
	else {
		dataxx->last_x = dataxx->start_x = event->xbutton.x;
		dataxx->last_y = dataxx->start_y = event->xbutton.y;
	
		clear_image_buf();		/* prepare for new image */
	
		draw_cross_hair();		/* clear draw area and put up cross hair */

	}	
}
	
/*******************************************************************
 * draw_next_segment -- event handler
 *
 * handles image drawing when mouse is moved (with button down) 
 * inside of the draw window
 *
 *******************************************************************/

void draw_next_segment(w, data, event)
Widget	w;
line_segment_data	*data;
XEvent			*event;
{
	if (helpflag == TRUE) {
	}
	else {
		/*
		 * Update the data points
		 */
	
		/* throw out bad points!!! */
		if ((event -> xbutton.x != ERROR) && (event -> xbutton.y != ERROR)) { 
			data->start_x = data->last_x;
			data->start_y = data->last_y;
			data->last_x = event->xbutton.x;
			data->last_y = event->xbutton.y;
	
			/*
			 * Draw the new line.
			 */
		
			XDrawLine(XtDisplay(w), XtWindow(w),
				data->gc, data->start_x,
				data->start_y, data->last_x, data->last_y);
		
			if (imagebuf.image_length < MAX_IMAGE_DOTS) {
				imagebuf.full_scale_image[imagebuf.image_length][XPOS] =
					event->xbutton.x;
				imagebuf.full_scale_image[imagebuf.image_length][YPOS] =
					event->xbutton.y;
				imagebuf.image_length++;	/* point to next image position */
			}
			else {
				cmdbuf.command = NEW_IMAGE;
				cmdbuf.subcmd1 = sizeof(imagebuf.image_length) +
								 sizeof(imagebuf.draw_rate) +
								 sizeof(imagebuf.rotate_rate) +
								 sizeof(imagebuf.converted_format) +
								 imagebuf.image_length * 2 * sizeof(short);
				cmdbuf.subcmd2 = NULL;
				imagebuf.converted_format = FALSE;
				imagebuf.draw_rate = (short) (current_draw_rate * MAX_DRAW_RATE) +
									MIN_DRAW_RATE;
				imagebuf.rotate_rate = 0;	/* for now!!!!! */
				send_cmd_to_laser();
			}
		}
	}
}
/*******************************************************************
 * draw_rate_Scrolled -- event handler
 *
 * increments/decrements laser draw rate when scrollbar thumb is scrolled
 *
 *******************************************************************/

void draw_rate_Scrolled(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	char *msg[80];
	if (helpflag == TRUE) {
			put_msg("This button is used to send a new draw rate to the laser.");
	}
	else {
		if (call_data < 0) call_data = - call_data;
		current_draw_rate = (1.0 - ((float)call_data/DRAW_RATE_SBAR_LENGTH));

			cmdbuf.command = NEW_DRAW_RATE;
			cmdbuf.subcmd1 = (short) (current_draw_rate * MAX_DRAW_RATE) +
							MIN_DRAW_RATE;
			cmdbuf.subcmd2 = NULL;

			sprintf(msgbuf,"Draw rate scrolled to %d dots/sec", cmdbuf.subcmd1);
			put_msg(msgbuf);	
			send_cmd_to_laser();

			XtScrollBarSetThumb(draw_rate_scroll,(1.0 - current_draw_rate),NULL);
	}
}

/*******************************************************************
 * draw_rate_Thumbed -- event handler
 *
 * sends draw rate factor to laser when scrollbar thumb is moved
 *******************************************************************/


void draw_rate_Thumbed(w, closure, top)
    Widget w;
    caddr_t closure;
    float top;
{
	float	last_draw_rate;

	if (helpflag == TRUE) {
		put_msg("This scroll bar thumb is used to set a new draw rate.");
	}
	else {

		last_draw_rate = current_draw_rate;
		current_draw_rate = (1.0 - top);

		if (current_draw_rate != last_draw_rate) {
			cmdbuf.command = NEW_DRAW_RATE;
			cmdbuf.subcmd1 = (short) (current_draw_rate * MAX_DRAW_RATE) +
							MIN_DRAW_RATE;
			cmdbuf.subcmd2 = NULL;
			sprintf(msgbuf,"Draw rate thumbed to %d dots/sec", cmdbuf.subcmd1);
			put_msg(msgbuf);	
	
			send_cmd_to_laser();
		}
	}
}

/*******************************************************************
 * X_gain_Scrolled -- event handler
 *
 * increments/decrements laser X gain when scrollbar thumb is scrolled
 *******************************************************************/

void X_gain_Scrolled(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
			put_msg("This button is used to send a new X gain to the laser.");
	}
	else {
		if (call_data < 0) call_data = - call_data;
		current_X_gain = (1.0 - ((float)call_data/GAIN_SBAR_LENGTH));

			cmdbuf.command = SCALE_IMAGE;
			cmdbuf.subcmd1 = (short)(current_X_gain * MAX_GAIN);
			if (XY_ganged){
				current_Y_gain = current_X_gain;
				cmdbuf.subcmd2 = cmdbuf.subcmd1;
				XtScrollBarSetThumb(Y_gain_scroll,(1.0 - current_X_gain),NULL);
				sprintf(msgbuf,"Scrolled to x: %d%%  y:%d%%",cmdbuf.subcmd1, cmdbuf.subcmd2);
				put_msg(msgbuf);	
			}
			else  {
				cmdbuf.subcmd2 = (short)(current_Y_gain * MAX_GAIN);
				sprintf(msgbuf,"X scrolled to %d%%",cmdbuf.subcmd1);
				put_msg(msgbuf);	
			}

			XtScrollBarSetThumb(X_gain_scroll,(1.0 - current_X_gain),NULL);
	
			send_cmd_to_laser();
	}
}

/*******************************************************************
 * X_gain_Thumbed -- event handler
 *
 * sends X gain factor to laser when scrollbar thumb is moved
 *******************************************************************/

void X_gain_Thumbed(w, closure, top)
    Widget w;
    caddr_t closure;
    float top;
{
	float	last_X_gain;

	if (helpflag == TRUE) {
		put_msg("This scroll bar thumb is used to set a new X gain factor.");
	}
	else {
		last_X_gain = current_X_gain;
		current_X_gain = 1.0 - top;

		if (current_X_gain != last_X_gain) {
			cmdbuf.command = SCALE_IMAGE;
			cmdbuf.subcmd1 = (short)(current_X_gain * MAX_GAIN);
			if (XY_ganged){
				current_Y_gain = current_X_gain;
				cmdbuf.subcmd2 = cmdbuf.subcmd1;
				XtScrollBarSetThumb(Y_gain_scroll,top,NULL);
			}
			else cmdbuf.subcmd2 = (short)(current_Y_gain * MAX_GAIN);
	
				sprintf(msgbuf,"Thumbed to x: %d%%  y:%d%%",cmdbuf.subcmd1, cmdbuf.subcmd2);
				put_msg(msgbuf);	
	
			send_cmd_to_laser();
		}
	}
}

/*******************************************************************
 * Y_gain_Scrolled -- event handler
 *
 * increments/decrements laser Y gain when scrollbar thumb is scrolled
 *
 *******************************************************************/

void Y_gain_Scrolled(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
			put_msg("This button is used to send a new Y gain to the laser.");
	}
	else {	/* do command */
		if (call_data < 0) call_data = - call_data; /* ignore sign */

			current_Y_gain = (1.0 - ((float)call_data/GAIN_SBAR_LENGTH));

			cmdbuf.command = SCALE_IMAGE;
			cmdbuf.subcmd2 = (short)(current_Y_gain * MAX_GAIN);
			if (XY_ganged){
				current_X_gain = current_Y_gain;
				cmdbuf.subcmd1 = cmdbuf.subcmd2;
				XtScrollBarSetThumb(X_gain_scroll,(1.0 - current_Y_gain),NULL);
				sprintf(msgbuf,"Scrolled to x: %d%%  y:%d%%\n",cmdbuf.subcmd1, cmdbuf.subcmd2);
				put_msg(msgbuf);	
			}
			else {
				cmdbuf.subcmd1 = (short)(current_X_gain * MAX_GAIN);
				sprintf(msgbuf,"Y Scrolled to %d%%", cmdbuf.subcmd2);
				put_msg(msgbuf);
			}

			XtScrollBarSetThumb(Y_gain_scroll,(1.0 - current_Y_gain),NULL);
			
	
	
			send_cmd_to_laser();
	}
}

/*******************************************************************
 * Y_gain_Thumbed -- event handler
 * sends Y gain factor to laser when scrollbar thumb is moved
 *******************************************************************/

void Y_gain_Thumbed(w, closure, top)
    Widget w;
    caddr_t closure;
    float top;
{
	float	last_Y_gain;
	if (helpflag == TRUE) {
		put_msg("This scroll bar thumb is used to set a new Y gain factor.");
	}
	else {
		last_Y_gain = current_Y_gain;
		current_Y_gain = 1.0 - top;

		if (current_Y_gain != last_Y_gain) {
			cmdbuf.command = SCALE_IMAGE;
			cmdbuf.subcmd2 = (short)(current_Y_gain * MAX_GAIN);
			if (XY_ganged){
				cmdbuf.subcmd1 = cmdbuf.subcmd2;
				current_X_gain = current_Y_gain;
				XtScrollBarSetThumb(X_gain_scroll,top,NULL);
			}
			else cmdbuf.subcmd1 = (short)(current_X_gain * MAX_GAIN);
			sprintf(msgbuf,"Thumbed to x:%d%%  y:%d%%",cmdbuf.subcmd1, cmdbuf.subcmd2);
			put_msg(msgbuf);
	
			send_cmd_to_laser();
		}
	}
}


/*******************************************************************
 * help_key -- event handler
 * displays helpful information about program and various buttons
 *******************************************************************/

void help_key(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
/* add code here to go back to normal video in box */
		helpflag = FALSE;
	    put_msg( "Exiting help..." );
		/* put some code here to clean up help */
		/* ie. close pop up window */
	}
	else{

/* add code here to go to reverse video */

		put_msg( "Select any button/scrollbar to get help. To exit, select HELP again.");
		helpflag = TRUE;
		/* put code here to open help database and a pop up window */
		/* for help info */
	}
}

/*******************************************************************
 * XY_gain_gang -- event handler
 * gangs the X and Y gain pots
 *******************************************************************/

void XY_gain_gang(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {		/* print help message */
		put_msg("This key is used to gang the X and Y gain pots together.\n");
	}
	else {	/* do the command asked for */

		if (!XY_ganged) { /* only gang them if they are not ganged already. */
			put_msg( "Ganging the X and Y gain pots together.");
			XY_ganged = TRUE;

			/* move Y gain thumb to same as X gain thumb */
			current_Y_gain = current_X_gain;
			XtScrollBarSetThumb(Y_gain_scroll,
							    (float)(1.0 - current_Y_gain), NULL);

			/* load up packet to send new gain to target */
			cmdbuf.command = SCALE_IMAGE;
			cmdbuf.subcmd1 = (short)(current_X_gain * MAX_GAIN);
			cmdbuf.subcmd2 = cmdbuf.subcmd1;

			send_cmd_to_laser();
		}
	}
}

/*******************************************************************
 * XY_gain_separate -- event handler
 * separates the X and Y gain pots
 *******************************************************************/

void XY_gain_separate(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {		/* print help message */
		put_msg("This key is used to separate the X and Y gain pots.");
	}
	else {	/* do the command asked for */

		if (XY_ganged) { /* only separate them if they are currently ganged.*/
			put_msg( "Separating the X and Y gain pots.");
			XY_ganged = FALSE;
		}
	}
}

/*******************************************************************
 * modified to write image to file fname (specfied elsewhere)
 * save_image -- event handler
 * writes image to a file
 *******************************************************************/

void save_image(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	int n;
	
	FILE  *fd, *fopen();

	if (helpflag == TRUE) {
		put_msg("This button writes the image to a disk file.");
	}
	else { /* put button handler here */

		/*	get_f(fname); */

		sprintf(msgbuf,"Saving the current image to the file %s",fname[0]);
		put_msg(msgbuf);
		imagebuf.converted_format = FALSE;
		imagebuf.draw_rate = (short) (current_draw_rate * MAX_DRAW_RATE) +
							MIN_DRAW_RATE;
		imagebuf.rotate_rate = 0;	/* for now!!!!! */

		if(( fd = fopen(fname[0],"w")) == NULL) {
			sprintf(msgbuf,"Can\'t create file %s",fname[0]);
			put_msg(msgbuf);
			return;
		}
		fprintf(fd,"Laser_Image_File\n");
		fprintf(fd,"%hd\n%hd\n%hd\n",
			imagebuf.image_length, 
			imagebuf.draw_rate,
			imagebuf.rotate_rate );
 
		for (n = 0 ; n < imagebuf.image_length-1; n++) {
			fprintf(fd,"%hd:%hd\n",
			imagebuf.full_scale_image[n][XPOS], 
			imagebuf.full_scale_image[n][YPOS]); 
		}
		fprintf(fd,"EOF\n");

		fclose(fd);
	}
}

/*******************************************************************
 * save_image -- event handler
 * saves image to disk 
 * THIS CODE IS NOT REALLY QUIT CODE, !!!!!!!!!!!!!
 * this code is a for new button practicing
 * old quit key -- modified
 *******************************************************************/

void quit_key(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	
	if (helpflag == TRUE) {
		put_msg("BLAH BLAH BLAH.");
	}
	else { /* put button handler here */

	}
}

/*******************************************************************
 * get_image -- event handler
 * gets image from disk 
 *******************************************************************/

void get_image(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	int n;
	ENET_IMAGE_PACKET	localbuf;

	static	char	file_id[100], eofcheck[20];
	FILE  *fd, *fopen();

	/* put inside laserglobals.h !!!!! */
	char *header = { "Laser_Image_File" };

	if (helpflag == TRUE) {
		put_msg("This button gets an image from the disk.");
	}
	else { /* put button handler here */

		/*	get_f(fname); */

		sprintf(msgbuf,"File %s to be opened..%s.",fname[0]);
		put_msg(msgbuf);

		if(( fd = fopen(fname[0],"r")) == NULL) {
			sprintf(msgbuf,"Can\'t open %s",fname[0]);
			put_msg(msgbuf);
			return;
		}
		put_msg("Reading in the data now");
	
		fscanf(fd,"%s",file_id);
		/*printf("file_id is \'%s\'\n",file_id); */

		if (strcmp(file_id,header) != 0) {
			sprintf(msgbuf,"Bad laser image file ID. Header should be %s, was %s.",header,file_id);
			put_msg(msgbuf);
			return;
		}
		
		fscanf(fd,"%hd", &localbuf.image_length); 
		fscanf(fd,"%hd", &localbuf.draw_rate);
		fscanf(fd,"%hd", &localbuf.rotate_rate );
		sprintf(msgbuf,"# dots: %d, draw rate: %d, rotation rate: %d",
			localbuf.image_length,localbuf.draw_rate,localbuf.rotate_rate);
		put_msg(msgbuf);

		for (n = 0 ; n < localbuf.image_length-1; n++) {
/* put in err check on read */
			fscanf(fd,"%hd:%hd", 
				&localbuf.full_scale_image[n][XPOS], 
				&localbuf.full_scale_image[n][YPOS]); 
		}

		fscanf(fd,"%s",eofcheck);

		if (strcmp(eofcheck,"EOF") != 0) {

			sprintf(msgbuf,"File format error. Expected \'EOF\' Got: %s",eofcheck);
			put_msg(msgbuf);
			return;
		}
 
		if (fclose(fd) == ERROR) {
			put_msg("Can\'t close file.");
			return;
		}

		/* copy localbuf to imagebuf */
		imagebuf.image_length = localbuf.image_length; 
		imagebuf.draw_rate = localbuf.draw_rate; 
		imagebuf.rotate_rate = localbuf.rotate_rate; 
		for (n = 0 ; n < imagebuf.image_length-1; n++) {
				imagebuf.full_scale_image[n][XPOS] = 
					localbuf.full_scale_image[n][XPOS];
				imagebuf.full_scale_image[n][YPOS] = 
					localbuf.full_scale_image[n][YPOS]; 
		}
		/* clear area and put up cross hair */
		draw_cross_hair();

		/* draw the image */
		for (n = 0 ; n < imagebuf.image_length-2; n++) {
			XDrawLine(XtDisplay(draw), XtWindow(draw),
				data.gc, 
				imagebuf.full_scale_image[n][XPOS], 
				imagebuf.full_scale_image[n][YPOS], 
				imagebuf.full_scale_image[n+1][XPOS], 
				imagebuf.full_scale_image[n+1][YPOS]); 
		}
		/* store values into current value holders */
		current_draw_rate = (float)((float)(imagebuf.draw_rate - MIN_DRAW_RATE)
			 / MAX_DRAW_RATE);

		XtScrollBarSetThumb(draw_rate_scroll,(1.0 - current_draw_rate),NULL);

		/* 
		 * send image to laser target 
		 */
		cmdbuf.command = NEW_IMAGE;
		cmdbuf.subcmd1 = sizeof(imagebuf.image_length) +
						 sizeof(imagebuf.draw_rate) +
						 sizeof(imagebuf.rotate_rate) +
						 sizeof(imagebuf.converted_format) +
						 imagebuf.image_length * 2 * sizeof(short);
		cmdbuf.subcmd2 = NULL;
		imagebuf.converted_format = FALSE;
		send_cmd_to_laser();		

	}
}

/*******************************************************************
 * quit -- event handler
 * exits program and closes anything necessary 
 *******************************************************************/

void quit_but_handler(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	char *s;

	if (helpflag == TRUE) {
		put_msg("This key is used to exit the program.");
	}
	else {
/**************
 ADD CODE TO ASK USER IF HE/SHE IS SURE
 ADD CODE TO ASK FOR A PASSWORD You don't want people walking by to
 be able to kill the program use same password as startup
***************/

	/*put a popup here */
 
		put_msg("Please enter the exit password (on console) --> ");

		scanf("%s",pword);
		if (strcmp(pword,*correct_password) == 0) {

			put_msg("\nPassword was correct, ending the laser light show.");
			cmdbuf.command = END_PROGRAM;
			cmdbuf.subcmd1 = END_PROGRAM;
			cmdbuf.subcmd2 = END_PROGRAM;
			send_cmd_to_laser();
			if (!user_interface_test) {
				put_msg("Pausing to let connections close.");
				sleep(2); /* try to pause and let the message get across */
				shutdown(serverSock,2);	/* force shutdown on socket */
				close(serverSock);		/* close connection to target */
				sprintf(msgbuf,"Socket %d to laser target closed.",serverSock);
				put_msg(msgbuf);
			}
			put_msg( "Exiting the laser program." );
			exit(0);
		}
		put_msg("\nSorry, invalid password. Program continuing onward.");
	}
}

/*******************************************************************
 * redisplay -- event handler
 * redisplays the image upon an exposure 
 *******************************************************************/

void redisplay(w,dataxx)
    Widget w;
	line_segment_data	*dataxx;
	/*line_segment_data	*data; */
{
	int n;

	/* clear area and draw cross hair */
	draw_cross_hair();

	/*for (n = 0 ; n < imagebuf.image_length-2; n++) {
		XDrawLine(XtDisplay(w), XtWindow(w),
			dataxx->gc, 
	*/
	for (n = 0 ; n < imagebuf.image_length-2; n++) {
		XDrawLine(XtDisplay(draw), XtWindow(draw),
			data.gc, 
			imagebuf.full_scale_image[n][XPOS], 
			imagebuf.full_scale_image[n][YPOS], 
			imagebuf.full_scale_image[n+1][XPOS], 
			imagebuf.full_scale_image[n+1][YPOS]); 
	}
}

/*******************************************************************
 * invert_X -- event handler
 * inverts X axis of picture 
 *******************************************************************/

void invert_X(w, closure, call_data)				/* inverts X axis */

    Widget w;
    caddr_t closure;
    int call_data;
{
	int n;
	if (helpflag == TRUE) {
		put_msg("This button is used to invert the X axis.");
	}
	else {

		cmdbuf.command = INVERT_IMAGE_X;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		/* put in code to do same in display area */
		put_msg("Inverting X axis.");
		for (n = 0; n <imagebuf.image_length; n++){
				imagebuf.full_scale_image[n][XPOS] =
					MAX_XWINDOW_X - imagebuf.full_scale_image[n][XPOS];
		}
		redisplay(draw,data);

	}
}

/*******************************************************************
 * invert_Y -- event handler
 * inverts Y axis of picture 
 *******************************************************************/

void invert_Y(w, closure, call_data)				/* inverts Y axis */

    Widget w;
    caddr_t closure;
    int call_data;
{
	int n;
	if (helpflag == TRUE) {
		put_msg("This button is used to invert the Y axis.");
	}
	else {

		cmdbuf.command = INVERT_IMAGE_Y;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		/* put in code to do same in display area */
		put_msg("Inverting Y axis.");
		for (n = 0; n <imagebuf.image_length; n++){
				imagebuf.full_scale_image[n][YPOS] =
					MAX_XWINDOW_Y - imagebuf.full_scale_image[n][YPOS];
		}
		redisplay(draw,data);
	}
}

/*******************************************************************
 * laser_on -- event handler
 * turns on the laser beam
 *******************************************************************/

void laser_on(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		put_msg("This button is used to turn on the laser beam.");
	}
	else {

		cmdbuf.command = LASER_ON;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		put_msg("Turning laser beam on.");
	}
}

/*******************************************************************
 * laser_off -- event handler
 * turns off the laser beam
 *******************************************************************/

void laser_off(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		put_msg("This button is used to turn off the laser beam.");
	}
	else {

		cmdbuf.command = LASER_OFF;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		put_msg("Turning laser beam off.");
	}
}


/*******************************************************************
 * spare -- event handler
 * your stuff goes here
 *******************************************************************/

void spare(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		put_msg("This button does...");
	}
	else {

		cmdbuf.command = NULL;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		put_msg("Doing...");
	}
}

/*******************************************************************
 * full_scale -- event handler
 * sets laser up to back to full scale image display
 *******************************************************************/

void full_scale(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		put_msg("This button is used to display the image in full scale.");
	}
	else {

		cmdbuf.command = GO_FULL_SCALE;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		/* set local indicators to full scale */
		current_Y_gain = 1.0;	
		current_X_gain = 1.0;

		XtScrollBarSetThumb(X_gain_scroll,(float)(1.0 - current_X_gain),NULL);
		XtScrollBarSetThumb(Y_gain_scroll,(float)(1.0 - current_Y_gain),NULL);

		put_msg("Going back to full gain on both X and Y axis.");
	}
}

/*******************************************************************
 * send_image -- event handler
 * sends current image to the laser target machine 
 *******************************************************************/

void send_image(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		put_msg("This button is used to send a new image to the laser.");
	}
	else {
		/* don't send image if zero length buffer */
		if (imagebuf.image_length != 0) {
			cmdbuf.command = NEW_IMAGE;
			cmdbuf.subcmd1 = sizeof(imagebuf.image_length) +
							 sizeof(imagebuf.draw_rate) +
							 sizeof(imagebuf.rotate_rate) +
							 sizeof(imagebuf.converted_format) +
							 imagebuf.image_length * 2 * sizeof(short);
			cmdbuf.subcmd2 = NULL;
			imagebuf.converted_format = FALSE;
			imagebuf.draw_rate = (short) (current_draw_rate * MAX_DRAW_RATE) +
								MIN_DRAW_RATE;
			imagebuf.rotate_rate = 0;	/* for now!!!!! */

			/* set local indicators to full scale */
			current_Y_gain = 1.0;	
			current_X_gain = 1.0;

			XtScrollBarSetThumb(X_gain_scroll,(float)(1.0 -
													current_X_gain),NULL);
			XtScrollBarSetThumb(Y_gain_scroll,(float)(1.0 - 
													current_Y_gain),NULL);
			put_msg("Sending a new image to the laser.");
			send_cmd_to_laser();
			/*	clear_image_buf(); */
	
			/* put in code to do same in display area */
		}
	}
}

/*******************************************************************
 * lissajous -- event handler
 * sends a lissajous pattern command to the laser target
 *******************************************************************/

void lissajous(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	double	radius,xtheta,ytheta;
	int 	theta;

	if (helpflag == TRUE) {
		put_msg("This button is used to send a lissajous command to the laser target.");
	}
	else {

	/*	cmdbuf.command = LISSAJOUS_PATTERN; */

		/*for now put in hard values, later take them from dialog or menu box */

		cmdbuf.subcmd1 = 3;
		cmdbuf.subcmd2 = 2;

	/*	send_cmd_to_laser(); */

		
		put_msg("Sending a lissajous pattern command to the laser target.");

		for (theta = 0; theta<360; theta++){

			/* set radius to minimum of width and height */
			radius = (MAX_XWINDOW_X < MAX_XWINDOW_Y) ? 
							MAX_XWINDOW_X/2 : MAX_XWINDOW_Y/2; 

			/* convert angles to radians & factor in lissajous ratio */
			xtheta = theta * cmdbuf.subcmd1 * RAD_PER_DEG;
			ytheta = theta * cmdbuf.subcmd2 * RAD_PER_DEG;

			/* store into display buffer */
			imagebuf.full_scale_image[theta][XPOS] =
				(u_short)(CENTER_OF_XWINDOW_X + (radius * cos(xtheta))); 
			imagebuf.full_scale_image[theta][YPOS] =
				(u_short)(CENTER_OF_XWINDOW_Y + (radius * sin(ytheta))); 
		}

		/* set defaults in output struc */
		imagebuf.image_length = 360;
		current_draw_rate = (float)((float)(DEFAULT_DRAW_RATE - MIN_DRAW_RATE)
			 / MAX_DRAW_RATE);

		current_X_gain = 1.0;
		current_Y_gain = 1.0;

		/* set scroll bars */
		XtScrollBarSetThumb(X_gain_scroll,(float)(1.0 - current_X_gain),NULL);
		XtScrollBarSetThumb(Y_gain_scroll,(float)(1.0 - current_Y_gain),NULL);
		XtScrollBarSetThumb(draw_rate_scroll,(1.0 - current_draw_rate),NULL);

		/* display the image */
		redisplay(draw,data);

		/* send the image to the laser */
		/* use callback routine w/o args */
		send_image(draw, (caddr_t)NULL, (int)NULL); 
	}
}

/*******************************************************************
 * clear_image -- event handler
 * clears the display and the image data structure 
 *******************************************************************/
extern	Widget	draw;

void clear_image(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		put_msg("This button is used to clear the drawing area and laser display.");
	}
	else {

		draw_cross_hair();

		put_msg("Clearing the drawing area.");

		/* could add code which sends a zero length image to laser */
		/* could add laserv command which zaps buffer. this would be faster */
		/* turn off the laser */
		cmdbuf.command = LASER_OFF;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();
	}
}

/*******************************************************************
 * canned_image -- event handler
 * commands laser target to display a canned image
 *******************************************************************/

void canned_image(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		put_msg("This button is used to send a canned image command to the laser target.");
	}
	else {

		cmdbuf.command = CANNED_IMAGE;
		cmdbuf.subcmd1 = 0;		/* for now just display image 0 */
		cmdbuf.subcmd2 = NULL;
		put_msg( "Sending a canned image command to laser target." );
		send_cmd_to_laser();
	}
}

/* the following isn't really an event but it seemed like a good place for it*/

/***************************************************************************/
/**************************** clear_image_buf ******************************/
/***************************************************************************/

int
clear_image_buf()
{
	int n;		/* loop counter */	
	/* initialize new image length to zero */
	imagebuf.image_length = 0;
	imagebuf.draw_rate = DEFAULT_DRAW_RATE;
	current_draw_rate = ( (float)(DEFAULT_DRAW_RATE - MIN_DRAW_RATE) / 
						  (float)MAX_DRAW_RATE );
	XtScrollBarSetThumb(draw_rate_scroll, (1.0 - current_draw_rate),NULL);

	imagebuf.rotate_rate = 0;
	imagebuf.converted_format = FALSE;	/* XWindow format */

	/* initialize image buffer to have all dots at window center*/
	for ( n = 0 ; n < MAX_IMAGE_DOTS ; n ++) {
		imagebuf.full_scale_image[n][XPOS] = CENTER_OF_XWINDOW_X;
		imagebuf.full_scale_image[n][YPOS] = CENTER_OF_XWINDOW_Y;
	}
}

