/*******************************************************************
 * laserevent.c -- Event handlers for the laser light show demo
 * 
 * Ron Wilder -- Integrated Solutions Inc.
 * 1 Sep 1989 -- Date of Birth
 *
 * This module contains all of the event handlers required for the 
 * laser light show demo program. 
 * Events include: scroll bar motion, button pressing, and drawing
 *******************************************************************/ 

#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <stdio.h>
#include "laserxt.h"
#include "laser.h"

#define ERROR -1

/* Misc Globals */
extern int serverSock;							/* server socket */
extern ENET_CMD_PACKET		cmdbuf;				/* command packet */
extern ENET_IMAGE_PACKET	imagebuf;			/* image packet */
extern float current_draw_rate;					/* laser motion speed */
extern float current_X_gain;					/* current gain factor */
extern float current_Y_gain;					/* current gain factor */
extern boolean user_interface_test;				/* bypass laser -- test X only*/
extern boolean helpflag;						/* help mode on if TRUE */
extern boolean XY_ganged;						/* gang gain pots if TRUE*/
extern boolean laser_on_flg;					/* laser beam on if TRUE*/
extern int laser_button;						/* holds button number */
extern Widget draw_rate_scroll;					/* Draw rate scroll bar widget*/
extern Widget X_gain_scroll;					/* X gain scrollbar widget */
extern Widget Y_gain_scroll;					/* Y gain scrollbar widget */
extern Widget button_labels[];
extern BUTTON button[];
extern char *fname[];
extern FILE *f1, *open();


/*extern unsigned short current_rot_rate;*/		/* current rotation rate */
/*extern unsigned short current_position;*/		/* current image index */

extern int send_cmd_to_laser();
extern int clear_image_buf();


/******************************************************************
 * start_image_capture -- event handler
 * handles image start when mouse button is pressed inside draw
 * window.  
 *******************************************************************/


void start_image_capture(w, data, event)
Widget	w;
line_segment_data	*data;
XEvent			*event;
{

	/* temporary for now ... put in a crosshair */


	data->last_x = data->start_x = event->xbutton.x;
	data->last_y = data->start_y = event->xbutton.y;

	/* actually, both of the next two functions go into the new image button */
	/* need pixmap clear command to clear drawing area */

	XClearWindow(XtDisplay(w), XtWindow(w));
	clear_image_buf();		/* prepare for new image */

	XDrawLine (XtDisplay(w), XtWindow(w),data->gc,
		CENTER_OF_XWINDOW_X, MIN_XWINDOW_Y,
		CENTER_OF_XWINDOW_X, MAX_XWINDOW_Y);

	XDrawLine (XtDisplay(w), XtWindow(w),data->gc,
		MIN_XWINDOW_X, CENTER_OF_XWINDOW_Y,
		MAX_XWINDOW_X, CENTER_OF_XWINDOW_Y);

}

/*******************************************************************
 * draw_next_segment -- event handler
 *
 * handles image drawing when mouse is moved (with button down) 
 * inside of the draw window
 *
 *******************************************************************/

void draw_next_segment(w, data, event)
Widget	w;
line_segment_data	*data;
XEvent			*event;
{
	static char label[50];
	static Arg args[] = { { XtNlabel, (XtArgVal) label } };
	/*
	 * Update the data points
	 */

	/* throw out bad points!!! */
	if ((event -> xbutton.x != ERROR) && (event -> xbutton.y != ERROR)) { 
		data->start_x = data->last_x;
		data->start_y = data->last_y;
		data->last_x = event->xbutton.x;
		data->last_y = event->xbutton.y;

		/*
		 * Draw the new line.
		 */
	
		XDrawLine(XtDisplay(w), XtWindow(w),
			data->gc, data->start_x,
			data->start_y, data->last_x, data->last_y);
	
		if (imagebuf.image_length < MAX_IMAGE_DOTS) {
			imagebuf.full_scale_image[imagebuf.image_length][XPOS] =
				event->xbutton.x;
			imagebuf.full_scale_image[imagebuf.image_length][YPOS] =
				event->xbutton.y;
			imagebuf.image_length++;	/* point to next image position */
		}
		else {
			cmdbuf.command = NEW_IMAGE;
			cmdbuf.subcmd1 = sizeof(imagebuf.image_length) +
							 sizeof(imagebuf.draw_rate) +
							 sizeof(imagebuf.rotate_rate) +
							 sizeof(imagebuf.converted_format) +
							 imagebuf.image_length * 2 * sizeof(short);
			cmdbuf.subcmd2 = NULL;
			imagebuf.converted_format = FALSE;
			imagebuf.draw_rate = (short) (current_draw_rate * MAX_DRAW_RATE) +
								MIN_DRAW_RATE;
			imagebuf.rotate_rate = 0;	/* for now!!!!! */
			send_cmd_to_laser();
			if (!laser_on_flg) {
				printf("Turning laser beam on.\n");
				laser_on_flg = TRUE;
		
				/* set the laser button label to indicate that it's on now */	
				strcpy( label, button[laser_button].toggle_label);
				XtSetValues(button_labels[laser_button],args,XtNumber(args));
			}
		}
	}
}
	
/*******************************************************************
 * draw_rate_Scrolled -- event handler
 *
 * increments/decrements laser draw rate when scrollbar thumb is scrolled
 *
 *******************************************************************/

void draw_rate_Scrolled(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		if (call_data < 0) {
			printf("This button is used to send a lissajous pattern.\n");
			/*printf("This button is used to decrement the draw rate.\n");*/
		}
		else 
			printf("This button is used to send a new image.\n");
			/*printf("This button is used to increment the draw rate.\n");*/
	}
	else {
		if (call_data < 0) {
			printf("Currently this button sends a 2:3 lissajous to laser.\n");
			cmdbuf.command = LISSAJOUS_PATTERN;
			cmdbuf.subcmd1 = 2;
			cmdbuf.subcmd2 = 3;

			send_cmd_to_laser();
		}
		else {
			/*printf( "Draw rate scrolled by %d pixels.\n", call_data ); */
		}
	}
}

/*******************************************************************
 * draw_rate_Thumbed -- event handler
 *
 * sends draw rate factor to laser when scrollbar thumb is moved
 *******************************************************************/


void draw_rate_Thumbed(w, closure, top)
    Widget w;
    caddr_t closure;
    float top;
{
	float	last_draw_rate;

	if (helpflag == TRUE) {
		printf("This scroll bar thumb is used to set a new draw rate.\n");
	}
	else {

		last_draw_rate = current_draw_rate;
		current_draw_rate = (1.0 - top);

		if (current_draw_rate != last_draw_rate) {
			cmdbuf.command = NEW_DRAW_RATE;
			cmdbuf.subcmd1 = (short) (current_draw_rate * MAX_DRAW_RATE) +
							MIN_DRAW_RATE;
			cmdbuf.subcmd2 = NULL;
			printf("Current draw rate is %f%%.  Value sent is %d\n",
					current_draw_rate, cmdbuf.subcmd1);
	
			send_cmd_to_laser();
		}
	}
}

/*******************************************************************
 * X_gain_Scrolled -- event handler
 *
 * increments/decrements laser X gain when scrollbar thumb is scrolled
 *******************************************************************/

void X_gain_Scrolled(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		if (call_data < 0)
			printf("This button is used to decrement the X gain factor.\n");
		else 
			printf("This button is used to increment the X gain factor.\n");
	}
	else {
	}
}

/*******************************************************************
 * X_gain_Thumbed -- event handler
 *
 * sends X gain factor to laser when scrollbar thumb is moved
 *******************************************************************/

void X_gain_Thumbed(w, closure, top)
    Widget w;
    caddr_t closure;
    float top;
{
	float	last_X_gain;

	if (helpflag == TRUE) {
		printf("This scroll bar thumb is used to set a new X gain factor.\n");
	}
	else {
		last_X_gain = current_X_gain;
		current_X_gain = 1.0 - top;

		if (current_X_gain != last_X_gain) {
			printf( "X gain thumbed to %f%%\n", current_X_gain);
			cmdbuf.command = SCALE_IMAGE;
			cmdbuf.subcmd1 = (short)(current_X_gain * MAX_GAIN);
			if (XY_ganged){
				current_Y_gain = current_X_gain;
				cmdbuf.subcmd2 = cmdbuf.subcmd1;
				XtScrollBarSetThumb(Y_gain_scroll,top,NULL);
			}
			else cmdbuf.subcmd2 = (short)(current_Y_gain * MAX_GAIN);
	
			printf("x: %d  y:%d\n",cmdbuf.subcmd1, cmdbuf.subcmd2);
	
			send_cmd_to_laser();
		}
	}
}

/*******************************************************************
 * Y_gain_Scrolled -- event handler
 *
 * increments/decrements laser Y gain when scrollbar thumb is scrolled
 *
 *******************************************************************/

void Y_gain_Scrolled(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		if (call_data < 0)
			printf("This button is used to decrement the Y gain factor.\n");
		else 
			printf("This button is used to increment the Y gain factor.\n");
	}
	else {

		printf( "Y gain scrolled by %d pixels.\n", call_data );
	}
}

/*******************************************************************
 * Y_gain_Thumbed -- event handler
 * sends Y gain factor to laser when scrollbar thumb is moved
 *******************************************************************/

void Y_gain_Thumbed(w, closure, top)
    Widget w;
    caddr_t closure;
    float top;
{
	float	last_Y_gain;
	if (helpflag == TRUE) {
		printf("This scroll bar thumb is used to set a new Y gain factor.\n");
	}
	else {
		last_Y_gain = current_Y_gain;
		current_Y_gain = 1.0 - top;

		if (current_Y_gain != last_Y_gain) {

			printf( "Y gain thumbed to %f%%\n", current_Y_gain);
			cmdbuf.command = SCALE_IMAGE;
			cmdbuf.subcmd2 = (short)(current_Y_gain * MAX_GAIN);
			if (XY_ganged){
				cmdbuf.subcmd1 = cmdbuf.subcmd2;
				current_X_gain = current_Y_gain;
				XtScrollBarSetThumb(X_gain_scroll,top,NULL);
			}
			else cmdbuf.subcmd1 = (short)(current_X_gain * MAX_GAIN);
			printf("x:%d  y:%d\n",cmdbuf.subcmd1, cmdbuf.subcmd2);
	
			send_cmd_to_laser();
		}
	}
}


/*******************************************************************
 * help_key -- event handler
 * displays helpful information about program and various buttons
 *******************************************************************/

void help_key(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		helpflag = FALSE;
	    printf( "Exiting help...\n" );
		/* put some code here to clean up help */
		/* ie. close pop up window */
	}
	else{
		printf( "Select any button/scrollbar to get help. To exit, select HELP again.\n");
		helpflag = TRUE;
		/* put code here to open help database and a pop up window */
		/* for help info */
	}
}

/*******************************************************************
 * XY_gain_gang -- event handler
 * gangs the X and Y gain pots
 *******************************************************************/

void XY_gain_gang(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {		/* print help message */
		printf("This key is used to gang the X and Y gain pots together.\n");
	}
	else {	/* do the command asked for */

		if (!XY_ganged) { /* only gang them if they are not ganged already. */
			printf( "Ganging the X and Y gain pots together.\n");
			XY_ganged = TRUE;

			/* move Y gain thumb to same as X gain thumb */
			current_Y_gain = current_X_gain;
			XtScrollBarSetThumb(Y_gain_scroll,
							    (float)(1.0 - current_Y_gain), NULL);

			/* load up packet to send new gain to target */
			cmdbuf.command = SCALE_IMAGE;
			cmdbuf.subcmd1 = (short)(current_X_gain * MAX_GAIN);
			cmdbuf.subcmd2 = cmdbuf.subcmd1;

			send_cmd_to_laser();
		}
	}
}

/*******************************************************************
 * XY_gain_separate -- event handler
 * separates the X and Y gain pots
 *******************************************************************/

void XY_gain_separate(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {		/* print help message */
		printf("This key is used to separate the X and Y gain pots.\n");
	}
	else {	/* do the command asked for */

		if (XY_ganged) { /* only separate them if they are currently ganged.*/
			printf( "Separating the X and Y gain pots.\n");
			XY_ganged = FALSE;
		}
	}
}

/*******************************************************************
 * modified to write image to file xxx
 * quit_key -- event handler
 * exits program and closes anything necessary 
 *******************************************************************/

void quit_key(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	int n;
	
	if (helpflag == TRUE) {
		printf("This button writes the image to the disk file \'xxx\;\n");
	}
	else { /* put button handler here */

		/*	get_f(fname); */

		/* open file for write */
		if ((f1 = open(fname[0],1)) == (FILE *) ERROR ) {
			printf("Can\'t create file %s\n",fname[0]);
			return;
		}

		/* get actual image byte count */
		n =  sizeof(imagebuf.image_length) +
			 sizeof(imagebuf.draw_rate) +
			 sizeof(imagebuf.rotate_rate) +
			 sizeof(imagebuf.converted_format) +
			 imagebuf.image_length * 2 * sizeof(short);
		
		printf( "Writing %d bytes to the image file \'%s\'.\n", n, fname[0]);

		if (write(f1, (char *)&imagebuf, n) != n) {
			printf("Trouble writing the file \'%s\'. #bytes= %d\n", fname[0]);
			return;
		}

		if (close(f1) == ERROR) {
			printf("Can\'t close file.\n");
			return;
		}
		printf("File successfully written\n");
	}
}

/*******************************************************************
 * save_image -- event handler
 * saves image to disk 
 *******************************************************************/

void save_image(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	int n;
	
	if (helpflag == TRUE) {
		printf("This button saves the current image to the disk.\n");
	}
	else { /* put button handler here */

		/*	get_f(fname); */

		/* open file for write */
		if ((f1 = open(fname[0],1)) == (FILE *) ERROR ) {
			printf("Can\'t create file %s\n",fname[0]);
			return;
		}

		/* get actual image byte count */
		n =  sizeof(imagebuf.image_length) +
			 sizeof(imagebuf.draw_rate) +
			 sizeof(imagebuf.rotate_rate) +
			 sizeof(imagebuf.converted_format) +
			 imagebuf.image_length * 2 * sizeof(short);
		
		printf( "Writing %d bytes to the image file \'%s\'.\n", n, fname[0]);

		if (write(f1, (char *)&imagebuf, n) != n) {
			printf("Trouble writing the file \'%s\'. #bytes= %d\n", fname[0]);
			return;
		}

		if (close(f1) == ERROR) {
			printf("Can\'t close file.\n");
			return;
		}
		printf("File successfully written\n");
	}
}

/*******************************************************************
 * quit -- event handler
 * exits program and closes anything necessary 
 *******************************************************************/

void quit_but_handler(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		printf("This key is used to exit the program.\n");
	}
	else {

		cmdbuf.command = END_PROGRAM;
		cmdbuf.subcmd1 = END_PROGRAM;
		cmdbuf.subcmd2 = END_PROGRAM;
		send_cmd_to_laser();
		if (!user_interface_test) {
			printf("Pausing to let connections close.\n");
			sleep(2); /* try to pause and let the message get across */
			shutdown(serverSock,2);	/* force shutdown on socket */
			close(serverSock);		/* close connection to target */
			printf("Socket %d to laser target closed.\n",serverSock);
		}
		printf( "Exiting the laser program.\n" );
		exit(0);
	}
}

/*******************************************************************
 * redisplay -- event handler
 * redisplays the image upon an exposure 
 *******************************************************************/

void redisplay(w,data)
    Widget w;
	line_segment_data	*data;
{
	int n;

	XClearWindow(XtDisplay(w), XtWindow(w));

	XDrawLine (XtDisplay(w), XtWindow(w),data->gc,
		CENTER_OF_XWINDOW_X, MIN_XWINDOW_Y,
		CENTER_OF_XWINDOW_X, MAX_XWINDOW_Y);

	XDrawLine (XtDisplay(w), XtWindow(w),data->gc,
		MIN_XWINDOW_X, CENTER_OF_XWINDOW_Y,
		MAX_XWINDOW_X, CENTER_OF_XWINDOW_Y);

	for (n = 0 ; n < imagebuf.image_length-1; n++) {
		XDrawLine(XtDisplay(w), XtWindow(w),
			data->gc, 
			imagebuf.full_scale_image[n][XPOS], 
			imagebuf.full_scale_image[n][YPOS], 
			imagebuf.full_scale_image[n+1][XPOS], 
			imagebuf.full_scale_image[n+1][YPOS]); 
	}
}

/*******************************************************************
 * invert_X -- event handler
 * inverts X axis of picture 
 *******************************************************************/

void invert_X(w, closure, call_data)				/* inverts X axis */

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		printf("This button is used to invert the X axis.\n");
	}
	else {

		cmdbuf.command = INVERT_IMAGE_X;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		/* put in code to do same in display area */
		printf("Inverting X axis.\n");
	}
}

/*******************************************************************
 * invert_Y -- event handler
 * inverts Y axis of picture 
 *******************************************************************/

void invert_Y(w, closure, call_data)				/* inverts Y axis */

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		printf("This button is used to invert the Y axis.\n");
	}
	else {

		cmdbuf.command = INVERT_IMAGE_Y;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		/* put in code to do same in display area */
		printf("Inverting Y axis.\n");
	}
}

/*******************************************************************
 * laser_on_off -- event handler
 * turns the laser beam on or off
 *******************************************************************/

void laser_on_off(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	static char label[BUTTON_BOX_WIDTH - (2 * INSIDE_BORDER)];
	static Arg args[] = { {XtNlabel, (XtArgVal)label} };
	
	if (helpflag == TRUE) {
		printf("This button is used to turn the laser beam on or off.\n");
	}
	else {


		if (!laser_on_flg) {
			printf("Turning laser beam on.\n");
			cmdbuf.command = LASER_ON;
			cmdbuf.subcmd1 = NULL;
			cmdbuf.subcmd2 = NULL;
			laser_on_flg = TRUE;
			send_cmd_to_laser();
		
			/* set the button label to indicate that  it's on now */	
			/* assumes that call_data was initialized to the button number */

			strcpy( label, button[laser_button].toggle_label);
		printf("label is %s\n",label);
		printf("laser button is number %d\n",laser_button);
		printf("args is %s\n",args[0]);
		printf("num args is %d\n",XtNumber(args));
			XtSetValues(button_labels[laser_button],args,XtNumber(args));

		}
		else  {
			printf("Turning laser beam off.\n");

			cmdbuf.command = LASER_OFF;
			cmdbuf.subcmd1 = NULL;
			cmdbuf.subcmd2 = NULL;
			laser_on_flg = FALSE;
			send_cmd_to_laser();
		
			/* set the button label to indicate that it's off now */	
			/* assumes that call_data was initialized to the button number */

			strcpy( label, button[laser_button].label);
			XtSetValues(button_labels[laser_button],args,XtNumber(args));
		}
	}
}

/*******************************************************************
 * spare -- event handler
 * your stuff goes here
 *******************************************************************/

void spare(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		printf("This button does...\n");
	}
	else {

		cmdbuf.command = NULL;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		printf("Doing...\n");
	}
}

/*******************************************************************
 * full_scale -- event handler
 * sets laser up to back to full scale image display
 *******************************************************************/

void full_scale(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	if (helpflag == TRUE) {
		printf("This button is used to display the image in full scale.\n");
	}
	else {

		cmdbuf.command = GO_FULL_SCALE;
		cmdbuf.subcmd1 = NULL;
		cmdbuf.subcmd2 = NULL;
		send_cmd_to_laser();

		/* set local indicators to full scale */
		current_Y_gain = 1.0;	
		current_X_gain = 1.0;

		XtScrollBarSetThumb(X_gain_scroll,(float)(1.0 - current_X_gain),NULL);
		XtScrollBarSetThumb(Y_gain_scroll,(float)(1.0 - current_Y_gain),NULL);

		printf("Going back to full gain on both axis.\n");
	}
}

/*******************************************************************
 * send_image -- event handler
 * sends current image to the laser target machine 
 *******************************************************************/

void send_image(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	static char label[50];
	static Arg args[] = { { XtNlabel, (XtArgVal) label } };
	if (helpflag == TRUE) {
		printf("This button is used to send a new image to the laser.\n");
	}
	else {
		/* don't send image if zero length buffer */
		if (imagebuf.image_length != 0) {
			cmdbuf.command = NEW_IMAGE;
			cmdbuf.subcmd1 = sizeof(imagebuf.image_length) +
							 sizeof(imagebuf.draw_rate) +
							 sizeof(imagebuf.rotate_rate) +
							 sizeof(imagebuf.converted_format) +
							 imagebuf.image_length * 2 * sizeof(short);
			cmdbuf.subcmd2 = NULL;
			imagebuf.converted_format = FALSE;
			imagebuf.draw_rate = (short) (current_draw_rate * MAX_DRAW_RATE) +
								MIN_DRAW_RATE;
			imagebuf.rotate_rate = 0;	/* for now!!!!! */

			/* set local indicators to full scale */
			current_Y_gain = 1.0;	
			current_X_gain = 1.0;

			XtScrollBarSetThumb(X_gain_scroll,(float)(1.0 -
													current_X_gain),NULL);
			XtScrollBarSetThumb(Y_gain_scroll,(float)(1.0 - 
													current_Y_gain),NULL);

			if (!laser_on_flg) {
				printf("Turning laser beam on.\n");
				laser_on_flg = TRUE;
		
				/* set the laser button label to indicate that it's on now */	
				strcpy(label, button[laser_button].toggle_label);
				XtSetValues(button_labels[laser_button],args,XtNumber(args));
			}

			printf("Sending a new image to the laser.\n");
			send_cmd_to_laser();
			/*	clear_image_buf(); */
	
			/* put in code to do same in display area */
				printf("Sending a new image to the laser.\n");
		}
	}
}

/*******************************************************************
 * lissajous -- event handler
 * sends a lissajous pattern command to the laser target
 *******************************************************************/

void lissajous(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
	static char label[50];
	static Arg args[] = { { XtNlabel, (XtArgVal) label } };
	if (helpflag == TRUE) {
		printf("This button is used to send a lissajous command to the laser target.\n");
	}
	else {

		cmdbuf.command = LISSAJOUS_PATTERN;
	/*for now put in hard values, later take them from dialog or menu box */
		cmdbuf.subcmd1 = 3;
		cmdbuf.subcmd2 = 2;
		send_cmd_to_laser();

		if (!laser_on_flg) {
			printf("Turning laser beam on.\n");
			laser_on_flg = TRUE;
	
			/* set the laser button label to indicate that it's on now */	
			strcpy( label, button[laser_button].toggle_label);
			XtSetValues(button_labels[laser_button],args,XtNumber(args));
		}

		printf("Sending a lissajous pattern command to the laser target.\n");
	}
}

/*******************************************************************
 * clear_image -- event handler
 * clears the display and the image data structure 
 *******************************************************************/
extern	Widget	draw;

void clear_image(w, closure, call_data)

    Widget w;
    caddr_t closure;
    int call_data;
{
/*
	Widget *widptr;
	widptr = &(Widget)draw;
*/
	if (helpflag == TRUE) {
		printf("This button is used to clear the drawing area.\n");
	}
	else {

		/* figure out why following command has trouble */
		XClearWindow(XtDisplay(draw), XtWindow(draw));
		/*XClearWindow(XtDisplay(*draw), XtWindow(*draw));*/

		/*XClearWindow(XtDisplay((Widget)closure), XtWindow((Widget)closure);*/
		clear_image_buf();		/* prepare for new image */
		printf("Clearing the drawing area.\n");
	}
}

/*******************************************************************
 * canned_image -- event handler
 * commands laser target to display a canned image
 *******************************************************************/

void canned_image(w, closure, call_data)
    Widget w;
    caddr_t closure;
    int call_data;
{
	static char label[50];
	static Arg args[] = { { XtNlabel, (XtArgVal) label } };
	if (helpflag == TRUE) {
		printf("This button is used to send a canned image command to the laser target.\n");
	}
	else {

		cmdbuf.command = CANNED_IMAGE;
		cmdbuf.subcmd1 = 0;		/* for now just display image 0 */
		cmdbuf.subcmd2 = NULL;
		printf( "Sending a canned image command to laser target.\n" );
		send_cmd_to_laser();

		if (!laser_on_flg) {
			printf("Turning laser beam on.\n");
			laser_on_flg = TRUE;
	
			/* set the laser button label to indicate that it's on now */	
			strcpy( label, button[laser_button].toggle_label);
			XtSetValues(button_labels[laser_button],args,XtNumber(args));
		}
	}
}

/* the following isn't really an event but it seemed like a good place for it*/

/***************************************************************************/
/**************************** clear_image_buf ******************************/
/***************************************************************************/

int
clear_image_buf()
{
	int n;		/* loop counter */	
	/* initialize new image length to zero */
	imagebuf.image_length = 0;
	imagebuf.draw_rate = DEFAULT_DRAW_RATE;
	current_draw_rate = ( (float)(DEFAULT_DRAW_RATE - MIN_DRAW_RATE) / 
						  (float)MAX_DRAW_RATE );
	XtScrollBarSetThumb(draw_rate_scroll, (1.0 - current_draw_rate),NULL);

	imagebuf.rotate_rate = 0;
	imagebuf.converted_format = FALSE;	/* XWindow format */

	/* initialize image buffer to have all dots at window center*/
	for ( n = 0 ; n < MAX_IMAGE_DOTS ; n ++) {
		imagebuf.full_scale_image[n][XPOS] = CENTER_OF_XWINDOW_X;
		imagebuf.full_scale_image[n][YPOS] = CENTER_OF_XWINDOW_Y;
	}
}

