#ifdef SYSV
#include <X/mitcopyright.h>
#include <string.h>
#include <time.h>
#else
#include <X/mit-copyright.h>
#include <strings.h>
#include <sys/time.h>
#endif
#include <stdio.h>
#include <signal.h>
#include <X/Xlib.h>
#include "bell.bm"
#include "xclock.cursor"
#include "slash.cursor"
#include "alarm.cursor"

extern char *getenv();
extern long time();
extern unsigned alarm(), sleep();
extern void exit();

#define PI 3.141592
#define TRUE 1
#define FALSE 0
#define DEF_INTERVAL 60
#define MIN_ANL_SIZE 75
#define DEF_FONT "6x10"
#define DEF_GEOMETRY "=100x100-0+0"
#define DEF_ANALOG_PAD 2
#define DEF_DIGITAL_PAD 3
#define DEF_BORDER_WIDTH 2

Window AnalogWindow, IconWindow, DateWindow, DayWindow, BellWindow;
Cursor ClockCursor, DateCursor, BellCursor;
Font f;
FontInfo fi;

char *geometry, *FontName, *DspName = NULL;
int rv, DigitalPad, AnalogPad, BorderWidth;
unsigned interval;

/*
 * Miscellany for use in drawing the analog clock.
 */
Vertex SegBuff[128];
Vertex *SegBuffPtr;
int NumSegs = 0;
int radius, XCenter, YCenter, sec_hand_len, min_hand_len, hour_hand_len;

int anl_width, anl_height;
int FgPixel = BlackPixel, BgPixel = WhitePixel;
short alarm_on = FALSE, ringing = FALSE, auto_alarm_off,
      Iconified = FALSE, AlarmSetMode = FALSE;
static struct {
    int hour, min;
} al_tm;
struct tm *tm;

static char *MonthName[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
static char *DayName[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};

handler () {
    (void) signal (SIGALRM, handler);
    ShowTime ();
    (void) alarm (ringing ? 1 : interval);
}

main (argc, argv)
int argc;
char **argv;
{
    ReadDefaults (*argv);
    ParseOptions (argc, argv);
    XWinSetup (*argv);
    SegBuffPtr = SegBuff;
    SetAnalogSize();
    DrawClockFace(sec_hand_len, radius);
    (void) signal (SIGALRM, handler);
    ShowTime ();
    (void) alarm (interval == 1 ? interval : 60 - tm->tm_sec);

    while (1) {
	XButtonEvent event;

	XNextEvent (&event);
	if (event.window == AnalogWindow) {
	    XExposeEvent *wevent = (XExposeEvent *) &event;
	    Iconified = FALSE;
	    if (wevent->width != anl_width || wevent->height != anl_height) {
		anl_width = wevent->width;
		anl_height = wevent->height;
		SetAnalogSize();
	    }
	    DrawClockFace(sec_hand_len, radius);
	    ShowTime ();
	}
	else if (event.window == IconWindow && event.type == ExposeWindow) {
	    Iconified = TRUE;
	    ShowTime ();
	}
	else if (event.window == BellWindow)
	    switch (event.type) {
	    case ButtonPressed:
		ringing = alarm_on = FALSE;
		XUnmapWindow (BellWindow);
		break;
	    case ExposeWindow:
		XBitmapBitsPut(BellWindow, 0, 0, bell_width, bell_height,
		    bell_bits, FgPixel, BgPixel, 0, GXcopy, AllPlanes);
		break;
	    }
	else if (event.type == EnterWindow) {
	    XClear (DateWindow);
	    AlarmSetMode = TRUE;
	    set_alarm ();
	    AlarmSetMode = FALSE;
	    ShowDate ();
	}
    }
}

/*
 * Get the defaults
 */
ReadDefaults (pname)
char *pname;
{
    char *def_value;

    if ((def_value = XGetDefault (pname, "BodyFont")) != NULL)
	FontName = def_value;
    else FontName = DEF_FONT;
    if ((def_value = XGetDefault (pname, "ReverseVideo")) != NULL &&
		strcmp (def_value, "on") == 0)
	rv = TRUE;
    else rv = FALSE;
    if ((def_value = XGetDefault (pname, "AnalogPad")) != NULL)
	AnalogPad = atoi (def_value);
    else AnalogPad = DEF_ANALOG_PAD;
    if ((def_value = XGetDefault (pname, "DigitalPad")) != NULL)
	DigitalPad = atoi (def_value);
    else DigitalPad = DEF_DIGITAL_PAD;
    if ((def_value = XGetDefault (pname, "BorderWidth")) != NULL)
	BorderWidth = atoi (def_value);
    else BorderWidth = DEF_BORDER_WIDTH;
    if ((def_value = XGetDefault (pname, "Update")) != NULL)
	interval = atoi (def_value);
    else interval = DEF_INTERVAL;
    if ((def_value = XGetDefault (pname, "AlarmTime")) != NULL)
	AlarmTime (def_value);
    if ((def_value = XGetDefault (pname, "ManualAlarm")) != NULL &&
		strcmp (def_value, "on") == 0)
	auto_alarm_off = FALSE;
    else auto_alarm_off = TRUE;
}

/*
 * Parse command line options.
 */
ParseOptions (argc, argv)
int argc;
char **argv;
{
    for (argv++; --argc; argv++) {
	if (**argv == '=') {
	    geometry = *argv;
	    continue;
	}
	if (strchr (*argv, ':') != NULL) {
	    DspName = *argv;
	    continue;
	}
	if (strcmp (*argv, "-rv") == 0 || strcmp (*argv, "-reverse") == 0) {
	    rv = TRUE;
	    continue;
	}
        if (strcmp (*argv, "-ma") == 0 || strcmp (*argv, "-manualalarm") == 0) {
	    auto_alarm_off = FALSE;
	    continue;
	}
	if (**argv != '-' || argc == 1)
	    continue;
	if (strcmp (*argv, "-fn") == 0 || strcmp (*argv, "-font") == 0) {
	    FontName = *++argv;
	    argc--;
	    continue;
	}
	if (strcmp (*argv, "-update") == 0 || strcmp (*argv, "-u") == 0) {
	    interval = atoi (*++argv);
	    argc--;
	    continue;
	}
	if (strcmp (*argv, "-bw") == 0 || strcmp (*argv, "-border") == 0) {
	    BorderWidth = atoi (*++argv);
	    argc--;
	    continue;
	}
	if (strcmp (*argv, "-ap") == 0 || strcmp (*argv, "-analogpad") == 0) {
	    AnalogPad = atoi (*++argv);
	    argc--;
	    continue;
	}
	if (strcmp (*argv, "-dp") == 0 || strcmp (*argv, "-digitalpad") == 0) {
	    DigitalPad = atoi (*++argv);
	    argc--;
	    continue;
	}
	if (strcmp (*argv, "-alarm") == 0) {
	    AlarmTime (*++argv);
	    argc--;
	    continue;
	}
    }
    if (rv) {
	FgPixel = WhitePixel;
	BgPixel = BlackPixel;
    }
    if (interval < 1)
	interval = 1;
}

AlarmTime (argv)
char *argv;
{
    al_tm.hour = atoi (argv);
    al_tm.min = atoi (strchr (argv, ':') + 1);
    alarm_on = TRUE;
}

/*
 * Set up all the various parts of the windowing system.
 */
XWinSetup (pname)
char *pname;
{
    OpaqueFrame frame;
    WindowInfo winfo;
    int IconWidth = interval == 1 ? 20 : 17;

    if (XOpenDisplay(DspName) == NULL || (f = XGetFont (FontName)) == 0)
	exit (1);
    XQueryFont (f, &fi);
    frame.bdrwidth = BorderWidth;
    frame.border = rv ? WhitePixmap : BlackPixmap;
    frame.background = rv ? BlackPixmap : WhitePixmap;
    AnalogWindow = XCreate ("Xclock", pname, geometry, DEF_GEOMETRY, &frame,
			MIN_ANL_SIZE, MIN_ANL_SIZE);
    XQueryWindow (AnalogWindow, &winfo);
    anl_width = winfo.width;
    anl_height = winfo.height;
    IconWindow = XCreateWindow (RootWindow, 0, 0,
		fi.width * IconWidth + 2 * DigitalPad,
		fi.height + 2 * DigitalPad, BorderWidth,
		rv ? WhitePixmap : BlackPixmap,
		rv ? BlackPixmap : WhitePixmap);
    DayWindow = XCreateTransparency (AnalogWindow,
		anl_width/2 - fi.width*3/2, anl_height/4 - fi.height/2,
		fi.width*3, fi.height);
    DateWindow = XCreateTransparency (AnalogWindow,
		anl_width/2 - fi.width*6/2,
		anl_height*3/4 - fi.height/2,
		fi.width*6, fi.height);
    BellWindow = XCreateWindow (AnalogWindow, 0, 0,
					  bell_width, bell_height, 0,
					  rv ? WhitePixmap : BlackPixmap,
					  rv ? WhitePixmap : BlackPixmap);
    XSetIconWindow(AnalogWindow,IconWindow);
    ClockCursor = XCreateCursor (xclock_width, xclock_height, xclock_bits,
	xclock_bits, xclock_x_hot, xclock_y_hot, FgPixel, BgPixel, GXcopy);
    XDefineCursor (AnalogWindow, ClockCursor);
    XDefineCursor (IconWindow, ClockCursor);
    DateCursor = XCreateCursor (alarm_width, alarm_height, alarm_bits,
	alarm_bits, alarm_x_hot, alarm_y_hot, BgPixel, FgPixel, GXcopy);
    XDefineCursor (DateWindow, DateCursor);
    BellCursor = XCreateCursor (slash_width, slash_height, slash_bits,
	slash_bits, slash_x_hot, slash_y_hot, FgPixel, BgPixel, GXcopy);
    XDefineCursor (BellWindow, BellCursor);
    XMapWindow (AnalogWindow);
    XMapWindow (DateWindow);
    XMapWindow (DayWindow);
    XSelectInput (AnalogWindow, ExposeWindow);
    XSelectInput (IconWindow, ExposeWindow|ButtonPressed|ButtonReleased);
    XSelectInput (DateWindow,
			EnterWindow|LeaveWindow|ButtonPressed|ButtonReleased);
    XSelectInput (BellWindow, ExposeWindow|ButtonPressed);
    if (alarm_on == TRUE)
	XMapWindow (BellWindow);
}

/*
 *  Draw the clock face (every fifth tick-mark is longer than the others).
 */
DrawClockFace(second_hand, Radius)
int second_hand;
int Radius;
{
	register int i;
	register int delta = (Radius - second_hand) / 3;
	
	XClear(AnalogWindow);
	for (i = 0; i < 60; i++)
		if ((i % 5) == 0)
			DrawHand(second_hand, Radius, ((double) i)/60.);
		else DrawHand((Radius - delta), Radius, ((double) i)/60.);
	Sync(FgPixel);
}

/*
 * This routine synchronizes the segment buffer contents with the
 * VS100 screen.  In effect, it flushes the buffer contents to the
 * screen.
 */
Sync(color)
int color;
{
	/*
	 * Call the X draw curve routine.
	 */
	XDraw(AnalogWindow, SegBuff, NumSegs, 1, 1, color, GXcopy, AllPlanes);

	/*
	 * Reset the segment buffer pointer and the segment counter.
	 */
	SegBuffPtr = SegBuff;
	NumSegs = 0;
}

/*
 * DrawHand - Draws (or erases) a hand.  Fraction_of_a_circle is a
 * fraction between 0 and 1 (inclusive) indicating how far around the
 * circle (clockwise) high noon.
 *
 * Blank_length is the distance from the center which the hand begins
 * (logically 0, but I have the feature, I might as well use it).
 * length is the maximum length of the hand.
 *
 * The blank_length feature is because I wanted to draw tick-marks around the
 * circle (for seconds).  The obvious means of drawing lines from the center
 * to the perimeter, then erasing all but the outside most pixels doesn't
 * work because of round-off error (sigh).
 */
DrawHand(blank_length, length, fraction_of_a_circle)
int blank_length, length;
double fraction_of_a_circle;
{

	extern double cos(), sin();
	double angle;

	/*
	 *  A full circle is 2 PI radians.
	 *  Angles are measured from 6 o'clock, not noon (or so it seems).
	 *  Thus, when fraction_of_a_circle is 0, we want angle to be PI,
	 *  and when fraction_of_a_circle is 1, we want angle to be 3 PI.
	 *
	 *  Also, angles increase counter-clockwise, not clockwise, so we
	 *  throw in a factor of -1 so our clock doesn't run backwards.
	 */
	angle = (-2 * PI * fraction_of_a_circle) + PI;

	/*
	 * Add the move instruction to the segment buffer and increment
	 * the "next" pointer.
	 */
	SegBuffPtr->x = XCenter + (int)((float)blank_length * sin(angle));
	SegBuffPtr->y = YCenter + (int)((float)blank_length * cos(angle));
	SegBuffPtr->flags = VertexDontDraw;
	SegBuffPtr++;
	NumSegs++;

	/*
	 * Add the new point to the buffer and increment the "next" pointer.
	 */
	SegBuffPtr->x = XCenter + (int)((float)length * sin(angle));
	SegBuffPtr->y = YCenter + (int)((float)length * cos(angle));
	SegBuffPtr->flags = VertexDrawLastPoint;
	SegBuffPtr++;
	NumSegs++;
}

set_alarm () {
    short minute_set, hour_set, buttons;
    int i_ignore;
    Window w_ignore;
    XButtonEvent event;

    XQueryMouseButtons (RootWindow, &i_ignore, &i_ignore, &w_ignore, &buttons);
    if (buttons & LeftMask)
	ActivateAlarm ();
    hour_set = buttons & MiddleMask;
    minute_set = buttons & RightMask;
    ShowAlarmTime ();
    do {
	if (XPending() || (hour_set == minute_set)) {
	    XNextEvent (&event);
	    if (event.type == ButtonPressed || event.type == ButtonReleased)
		switch (event.detail & ValueMask) {
		case LeftButton:
		    ActivateAlarm();
		    break;
		case MiddleButton:
		    hour_set = (event.type == ButtonPressed);
		    break;
		case RightButton:
		    minute_set = (event.type == ButtonPressed);
		    break;
		}
	}

	if (hour_set && minute_set)
	    al_tm.min = al_tm.hour = 0;
	else if (minute_set && ++al_tm.min > 59)
	    al_tm.min = 0;
	else if (hour_set && ++al_tm.hour > 23)
	    al_tm.hour = 0;
	ShowAlarmTime ();

	if (!XPending() && (minute_set != hour_set)) {
	    XFlush ();
	    (void) sleep (1);
	}

    } while (event.type != LeaveWindow);
}

ActivateAlarm () {
    alarm_on = TRUE;
    XMapWindow (BellWindow);
    XBitmapBitsPut(BellWindow, 0, 0, bell_width, bell_height,
		    bell_bits, FgPixel, BgPixel, 0, GXcopy, AllPlanes);
}

ShowTime () {
    long clock;

    clock = time ((long *) 0);
    tm = localtime (&clock);
    if (Iconified)
	ShowIcon ();
    else ShowHands ();
    if (ringing == TRUE) {
	if (auto_alarm_off == TRUE && tm->tm_min != al_tm.min) {
	    ringing = FALSE;
	    XUnmapWindow (BellWindow);
	}
	else RingAlarm ();
    }
    else if (alarm_on && tm->tm_hour == al_tm.hour && tm->tm_min == al_tm.min)
	ringing = TRUE;
    XFlush ();
}

RingAlarm () {
    if (tm->tm_sec & 1)
	XBitmapBitsPut(BellWindow, 0, 0, bell_width, bell_height,
		    bell_bits, FgPixel, BgPixel, 0, GXcopy, AllPlanes);
    else XBitmapBitsPut(BellWindow, 0, 0, bell_width, bell_height,
		    bell_bits, BgPixel, FgPixel, 0, GXcopy, AllPlanes);
    XFeep (0);
}

ShowHands () {
    static struct tm old_time;

    if (interval == 1)
	DrawHand(1, sec_hand_len, ((double) old_time.tm_sec)/60.0);
    if (old_time.tm_min != tm->tm_min) {
	DrawHand(1, min_hand_len, ((double) old_time.tm_min)/60.0);
	DrawHand(1, hour_hand_len, (((double)old_time.tm_hour) + 
			(((double)old_time.tm_min)/60.0))/12.0);
    }
    Sync(BgPixel);
    if (AlarmSetMode)
	ShowAlarmTime ();
    else ShowDate ();
    ShowDay ();
    old_time = *tm;
    if (interval == 1)
	DrawHand(1, sec_hand_len, ((double) tm->tm_sec)/60.0);
    DrawHand(1, min_hand_len, ((double) tm->tm_min)/60.0);
    DrawHand(1, hour_hand_len, (((double)tm->tm_hour) + 
			(((double)tm->tm_min)/60.0))/12.0);
    Sync(FgPixel);
}

ShowIcon () {
    char buf[21];

    if (interval == 1)
	(void) sprintf (buf, "%s, %d %s %2d:%.2d:%.2d",
		DayName[tm->tm_wday], tm->tm_mday, MonthName[tm->tm_mon],
		tm->tm_hour, tm->tm_min, tm->tm_sec);
    else (void) sprintf (buf, "%s, %d %s %2d:%.2d",
		DayName[tm->tm_wday], tm->tm_mday,
		MonthName[tm->tm_mon], tm->tm_hour, tm->tm_min);
    XText (IconWindow,DigitalPad,DigitalPad,buf,strlen(buf),f,FgPixel,BgPixel);
}

ShowDate () {
    char buf[7];

    (void) sprintf (buf, "%2d %s", tm->tm_mday, MonthName[tm->tm_mon]);
    XText (DateWindow, 0, 0, buf, 6, f, FgPixel, BgPixel);
}

ShowDay () {
    XText (DayWindow, 0, 0, DayName[tm->tm_wday], 3, f, FgPixel, BgPixel);
}

ShowAlarmTime () {
    char buf[6];

    (void) sprintf (buf, "%2d:%.2d", al_tm.hour, al_tm.min);
    if (Iconified)
	XText (IconWindow, DigitalPad, DigitalPad, buf, 5, f, BgPixel, FgPixel);
    else XText (DateWindow, 0, 0, buf, 5, f, BgPixel, FgPixel);
}

SetAnalogSize () {
    int face_size;

    XCenter = anl_width / 2;
    YCenter = anl_height / 2;
    face_size = anl_width > anl_height ? anl_height : anl_width;
    radius = (face_size - AnalogPad) / 2;
    sec_hand_len = 90 * radius / 100;
    min_hand_len = 70 * radius / 100;
    hour_hand_len = 40 * radius / 100;
    XMoveWindow (DateWindow, anl_width/2 - fi.width*6/2,
				anl_height*3/4 - fi.height/2);
    XMoveWindow (DayWindow, anl_width/2 - fi.width*3/2,
				anl_height/4 - fi.height/2);
}
