/* sysALib.s - Motorola MVME-147 system dependent assembly language routines */

/*
modification history
--------------------
01c,07sep88,gae  documentation.
01b,26aug88,gae  documentation.
01a,03may88,dfm  written by modifying 01f mv133a version.
*/

/*
DESCRIPTION
This module contains system-dependent routines which must be written
in assembly language.  They perform functions such as locking and
unlocking interrupts, trapping to the ROM monitor, etc.

The first routine in this module is the system start-up code.
It is the first code executed after booting.

This module MUST be the first specified on the "ld" command that is used
to build the system.  sysInit is the entry point for UniWorks.

INTERNAL
Many routines in this module "link" and "unlk" the "c" frame pointer
a6@ although they don't use it in any way!  This is only for the benefit of
the stacktrace facility to allow it to properly trace tasks executing within
these routines.
*/

#define ASMLANGUAGE
#include "UniWorks.h"
#include "sysLib.h"
#include "config.h"

	/* internals */

	.globl	_sysInit	/* start of system code */
	.globl  _sysKernelTrap
	.globl  _sysVwTrap
	.globl  _sysVwTrapRtn
	.globl	_sysCacheEnable
	.globl	_sysAcfailInt
	.globl	_sysAbortInt
	.globl	_sysBerrInt

	/* externals */

	.globl	_usrInit	/* system initialization routine */
	.globl	_logMsg		/* error message logger */
	.globl  _reboot
	.globl  _excToDoAdd
	.globl  _intSetVec
	.globl  _intGetVec

	.data
AcfMsg:	.asciz	"ACFAIL detected!\012"

	.text
	.even

/*******************************************************************************
*
* sysInit - start after boot
*
* This is the system start-up entry point for UniWorks in ram.
* It is the first code executed after booting.
* It disables the interrupts, sets up the stack,
* and jumps to the C routine usrInit (2) in usrConfig (1).
*
* The initial stack is set to grow down from sysInit.
* Note that this initial stack is used only by usrInit,
* then is never used again.
* Memory for the stack needs to be accounted for when determining the load
* address of the system.  THIS IS NOT A CALLABLE ROUTINE.

* sysInit ()		/* THIS IS NOT A CALLABLE ROUTINE *

*/

_sysInit:
	movew	#0x3700,sr		/* disable interrupts, turn on M bit */
	movel	#_sysInit,a7		/* set stack to grow down from code */
	movel	#BOOT_WARM_AUTOBOOT,a7@- /* push start type arg = WARM_BOOT */
	jsr	_usrInit		/* never returns - starts up kernel */

/*******************************************************************************
*
* sysKernelTrap - trap to kernel function
*
* This routine traps to the kernel. It can only be called from assembly
* language.
* The trap number for the kernel is defined in config.h.
*
* NOMANUAL

* sysKernelTrap ()

*/
_sysKernelTrap:
	trap	#TRAP_KERNEL
	rts

/*******************************************************************************
*
* sysVwTrap - trap to UniWorks function
*
* This routine traps to UniWorks. It can only be called from assembly
* language.
* The trap number for UniWorks is defined in config.h.
*
* NOMANUAL

* sysVwTrap ()

*/
_sysVwTrap:
	trap	#TRAP_UNIWORKS
	rts

/*
 * This routine calls the routine whose address is in a0.  Since the system
 * is now in supervisor state (since we got here from a trap) this can be
 * used to call a routine in supervisor state.
 * THIS IS NOT A C CALLABLE ROUTINE!!
 */

_sysVwTrapRtn:
	jsr	a0@		/* vector to proper routine. */
	rte			/* return from trap */

/*******************************************************************************
*
* sysCacheEnable - enable cache memory
*
* Enable or disable the 68030 on-chip caches.
*
* SEE ALSO: "MC68030 32-Bit Microprocessor User's Manual"
*

* VOID sysCacheEnable (enable)
*     BOOL enable;		/* enable if TRUE, otherwise disable *

*/

_sysCacheEnable:

	link	a6,#0
	movel	a6@(0x08),d1		/* put enable in d1 */

	cmpl	#1,d1			/* does enable == TRUE? */
	bne	sce0			/* if not, disable cache */
	movel	#0x1919,d0		/* else, clear and enable caches */
	bra	sce1			/* burst mode, instr & data caches */

sce0:	movel	#0,d0			/* disable caches */

	/* Enable or disable the caches by writing to the cache control
	   register.  Since the assemblers differ on the syntax for this one
	   (if they even have it), it's done with two .word's.  Yucko. */

sce1:
	.word 0x4e7b,0x0002		/* movec d0,cacr */

	unlk	a6
	rts

/*******************************************************************************
*
* sysAcfailInt - ACFAIL interrupt routine
*
* This routine handles the ACFAIL interrupt on the MVME-147.
* It clears the fail bit in the PCC and then logs a message.
*
* NOMANUAL

* VOID sysAcfailInt ()

*/

_sysAcfailInt:

	link	a6,#0
	moveml	d0-d7/a0-a5,a7@-	/* save all registers */

	orb	#0x80,0xfffe101c	/* reset the intstat bit */

	pea	AcfMsg
	jsr	_logMsg			/* ("ACFAIL detected!\n") */
	addql	#4,a7

	/* Anything that can be done to cope should be either done or
	 * scheduled here */

	moveml	a7@+,d0-d7/a0-a5	/* restore all registers */
	unlk	a6
	rte

/*******************************************************************************
*
* sysAbortInt - field PCC generated abort interrupt
*
* This routine fields the PCC abort interrupt.  The PCC debounces the
* abort button and, if its interrupt is enabled, requests an interrupt
* at vector number PCC_INT_VEC_BASE + PCC_INT_ABORT.  This routine
* has been connected to that vector, and will clear the abort interrupt
* bit in the corresponding PCC control register, and then call the
* normal abort handler previously set by tySetAbortFunc, simulating
* an abort from the terminal.
*
* NOTE
* Probably shouldn't bother trying to reboot -- just call sysToMonitor.
*
* NOMANUAL

* VOID sysAbortInt ()

*/
_sysAbortInt:

	link	a6,#0
	moveml	d0-d7/a0-a5,a7@-	/* save all registers */

	orb	#0x80,0xfffe1024	/* reset the intstat bit */

	movel	#BOOT_WARM_NO_AUTOBOOT,a7@-
	movel	#_reboot, a7@-
	jsr	_excToDoAdd
	addl	#8, a7
sai0:
	moveml	a7@+,d0-d7/a0-a5	/* restore all registers */
	unlk	a6
	rte

/*******************************************************************************
*
* sysBerrInt - field PCC generated BERR interrupt
*
* This routine fields the PCC BERR interrupt.  The PCC intercepts the
* BERR line and, if its interrupt is enabled, requests an interrupt
* at vector number PCC_INT_VEC_BASE + PCC_INT_BERR.  This routine
* has been connected to that vector, and will clear the BERR interrupt
* bit in the corresponding PCC control register, and then jump to the
* normal exception handler simulating an interrupt to vector 0x02 (the
* 68030 bus error interrupt vector).
*
* NOMANUAL

* sysBerrInt ()

*/
_sysBerrInt:
	movel	d0,a7@-			/* save space for pc */
	movew	cc,a7@-			/* save condition codes for rtr below */

	link	a6,#0
	moveml	d0-d7/a0-a5,a7@-	/* save all registers */

	orb	#0x08,0xfffe1022	/* reset the intstat bit */

	moveq	#0x08,d0		/* bus error interrupt at vec 2 */
	movel	d0,a7@-
	jsr	_intGetVec
	addql	#4,a7
	movel	d0,a6@(0x06)		/* save vector as new pc */

	movew	a6@(0x10),d0		/* get stack frame vector word */
	andw	#0xf000,d0		/* preserve frame type bits */
	orw	#0x08,d0		/* change to vector address for 2 */
	movew	d0,a6@(0x10)		/* and put back

	moveml	a7@+,d0-d7/a0-a5	/* restore all registers */
	unlk	a6

	rtr				/* reload condition codes & pc */

