/* tyCoDrv.c - The Matrix MS-CPU100 DUART tty handler */

static char *copyright = "Copyright 1988, Wind River Systems";

/*
modification history
--------------------
01a,16aug88,dmp  written by modifying v01a of mz7120 version.
*/

/*
DESCRIPTION
This is the driver for the MS-CPU100 DUART.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available in the mode that we use the chip in are 75, 110, 134.5,
150, 300, 600, 1200, 2000, 2400, 4800, 1800, 9600, 19200.  In the other mode,
you lose 19200 but gain 38400; others are also affected.

*/

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "tyLib.h"
#include "config.h"


typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char csrVal;	/* rate to write to the csr reg to get that baud rate */
    } BAUD;

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *dr;		/* data port */
    char *csr;		/* clock select register */
    char *cr;		/* control reg */
    char rem;		/* bit for receive-enable mask */
    char tem;		/* bit for transmit-enable mask */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [N_CHANNELS] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE, DUART_RHRA, DUART_CSRA, DUART_CRA,
	       RX_RDY_A_INT, TX_RDY_A_INT},
    {{{{NULL}}}, FALSE, DUART_RHRB, DUART_CSRB, DUART_CRB,
	       RX_RDY_B_INT, TX_RDY_B_INT}
    };

/* baudTable is a table of the available baud rates, and the values to write
   to the csr reg to get those rates */

LOCAL BAUD baudTable [] =		/* BRG bit = 1 */
    {
    {75,	RX_CLK_75    | TX_CLK_75 },
    {110,	RX_CLK_110   | TX_CLK_110 },
    {134,	RX_CLK_134_5 | TX_CLK_134_5 },
    {150,	RX_CLK_150   | TX_CLK_150 },
    {300,	RX_CLK_300   | TX_CLK_300 },
    {600,	RX_CLK_600   | TX_CLK_600 },
    {1200,	RX_CLK_1200  | TX_CLK_1200 },
    {2000,	RX_CLK_2000  | TX_CLK_2000 },
    {2400,	RX_CLK_2400  | TX_CLK_2400 },
    {4800,	RX_CLK_4800  | TX_CLK_4800 },
    {1800,	RX_CLK_1800  | TX_CLK_1800 },
    {9600,	RX_CLK_9600  | TX_CLK_9600 },
    {19200,	RX_CLK_19200 | TX_CLK_19200}
    };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();
LOCAL VOID tyTxInt ();
LOCAL VOID tyRxInt ();


/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the driver, sets up interrupt vectors,
* and performs hardware initialization of the ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* DUART interrupts on autovector 6 */

    sysDuartConnect (tyRxInt, tyTxInt);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);
    return (OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 or 1) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    /* if there is a device already on this channel, don't do it */

    if (tyCoDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    if (tyDevInit ((TY_DEV_ID) &tyCoDv [channel], 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)

	return (ERROR);

    sysImrSet (tyCoDv [channel].rem, 0);

    /* Mark the device as having been created, and add the device to
     * the io sys */

    tyCoDv [channel].created = TRUE;
    return (iosDevAdd ((DEV_HDR *) &tyCoDv [channel], name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the duart
*
* This routine initializes the duart for the UniWorks environment. 
* This routine must be called in supervisor mode, since it accesses I/O
* space.
*/

LOCAL VOID tyCoHrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();

    /* 8 data bits, 1 stop bit, no parity, set for 9600 baud */

    *DUART_CRA  = RST_BRK_INT_CMD;
    *DUART_CRA  = RST_ERR_STS_CMD;
    *DUART_CRA  = RST_TX_CMD;
    *DUART_CRA  = RST_RX_CMD;
    *DUART_CRA  = RST_MR_PTR_CMD;
    *DUART_MRA  = PAR_MODE_NO | BITS_CHAR_8;	/* mode A: 1 */
    *DUART_MRA  = STOP_BITS_1;			/* mode A: 2 */
    *DUART_CSRA = RX_CLK_9600 | TX_CLK_9600;	/* clock A */

    *DUART_CRB  = RST_BRK_INT_CMD;
    *DUART_CRB  = RST_ERR_STS_CMD;
    *DUART_CRB  = RST_TX_CMD;
    *DUART_CRB  = RST_RX_CMD;
    *DUART_CRB  = RST_MR_PTR_CMD;
    *DUART_MRB  = PAR_MODE_NO | BITS_CHAR_8;	/* mode B: 1 */
    *DUART_MRB  = STOP_BITS_1;			/* mode B: 2 */
    *DUART_CSRB = RX_CLK_9600 | TX_CLK_9600;	/* clock B */

    *DUART_SOPBC = 3;				/* drive both A & B CTS* low */

    /* enable the receivers and transmitters on both channels */

    *DUART_CRA  = RX_ENABLE | TX_ENABLE;	/* command register A */
    *DUART_CRB  = RX_ENABLE | TX_ENABLE;	/* command register B */

    /* all interrupts are masked out: the receiver interrupt will be enabled
       in the tyCoDevCreate */

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyCoOpen - open file to duart
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - Task level read routine for duart
*
* This routine fields all read calls to the duart.  It
* calls tyRead with a pointer to the appropriate element of ty_dev.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - Task level write routine for duart
*
* This routine fields all write calls to the duart.  It
* calls tyWrite with a pointer to the appropriate element of ty_dev.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:
	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)
		    {
		    *pTyCoDv->csr = baudTable [i].csrVal;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}
    return (status);
    }

/*******************************************************************************
*
* tyRxInt - handle a receiver interrupt
*
* This routine gets called by sysBoardInt to handle a receiver interrupt.
*/

LOCAL VOID tyRxInt (channel)
    FAST int channel;

    {
    tyIRd ((TY_DEV_ID) &tyCoDv [channel], *tyCoDv [channel].dr);
    }
/*******************************************************************************
*
* tyTxInt - handle a transmitter interrupt
*
* This routine gets called by sysBoardInt to handle a xmitter interrupt.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID tyTxInt (channel)
    FAST int channel;

    {
    char outChar;

    if ((tyCoDv [channel].created) &&
	(tyITx ((TY_DEV_ID) &tyCoDv [channel], &outChar) == OK))

	*(tyCoDv [channel].dr) = outChar;

    else

	/* turn off the transmitter */

	sysImrSet (0, tyCoDv [channel].tem);
    }
/*******************************************************************************
*
* tyCoStartup - Transmitter startup routine
*
* Call interrupt level character output routine for MS-CPU100 duart.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    /* all we gotta do is enable the transmitter and it should interrupt
     * to write the next char out. */

    sysImrSet (pTyCoDv->tem, 0);
    }
