/* memLib.c - the UniWorks memory manager */

static char *copyright = "Copyright 1986-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
The library memLib provides a first-fit memory management facility.
Memory is allocated (malloc) from a pool of free memory blocks;
adjacent blocks are coalesced when they are freed (free).
No attempt is made to reduce fragmentation - there is no garbage collection. 
The package is initialized by calling memInit with a pointer to a block
of memory from which memory will be allocated.
The routine memShow displays statistics about the free pool.

The system memory manager is built on top of a generic
"partition" manager which may be used to create private pools of memory.
This is useful for drivers that control devices that have on board memory
which must be configured and managed by the host processor.  

Various error checking and error action options can be selected with a call
to memOptionsSet (and memPartOptionsSet).
The default is to check blocks being freed and to log an error message
and suspend the caller if a bad block is discovered.  Also default is
the logging of attempts to allocate more memory than is available.

CAVEATS
malloc always rounds up to a 4 byte boundary and there
is a 12 byte overhead for each block allocated.

INCLUDE FILE: memLib.h

INTERNAL
Blocks allocated by malloc are actually larger than the size requested by
the user.  Each block is prefaced by a header which contains the size and
status of that block and a pointer to the previous block.
The pointer returned to the user points just past this header.
Likewise when a block is freed, the header is found just in front of
the block pointer passed by the user.

The header data is arranged so that the pointer to the previous
block comes first and is therefore adjacent to that previous block.
Thus each block has 4 bytes of redundant information on either side of
it (the 4 bytes of size and status before it in its own header, and the 4 byte
pointer after it in the next header).
This redundant information is optionally used used in free() and realloc()
to make a consistency check on blocks specified by the user.
This mechanism helps to detect two common errors: (1) bad block pointers
passed to free() or realloc() are usually detected, and (2) trashing
up to 4 bytes on either side of the block will only affect that block
and will also be detected by free() or realloc().

There is a minimum block size which malloc allocates;  this is to insure
that there will be enough room for the free list links that must be used
when the block is freed if it cannot be coalesced with an adjacent block.

malloc and realloc always force the requested block size to be multiple of 4;
since memInit forces the initial pool to be on a 4 byte boundary, this causes
all blocks to lie on a 4 byte boundaries, thus insuring that the block
passed back to the user will always lie on a 4 byte boundary.

The memory partition semaphore is a structure in the partition descriptor
rather than a pointer (SEM_ID) to a dynamically created semaphore structure.
This is because of the chicken-and-the-egg problem of memLib using semaphores
and semCreate calling malloc.  Instead the structure is simply declared
directly in the partition structure and we call semInit() instead of
semCreate().
*/

#include "UniWorks.h"
#include "semLib.h"
#include "lstLib.h"
#include "memLib.h"


/* BLOCK_HDR is the header that is at the front of every block */

typedef struct blockHdr		/* BLOCK_HDR */
    {
    struct blockHdr *pPrevHdr;	/* pointer to previous block hdr */
    unsigned nWords : 31;	/* size in words of this block */
    unsigned free   : 1;	/* TRUE = this block is free */
    } BLOCK_HDR;

/* FREE_BLOCK is the structure for a free block.  It includes the freelist
 * node in addition to the usual header. */

typedef struct			/* FREE_BLOCK */
    {
    BLOCK_HDR hdr;		/* normal block header */
    NODE node;			/* freelist links */
    } FREE_BLOCK;


/* macros for rounding: all allocated blocks are aligned on a 4 byte boundary */

#define MEM_ROUND_UP(x)		(((int) (x) + 3) & ~3)
#define MEM_IS_ROUND(x)		(((int) (x) & 3) == 0)


/* macros for getting to next and previous blocks */

#define NEXT_HDR(pHdr) \
		((BLOCK_HDR *) ((char *) (pHdr) + (2 * (pHdr)->nWords)))
#define PREV_HDR(pHdr)	((pHdr)->pPrevHdr)


/* macros for converting between the "block" that caller knows
 * (actual available data area) and the block header in front of it */

#define HDR_TO_BLOCK(pHdr)	((char *) ((int) pHdr + sizeof (BLOCK_HDR)))
#define BLOCK_TO_HDR(pBlock)	((BLOCK_HDR *) ((int) pBlock - \
						sizeof(BLOCK_HDR)))


/* macros for converting between the "node" that is strung on the freelist
 * and the block header in front of it */

#define HDR_TO_NODE(pHdr)	(& ((FREE_BLOCK *) pHdr)->node)
#define NODE_TO_HDR(pNode)	((BLOCK_HDR *) ((int) pNode - \
						OFFSET (FREE_BLOCK, node)))

/* local variables */

PARTITION memSysPartition;	/* system partition used by malloc... */


/* forward declarations */

LOCAL void memPartAllocError ();
LOCAL void memPartBlockError ();
LOCAL BOOL memPartBlockIsValid ();
LOCAL BLOCK_HDR *memBlockSplit ();
LOCAL BLOCK_HDR *memBlockSplit2 ();


/*******************************************************************************
*
* memPartInit - create a memory partition
*
* memPartInit creates a new memory partition containing the specified
* memory pool.  It returns a partition ID which can then be passed to
* other memPart... routines to manage the partition, i.e. to allocate and free
* memory blocks in the partition.  Partitions can be created to manage
* any number of separate memory pools.
*
* NOTE
* The descriptor for the new partition is allocated out of the system memory
* pool (i.e. with malloc).
*
* RETURNS:
*    partition id, or
*    NULL if insufficient memory in system memory pool for new
*	  partition descriptor
*/

PART_ID memPartInit (pPool, poolSize)
    char *pPool;	/* pointer to memory pool */
    unsigned poolSize;	/* pool size in bytes */

    {
    FAST PART_ID pPart;

    /* allocate a partition structure from the system memory partition */

    pPart = (PART_ID) malloc (sizeof (PARTITION));

    if (pPart != NULL)
	memPInit (pPart, pPool, poolSize);

    return (pPart);
    }
/*******************************************************************************
*
* memPInit - initialize a memory partition
*
* memPInit initializes a partition free list, seeding it with the 
* memory pool passed as an argument.  It must be called exactly once 
* for each memory partition created.
*
* SEE ALSO: memPartInit (2)
*/

VOID memPInit (partId, pPool, poolSize)
    FAST PART_ID partId;	/* partition to initialize */
    char *pPool;		/* pointer to memory pool */
    unsigned poolSize;		/* pool size in bytes */

    {
    /* initialize partition descriptor */

    bzero ((char *) partId, sizeof (*partId));

    partId->options = MEM_ALLOC_ERROR_LOG_MSG
                      | MEM_BLOCK_ERROR_LOG_AND_SUSPEND
                      | MEM_BLOCK_CHECK;

    partId->minBlockWords = sizeof (FREE_BLOCK) >> 1;

    /* initialize partition semaphore and free list and add memory to pool */

    semInit (&partId->sem);
    semGive (&partId->sem);

    lstInit (&partId->freeList);

    (void) memPartAddToPool (partId, pPool, poolSize);
    }
/*******************************************************************************
*
* memPartAddToPool - add memory to a memory partition
*
* This routine allows additional memory to be added to a memory
* partition after the initial memPartInit (2).
*
* SEE ALSO: memPartInit (2)
*/

VOID memPartAddToPool (partId, pPool, poolSize)
    FAST PART_ID partId;		/* partition to initialize */
    FAST char *pPool;			/* pointer to memory pool */
    FAST unsigned poolSize;		/* pool size in bytes */

    {
    FAST BLOCK_HDR *pHdrStart;
    FAST BLOCK_HDR *pHdrMid;
    FAST BLOCK_HDR *pHdrEnd;
    char *tmp;

    /* insure that the pool starts on an even byte boundry */

    tmp       = (char *) MEM_ROUND_UP (pPool);	/* get actual start */
    poolSize -= tmp - pPool;			/* adjust length */
    pPool     = tmp;

    /* initialize three blocks -
     * one at each end of the pool for end cases and real initial free block */

    pHdrStart		= (BLOCK_HDR *) pPool;
    pHdrStart->pPrevHdr = NULL;
    pHdrStart->free     = FALSE;
    pHdrStart->nWords   = sizeof (BLOCK_HDR) >> 1;

    pHdrMid		= NEXT_HDR (pHdrStart);
    pHdrMid->pPrevHdr   = pHdrStart;
    pHdrMid->free       = TRUE;
    pHdrMid->nWords     = (poolSize - 2 * sizeof (BLOCK_HDR)) >> 1;

    pHdrEnd		= NEXT_HDR (pHdrMid);
    pHdrEnd->pPrevHdr   = pHdrMid;
    pHdrEnd->free       = FALSE;
    pHdrEnd->nWords     = sizeof (BLOCK_HDR) >> 1;

    semTake (&partId->sem);

    lstInsert (&partId->freeList, (NODE *) NULL, HDR_TO_NODE (pHdrMid));
    partId->totalWords += (poolSize >> 1);

    semGive (&partId->sem);
    }
/*******************************************************************************
*
* memPartOptionsSet - select memory management options
*
* There are two kinds of errors that the memory manager can detect.
* The default is to check blocks being freed and to log an error message
* and suspend the caller if a bad block is discovered.  Also default is
* the logging of attempts to allocate more memory than is available.
* When a corrupted block is detected by free, the task that is suspended
* may, or may not, be the corruptor.
*/

VOID memPartOptionsSet (partId, options)
    PART_ID partId;	/* partition for which to set option */
    unsigned options;	/* memory management options */

    {
    partId->options = options;
    }
/*******************************************************************************
*
* memPartAlloc - allocate a block of memory from a partition 
*
* memPartAlloc allocates a block of memory from a partition,
* which is greater than or equal to the size specified by the parameter.
* The partition must have been previously initialized with memPInit (2).
*
* RETURNS:
*   pointer to block, or
*   NULL if unsuccessful.
*/

char *memPartAlloc (partId, nBytes)
    FAST PART_ID partId;	/* memory partition to allocate out of */
    unsigned nBytes;		/* number of bytes to allocate */

    {
    FAST unsigned nWords;
    FAST NODE *pNode;
    FAST BLOCK_HDR *pHdr;
    BLOCK_HDR *pNewHdr;

    /* get actual size to allocate; add overhead, round-up, check for minimum */

    nWords = (MEM_ROUND_UP (nBytes) + sizeof (BLOCK_HDR)) >> 1;
    if (nWords < partId->minBlockWords)
	nWords = partId->minBlockWords;

    /* get exclusive access to the partition */

    semTake (&partId->sem);

    /* first fit */

    pNode = lstFirst (&partId->freeList);

    while ((pNode != NULL) && (NODE_TO_HDR (pNode)->nWords < nWords))
	pNode = lstNext (pNode);

    if (pNode == NULL)
	{
	semGive (&partId->sem);
	memPartAllocError (partId, nBytes);
	return (NULL);
	}

    pHdr = NODE_TO_HDR (pNode);


    /* now we split off from this block, the amount required by the user;
     * note that the piece we are giving the user is at the end of the
     * block so that the remaining piece is still the piece that is
     * linked in the free list;
     * if the user is taking the whole block, then we just delete it
     * from free list */

    pNewHdr = memBlockSplit (pHdr, nWords, partId->minBlockWords);
    if (pNewHdr != NULL)
	pHdr = pNewHdr;				/* give split off block */
    else
	lstDelete (&partId->freeList, HDR_TO_NODE (pHdr));	/* not split */

    /* mark the block we're giving as not free  */

    pHdr->free = FALSE;

    /* update allocation statistics */

    partId->curBlocksAllocated++;
    partId->cumBlocksAllocated++;
    partId->curWordsAllocated += pHdr->nWords;
    partId->cumWordsAllocated += pHdr->nWords;

    semGive (&partId->sem);

    return (HDR_TO_BLOCK (pHdr));
    }
/*******************************************************************************
*
* memPartRealloc - reallocate a block of memory in a given partition
*
* memPartRealloc changes the size of the given block and returns a 
* pointer to the (possibly moved) block.  The contents will be unchanged 
* up to the lesser of the new and old sizes.
*
* RETURNS:
*   Pointer to new block of memory, or
*   NULL if unsuccessful.
*/

char *memPartRealloc (partId, pBlock, nBytes)
    PART_ID partId;		/* partition ID */
    char *pBlock;		/* block to be reallocated */
    unsigned nBytes;		/* new block size in bytes */

    {
    FAST BLOCK_HDR *pHdr = BLOCK_TO_HDR (pBlock);
    FAST BLOCK_HDR *pNextHdr;
    FAST unsigned nWords;
    char *pNewBlock;

    /* get actual new size; round-up, add overhead, check for minimum */

    nWords = (MEM_ROUND_UP (nBytes) + sizeof (BLOCK_HDR)) >> 1;
    if (nWords < partId->minBlockWords)
	nWords = partId->minBlockWords;

    /* optional check for validity of block */

    if ((partId->options & MEM_BLOCK_CHECK) &&
        !memPartBlockIsValid (partId, pHdr, FALSE))
	{
	memPartBlockError (partId, pBlock, "memPartRealloc");
	return (NULL);
	}


    /* get exclusive access to the partition */

    semTake (&partId->sem);

    /* test if we are trying to increase size of block */

    if (nWords > pHdr->nWords)
	{
	/* increase size of block -
	 * check if next block is free and is big enough to satisfy request*/

	pNextHdr = NEXT_HDR (pHdr);

	if (!pNextHdr->free || ((pHdr->nWords + pNextHdr->nWords) < nWords))
	    {
	    /* can't use adjacent free block -
	     * allocate an entirely new block and copy data */

	    semGive (&partId->sem);

	    if ((pNewBlock = memPartAlloc (partId, nBytes)) == NULL)
		return (NULL);
	    
	    bcopy (pBlock, pNewBlock, (int) (2 * pHdr->nWords -
					     sizeof (BLOCK_HDR)));

	    (void) memPartFree (partId, pBlock);

	    return (pNewBlock);		/* RETURN, don't fall through */
	    }
	else
	    {
	    /* append next block to this one -
	     *  - delete next block from freelist,
	     *  - add its size to this block,
	     *  - update allocation statistics,
	     *  - fix prev info in new "next" block header */

	    lstDelete (&partId->freeList, HDR_TO_NODE (pNextHdr));

	    pHdr->nWords += pNextHdr->nWords;			/* fix size */

	    partId->curWordsAllocated += pNextHdr->nWords;	/* fix stats */
	    partId->cumWordsAllocated += pNextHdr->nWords;

	    NEXT_HDR (pHdr)->pPrevHdr = pHdr;			/* fix next */

	    /* if this combined block is too big, it will get fixed below */
	    }
	}


    /* split off any extra and give it back;
     * note that this covers both the case of a realloc for smaller space
     * and the case of a realloc for bigger space that caused a coalesce
     * with the next block that resulted in larger than required block */

    pNextHdr = memBlockSplit2 (pHdr, nWords, partId->minBlockWords);

    semGive (&partId->sem);

    if (pNextHdr != NULL)
	(void) memPartFree (partId, HDR_TO_BLOCK (pNextHdr));

    return (pBlock);
    }
/*******************************************************************************
*
* memPartFree - free a block of memory in a partition
*
* memPartFree takes a block of memory previously allocated with
* memPartAlloc (2) and returns it to the given partition's free memory pool.
*
* RETURNS: OK or ERROR if invalid block
*/

STATUS memPartFree (partId, pBlock)
    PART_ID partId;	/* memory partition to add the block to */
    char *pBlock;	/* pointer to block of memory to be freed */

    {
    FAST BLOCK_HDR *pHdr  = BLOCK_TO_HDR (pBlock); 
    FAST unsigned  nWords = pHdr->nWords;
    FAST BLOCK_HDR *pNextHdr;

    /* optional check for validity of block */

    if ((partId->options & MEM_BLOCK_CHECK) &&
        !memPartBlockIsValid (partId, pHdr, FALSE))
	{
	memPartBlockError (partId, pBlock, "memPartFree");
	return (ERROR);
	}


    /* get exclusive access to the partition */

    semTake (&partId->sem);

    /* check if we can coalesce with previous block;
     * if so, then we just extend the previous block,
     * otherwise we have to add this as a new free block */

    if (PREV_HDR (pHdr)->free)
	{
	pHdr = PREV_HDR (pHdr);			/* coalesce with prev block */
	pHdr->nWords += nWords;
	}
    else
	{
	pHdr->free = TRUE;			/* add new free block */
	lstInsert (&partId->freeList, (NODE *) NULL, HDR_TO_NODE (pHdr));
	}

    /* check if we can coalesce with next block;
     * if so, then we can extend our block delete next block from free list */

    pNextHdr = NEXT_HDR (pHdr);
    if (pNextHdr->free)
	{
	pHdr->nWords += pNextHdr->nWords;	/* coalesce with next */
	lstDelete (&partId->freeList, HDR_TO_NODE (pNextHdr));
	}


    /* fix up prev info of whatever block is now next */

    NEXT_HDR (pHdr)->pPrevHdr = pHdr;

    /* adjust allocation stats */

    partId->curBlocksAllocated--;
    partId->curWordsAllocated -= nWords;

    semGive (&partId->sem);

    return (OK);
    }
/*******************************************************************************
*
* memPartFindMax - find size of largest available free block
*
* Search the memory partition free list for the largest block.
*
* RETURNS: largest available block in bytes
*/

int memPartFindMax (partId)
    FAST PART_ID partId;	/* partition ID */

    {
    FAST BLOCK_HDR *pHdr;
    FAST NODE *pNode;
    FAST unsigned biggestWords = 0;

    semTake (&partId->sem);

    /* go through free list and find largest free */

    for (pNode = lstFirst (&partId->freeList);
	 pNode != NULL;
	 pNode = lstNext (pNode))
	{
	pHdr = NODE_TO_HDR (pNode);
	if (pHdr->nWords > biggestWords)
	    biggestWords = pHdr->nWords;
	}

    semGive (&partId->sem);

    return (2 * biggestWords - sizeof (BLOCK_HDR));
    }
/*******************************************************************************
*
* memPartShow - show partition blocks and statistics
*
* For the given partition, memPartShow prints on standard output the total 
* amount of free space in the pool, the number of blocks,
* the average block size, and the maximum block size.
* It also prints out the number of blocks currently allocated, and
* the average allocated block size.
*
* In addition, if `type' is 1 a list of all the blocks in
* the free list of the given partition is printed on standard output.
*
* SEE ALSO: memShow (2)
*/

VOID memPartShow (partId, type)
    FAST PART_ID partId;	/* partition ID */
    int type;			/* 0 = statistics, 1 = statistics & list */

    {
    FAST BLOCK_HDR *pHdr;
    FAST NODE *pNode;
    FAST unsigned totalBytes   = 0;
    FAST unsigned biggestWords = 0;
    int numBlocks;
    int ix = 1;

    /* print out list header if we are going to print list */

    if (type == 1)
	{
	printf ("\nFREE LIST:\n");
	printf ("  num     addr      size\n");
	printf ("  --- ---------- ----------\n");
	}

    semTake (&partId->sem);

    /* go through free list and find total free and largest free,
     * and print each block if specified */

    for (pNode = lstFirst (&partId->freeList);
	 pNode != NULL;
	 pNode = lstNext (pNode))
	{
	pHdr = NODE_TO_HDR (pNode);

	/* check consistency and delete if not */

	if (!memPartBlockIsValid (partId, pHdr, TRUE))
	    {
	    printf ("  invalid block at %#x deleted\n", pHdr);
	    lstDelete (&partId->freeList, HDR_TO_NODE (pHdr));
	    }
	else
	    {
	    totalBytes += 2 * pHdr->nWords;

	    if (pHdr->nWords > biggestWords)
		biggestWords = pHdr->nWords;

	    if (type == 1)
		printf ("  %3d %#10x %10d\n", ix++, pHdr, 2 * pHdr->nWords);
	    }
	}

    if (type == 1)
	printf ("\n\n");


    numBlocks = lstCount (&partId->freeList);

    if (type == 1)
	printf ("SUMMARY:\n");
    printf (" status   bytes    blocks   ave block  max block\n");
    printf (" ------ --------- -------- ---------- ----------\n");
    printf ("current\n");

    if (numBlocks != 0)
	printf ("   free  %8d  %7d  %9d %9d\n", totalBytes, numBlocks,
		totalBytes / numBlocks, 2 * biggestWords);
    else
	printf ("   no free blocks\n");

    if (partId->curBlocksAllocated != 0)
	printf ("  alloc  %8d  %7d  %9d        -\n",  
		2 * partId->curWordsAllocated, partId->curBlocksAllocated,
	        (2 * partId->curWordsAllocated) / partId->curBlocksAllocated);
    else 
	printf ("   no allocated blocks\n");

    printf ("cumulative\n");

    if (partId->curBlocksAllocated != 0)
	printf ("  alloc  %8d  %7d  %9d        -\n",      
		2 * partId->cumWordsAllocated, partId->cumBlocksAllocated,
		(2 * partId->cumWordsAllocated) / partId->cumBlocksAllocated);
    else 
	printf ("   no allocated blocks\n");

    semGive (&partId->sem);
    }
/*******************************************************************************
*
* memPartAllocError - handle allocation error
*/

LOCAL void memPartAllocError (pPart, nBytes)
    PART_ID pPart;
    unsigned nBytes;

    {
    errnoSet (S_memLib_NOT_ENOUGH_MEMORY);

    switch (pPart->options & MEM_ALLOC_ERROR_MASK)
	{
	case MEM_ALLOC_ERROR_LOG_MSG:
	    logMsg ( "memPartAlloc: block too big - %d in partition %#x.\n",
		    nBytes, (int) pPart);
	    break;

	case MEM_ALLOC_ERROR_LOG_AND_SUSPEND:
	    logMsg (
	"memPartAlloc: block too big - %d in partition %#x; suspending task.\n",
		    nBytes, (int) pPart);
	    taskSuspend (0);
	    break;
	}
    }
/*******************************************************************************
*
* memPartBlockError - handle invalid block error
*/

LOCAL void memPartBlockError (pPart, pBlock, label)
    PART_ID pPart;
    char *pBlock;
    char *label;

    {
    errnoSet (S_memLib_BLOCK_ERROR);

    switch (pPart->options & MEM_BLOCK_ERROR_MASK)
	{
	case MEM_BLOCK_ERROR_LOG_MSG:
	    logMsg ("%s: invalid block %#x in partition %#x.\n",
		    label, (int) pBlock, (int) pPart);
	    break;

	case MEM_BLOCK_ERROR_LOG_AND_SUSPEND:
	    logMsg ("%s: invalid block %#x in partition %#x; suspending task\n",
		    label, (int) pBlock, (int) pPart);
	    taskSuspend (0);
	    break;
	}
    }
/*******************************************************************************
*
* memPartBlockIsValid - check validity of block
*
* NOTE: This routine should NOT be called with the partition semaphore taken
* if possible, because if the block IS screwed up, then a bus error is
* not unlikely and we would hate to have the task die with the semaphore
* taken, thus hanging the partition.
*/

LOCAL BOOL memPartBlockIsValid (partId, pHdr, isFree)
    PART_ID partId;
    FAST BLOCK_HDR *pHdr;
    BOOL isFree;		/* expected status */

    {
    if (!MEM_IS_ROUND (pHdr)			/* not aligned */
	|| !MEM_IS_ROUND (2 * pHdr->nWords)	/* size not round */
        || (pHdr->nWords > partId->totalWords)	/* bigger than partition */
	|| (pHdr->free != isFree)		/* not right allocated-ness */
	|| (pHdr != PREV_HDR (NEXT_HDR (pHdr)))	/* doesn't match next block */
	|| (pHdr != NEXT_HDR (PREV_HDR (pHdr))))/* doesn't match prev block */
	{
	return (FALSE);
	}

    return (TRUE);
    }
/*****************************************************************************
*
* memBlockSplit - split a block into two blocks for malloc
*
* NOTE:
*
*	realloc needs a separate block split routine from malloc because
*	malloc splits the block and gives the caller the split off
*	portion, but realloc gives the caller the beginning portion
*	of the block.
*
*	example:
*
*	memBlockSplit2 (pHdr, 500, 50)
*
*	pHdr.nWords = 600
*	wordsLeft = 600 - 500 = 100
*	pHdr.nWords = 100	 	freed
*	pNewHdr.nWords = 500	 	given back to user 
*/

LOCAL BLOCK_HDR *memBlockSplit (pHdr, nWords, minWords)
    FAST BLOCK_HDR *pHdr;
    unsigned nWords;
    unsigned minWords;

    {
    FAST unsigned wordsLeft;
    FAST BLOCK_HDR *pNewHdr;

    /* check if block can be split */

    if ((wordsLeft = (pHdr->nWords - nWords)) < minWords)
	return (NULL);			/* not enough left */

    /* adjust original block size and create new block */

    pHdr->nWords = wordsLeft;

    pNewHdr = NEXT_HDR (pHdr);

    pNewHdr->pPrevHdr = pHdr;
    pNewHdr->nWords   = nWords;
    pNewHdr->free     = pHdr->free;

    /* fix next block */

    NEXT_HDR (pNewHdr)->pPrevHdr = pNewHdr;

    return (pNewHdr);
    }

/*******************************************************************************
*
* memInit - initialize the system memory pool
*
* memInit initializes the system partition free list, seeding it with the 
* memory pool passed as an argument.  It must be called exactly once 
* before any other routine in memLib (1).  This is normally done in
* usrRoot (2) in usrConfig (1).
*/

VOID memInit (pPool, poolSize)
    char *pPool;	/* pointer to memory pool */
    unsigned poolSize;	/* pool size in bytes */

    {
    memPInit (&memSysPartition, pPool, poolSize);
    }
/******************************************************************************
*
* memAddToPool - add memory to system memory pool
*
* This routine allows additional memory to be added to the system memory
* pool after the initial memInit (2).
*
* SEE ALSO: memPartAddToPool (2)
*/  

VOID memAddToPool (pPool, poolSize)
    FAST char *pPool;		/* pointer to memory pool */
    FAST unsigned poolSize;	/* pool size in bytes */

    {
    (void) memPartAddToPool (&memSysPartition, pPool, poolSize);
    }
/*******************************************************************************
*
* memOptionsSet - turn debugging on and off
*
* Set the debug options for the system memory pool.
*
* SEE ALSO: memPartOptionsSet (2)
*/

VOID memOptionsSet (options)
    unsigned options;	/* options for system partition */

    {
    memPartOptionsSet (&memSysPartition, options);
    }
/*******************************************************************************
*
* malloc - allocate a block of memory from the system memory pool
*
* malloc allocates a block of memory from the free pool which is greater
* than or equal to the size specified by the parameter.
*
* RETURNS:
*   pointer to block, or
*   NULL if unsuccessful.
*/

char *malloc (nBytes)
    unsigned nBytes;		/* number of bytes to allocate */

    {
    return (memPartAlloc (&memSysPartition, nBytes));
    }
/*******************************************************************************
*
* calloc - allocate space for an array
*
* The routine calloc allocates space for an array of `elemNum' elements
* of size `elemSize'.  The space is initialized to zeros.
*
* RETURNS:
*   pointer to block, or
*   NULL if unsuccessful.
*/

char *calloc (elemNum, elemSize)
    unsigned elemNum;	/* number of elements */
    unsigned elemSize;	/* size of elements */

    {
    FAST char *pMem;
    FAST unsigned nBytes = elemNum * elemSize;
    
    if ((pMem = memPartAlloc (&memSysPartition, nBytes)) != NULL)
	bzero (pMem, (int) nBytes);

    return (pMem);
    }
/*******************************************************************************
*
* realloc - reallocate a block of memory
*
* realloc changes the size of the given block and returns a pointer to the 
* (possibly moved) block.  The contents will be unchanged up to the lesser
* of the new and old sizes.
*
* RETURNS:
*   Pointer to new block of memory, or
*   NULL if unsuccessful.
*/

char *realloc (pBlock, newSize)
    char *pBlock;	/* block to be reallocated */
    unsigned newSize;	/* new block size */

    {
    return (memPartRealloc ((PART_ID) &memSysPartition, pBlock, newSize));
    }
/*******************************************************************************
*
* free - free a block of memory
*
* free takes a block of memory previously allocated with malloc (2) or
* calloc (2) and returns it to the free memory pool.
*
* It is an error if the block free'd was not malloc'd first.
*
* RETURNS: OK or ERROR if invalid block
*/

STATUS free (pBlock)
    char *pBlock;	/* pointer to block of memory to be freed */

    {
    return (memPartFree (&memSysPartition, pBlock));
    }
/*******************************************************************************
*
* cfree - free a block of memory
*
* cfree takes a block of memory previously allocated with calloc (2)
* and returns it to the free memory pool.
*
* It is an error if the block free'd was not calloc'd first.
*
* RETURNS: OK or ERROR if invalid block
*/

STATUS cfree (pBlock)
    char *pBlock;	/* pointer to block of memory to be freed */

    {
    return (memPartFree (&memSysPartition, pBlock));
    }
/*******************************************************************************
*
* memFindMax - find size of largest available free block in system pool
*
* Search the system memory partition free list for the largest block.
*
* RETURNS: largest available block in bytes
*
* SEE ALSO: memPartFindMax (2)
*/

int memFindMax ()
    {
    return (memPartFindMax (&memSysPartition));
    }
/*******************************************************************************
*
* memShow - show system partition blocks and statistics
*
* memShow prints on standard output the total 
* amount of free space in the pool, the number of blocks,
* the average block size, and the maximum block size.
* It also prints out the number of blocks currently allocated, and
* the average allocated block size.
*
* In addition, if `type' is 1 a list of all the blocks in
* the free list of the system partition is printed on standard output.
*
* EXAMPLE
* .CS
*    -> memShow 1
*
*    FREE LIST:
*      num     addr      size
*      --- ---------- ----------
*        1   0x3fee18         16
*        2   0x3b1434         20
*        3    0x4d188    2909400
*
*    SUMMARY:
*     status   bytes    blocks   ave block  max block
*     ------ --------- -------- ---------- ---------- 
*    current
*       free   2909436        3     969812   2909400
*      alloc    969060    16102         60        -
*    cumulative
*      alloc   1143340    16365         69        -
*    value = 12288 = 0x3000 = cFwd + 0x20
*    ->
* .CE
*
* SEE ALSO: memPartShow (2)
*/

VOID memShow (type)
    int type;

    {
    memPartShow (&memSysPartition, type);
    }

/*****************************************************************************
*
* memBlockSplit2 - split a block into two blocks for realloc
*
* NOTE:
*
*	realloc needs a separate block split routine from malloc because
*	malloc splits the block and gives the caller the split off
*	portion, but realloc gives the caller the beginning portion
*	of the block.
*
*	example:
*
*	memBlockSplit2 (pHdr, 500, 50)
*
*	pHdr.nWords = 600
*	wordsLeft = 600 - 500 = 100
*	pHdr.nWords = 500		given back to user
*	pNewHdr.nWords = 100		freed
*
*/
BLOCK_HDR *memBlockSplit2 (pHdr, nWords, minWords)
    FAST BLOCK_HDR *pHdr;
    unsigned nWords;
    unsigned minWords;

    {
    FAST unsigned wordsLeft;
    FAST BLOCK_HDR *pNewHdr;

    /* check if block can be split */

    if ((wordsLeft = (pHdr->nWords - nWords)) < minWords)
	return (NULL);			/* not enough left */

    /* adjust original block size and create new block */

    pHdr->nWords = nWords;

    pNewHdr = NEXT_HDR (pHdr);

    pNewHdr->pPrevHdr = pHdr;
    pNewHdr->nWords   = wordsLeft;
    pNewHdr->free     = pHdr->free;

    /* fix next block */

    NEXT_HDR (pNewHdr)->pPrevHdr = pNewHdr;

    return (pNewHdr);
    }
