/*
 * hex - translate object files into ASCII formats suitable for Motorola 
 *		S-record downloading
 *
 * 	Usage:
 *	    hex [-b] [-f] [-l] [-n#] [-r] [-s0] [-s2] [-s3] [-ns8] [-split[4]] [-psz#] [+saddr] ifile
 *	where:
 *		b	generate records for bss segment.
 *		f	just ship ifile as data, not as an executable file
 *		l	output "Loading at" message
 *		n#	number of characters to output
 *		r	output a \r instead of a \n after each record
 *		s0	output a leading s0 record
 *		s2	S2 records only
 *		s3	S3 records only
 *		ns8	do not output a trailing s8 record
 *		split	split into byte0/byte1 output files
 *		split4	split into byte0/byte1/byte2/byte3 output files
 *		psz#	size of proms (in Kbytes (i.e. 1024 bytes)
 *		saddr	starting load address (in hex). files starting address
 *			is used if saddr is not present
 *
 *     If sz# is used, output will be placed in files: ifile.HEX.#
 *     If split is used, output will be placed in files: ifile.HEX.#.byte[01]
 *     If split4 is used, output will be placed in files: ifile.HEX.#.byte[0-3]
 *     Otherwise, output is directed to stdout.
 */

#include <stdio.h>
#include <a.out.h>
#include <sys/types.h>
#include <sys/stat.h>

#define MAXOUTFILE	16
#define MAXRECSIZ	1000
#define RECSIZ		32

char *ifile;
struct exec exec;
char tbuf[MAXRECSIZ*4+10];
char recbuf[MAXRECSIZ+10];
int recsiz = RECSIZ;
int bflg;
int lflg;
int fflg;
int rflg;
int s0flg;
int s2flg;
int s3flg;
int ns8flg;
int splitflg = 1;
int psize;
int csum;
FILE *outf[MAXOUTFILE];
FILE *inp;

char hex[] = {
'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
};

#define FATAL 1	
#define NONFATAL 0
#define A16	((unsigned) 0x0000ffff)
#define A24	((unsigned) 0x00ffffff)

/* 
 * main - process args, open files
 */
main(argc, argv)
char **argv;
{
	register i, fn, pn, isize, npbas;
	int saddr = -1;
	char name[40];

	ifile = NULL;
	if (argc == 1)
bad:
	    diagnostic(FATAL, "usage:hex [-f] [-l] [-n#] [-r] [-s0] [-s2] [-ns8] [-split[4]] [-psz#] [+saddr] ifile", NULL);
	while (argc > 1) {
		if (strcmp("-f", argv[1]) == 0) {
			fflg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-l", argv[1]) == 0) {
			lflg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-r", argv[1]) == 0) {
			rflg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-s0", argv[1]) == 0) {
			s0flg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-s2", argv[1]) == 0) {
			s2flg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-s3", argv[1]) == 0) {
			s3flg++;
			--argc; ++argv;
			continue;
		}
		if (strcmp("-ns8", argv[1]) == 0) {
			ns8flg++;
			--argc; ++argv;
			continue;
		}
		if (strncmp("-split", argv[1], 6) == 0) {
			if (argv[1][6] == '4')
				splitflg = 4;
			else
				splitflg = 2;
			--argc; ++argv;
			continue;
		}
		if (strncmp("-psz", argv[1], 4) == 0) {
			psize = atoi(&argv[1][4]) * 1024;
			if (psize == 0)
				diagnostic(FATAL, "hex:psz parameter must be non-zero", NULL);
			--argc; ++argv;
			continue;
		}
		if (strncmp("-n", argv[1], 2) == 0) {
			recsiz = atoi(&argv[1][2]);
			if (recsiz >= MAXRECSIZ)
				diagnostic(FATAL, "hex:n parameter must be less than %d", MAXRECSIZ, NULL);
			--argc; ++argv;
			continue;
		}
		if (*argv[1] == '+') {
			sscanf(argv[1] , "+%lx" , &saddr);
			--argc; ++argv;
			continue;
		}
		if (ifile != NULL)
			goto bad;
		ifile = argv[1];
		--argc;
		++argv;
	}

	if (ifile == NULL)
		goto bad;

	if ((inp = fopen(ifile, "r")) == NULL) 		
		diagnostic(FATAL, "Can't open ", ifile, NULL);

	if (fflg) {
		if ((isize = siz()) == 0)
			diagnostic(FATAL, "can't stat file or zero length file", NULL);
	} else {
		seekfile(0);
		if (getbytes(&exec, sizeof(struct exec)) != sizeof(struct exec))
			diagnostic(FATAL, "Can't fetch file header", NULL);

		if (N_BADMAG(exec))
			diagnostic(FATAL, "Bad file type", NULL);
		if (bflg)
		    isize = exec.a_text + exec.a_data + exec.a_bss;
		else
		    isize = exec.a_text + exec.a_data;
	}
	if (psize == 0)
		psize = isize;

	npbas = (((isize + psize - 1) / psize) + splitflg - 1) / splitflg;
	if (saddr == -1)
		if (fflg) {
			saddr = 0;
		} else {
			if (npbas != 1)
				diagnostic(FATAL, "need saddr for multiple proms", NULL);
			else
				saddr = exec.a_entry;
		}
	if (npbas == 1 && splitflg == 1)
	    outf[0] = stdout;
	else
	    for (pn = 0; pn < npbas; pn++)
		for (fn = 0; fn < splitflg; fn++) {
		    if (splitflg > 1)
			sprintf(name, "%s.HEX.%d.byte%d", ifile, pn, fn);
		    else
			sprintf(name, "%s.HEX.%d", ifile, pn);
		    if ((outf[pn*splitflg+fn] = fopen(name, "w+")) == NULL)
			diagnostic(FATAL, "Can't open ", outf[pn*splitflg+fn], NULL);
		}
	if (lflg)
		fprintf(stderr, "Load starting at 0x%lx\n", saddr);

	if (s0flg)
		for (fn = 0; fn < npbas * splitflg; fn++)
			sends0(ifile, outf[fn]);
	if (fflg) {
		senddata(saddr, isize);
	} else {
		seekfile(N_TXTOFF(exec));
		senddata(saddr, exec.a_text);

		seekfile(N_TXTOFF(exec) + exec.a_text);
		senddata(saddr + exec.a_text, exec.a_data);

		if (bflg) {
		    seekfile(N_TXTOFF(exec) + exec.a_text + exec.a_data);	
		    senddata(saddr + exec.a_text + exec.a_data, exec.a_bss);
		}
	}
	if (ns8flg == 0)
		for (fn = 0; fn < npbas * splitflg; fn++)
			sends8(saddr, outf[fn]);
	exit(0);
}

siz()
{
	struct stat sb;

	if (fstat(fileno(inp), &sb) < 0)
		return(0);
	return(sb.st_size);
}

diagnostic(fatal, args)
char *args;
{
	register char **a;

	for (a = &args; *a != NULL; a++)
		fprintf(stderr, "%s", *a);
	fprintf(stderr, "\n");
	if (fatal)
		exit(0);
}


seekfile(loc)
{
	fseek(inp, loc, 0);
}

getbytes(addr, ct)
register char *addr;
register int ct;
{
	register int n;

	n = 0;
	while (ct-- > 0) {
		*addr = getc(inp);
		if (feof(inp) || ferror(inp))
			break;
		addr++;
		n++;
	}
	return(n);
}

sends0(fname, ofp)
register char *fname;
FILE *ofp;
{
	csum = 0;
	putc('S', ofp);
	putc('0', ofp);
	sendbyte(strlen(fname)+1, ofp);
	while (*fname != 0)
		sendbyte(*fname++, ofp);
	sendbyte((255-csum) & 0xff, ofp);
	sendend(ofp);
}

sends8(addr, ofp)
FILE *ofp;
{
	csum = 0;
	if (s3flg || addr > A24) {
		putc('S', ofp);
		putc('7', ofp);
		sendbyte(4+1, ofp);
		send32(addr, ofp);
	} else if (s2flg || addr > A16) {
		putc('S', ofp);
		putc('8', ofp);
		sendbyte(3+1, ofp);
		send24(addr, ofp);
	} else {
		putc('S', ofp);
		putc('9', ofp);
		sendbyte(2+1, ofp);
		send16((int) addr, ofp);
	}
	sendbyte((255-csum) & 0xff, ofp);
	sendend(ofp);
}

senddata(addr, ct)
register int addr, ct;
{
	register int n, i, fn;
	char	*c;
	static pn = 0;

	addr %= psize;
	for (;ct > 0; ct -= n) { 
		if (addr >= psize) {
			pn++;
			addr %= psize;
		}
		n = ct;
		if (n > (recsiz * splitflg))
			n = recsiz * splitflg;
		if (n > ((psize - addr) * splitflg))
			n = (psize - addr) * splitflg;
		if ((n = getbytes(tbuf, n)) != n)
			diagnostic(FATAL, "Unexpected end of file", NULL);

		if (splitflg > 1)
		    for (fn = 0; fn < splitflg; fn++) {
			c = recbuf;
			for (i=fn; i<n; i+=splitflg)
		    	    *c++ = tbuf[i];
			if (c != recbuf)
			    sendrec(addr, recbuf, c - recbuf,
				outf[pn*splitflg+fn]);
		    }
		else
			sendrec(addr, tbuf, n, outf[pn]);

		addr += n/splitflg;
	}
}

sendrec(addr, buf, n, ofp)
register int addr, n;
char *buf;
FILE *ofp;
{
	register int i;

	csum = 0;
	if (s3flg || addr > A24) {
		putc('S', ofp);
		putc('3', ofp);
		sendbyte(4+n+1, ofp);
		send32(addr, ofp);
	} else if (s2flg || addr > A16) {
		putc('S', ofp);
		putc('2', ofp);
		sendbyte(3+n+1, ofp);
		send24(addr, ofp);
	} else {
		putc('S', ofp);
		putc('1', ofp);
		sendbyte(2+n+1, ofp);
		send16((int) addr, ofp);
	}
	for (i = 0; i < n; i++)
		sendbyte(buf[i], ofp);
	sendbyte((255-csum) & 0xff, ofp);
	sendend(ofp);
}

sendend(ofp)
FILE *ofp;
{
	if (rflg)
		putc('\r', ofp);
	else
		putc('\n', ofp);
}

sendbyte(c, ofp)
register int c;
FILE *ofp;
{
	csum += c;
	putc(hex[(c>>4)&0xf], ofp);
	putc(hex[c&0xf], ofp);
}


send16(x, ofp)
register int x;
FILE *ofp;
{
	sendbyte(x >> 8, ofp);
	sendbyte(x, ofp);
}


send24(x, ofp)
register int x;
FILE *ofp;
{
	sendbyte(x >> 16, ofp);
	sendbyte(x >> 8, ofp);
	sendbyte(x, ofp);
}

send32(x, ofp)
register int x;
FILE *ofp;
{
	sendbyte(x >> 24, ofp);
	sendbyte(x >> 16, ofp);
	sendbyte(x >> 8, ofp);
	sendbyte(x, ofp);
}
