(*
 * recvtp -- draw, move, and delete rectangles using libvtp
 *
 * compile with:
 *	pc -o recvtp -w -O recvtp.p -lvtp
 *)
program main(input,output);

const
#include "/usr/include/pascal/gconst.h"
	Space		= 10;	(* inter-object space		*)
	BoxLine		= 2;	(* linesize			*)
	BoxWidth	= 50;	(* width of color box		*)
	BoxHeight	= 50;	(* height of color box		*)
	Ncolors		= 9;	(* number of available colors	*)
	ColorBoxX	= 10;   (* color selection boxes	*)
	ColorBoxY	= 10;
	ColorBoxW	= 450;
	ColorBoxH	= 50;
	SelectBoxX	= 470;	(* current color		*)
	SelectBoxY	= 10;
	SelectBoxW	= 50;
	SelectBoxH	= 50;
	Nmode		= 3;	(* mode selection boxes		*)
	ModeBoxX	= 580;
	ModeBoxY	= 10;
	ModeBoxW	= 300;
	ModeBoxH	= 50;
	DrawMode	= 0;	(* mode values *)
	MoveMode	= 1;
	DeleteMode	= 2;
	DrawBoxX	= 10;  (* box drawing area *)
	DrawBoxY	= 70;

type
#include "/usr/include/pascal/gtype.h"
rectangle =
	packed record
	   x,y,w,h,color:short;	(* info describing any rectangle *)
end;
recarray = array [0..499] of rectangle;
colorarray = array [0..8] of integer;
marray = packed array [0..200] of char;

var
	istate : wstate;	(* initial window state		*)
	inmouse : vtseq;	(* for getvtseq inmouse		*)
	dlist : recarray;	(* display list of rectangles	*)
	DrawBoxW : integer;	(* drawing area width		*)
	DrawBoxH : integer;	(* drawing area height		*)
	rightactive : integer;	(* activates right button	*)
	ActiveColor : integer;	(* current color		*)
	ActiveMode : integer;	(* current mode			*)
	npoints : integer;	(* number of rectangles		*)
	colors : colorarray;	(* available colors		*)
	void : integer;		(* indicates return value unused*)
	message : marray;	(* message array		*)

#include "/usr/include/pascal/gproc.h"

procedure Background;
var
   i : integer;
begin
    SetPosition(1, 0, 0);   (* clear screen to white *)
    SetColor(1, VTWhite);
    PaintRectangleInterior(1, istate.width, istate.height);

    SetPosition(1, ColorBoxX, ColorBoxY);	(* display available colors *)
    for i:=0 to (Ncolors-1) do begin
	SetColor(1, colors[i]);
	PaintRectangleInterior(1, BoxWidth, BoxHeight);
	BumpXPosition(1, BoxWidth);
    end;

    SetColor(1, ActiveColor);			(* selected color	*)
    SetPosition(1, SelectBoxX, SelectBoxY);	(* select color box	*)
    PaintRectangleInterior(1, SelectBoxW, SelectBoxH);

    SetColor(1, VTBlack);		(* available color selections *)
    SetPosition(1, ColorBoxX, ColorBoxY);
    PaintRectangleBorder(1, ColorBoxW, ColorBoxH);
    i := ColorBoxX + BoxWidth;
    while (i < (ColorBoxX + ColorBoxW)) do begin
	SetPosition(1, i, ColorBoxY);
	PaintLine(1, 0, BoxHeight);
	i := i + BoxWidth;
    end;

    SetPosition(1, ModeBoxX, ModeBoxY);
    PaintRectangleBorder(1, ModeBoxW, ModeBoxH);
    i := ModeBoxX + 2*BoxWidth;
    while (i < (ModeBoxX + ModeBoxW)) do begin
	SetPosition(1, i, ModeBoxY);
	PaintLine(1, 0, BoxHeight);
	i := i + 2*BoxWidth;
    end;
    SetJustification(1, VTCENTER);
    SetPosition(1, (ModeBoxX+BoxWidth), (ModeBoxY+(ModeBoxH div 2)));
    message := 'Draw';
    message[4] := chr(0);
    PaintString(1, VTSTREND, message[0]);
    SetPosition(1, (ModeBoxX+(3*BoxWidth)), (ModeBoxY+(ModeBoxH div 2)));
    message := 'Move';
    message[4] := chr(0);
    PaintString(1, VTSTREND, message[0]);
    SetPosition(1, (ModeBoxX+(5*BoxWidth)), (ModeBoxY+(ModeBoxH div 2)));
    message := 'Delete';
    message[6] := chr(0);
    PaintString(1, VTSTREND, message[0]);
    InvertRegion(1, (ModeBoxX + ActiveMode * (2*BoxWidth)),
			ModeBoxY, (2*BoxWidth), ModeBoxH);

    SetPosition(1, SelectBoxX, SelectBoxY);
    PaintRectangleBorder(1, SelectBoxW, SelectBoxH);
end;

procedure Finish;
begin
    SetPermanentClipping(1, 0, 0, 10000, 10000);
    SetPosition(1, 0, 0);
    SetColor(1, VTWhite);
    PaintRectangleInterior(1, 10000, 10000);
    SetWindowState(1, istate);
    Flush(1);
end;

procedure DrawRectangle(var r:rectangle);
begin
    SetPosition(1, r.x, r.y);
    SetColor(1, r.color);
    PaintRectangleInterior(1, r.w, r.h);
    SetColor(1, VTBlack);
    PaintRectangleBorder(1, r.w, r.h);
end;

function UseableCoordinates : integer;
var
    r : integer;
begin
    r := 0;
    if ((inmouse.x > DrawBoxX)			and
	(inmouse.x < (DrawBoxX+DrawBoxW))	and
	(inmouse.y > DrawBoxY)			and
	(inmouse.y < (DrawBoxY+DrawBoxH)))
    then begin
	r := 1;
    end
    else if ((inmouse.x > ColorBoxX)		and
	(inmouse.x < (ColorBoxX+ColorBoxW))	and
	(inmouse.y > ColorBoxY)			and
	(inmouse.y < (ColorBoxY+ColorBoxH)))
    then begin
	ActiveColor := colors[(inmouse.x-ColorBoxX) div BoxWidth];
	SetColor(1, ActiveColor);
	SetPosition(1, SelectBoxX, SelectBoxY);
	PaintRectangleInterior(1, SelectBoxW, SelectBoxH);
	SetColor(1, VTBlack);
    end
    else if ((inmouse.x > ModeBoxX)		and
	(inmouse.x < (ModeBoxX+ModeBoxW))	and
	(inmouse.y > ModeBoxY)			and
	(inmouse.y < (ModeBoxY+ModeBoxH))) 
    then begin
	InvertRegion(1, (ModeBoxX + ActiveMode * (2*BoxWidth)),
			    ModeBoxY, (2*BoxWidth), ModeBoxH);
	ActiveMode := (inmouse.x - ModeBoxX) div (2*BoxWidth);
	InvertRegion(1, (ModeBoxX + ActiveMode * (2*BoxWidth)),
			    ModeBoxY, (2*BoxWidth), ModeBoxH);
    end;
    UseableCoordinates := r;
end;

function IdentifyBox:integer;
var
    index , k: short;
begin
    k := -1;
    index := npoints;
    while ((k < 0)and(index >= 0)) do begin
	if ((inmouse.x >= dlist[index].x-BoxLine)			and
	  (inmouse.x<=(dlist[index].x+dlist[index].w+2*BoxLine))	and
	  (inmouse.y>=dlist[index].y-BoxLine)				and
	  (inmouse.y<=(dlist[index].y+dlist[index].h+2*BoxLine)))
	then  k := index;
	index := index - 1;
    end;
    IdentifyBox := k;
end;

procedure RefreshBoxes(x, y, w, h:short);
var
   cx, cy, cw, ch, i : short;
begin
    GetPermanentClipping(1, cx, cy, cw, ch);
    SetPermanentClipping(1, DrawBoxX-BoxLine, DrawBoxY-BoxLine,
		    DrawBoxW+(2*BoxLine), DrawBoxH+(2*BoxLine));
    RestrictPermanentClipping(1, x, y, w, h);
    SetColor(1, VTWhite);
    SetPosition(1, x, y);
    PaintRectangleInterior(1, w, h);
    for i:=0 to npoints do DrawRectangle(dlist[i]);
    SetColor(1, VTBlack);
    SetPosition(1, DrawBoxX, DrawBoxY);
    PaintRectangleBorder(1, DrawBoxW, DrawBoxH);
    SetPermanentClipping(1, cx, cy, cw, ch);
end;

procedure MoveBox;
var
    box, x, y, w, h, i, xt, yt: short;
begin

    box := IdentifyBox;
    if (box >= 0) then begin
	x := dlist[box].x - BoxLine;
	y := dlist[box].y - BoxLine;
	w := dlist[box].w + (BoxLine*2);
	h := dlist[box].h + (BoxLine*2);
	xt := dlist[box].x;
	yt := dlist[box].y;
	void := TrackFixedBox(1, xt, yt, dlist[box].w, dlist[box].h,
	    DrawBoxX, DrawBoxY, DrawBoxW, DrawBoxH, BoxLine);
	dlist[box].x := xt;
	dlist[box].y := yt;
	dlist[npoints+1] := dlist[box];
	for i:=box to npoints do dlist[i] := dlist[i+1];
	RefreshBoxes(x, y, w, h);
	DrawRectangle(dlist[npoints]);
    end
    else begin
	message := 'not a box';
	message[9] := chr(0);
	DisplayStatus(1, message[0]);
    end;
end;

procedure Initialize;
begin
    rightactive := 0;
    npoints := -1;
    ActiveColor := VTWhite;
    ActiveMode := DrawMode;
    colors[0] := VTBlack;
    colors[1] := VTGray12;
    colors[2] := VTGray25;
    colors[3] := VTGray37;
    colors[4] := VTGray50;
    colors[5] := VTGray62;
    colors[6] := VTGray75;
    colors[7] := VTGray87;
    colors[8] := VTWhite;
    void := GetWindowState(1, istate);
    void := SetLineDisc(1, TWSDISC);
    void := BlockRefreshAdjust(1);
    SetRefresh(1,0);
    SetBuf(1, 1024);
    SetBColor(1, VTWhite);
    SetMouseMode(1, [VTMOUSEDOWN]);
    SetThickness(1, BoxLine);
    DrawBoxW := istate.width - DrawBoxX - (Space*2);
    DrawBoxH := istate.height - DrawBoxY - (Space*2);
    Background;
    RefreshBoxes(DrawBoxX-BoxLine, DrawBoxY-BoxLine,
		DrawBoxW+(2*BoxLine), DrawBoxH+(2*BoxLine));
end;

procedure VTRefresh(fd,id:integer; x, y, w, h:short);
var
    cx, cy, cw, ch : short;
begin
    GetPermanentClipping(1, cx, cy, cw, ch);
    SetPermanentClipping(1, x, y, w, h);
    Background;
    RefreshBoxes(x, y, w, h);
    SetPermanentClipping(1, cx, cy, cw, ch);
end;

procedure DeleteBox;
var
    box, x, y, w, h, i : short;
begin
    box := IdentifyBox;
    if (box > = 0) then begin
        x := dlist[box].x - BoxLine;
        y := dlist[box].y - BoxLine;
        w := dlist[box].w + (BoxLine*2);
        h := dlist[box].h + (BoxLine*2);
	for i:=box to (npoints-1) do dlist[i] := dlist[i+1];
        npoints := npoints - 1;
        RefreshBoxes(x, y, w, h);
    end
    else begin
	message := 'not a box';
	message[9] := chr(0);
	DisplayStatus(1, message[0]);
    end;
end;

procedure DrawBox;
var
   tw, th : short;
begin
    npoints := npoints + 1;
    dlist[npoints].x := inmouse.x;
    dlist[npoints].y := inmouse.y;
    dlist[npoints].w := 0;
    dlist[npoints].h := 0;
    dlist[npoints].color := ActiveColor;
    if (rightactive <> 0) then begin
	if (npoints > 0) then begin
	    dlist[npoints].w := dlist[npoints-1].w;
	    dlist[npoints].h := dlist[npoints-1].h;
	end;
    end
    else begin
	tw := 0;
	th := 0;
	void := TrackRubberBox(1, dlist[npoints].x, dlist[npoints].y, tw, th,
		    DrawBoxX, DrawBoxY, DrawBoxW, DrawBoxH, BoxLine);
	dlist[npoints].w := tw;
	dlist[npoints].h := th;
    end;
	(* width should always be positive *)
    if (dlist[npoints].w < 0) then begin
	dlist[npoints].w := -dlist[npoints].w;
	dlist[npoints].x := dlist[npoints].x - dlist[npoints].w;
    end;
	(* height should always be positive *)
    if (dlist[npoints].h < 0) then begin
	dlist[npoints].h := -dlist[npoints].h;
	dlist[npoints].y := dlist[npoints].y - dlist[npoints].h;
    end;
    DrawRectangle(dlist[npoints]);
end;

procedure TouchBoxes;
var
    j, i, done: short;
begin
    done := 0;
    while (done = 0) do begin
	i := getvtseq(1,inmouse);
	case i of
	    VTMOUSE: begin
	        rightactive := 0;
		if (VTMOUSERIGHT in inmouse.buttons) then rightactive := 1;
		if ((VTMOUSELEFT in inmouse.buttons) or
		    (VTMOUSERIGHT in inmouse.buttons)) then
		begin
		   j := UseableCoordinates;
		   if (j = 1) then begin
		      SetPermanentClipping(1,DrawBoxX,DrawBoxY,
					     DrawBoxW, DrawBoxH);
	              case ActiveMode of
			 DrawMode:	DrawBox;
			 MoveMode:	MoveBox;
			 DeleteMode:	DeleteBox;
		      end;
		      SetPermanentClipping(1, 0, 0, 10000, 10000);
		   end;
		end
		else if (VTMOUSEMIDDLE in inmouse.buttons) then begin
		    SetPosition(1, inmouse.x, inmouse.y);
		    message := 'Do@Clear@Exit@@';
		    message[2] := chr(0);
		    message[8] := chr(0);
		    message[13] := chr(0);
		    message[14] := chr(0);
		    j := DisplayPopUp(1,message[0]);
		    case j of
		       0 : (* Ignore *)
			   ;
		       1 : (* Clear *)
			   begin
		              npoints := -1;
		              SetPosition(1, DrawBoxX, DrawBoxY);
		              SetColor(1, VTWhite);
		              PaintRectangleInterior(1, DrawBoxW, DrawBoxH);
		              SetColor(1, ActiveColor);
			   end;
		       2 : (* Exit *)
		           done := 1;
		    end;
		end;
	    end;
	    VTEOF,VTASCII,VTHARDKEY : begin
	       message := 'use the mouse';
	       message[13] := chr(0);
	       DisplayStatus(1, message[0]);
	    end;
	end;
    end;
end;

begin
    Initialize;
    TouchBoxes;
    Finish;
end.
