/*
 * recvt -- draw, move, and delete rectangles using libvt
 *
 * compile with:
 *	cc -o recvt -O recvt.c -lvt
 */

#include <vt.h>

#define MAX_RECTANGLE	500	/* arbitrary maximum number of rectangles */
#define SPACE		10	/* inter-object space */
#define BOX_LINE	2	/* linesize */
#define BOX_WIDTH	50	/* width and height of color_box */
#define BOX_HEIGHT	50
#define NCOLORS		9	/* number of available colors */
#define COLOR_BOX_X	SPACE   /* color selection boxes */
#define COLOR_BOX_Y	SPACE
#define COLOR_BOX_W	(NCOLORS*BOX_WIDTH)
#define COLOR_BOX_H	BOX_HEIGHT
#define SELECT_BOX_X	(COLOR_BOX_X + COLOR_BOX_W + SPACE) /* current color */
#define SELECT_BOX_Y	SPACE
#define SELECT_BOX_W	BOX_WIDTH
#define SELECT_BOX_H	BOX_HEIGHT
#define NMODE		3	/* mode selection boxes */
#define MODE_BOX_X	(SELECT_BOX_X + SELECT_BOX_W + SPACE + BOX_WIDTH)
#define MODE_BOX_Y	SPACE
#define MODE_BOX_W	(NMODE*2*BOX_WIDTH)
#define MODE_BOX_H	BOX_HEIGHT
#define DRAW_MODE	0	/* mode values */
#define MOVE_MODE	1
#define DELETE_MODE	2
#define DRAW_BOX_X	(COLOR_BOX_X)  /* box drawing area */
#define DRAW_BOX_Y	(COLOR_BOX_Y + COLOR_BOX_H + SPACE)

struct wstate	istate;		/* initial window state */
struct vtseq	input;		/* for getvtseq input */
struct rectangle {
    short x;
    short y;
    short w;
    short h;
    short color;
}   dlist[MAX_RECTANGLE];		/* display list of rectangles */
short	DRAW_BOX_W, DRAW_BOX_H;		/* drawing area */
short	npoints = -1;			/* number of rectangles */
short	current_color = VT_White;	/* current color */
short	current_mode  = DRAW_MODE;	/* current mode */
short	repeat = 0;
short	colors[NCOLORS] = {		/* available colors */
    VT_Black,  VT_Gray12, VT_Gray25,
    VT_Gray37, VT_Gray50, VT_Gray62,
    VT_Gray75, VT_Gray87, VT_White
};
struct gconfig	gc;

main(argc, argv)
int	argc;
char	*argv[];
{
    if (GetGraphicsConfig(1, &gc) == -1) {
	    printf("%s: not a window\n", argv[0]);
	    exit(1);
    }
    Initialize();
    TouchBoxes();
    Finish();
}

Initialize()
{
    void Refresh(), RefreshBoxes();

    if (gc.realcolors >= sizeof(colors)/sizeof(short)) {
	/* use real colors on a color workstation */
	int	c;
	for (c=0; c<sizeof(colors)/sizeof(short); c++)
	    colors[c] = c;
    }

    GetWindowState(1, &istate);
    SetLineDisc(1, TWSDISC);
    BlockRefreshAdjust(1);
    SetRefresh(1, 0, Refresh);
    SetBuf(1, 512);
    SetBColor(1, VT_White);
    SetMouseMode(1, VT_MOUSE_DOWN);
    SetThickness(1, BOX_LINE);
    DRAW_BOX_W = istate.width - DRAW_BOX_X - (SPACE*2);
    DRAW_BOX_H = istate.height - DRAW_BOX_Y - (SPACE*2);
    Background();
    RefreshBoxes(DRAW_BOX_X-BOX_LINE, DRAW_BOX_Y-BOX_LINE,
		DRAW_BOX_W+(2*BOX_LINE), DRAW_BOX_H+(2*BOX_LINE));
}

Background()
{
    short   i;
    SetPosition(1, 0, 0);   /* clear screen to white */
    SetColor(1, VT_White);
    PaintRectangleInterior(1, istate.width, istate.height);

    SetPosition(1, COLOR_BOX_X, COLOR_BOX_Y);	/* display available colors */
    for (i=0; i<NCOLORS; i++) {
	SetColor(1, colors[i]);
	PaintRectangleInterior(1, BOX_WIDTH, BOX_HEIGHT);
	BumpXPosition(1, BOX_WIDTH);
    }

    SetColor(1, current_color);		/* selected color */
    SetPosition(1, SELECT_BOX_X, SELECT_BOX_Y);	 /* select color box */
    PaintRectangleInterior(1, SELECT_BOX_W, SELECT_BOX_H);

    SetColor(1, VT_Black);		/* available color selections */
    SetPosition(1, COLOR_BOX_X, COLOR_BOX_Y);
    PaintRectangleBorder(1, COLOR_BOX_W, COLOR_BOX_H);
    for (i=(COLOR_BOX_X+BOX_WIDTH); i<(COLOR_BOX_X+COLOR_BOX_W); i+=BOX_WIDTH) {
	SetPosition(1, i, COLOR_BOX_Y);
	PaintLine(1, 0, BOX_HEIGHT);
    }

    SetPosition(1, MODE_BOX_X, MODE_BOX_Y);
    PaintRectangleBorder(1, MODE_BOX_W, MODE_BOX_H);
    for (i=(MODE_BOX_X+(2*BOX_WIDTH)); i<(MODE_BOX_X+MODE_BOX_W);
		i+=(2*BOX_WIDTH)) {
	SetPosition(1, i, MODE_BOX_Y);
	PaintLine(1, 0, BOX_HEIGHT);
    }
    SetJustification(1, VT_CENTER);
    SetPosition(1, (MODE_BOX_X+BOX_WIDTH), (MODE_BOX_Y+(MODE_BOX_H/2)));
    PaintString(1, VT_STREND, "Draw");
    SetPosition(1, (MODE_BOX_X+(3*BOX_WIDTH)), (MODE_BOX_Y+(MODE_BOX_H/2)));
    PaintString(1, VT_STREND, "Move");
    SetPosition(1, (MODE_BOX_X+(5*BOX_WIDTH)), (MODE_BOX_Y+(MODE_BOX_H/2)));
    PaintString(1, VT_STREND, "Delete");
    InvertRegion(1, (MODE_BOX_X + current_mode * (2*BOX_WIDTH)),
			MODE_BOX_Y, (2*BOX_WIDTH), MODE_BOX_H);

    SetPosition(1, SELECT_BOX_X, SELECT_BOX_Y);
    PaintRectangleBorder(1, SELECT_BOX_W, SELECT_BOX_H);
}

void RefreshBoxes(x, y, w, h)
short	x, y, w, h;
{
    short	cx, cy, cw, ch, i;

    GetPermanentClipping(1, &cx, &cy, &cw, &ch);
    SetPermanentClipping(1, DRAW_BOX_X-BOX_LINE, DRAW_BOX_Y-BOX_LINE,
		    DRAW_BOX_W+(2*BOX_LINE), DRAW_BOX_H+(2*BOX_LINE));
    RestrictPermanentClipping(1, x, y, w, h);
    SetColor(1, VT_White);
    SetPosition(1, x, y);
    PaintRectangleInterior(1, w, h);
    for (i=0; i<=npoints; i++) DrawRectangle(&dlist[i]);
    SetColor(1, VT_Black);
    SetPosition(1, DRAW_BOX_X, DRAW_BOX_Y);
    PaintRectangleBorder(1, DRAW_BOX_W, DRAW_BOX_H);
    SetPermanentClipping(1, cx, cy, cw, ch);
}

void Refresh(id, x, y, w, h)
int	id;
short	x, y, w, h;
{
    short	cx, cy, cw, ch;

    GetPermanentClipping(1, &cx, &cy, &cw, &ch);
    SetPermanentClipping(1, x, y, w, h);
    Background();
    RefreshBoxes(x, y, w, h);
    SetPermanentClipping(1, cx, cy, cw, ch);
}

Finish()
{
    SetPermanentClipping(1, 0, 0, 10000, 10000);
    SetPosition(1, 0, 0);
    SetColor(1, VT_White);
    PaintRectangleInterior(1, 10000, 10000);
    SetWindowState(1, &istate);
}

TouchBoxes()
{
    for (;;) {
	switch (getvtseq(1, &input)) {
	case VT_MOUSE:
	    repeat = 0;
	    switch(input.u.mouse.buttons &
		(VT_MOUSE_LEFT|VT_MOUSE_MIDDLE|VT_MOUSE_RIGHT) ) {
	    case VT_MOUSE_RIGHT:
		repeat = 1;
	    case VT_MOUSE_LEFT:
		if (UseableCoordinates()){
		    SetPermanentClipping(1, DRAW_BOX_X, DRAW_BOX_Y,
					    DRAW_BOX_W, DRAW_BOX_H);
		    switch (current_mode) {
		    case DRAW_MODE:
			DrawBox();
			break;
		    case MOVE_MODE:
			MoveBox();
			break;
		    case DELETE_MODE:
			DeleteBox();
			break;
		    default:
			break;
		    }
		    SetPermanentClipping(1, 0, 0, 10000, 10000);
		}
		break;
	    case VT_MOUSE_MIDDLE:
		SetPosition(1, input.u.mouse.x, input.u.mouse.y);
		switch(DisplayPopUp(1, "Do\0Clear\0Exit\0")) {
		case 1:		  /* Clear */
		    npoints = -1;
		    SetPosition(1, DRAW_BOX_X, DRAW_BOX_Y);
		    SetColor(1, VT_White);
		    PaintRectangleInterior(1, DRAW_BOX_W, DRAW_BOX_H);
		    SetColor(1, current_color);
		    break;
		case 2:		  /* Exit */
		    return;
		    break;
		}
		break;
	    }
	    break;
	default:
	    DisplayStatus(1, "use the mouse");
	    break;
	}
    }
}

DrawRectangle(r)
struct rectangle    *r;
{
    SetPosition(1, r->x, r->y);
    SetColor(1, r->color);
    PaintRectangleInterior(1, r->w, r->h);
    SetColor(1, VT_Black);
    PaintRectangleBorder(1, r->w, r->h);
}

int UseableCoordinates()
{
    if ((input.u.mouse.x > DRAW_BOX_X)			&&
	(input.u.mouse.x < (DRAW_BOX_X+DRAW_BOX_W))	&&
	(input.u.mouse.y > DRAW_BOX_Y)			&&
	(input.u.mouse.y < (DRAW_BOX_Y+DRAW_BOX_H))) {
	return(1);
    } else if ((input.u.mouse.x > COLOR_BOX_X)		&&
	(input.u.mouse.x < (COLOR_BOX_X+COLOR_BOX_W))	&&
	(input.u.mouse.y > COLOR_BOX_Y)			&&
	(input.u.mouse.y < (COLOR_BOX_Y+COLOR_BOX_H))) {
	current_color = colors[(input.u.mouse.x-COLOR_BOX_X)/BOX_WIDTH];
	SetColor(1, current_color);
	SetPosition(1, SELECT_BOX_X, SELECT_BOX_Y);
	PaintRectangleInterior(1, SELECT_BOX_W, SELECT_BOX_H);
	SetColor(1, VT_Black);
    } else if ((input.u.mouse.x > MODE_BOX_X)		&&
	(input.u.mouse.x < (MODE_BOX_X+MODE_BOX_W))	&&
	(input.u.mouse.y > MODE_BOX_Y)			&&
	(input.u.mouse.y < (MODE_BOX_Y+MODE_BOX_H))) {
	InvertRegion(1, (MODE_BOX_X + current_mode * (2*BOX_WIDTH)),
			    MODE_BOX_Y, (2*BOX_WIDTH), MODE_BOX_H);
	current_mode = (input.u.mouse.x - MODE_BOX_X)/(2*BOX_WIDTH);
	InvertRegion(1, (MODE_BOX_X + current_mode * (2*BOX_WIDTH)),
			    MODE_BOX_Y, (2*BOX_WIDTH), MODE_BOX_H);
    }
    return(0);
}

int IdentifyBox()
{
    short	index;

    for (index=npoints; index>=0; index--) {
	if ((input.u.mouse.x >= dlist[index].x-BOX_LINE)		&&
	  (input.u.mouse.x<=(dlist[index].x+dlist[index].w+2*BOX_LINE))	&&
	  (input.u.mouse.y>=dlist[index].y-BOX_LINE)			&&
	  (input.u.mouse.y<=(dlist[index].y+dlist[index].h+2*BOX_LINE)))
		break;
    }
    return(index);
}

MoveBox()
{
    short	box, x, y, w, h, i;

    if ((box=IdentifyBox()) >= 0) {
	x = dlist[box].x - BOX_LINE;
	y = dlist[box].y - BOX_LINE;
	w = dlist[box].w + (BOX_LINE*2);
	h = dlist[box].h + (BOX_LINE*2);
	TrackFixedBox(1,
	    &dlist[box].x, &dlist[box].y, dlist[box].w, dlist[box].h,
	    DRAW_BOX_X, DRAW_BOX_Y, DRAW_BOX_W, DRAW_BOX_H, BOX_LINE);
	dlist[npoints+1] = dlist[box];
	for (i=box; i<=npoints; i++) {
	    dlist[i] = dlist[i+1];
	}
	RefreshBoxes(x, y, w, h);
	DrawRectangle(&dlist[npoints]);
    } else {
	DisplayStatus(1, "not a box");
    }
}

DeleteBox()
{
    short box, x, y, w, h, i;

    if ((box=IdentifyBox()) >= 0) {
        x = dlist[box].x - BOX_LINE;
        y = dlist[box].y - BOX_LINE;
        w = dlist[box].w + (BOX_LINE*2);
        h = dlist[box].h + (BOX_LINE*2);
        for (i=box; i<npoints; i++) {
	    dlist[i] = dlist[i+1];
        }
        npoints--;
        RefreshBoxes(x, y, w, h);
    } else {
	DisplayStatus(1, "not a box");
    }
}

DrawBox()
{
    npoints ++;
    dlist[npoints].x = input.u.mouse.x;
    dlist[npoints].y = input.u.mouse.y;
    dlist[npoints].w = 0;
    dlist[npoints].h = 0;
    dlist[npoints].color = current_color;
    if (repeat != 0) {
	if (npoints > 0) {
	    dlist[npoints].w = dlist[npoints-1].w;
	    dlist[npoints].h = dlist[npoints-1].h;
	}
    }
    else {
	TrackRubberBox(1, dlist[npoints].x, dlist[npoints].y,
		    &dlist[npoints].w, &dlist[npoints].h,
		    DRAW_BOX_X, DRAW_BOX_Y, DRAW_BOX_W, DRAW_BOX_H, BOX_LINE);
    }
    if (dlist[npoints].w < 0) { /* width should always be positive */
	dlist[npoints].w = -dlist[npoints].w;
	dlist[npoints].x -= dlist[npoints].w;
    }
    if (dlist[npoints].h < 0) { /* height should always be positive */
	dlist[npoints].h = -dlist[npoints].h;
	dlist[npoints].y -= dlist[npoints].h;
    }
    DrawRectangle(&dlist[npoints]);
}
