(*
 * recbmp -- draw, move, and delete rectangles using libvtp and libbmp
 *
 * compile with:
 *	pc -o recbmp -w -O recbmp.p -lbmp -lvtp
 *)
program main(input,output);

const
#include "/usr/include/pascal/gconst.h"
	Space		= 10;	(* inter-object space		*)
	BoxLine		= 2;	(* linesize			*)
	BoxWidth	= 50;	(* width of color box		*)
	BoxHeight	= 50;	(* height of color box		*)
	Ncolors		= 9;	(* number of available colors	*)
	ColorBoxX	= 10;   (* color selection boxes	*)
	ColorBoxY	= 10;
	ColorBoxW	= 450;
	ColorBoxH	= 50;
	SelectBoxX	= 470;	(* current color		*)
	SelectBoxY	= 10;
	SelectBoxW	= 50;
	SelectBoxH	= 50;
	Nmode		= 3;	(* mode selection boxes		*)
	ModeBoxX	= 580;
	ModeBoxY	= 10;
	ModeBoxW	= 300;
	ModeBoxH	= 50;
	DrawMode	= 0;	(* mode values *)
	MoveMode	= 1;
	DeleteMode	= 2;
	DrawBoxX	= 10;  (* box drawing area *)
	DrawBoxY	= 70;

type
#include "/usr/include/pascal/gtype.h"
rectangle =
	packed record
	   x,y,w,h,color:short;	(* info describing any rectangle *)
end;
recarray = array [0..499] of rectangle;
colorarray = array [0..8] of integer;
marray = packed array [0..200] of char;

var
	istate : wstate;	(* initial window state		*)
	inmouse : vtseq;	(* for getvtseq inmouse		*)
	dlist : recarray;	(* display list of rectangles	*)
	DrawBoxW : integer;	(* drawing area width		*)
	DrawBoxH : integer;	(* drawing area height		*)
	rightactive : integer;	(* activates right button	*)
	ActiveColor : integer;	(* current color		*)
	ActiveMode : integer;	(* current mode			*)
	npoints : integer;	(* number of rectangles		*)
	colors : colorarray;	(* available colors		*)
	void : integer;		(* indicates return value unused*)
	message : marray;	(* message array		*)
	bmd : BMD;		(* bitmap descriptor		*)

#include "/usr/include/pascal/gproc.h"

procedure VTRefresh(fd,id:integer; x, y, w, h:short);
begin
    if ((x+w) > bmd^.width) then begin
	w := bmd^.width - x;
    end;
    if ((y+h) > bmd^.height) then begin
	h := bmd^.height - y;
    end;
    void := bmDisplayBitmap(1,COPYRASTER,bmd,x,y,x,y,w,h,0);
end;

procedure Background;
var
   i : integer;
begin
    bmClearRegion(bmd,0,0,istate.width,istate.height,VTWhite);

    bmSetPosition(bmd, ColorBoxX, ColorBoxY);	(* display available colors *)
    for i:=0 to (Ncolors-1) do begin
	bmSetColor(bmd, colors[i]);
	bmPaintRectangleInterior(bmd, BoxWidth, BoxHeight);
	bmBumpXPosition(bmd, BoxWidth);
    end;

    bmSetColor(bmd, ActiveColor);		(* selected color	*)
    bmSetPosition(bmd, SelectBoxX, SelectBoxY);	(* select color box	*)
    bmPaintRectangleInterior(bmd, SelectBoxW, SelectBoxH);

    bmSetColor(bmd, VTBlack);		(* available color selections *)
    bmSetPosition(bmd, ColorBoxX, ColorBoxY);
    bmPaintRectangleBorder(bmd, ColorBoxW, ColorBoxH);
    i := ColorBoxX + BoxWidth;
    while (i < (ColorBoxX + ColorBoxW)) do begin
	bmSetPosition(bmd, i, ColorBoxY);
	bmPaintLine(bmd, 0, BoxHeight);
	i := i + BoxWidth;
    end;

    bmSetPosition(bmd, ModeBoxX, ModeBoxY);
    bmPaintRectangleBorder(bmd, ModeBoxW, ModeBoxH);
    i := ModeBoxX + 2*BoxWidth;
    while (i < (ModeBoxX + ModeBoxW)) do begin
	bmSetPosition(bmd, i, ModeBoxY);
	bmPaintLine(bmd, 0, BoxHeight);
	i := i + 2*BoxWidth;
    end;
    bmSetJustification(bmd, VTCENTER);
    bmSetPosition(bmd, (ModeBoxX+BoxWidth), (ModeBoxY+(ModeBoxH div 2)));
    message := 'Draw';
    message[4] := chr(0);
    bmPaintString(bmd, VTSTREND, message[0]);
    bmSetPosition(bmd, (ModeBoxX+(3*BoxWidth)), (ModeBoxY+(ModeBoxH div 2)));
    message := 'Move';
    message[4] := chr(0);
    bmPaintString(bmd, VTSTREND, message[0]);
    bmSetPosition(bmd, (ModeBoxX+(5*BoxWidth)), (ModeBoxY+(ModeBoxH div 2)));
    message := 'Delete';
    message[6] := chr(0);
    bmPaintString(bmd, VTSTREND, message[0]);
    bmInvertRegion(bmd, (ModeBoxX + ActiveMode * (2*BoxWidth)),
			ModeBoxY, (2*BoxWidth), ModeBoxH);

    bmSetPosition(bmd, SelectBoxX, SelectBoxY);
    bmPaintRectangleBorder(bmd, SelectBoxW, SelectBoxH);
    VTRefresh(1,0,0,0,istate.width,istate.height);
end;

procedure Finish;
begin
    SetPermanentClipping(1, 0, 0, 10000, 10000);
    SetPosition(1, 0, 0);
    SetColor(1, VTWhite);
    PaintRectangleInterior(1, 10000, 10000);
    SetWindowState(1, istate);
    Flush(1);
end;

procedure DrawRectangle(var r:rectangle);
begin
    bmSetPermanentClipping(bmd,DrawBoxX,DrawBoxY,DrawBoxW,DrawBoxH);
    bmSetPosition(bmd, r.x, r.y);
    bmSetColor(bmd, r.color);
    bmPaintRectangleInterior(bmd, r.w, r.h);
    bmSetColor(bmd, VTBlack);
    bmPaintRectangleBorder(bmd, r.w, r.h);
    bmSetPermanentClipping(bmd,0,0,bmd^.width,bmd^.height);
end;

function UseableCoordinates : integer;
var
    r : integer;
begin
    r := 0;
    if ((inmouse.x > DrawBoxX)			and
	(inmouse.x < (DrawBoxX+DrawBoxW))	and
	(inmouse.y > DrawBoxY)			and
	(inmouse.y < (DrawBoxY+DrawBoxH)))
    then begin
	r := 1;
    end
    else if ((inmouse.x > ColorBoxX)		and
	(inmouse.x < (ColorBoxX+ColorBoxW))	and
	(inmouse.y > ColorBoxY)			and
	(inmouse.y < (ColorBoxY+ColorBoxH)))
    then begin
	ActiveColor := colors[(inmouse.x-ColorBoxX) div BoxWidth];
	bmSetColor(bmd, ActiveColor);
	bmSetPosition(bmd, SelectBoxX, SelectBoxY);
	bmPaintRectangleInterior(bmd, SelectBoxW, SelectBoxH);
	bmSetColor(bmd, VTBlack);
	VTRefresh(1,0,SelectBoxX,SelectBoxY,SelectBoxW,SelectBoxH);
    end
    else if ((inmouse.x > ModeBoxX)		and
	(inmouse.x < (ModeBoxX+ModeBoxW))	and
	(inmouse.y > ModeBoxY)			and
	(inmouse.y < (ModeBoxY+ModeBoxH))) 
    then begin
	bmInvertRegion(bmd, (ModeBoxX + ActiveMode * (2*BoxWidth)),
			    ModeBoxY, (2*BoxWidth), ModeBoxH);
	VTRefresh(1,0, (ModeBoxX + ActiveMode * (2*BoxWidth)),
			    ModeBoxY, (2*BoxWidth), ModeBoxH);
	ActiveMode := (inmouse.x - ModeBoxX) div (2*BoxWidth);
	bmInvertRegion(bmd, (ModeBoxX + ActiveMode * (2*BoxWidth)),
			    ModeBoxY, (2*BoxWidth), ModeBoxH);
	VTRefresh(1,0, (ModeBoxX + ActiveMode * (2*BoxWidth)),
			    ModeBoxY, (2*BoxWidth), ModeBoxH);
    end;
    UseableCoordinates := r;
end;

function IdentifyBox:integer;
var
    index , k: short;
begin
    k := -1;
    index := npoints;
    while ((k < 0)and(index >= 0)) do begin
	if ((inmouse.x >= dlist[index].x-BoxLine)			and
	  (inmouse.x<=(dlist[index].x+dlist[index].w+2*BoxLine))	and
	  (inmouse.y>=dlist[index].y-BoxLine)				and
	  (inmouse.y<=(dlist[index].y+dlist[index].h+2*BoxLine)))
	then  k := index;
	index := index - 1;
    end;
    IdentifyBox := k;
end;

procedure RefreshBoxes(x, y, w, h:short);
var
   cx, cy, cw, ch, i : short;
begin
    bmGetPermanentClipping(bmd, cx, cy, cw, ch);
    bmSetPermanentClipping(bmd, DrawBoxX-BoxLine, DrawBoxY-BoxLine,
		    DrawBoxW+(2*BoxLine), DrawBoxH+(2*BoxLine));
    bmRestrictPermanentClipping(bmd, x, y, w, h);
    bmSetColor(bmd, VTWhite);
    bmSetPosition(bmd, x, y);
    bmPaintRectangleInterior(bmd, w, h);
    for i:=0 to npoints do DrawRectangle(dlist[i]);
    bmSetColor(bmd, VTBlack);
    bmSetPosition(bmd, DrawBoxX, DrawBoxY);
    bmPaintRectangleBorder(bmd, DrawBoxW, DrawBoxH);
    bmSetPermanentClipping(bmd, cx, cy, cw, ch);
    void := bmDisplayBitmap(1,COPYRASTER,bmd,x,y,x,y,w,h,0);
end;

procedure MoveBox;
var
    box, x, y, w, h, i, xt, yt: short;
begin

    box := IdentifyBox;
    if (box >= 0) then begin
	x := dlist[box].x - BoxLine;
	y := dlist[box].y - BoxLine;
	w := dlist[box].w + (BoxLine*2);
	h := dlist[box].h + (BoxLine*2);
	xt := dlist[box].x;
	yt := dlist[box].y;
	void := TrackFixedBox(1, xt, yt, dlist[box].w, dlist[box].h,
	    DrawBoxX, DrawBoxY, DrawBoxW, DrawBoxH, BoxLine);
	dlist[box].x := xt;
	dlist[box].y := yt;
	dlist[npoints+1] := dlist[box];
	for i:=box to npoints do dlist[i] := dlist[i+1];
	RefreshBoxes(x, y, w, h);
	DrawRectangle(dlist[npoints]);
    end
    else begin
	message := 'not a box';
	message[9] := chr(0);
	DisplayStatus(1, message[0]);
    end;
    VTRefresh(1,0,dlist[npoints].x-BoxLine,dlist[npoints].y-BoxLine,
		dlist[npoints].w+2*BoxLine,dlist[npoints].h+2*BoxLine);
end;

procedure Initialize;
begin
    void := GetWindowState(1, istate);
    if ( istate.width < (ModeBoxX + ModeBoxW + 2*BoxLine)) then
          istate.width := (ModeBoxX + ModeBoxW + 2*BoxLine);
    if ( istate.height < (ColorBoxY + Space + 2*ColorBoxH)) then 
          istate.height := (ColorBoxY + Space + 2*ColorBoxH);
    bmd := bmAllocate(istate.width,istate.height);
    void := SetLineDisc(1, TWSDISC);
    if (bmd <> nil) then begin
       rightactive := 0;
       npoints := -1;
       ActiveColor := VTWhite;
       ActiveMode := DrawMode;
       colors[0] := VTBlack;
       colors[1] := VTGray12;
       colors[2] := VTGray25;
       colors[3] := VTGray37;
       colors[4] := VTGray50;
       colors[5] := VTGray62;
       colors[6] := VTGray75;
       colors[7] := VTGray87;
       colors[8] := VTWhite;
       bmSetAddressing(bmd, VTRELATIVE);
       bmClearRegion(bmd, 0, 0, istate.width, istate.height, VTWhite);
       bmSetThickness(bmd, BoxLine);
       void := BlockRefreshAdjust(1);
       SetRefresh(1,0);
       SetMouseMode(1, [VTMOUSEDOWN]);
       SetThickness(1, BoxLine);
       DrawBoxW := istate.width - DrawBoxX - (Space*2);
       DrawBoxH := istate.height - DrawBoxY - (Space*2);
       Background;
       RefreshBoxes(DrawBoxX-BoxLine, DrawBoxY-BoxLine,
		DrawBoxW+(2*BoxLine), DrawBoxH+(2*BoxLine));
    end;
end;

procedure DeleteBox;
var
    box, x, y, w, h, i : short;
begin
    box := IdentifyBox;
    if (box > = 0) then begin
        x := dlist[box].x - BoxLine;
        y := dlist[box].y - BoxLine;
        w := dlist[box].w + (BoxLine*2);
        h := dlist[box].h + (BoxLine*2);
	for i:=box to (npoints-1) do dlist[i] := dlist[i+1];
        npoints := npoints - 1;
        RefreshBoxes(x, y, w, h);
    end
    else begin
	message := 'not a box';
	message[9] := chr(0);
	DisplayStatus(1, message[0]);
    end;
end;

procedure DrawBox;
var
   tw, th : short;
begin
    npoints := npoints + 1;
    dlist[npoints].x := inmouse.x;
    dlist[npoints].y := inmouse.y;
    dlist[npoints].w := 0;
    dlist[npoints].h := 0;
    dlist[npoints].color := ActiveColor;
    if (rightactive <> 0) then begin
	if (npoints > 0) then begin
	    dlist[npoints].w := dlist[npoints-1].w;
	    dlist[npoints].h := dlist[npoints-1].h;
	end;
    end
    else begin
	tw := 0;
	th := 0;
	void := TrackRubberBox(1, dlist[npoints].x, dlist[npoints].y, tw, th,
		    DrawBoxX, DrawBoxY, DrawBoxW, DrawBoxH, BoxLine);
	dlist[npoints].w := tw;
	dlist[npoints].h := th;
    end;
	(* width should always be positive *)
    if (dlist[npoints].w < 0) then begin
	dlist[npoints].w := -dlist[npoints].w;
	dlist[npoints].x := dlist[npoints].x - dlist[npoints].w;
    end;
	(* height should always be positive *)
    if (dlist[npoints].h < 0) then begin
	dlist[npoints].h := -dlist[npoints].h;
	dlist[npoints].y := dlist[npoints].y - dlist[npoints].h;
    end;
    DrawRectangle(dlist[npoints]);
    VTRefresh(1,0,dlist[npoints].x-BoxLine,dlist[npoints].y-BoxLine,
		dlist[npoints].w+2*BoxLine,dlist[npoints].h+2*BoxLine);
end;

procedure TouchBoxes;
var
    j, i, done: short;
begin
    done := 0;
    while (done = 0) do begin
	i := getvtseq(1,inmouse);
	RemoveStatus(1);
	case i of
	    VTMOUSE: begin
	        rightactive := 0;
		if (VTMOUSERIGHT in inmouse.buttons) then rightactive := 1;
		if ((VTMOUSELEFT in inmouse.buttons) or
		    (VTMOUSERIGHT in inmouse.buttons)) then
		begin
		   j := UseableCoordinates;
		   if (j = 1) then begin
		      SetPermanentClipping(1,DrawBoxX,DrawBoxY,
					     DrawBoxW, DrawBoxH);
	              case ActiveMode of
			 DrawMode:	DrawBox;
			 MoveMode:	MoveBox;
			 DeleteMode:	DeleteBox;
		      end;
		      SetPermanentClipping(1, 0, 0, 10000, 10000);
		   end;
		end
		else if (VTMOUSEMIDDLE in inmouse.buttons) then begin
		    SetPosition(1, inmouse.x, inmouse.y);
		    message := 'Do@Clear@Exit@@';
		    message[2] := chr(0);
		    message[8] := chr(0);
		    message[13] := chr(0);
		    message[14] := chr(0);
		    j := DisplayPopUp(1,message[0]);
		    case j of
		       0 : (* Ignore *)
			   ;
		       1 : (* Clear *)
			   begin
		              npoints := -1;
			      bmClearRegion(bmd, 0, 0, bmd^.width,
					    bmd^.height, VTWhite);
		              SetPosition(1, DrawBoxX, DrawBoxY);
		              SetColor(1, VTWhite);
		              PaintRectangleInterior(1, DrawBoxW, DrawBoxH);
			   end;
		       2 : (* Exit *)
		           done := 1;
		    end;
		end;
	    end;
	    VTEOF,VTASCII,VTHARDKEY : begin
	       message := 'use the mouse';
	       message[13] := chr(0);
	       DisplayStatus(1, message[0]);
	    end;
	end;
    end;
end;

begin
    Initialize;
    if (bmd <> nil) then begin
       TouchBoxes;
       Finish;
    end
    else begin
       writeln(' Pascal Bitmap Rectangle Program: ');
       writeln('     error: unable to allocate bitmap');
    end;
end.
