/*
 * recbm -- draw, move, and delete rectangles using libbm and libvt
 *
 * compile with:
 *	cc -o recbm -O recbm.c -lbm -lvt
 */

#include <vt.h>
#include <bitmap.h>

#define MAX_RECTANGLE		500	/* arbitrary max number of rectangles */
#define SPACE			10	/* inter-object space */
#define BOX_LINE		2	/* linesize */
#define BOX_WIDTH		50	/* width and height of color_box */
#define BOX_HEIGHT		50
#define NCOLORS			9	/* number of available colors */
#define COLOR_BOX_X		SPACE	/* color selection boxes */
#define COLOR_BOX_Y		SPACE
#define COLOR_BOX_W		(NCOLORS*BOX_WIDTH)
#define COLOR_BOX_H		BOX_HEIGHT
#define SELECT_BOX_X		(COLOR_BOX_X + COLOR_BOX_W + SPACE)
					/* current color */
#define SELECT_BOX_Y		SPACE
#define SELECT_BOX_W		BOX_WIDTH
#define SELECT_BOX_H		BOX_HEIGHT
#define NMODE			3	 /* mode selection boxes */
#define MODE_BOX_X		(SELECT_BOX_X+SELECT_BOX_W+SPACE+BOX_WIDTH)
#define MODE_BOX_Y		SPACE
#define MODE_BOX_W		(NMODE*2*BOX_WIDTH)
#define MODE_BOX_H		BOX_HEIGHT
#define DRAW_MODE		0	/* mode values */
#define MOVE_MODE		1
#define DELETE_MODE		2
#define DRAW_BOX_X		(COLOR_BOX_X)	/* box drawing area */
#define DRAW_BOX_Y		(COLOR_BOX_Y + COLOR_BOX_H + SPACE)
	
struct wstate	istate;		/* initial window state */
struct vtseq	input;		/* for getvtseq input */
struct rectangle {
    short x;
    short y;
    short w;
    short h;
    short *color;
}   dlist[MAX_RECTANGLE];	/* display list of rectangles */
short		DRAW_BOX_W, DRAW_BOX_H;		/* drawing area */
struct BMD	*bmd;
short		npoints = -1;			/* number of rectangles */
short		current_mode = DRAW_MODE;	/* current mode */
short		repeat = 0;
short		*current_color = BMP00;		/* current color */
short		*colors[NCOLORS] = {BMP01, BMP02, BMP03,
				    BMP04, BMP05, BMP06,
				    BMP07, BMP08, BMP00 };

main(argc, argv)
int	argc;
char	*argv[];
{
    struct gconfig	gc;

    if (GetGraphicsConfig(1, &gc) == -1) {
	    printf("%s: not a window\n", argv[0]);
	    exit(1);
    }
    Initialize();
    TouchBoxes();
    Finish();
}

Initialize()
{
    void Refresh(), RefreshBoxes();

    GetWindowState(1, &istate);
    if ( istate.width < (MODE_BOX_X + MODE_BOX_W + 2*BOX_LINE))
         istate.width = (MODE_BOX_X + MODE_BOX_W + 2*BOX_LINE);
    if ( istate.height < (COLOR_BOX_Y + SPACE + 2*COLOR_BOX_H)) 
         istate.height = (COLOR_BOX_Y + SPACE + 2*COLOR_BOX_H);
    if ((bmd = BM_Allocate(istate.width,istate.height)) == 0) {
       DisplayStatus(1, "Unable to Allocate bitmap");
       exit(1);
    }
    BM_SetAddressing(bmd, VT_RELATIVE);
    BM_ClearRegion(bmd, 0, 0, istate.width, istate.height, VT_White);
    BM_SetThickness(bmd, BOX_LINE);

    SetLineDisc(1, TWSDISC);
    BlockRefreshAdjust(1);
    SetRefresh(1, 0, Refresh);
    SetBColor(1, VT_White);
    SetMouseMode(1, VT_MOUSE_DOWN);

    DRAW_BOX_W = istate.width - DRAW_BOX_X - (SPACE*2);
    DRAW_BOX_H = istate.height - DRAW_BOX_Y - (SPACE*2);
    Background();
    RefreshBoxes(DRAW_BOX_X-BOX_LINE, DRAW_BOX_Y-BOX_LINE,
		DRAW_BOX_W+(2*BOX_LINE), DRAW_BOX_H+(2*BOX_LINE));
}

Background()
{
    short   i;

    BM_SetPosition(bmd, 0, 0);   /* clear region to white */
    BM_ClearRegion(bmd, 0, 0, istate.width, istate.height, VT_White);

    BM_SetPosition(bmd, COLOR_BOX_X, COLOR_BOX_Y);
    for (i=0; i<NCOLORS; i++) {
	BM_SetPattern(bmd, colors[i]);
	BM_PaintRectangleInterior(bmd, BOX_WIDTH, BOX_HEIGHT);
	BM_BumpXPosition(bmd, BOX_WIDTH);
    }

    BM_SetPattern(bmd, current_color);
    BM_SetPosition(bmd, SELECT_BOX_X, SELECT_BOX_Y);
    BM_PaintRectangleInterior(bmd, SELECT_BOX_W, SELECT_BOX_H);

    BM_SetPosition(bmd, COLOR_BOX_X, COLOR_BOX_Y);
    BM_SetColor(bmd,VT_Black);
    BM_PaintRectangleBorder(bmd, COLOR_BOX_W, COLOR_BOX_H);
    for (i=(COLOR_BOX_X+BOX_WIDTH); i<(COLOR_BOX_X+COLOR_BOX_W); i+=BOX_WIDTH) {
	BM_SetPosition(bmd, i, COLOR_BOX_Y);
	BM_PaintLine(bmd, 0, BOX_HEIGHT);
    }
    BM_SetPattern(bmd, current_color);

    BM_SetPosition(bmd, MODE_BOX_X, MODE_BOX_Y);
    BM_SetColor(bmd,VT_Black);
    BM_PaintRectangleBorder(bmd, MODE_BOX_W, MODE_BOX_H);
    for (i=(MODE_BOX_X+(2*BOX_WIDTH)); i<(MODE_BOX_X+MODE_BOX_W);
		i+=(2*BOX_WIDTH)) {
	BM_SetPosition(bmd, i, MODE_BOX_Y);
	BM_PaintLine(bmd, 0, BOX_HEIGHT);
    }
    BM_SetPattern(bmd, current_color);
    BM_SetJustification(bmd, VT_CENTER);
    BM_SetPosition(bmd, (MODE_BOX_X+BOX_WIDTH), (MODE_BOX_Y+(MODE_BOX_H/2)));
    BM_PaintString(bmd, VT_STREND, "Draw");
    BM_SetPosition(bmd, 
	(MODE_BOX_X+(3*BOX_WIDTH)), (MODE_BOX_Y+(MODE_BOX_H/2)));
    BM_PaintString(bmd, VT_STREND, "Move");
    BM_SetPosition(bmd, 
	(MODE_BOX_X+(5*BOX_WIDTH)), (MODE_BOX_Y+(MODE_BOX_H/2)));
    BM_PaintString(bmd, VT_STREND, "Delete");
    BM_InvertRegion(bmd, 
       (MODE_BOX_X + current_mode * (2*BOX_WIDTH)),
        MODE_BOX_Y, (2*BOX_WIDTH), MODE_BOX_H);

    BM_SetPosition(bmd, SELECT_BOX_X, SELECT_BOX_Y);
    BM_SetColor(bmd,VT_Black);
    BM_PaintRectangleBorder(bmd, SELECT_BOX_W, SELECT_BOX_H);
    BM_SetPattern(bmd, current_color);
    Refresh(1,0,0,istate.width,istate.height);
}

void RefreshBoxes(x, y, w, h)
short	x, y, w, h;
{
    short	cx, cy, cw, ch, i;

    BM_GetPermanentClipping(bmd, &cx, &cy, &cw, &ch);
    BM_SetPermanentClipping(bmd, 
       (DRAW_BOX_X-BOX_LINE),     (DRAW_BOX_Y-BOX_LINE),
       (DRAW_BOX_W+(2*BOX_LINE)), (DRAW_BOX_H+(2*BOX_LINE)));
    BM_RestrictPermanentClipping(bmd, x, y, w, h);
    BM_ClearRegion(bmd, x, y, w, h, VT_White);
    for (i=0; i<=npoints; i++) DrawRectangle(&dlist[i]);
    BM_SetPosition(bmd, DRAW_BOX_X, DRAW_BOX_Y);
    BM_SetColor(bmd,VT_Black);
    BM_PaintRectangleBorder(bmd, DRAW_BOX_W, DRAW_BOX_H);
    BM_SetPattern(bmd, current_color);
    BM_SetPermanentClipping(bmd, cx, cy, cw, ch);
    Refresh(1,x,y,w,h);
}

void Refresh(id, x, y, w, h)
int	id;
short	x, y, w, h;
{
    if (x > bmd->width) return;
    if (y > bmd->height) return;
    if ((x+w) > bmd->width)  w = bmd->width - x;
    if ((y+h) > bmd->height) h = bmd->height - y;
    BM_DisplayBitmap(1, COPYRASTER, bmd, x, y, x, y, w, h, 0);
}

Finish()
{
    BM_Deallocate(bmd);
    SetPermanentClipping(1, 0, 0, 10000, 10000);
    SetPosition(1, 0, 0);
    SetColor(1, VT_White);
    PaintRectangleInterior(1, 10000, 10000);
    SetWindowState(1, &istate);
    Flush(1);
}

TouchBoxes()
{
    for (;;) {
	switch (getvtseq(1, &input)) {
	case VT_MOUSE:
	    repeat = 0;
	    switch(input.u.mouse.buttons &
		(VT_MOUSE_LEFT|VT_MOUSE_MIDDLE|VT_MOUSE_RIGHT) ) {
	    case VT_MOUSE_RIGHT:
		repeat = 1;
	    case VT_MOUSE_LEFT:
		if (UseableCoordinates()){
		    BM_SetPermanentClipping(bmd, 
		       DRAW_BOX_X, DRAW_BOX_Y, DRAW_BOX_W, DRAW_BOX_H);
		    switch (current_mode) {
		    case DRAW_MODE:
			DrawBox();
			break;
		    case MOVE_MODE:
			MoveBox();
			break;
		    case DELETE_MODE:
			DeleteBox();
			break;
		    default:
			break;
		    }
		    BM_SetPermanentClipping(bmd, 0, 0, 10000, 10000);
		}
		break;
	    case VT_MOUSE_MIDDLE:
		SetPosition(1, input.u.mouse.x, input.u.mouse.y);
		switch(DisplayPopUp(1, "Do\0Clear\0Exit\0")) {
		case 1:		  /* Clear */
		    npoints = -1;
		    BM_ClearRegion(bmd,
		       DRAW_BOX_X, DRAW_BOX_Y, DRAW_BOX_W, DRAW_BOX_H, VT_White);
                    Refresh(1,DRAW_BOX_X,DRAW_BOX_Y,DRAW_BOX_W,DRAW_BOX_H);
		    break;
		case 2:		  /* Exit */
		    return;
		    break;
		}
		default: break;
	    }
	    break;
	default:
	    DisplayStatus(1, "use the mouse");
	    break;
	}
    }
}

DrawRectangle(r)
struct rectangle    *r;
{
    BM_SetPosition(bmd, r->x, r->y);
    BM_SetPattern(bmd, r->color);
    BM_PaintRectangleInterior(bmd, r->w, r->h);
    BM_SetColor(bmd,VT_Black);
    BM_PaintRectangleBorder(bmd, r->w, r->h);
    BM_SetPattern(bmd, r->color);
}

int UseableCoordinates()
{
    if ((input.u.mouse.x > DRAW_BOX_X)			&&
	(input.u.mouse.x < (DRAW_BOX_X+DRAW_BOX_W))	&&
	(input.u.mouse.y > DRAW_BOX_Y)			&&
	(input.u.mouse.y < (DRAW_BOX_Y+DRAW_BOX_H))) {
	return(1);
    } else if ((input.u.mouse.x > COLOR_BOX_X)		&&
	(input.u.mouse.x < (COLOR_BOX_X+COLOR_BOX_W))	&&
	(input.u.mouse.y > COLOR_BOX_Y)			&&
	(input.u.mouse.y < (COLOR_BOX_Y+COLOR_BOX_H))) {
	current_color = colors[(input.u.mouse.x-COLOR_BOX_X)/BOX_WIDTH];
	BM_SetPattern(bmd, current_color);
	BM_SetPosition(bmd, SELECT_BOX_X, SELECT_BOX_Y);
	BM_PaintRectangleInterior(bmd, SELECT_BOX_W, SELECT_BOX_H);
        Refresh(1,SELECT_BOX_X,SELECT_BOX_Y,SELECT_BOX_W,SELECT_BOX_H);
    } else if ((input.u.mouse.x > MODE_BOX_X)		&&
	(input.u.mouse.x < (MODE_BOX_X+MODE_BOX_W))	&&
	(input.u.mouse.y > MODE_BOX_Y)			&&
	(input.u.mouse.y < (MODE_BOX_Y+MODE_BOX_H)))
    {

	BM_SetPermanentClipping(bmd, 0, 0, istate.width, istate.height);
	BM_InvertRegion(bmd, 
	   (MODE_BOX_X + current_mode * (2*BOX_WIDTH)),
           MODE_BOX_Y, (2*BOX_WIDTH), MODE_BOX_H);
	current_mode = (input.u.mouse.x - MODE_BOX_X)/(2*BOX_WIDTH);
	BM_InvertRegion(bmd,
	   (MODE_BOX_X + current_mode * (2*BOX_WIDTH)),
           MODE_BOX_Y, (2*BOX_WIDTH), MODE_BOX_H);
        Refresh(1,MODE_BOX_X,MODE_BOX_Y,MODE_BOX_W,MODE_BOX_H);
    }
    return(0);
}

int IdentifyBox()
{
    short	index;

    for (index=npoints; index>=0; index--) {
	if ((input.u.mouse.x >= dlist[index].x-BOX_LINE)		&&
	  (input.u.mouse.x<=(dlist[index].x+dlist[index].w+2*BOX_LINE))	&&
	  (input.u.mouse.y>=dlist[index].y-BOX_LINE)			&&
	  (input.u.mouse.y<=(dlist[index].y+dlist[index].h+2*BOX_LINE)))
		break;
    }
    return(index);
}

MoveBox()
{
    short	box, x, y, w, h, i;

    if ((box=IdentifyBox()) >= 0) {
	x = dlist[box].x - BOX_LINE;
	y = dlist[box].y - BOX_LINE;
	w = dlist[box].w + (BOX_LINE*2);
	h = dlist[box].h + (BOX_LINE*2);
	TrackFixedBox(1,
	    &dlist[box].x, &dlist[box].y, dlist[box].w, dlist[box].h,
	    DRAW_BOX_X, DRAW_BOX_Y, DRAW_BOX_W, DRAW_BOX_H, BOX_LINE);
	dlist[npoints+1] = dlist[box];
	for (i=box; i<=npoints; i++) {
	    dlist[i] = dlist[i+1];
	}
	RefreshBoxes(x, y, w, h);
	DrawRectangle(&dlist[npoints]);
        Refresh(1,(dlist[npoints].x - BOX_LINE), 
		  (dlist[npoints].y - BOX_LINE),
	          (dlist[npoints].w + (BOX_LINE<<1)),
	          (dlist[npoints].h + (BOX_LINE<<1)));
    } else {
	DisplayStatus(1, "not a box");
    }
}

DeleteBox()
{
    short box, x, y, w, h, i;

    if ((box=IdentifyBox()) >= 0) {
        x = dlist[box].x - BOX_LINE;
        y = dlist[box].y - BOX_LINE;
        w = dlist[box].w + (BOX_LINE*2);
        h = dlist[box].h + (BOX_LINE*2);
        for (i=box; i<npoints; i++) {
	    dlist[i] = dlist[i+1];
        }
        npoints--;
        RefreshBoxes(x, y, w, h);
    } else {
	DisplayStatus(1, "not a box");
    }
}

DrawBox()
{
    npoints ++;
    dlist[npoints].x = input.u.mouse.x;
    dlist[npoints].y = input.u.mouse.y;
    dlist[npoints].w = 0;
    dlist[npoints].h = 0;
    dlist[npoints].color = current_color;
    if (repeat != 0) {
	if (npoints > 0) {
	    dlist[npoints].w = dlist[npoints-1].w;
	    dlist[npoints].h = dlist[npoints-1].h;
	}
    }
    else {
	TrackRubberBox(1, dlist[npoints].x, dlist[npoints].y,
		    &dlist[npoints].w, &dlist[npoints].h,
		    DRAW_BOX_X, DRAW_BOX_Y, DRAW_BOX_W, DRAW_BOX_H, BOX_LINE);
    }
    if (dlist[npoints].w < 0) { /* width should always be positive */
	dlist[npoints].w = -dlist[npoints].w;
	dlist[npoints].x -= dlist[npoints].w;
    }
    if (dlist[npoints].h < 0) { /* height should always be positive */
	dlist[npoints].h = -dlist[npoints].h;
	dlist[npoints].y -= dlist[npoints].h;
    }
    DrawRectangle(&dlist[npoints]);
    Refresh(1,(dlist[npoints].x - BOX_LINE),
	      (dlist[npoints].y - BOX_LINE),
	      (dlist[npoints].w + (BOX_LINE<<1)),
	      (dlist[npoints].h + (BOX_LINE<<1)));
}
