(*
 clkvtp -- an analog clock program using libvtp

 compile with:
	pc -o clkvtp -w -O clkvtp.p -lvtp

=============================================================
note: 
   The "unixprocs.h" include file must have the following
   format:
procedure sleep(len:integer);external;
function  time(tloc:integer):integer;external;
function localtime(var tloc:integer):TM;external;
=============================================================
 *)

program main(input,output);

const
#include "/usr/include/pascal/gconst.h"
	SCALE		= 1000000;
	NTICKS		= 60;
	BACKCOLOR	= VTGray37;	(* background color		*)
	SHADOWCOLOR	= VTBlack;	(* clock shadow color		*)
	BORDERCOLOR	= VTBlack;	(* clock border color		*)
	SURFCOLOR	= VTWhite;	(* clock surface color		*)
	HANDCOLOR	= VTBlack;	(* clock hand color		*)
	MARKCOLOR	= VTBlack;	(* clock marker color		*)
	POSTBCOLOR	= VTBlack;	(* clock post background color	*)
	POSTFCOLOR	= VTWhite;	(* clock post foreground color	*)

type
#include "/usr/include/pascal/gtype.h"
TM = ^TMRECORD;

TMRECORD = record;
	tmsec,
	tmmin,
	tmhour,
	tmmday,
	tmmon,
	tmyear,
	tmwday,
	tmyday,
	tmisdst : integer;
end;

TRIG = record
	   y, x : integer;	(* x and y coordinates of *)
end;				(* points on a large face *)
trigarray = array [0..NTICKS] of TRIG;
POS = record
	hourhand, 
	minutehand : short;
end;

var
   save : wstate;
   void  : integer;
   tm : TM;			(* time structure		*)
   table : trigarray;
   clockx, clocky : short;	(* clock position		*)
   clockr : short;		(* clock radius			*)
   pos, opos : POS;		(* cur and prev hand positions	*)
   rminute : short;		(* minute hand radius		*)
   rhour : short;		(* hour hand radius		*)
   rmarker : short;		(* radial position of markers	*)
   thickness : short;		(* thickness of face objects	*)
   curtime : integer;

#include "/usr/include/pascal/gproc.h"
#include "unixprocs.h"

procedure loadtable;
begin
	curtime := 0;
	(**************************************)
        (* note: min = minutes, deg = degrees *)
	(**************************************)
	table[0].y := 1000000;	table[0].x := 000000;	(* 0 min (90 deg) *)
	table[1].y := 994522;	table[1].x := 104529;	(* 1 min (84 deg) *)
	table[2].y := 978147;	table[2].x := 207912;	(* 2 min (78 deg) *)
	table[3].y := 951056;	table[3].x := 309017;	(* 3 min (72 deg) *)
	table[4].y := 913545;	table[4].x := 406737;	(* 4 min (66 deg) *)
	table[5].y := 866025;	table[5].x := 500000;	(* 5 min (60 deg) *)
	table[6].y := 809017;	table[6].x := 587786;	(* 6 min (54 deg) *)
	table[7].y := 743145;	table[7].x := 669131;	(* 7 min (48 deg) *)
	table[8].y := 669130;	table[8].x := 743145;	(* 8 min (42 deg) *)
	table[9].y := 587785;	table[9].x := 809017;	(* 9 min (36 deg) *)
	table[10].y := 500000;	table[10].x := 866025;	(* 10 min (30 deg) *)
	table[11].y := 406737;	table[11].x := 913546;	(* 11 min (24 deg) *)
	table[12].y := 309017;	table[12].x := 951057;	(* 12 min (18 deg) *)
	table[13].y := 207912;	table[13].x := 978148;	(* 13 min (12 deg) *)
	table[14].y := 104528;	table[14].x := 994522;	(* 14 min (6 deg) *)
	table[15].y := 000000;	table[15].x := 1000000;	(* 15 min (0 deg) *)
	table[16].y := -104531;	table[16].x := 994522;	(* 16 min (354 deg) *)
	table[17].y := -207914;	table[17].x := 978147;	(* 17 min (348 deg) *)
	table[18].y := -309019;	table[18].x := 951056;	(* 18 min (342 deg) *)
	table[19].y := -406739;	table[19].x := 913545;	(* 19 min (336 deg) *)
	table[20].y := -500002;	table[20].x := 866024;	(* 20 min (330 deg) *)
	table[21].y := -587787;	table[21].x := 809016;	(* 21 min (324 deg) *)
	table[22].y := -669132;	table[22].x := 743143;	(* 22 min (318 deg) *)
	table[23].y := -743146;	table[23].x := 669129;	(* 23 min (312 deg) *)
	table[24].y := -809018;	table[24].x := 587784;	(* 24 min (306 deg) *)
	table[25].y := -866026;	table[25].x := 499998;	(* 25 min (300 deg) *)
	table[26].y := -913546;	table[26].x := 406735;	(* 26 min (294 deg) *)
	table[27].y := -951057;	table[27].x := 309015;	(* 27 min (288 deg) *)
	table[28].y := -978148;	table[28].x := 207910;	(* 28 min (282 deg) *)
	table[29].y := -994522;	table[29].x := 104527;	(* 29 min (276 deg) *)
	table[30].y := -1000000;table[30].x := 000000;	(* 30 min (270 deg) *)
	table[31].y := -994522;	table[31].x := -104530;	(* 31 min (264 deg) *)
	table[32].y := -978147;	table[32].x := -207913;	(* 32 min (258 deg) *)
	table[33].y := -951056;	table[33].x := -309018;	(* 33 min (252 deg) *)
	table[34].y := -913545;	table[34].x := -406738;	(* 34 min (246 deg) *)
	table[35].y := -866025;	table[35].x := -500001;	(* 35 min (240 deg) *)
	table[36].y := -809016;	table[36].x := -587786;	(* 36 min (234 deg) *)
	table[37].y := -743144;	table[37].x := -669132;	(* 37 min (228 deg) *)
	table[38].y := -669130;	table[38].x := -743146;	(* 38 min (222 deg) *)
	table[39].y := -587784;	table[39].x := -809018;	(* 39 min (216 deg) *)
	table[40].y := -499999;	table[40].x := -866026;	(* 40 min (210 deg) *)
	table[41].y := -406735;	table[41].x := -913546;	(* 41 min (204 deg) *)
	table[42].y := -309016;	table[42].x := -951057;	(* 42 min (198 deg) *)
	table[43].y := -207911;	table[43].x := -978148;	(* 43 min (192 deg) *)
	table[44].y := -104527;	table[44].x := -994522;	(* 44 min (186 deg) *)
	table[45].y := 000000;	table[45].x := -1000000;(* 45 min (180 deg) *)
	table[46].y := 104530;	table[46].x := -994522;	(* 46 min (174 deg) *)
	table[47].y := 207913;	table[47].x := -978147;	(* 47 min (168 deg) *)
	table[48].y := 309018;	table[48].x := -951056;	(* 48 min (162 deg) *)
	table[49].y := 406738;	table[49].x := -913545;	(* 49 min (156 deg) *)
	table[50].y := 500001;	table[50].x := -866025;	(* 50 min (150 deg) *)
	table[51].y := 587786;	table[51].x := -809016;	(* 51 min (144 deg) *)
	table[52].y := 669131;	table[52].x := -743144;	(* 52 min (138 deg) *)
	table[53].y := 743145;	table[53].x := -669130;	(* 53 min (132 deg) *)
	table[54].y := 809017;	table[54].x := -587785;	(* 54 min (126 deg) *)
	table[55].y := 866026;	table[55].x := -499999;	(* 55 min (120 deg) *)
	table[56].y := 913546;	table[56].x := -406736;	(* 56 min (114 deg) *)
	table[57].y := 951057;	table[57].x := -309016;	(* 57 min (108 deg) *)
	table[58].y := 978148;	table[58].x := -207911;	(* 58 min (102 deg) *)
	table[59].y := 994522;	table[59].x := -104528;	(* 59 min (96 deg) *)
end;

procedure NewPosition;
begin
	(* Save the previous positions of the hands. *)
	opos.minutehand := pos.minutehand;
	opos.hourhand   := pos.hourhand;

	(* Get the current time. *)
	curtime := time(0);

	(* Break it into components. *)
	tm := localtime(curtime);

	(* Set the new positions of the hands. *)
	pos.minutehand := tm^.tmmin;
	pos.hourhand := ((tm^.tmhour mod 12) * 5) + (tm^.tmmin div 12);
end;

procedure DrawShadow(
   x,	(* clock x coordinate	*)
   y,	(* clock y coordinate	*)
   r	(* radius of clock	*)
   : short );
begin
	SetPosition(1, x+2*thickness, y+2*thickness);	(* offset the shadow *)
	SetColor(1, SHADOWCOLOR);
	PaintCircleInterior(1, r);
end;

procedure DrawBorder(
   x,	(* clock x coordinate	*)
   y,	(* clock y coordinate	*)
   r 	(* radius of clock	*)
   : short );
begin
	SetPosition(1, x, y);
	SetColor(1, BORDERCOLOR);
	SetThickness(1, thickness);
	PaintCircleBorder(1, r);
end;

procedure DrawSurface(
   x,	(* clock x coordinate	*)
   y,	(* clock y coordinate	*)
   r 	(* radius of clock	*)
   : short );
begin
	SetPosition(1, x, y);
	SetColor(1, SURFCOLOR);
	PaintCircleInterior(1, r);
end;

procedure DrawBackground;
begin
	SetPosition(1, 0, 0);
	SetColor(1, BACKCOLOR);
	PaintRectangleInterior(1, 2*clockx, 2*clocky);
end;

procedure DrawHand(
   cx,		(* clock x coordinate	*)
   cy,		(* clock y coordinate	*)
   rpos,	(* hand radial position	*)
   radius,	(* length of hand	*)
   color 	(* color of hand	*)
   : short );
type
   shortarray = packed array [1..3] of short;
var 
   x , y: shortarray;
   j : integer;
begin
	SetPosition(1, cx, cy);
	SetColor(1, color);
	(* points on polygon defining the hand *)
	x[1] := cx + ((table[(rpos+15) mod 60].x * 2*thickness) div SCALE);
	y[1] := cy - ((table[(rpos+15) mod 60].y * 2*thickness) div SCALE);
	x[2] := cx + ((table[(rpos+45) mod 60].x * 2*thickness) div SCALE);
	y[2] := cy - ((table[(rpos+45) mod 60].y * 2*thickness) div SCALE);
	x[3] := cx + ((table[rpos].x * radius) div SCALE);
	y[3] := cy - ((table[rpos].y * radius) div SCALE);
	SetAddressing(1, VTABSOLUTE);	(* need absolute here *)
	PaintPolygonInterior(1, 3, x[1], y[1]);
	SetAddressing(1, VTRELATIVE);	(* set back to relative *)
end;

procedure DrawMarkers(
   x,	(* clock x coordinate	*)
   y,	(* clock y coordinate	*)
   r 	(* radius of clock	*)
   : short );
var
   i : integer;
   px , py : integer;
begin
	SetColor(1, MARKCOLOR);
	for i := 0 to 59 do begin
		px := x + ((table[i].x*r) div SCALE);
		py := y - ((table[i].y*r) div SCALE);
		SetPosition(1, px, py);
		(* large dots *) if ((i mod 5) = 0) then
			PaintCircleInterior(1, thickness)
		(* small dots *) else
			PaintCircleInterior(1, (thickness div 3));
	end;
end;

procedure DrawPost(
   x,	(* clock x coordinate	*)
   y 	(* clock y coordinate	*)
   : short );
begin
	SetPosition(1, x, y);
	SetThickness(1, thickness);
	SetColor(1, POSTBCOLOR);
	PaintCircleBorder(1, thickness);
	SetColor(1, POSTFCOLOR);
	PaintCircleInterior(1, thickness);
end;

procedure NewSize(
   w,	(* new width of window	*)
   h 	(* new height of window	*)
   : short );
var
   small : short;
begin
	(* center the clock *)
	clockx := w div 2;
	clocky := h div 2;

	(* radius is 45% of smaller window dimension, but at least 1 *)
	if (w <= h) then small := w
	else	    small := h;
	clockr := (45 * small) div 100;
	if (clockr <= 0) then clockr := 1;

	(* minute hand radius is 90% of clock radius, but at least 1 *)
	rminute := (90 * clockr) div 100;
	if (rminute <= 0) then rminute := 1;

	(* marker radius is at tip of minute hand *)
	rmarker := rminute;

	(* hour hand radius is 60% of clock radius, but at least 1 *)
	rhour := ((60 * clockr) div 100);
	if (rhour <= 0) then rhour := 1;

	(* thickness is 1/25th of clock radius, but at least 2 *)
	thickness := clockr div 25;
	if (thickness <= 1) then thickness := 2;
end;

procedure DrawFace(
   x,	(* x coordinate of clock	*)
   y,	(* y coordinate of clock	*)
   h,	(* position of hour hand	*)
   m 	(* position of minute hand	*)
   : short );
begin
	DrawHand(x, y, m, rminute, HANDCOLOR);	(* draw the minute hand *)
	DrawHand(x, y, h, rhour, HANDCOLOR);	(* draw the hour hand	*)
	DrawPost(x, y);				(* draw the center post	*)
	DrawMarkers(x, y, rmarker);		(* draw the hour/minute marks *)
end;

procedure DrawClock(
   x,	(* x position of clock	*)
   y,	(* y position of clock	*)
   r,	(* radius of clock	*)
   h,	(* hour hand position in minutes from vertical		*)
   m	(* minute hand position in minutes from vertical	*)
   : short );
begin
	DrawShadow(x, y, r);	(* draw the shadow of the clock *)
	DrawBorder(x, y, r);	(* draw the border of the clock *)
	DrawSurface(x, y, r);	(* draw the clock surface 	*)
	DrawFace(x, y, h, m);	(* draw the face of the clock	*)
end;

procedure VTRefresh(
   fd : integer;	(* window descriptor	*)
   id : integer;	(* refresh id param	*)
   x,y,w,h 		(* refresh rectangle	*)
   : short);
var
  ws : wstate;
begin
	void  := GetWindowState(fd, ws);	(* save current state	     *)
	SetTemporaryClipping(fd, x, y, w, h);	(* area to refresh	     *)
	DrawBackground;				(* draw background and clock *)
	Flush(fd);
	DrawClock(clockx, clocky, clockr, pos.hourhand, pos.minutehand);
	SetWindowState(fd, ws);			(* restore previous state    *)
end;

procedure VTAdjust(
   fd : integer;
   id : integer;
   w, h 
   : short);
begin
	(*
	 * Refresh will redraw the screen.  Just need to get the
	 * new size and clock position into the display list.
	 *)
	 NewSize(w, h);
end;

begin

(*
 * Initialize the screen.
 *)

void := GetWindowState(1, save);(* mostly interested in window size	*)
void := SetLineDisc(1, TWSDISC);(* want the graphics line discipline	*)
SetBuf(1, 2000);		(* buffer the outputi			*)
SetBColor(1, VTGray37);
void := BlockRefreshAdjust(1);	(* no refresh/adjust until initialize	*)
SetRefresh(1,0);		(* specify the refresh routine		*)
SetAdjust(1,0);			(* specify the adjust routine		*)
SetAddressing(1, VTRELATIVE);	(* addressing relative to cur position	*)
loadtable;			(* load up the trig table		*)

(*
 * Draw the initial clock.
 *)

NewPosition;			(* set initial position and size; draw clock *)
NewSize(save.width, save.height);
DrawBackground;		(* fill in the background *)
DrawClock(clockx, clocky, clockr, pos.hourhand, pos.minutehand);
Flush(1);			(* flush the output so it is displayed *)


(*
 * Update the clock hands every minute.
 *)

   while (1 = 1) do begin    (* time NEVER stops *)
	(* Wait until the next change in minutes. *)
	void := BlockRefreshAdjust(0);	(* allow refresh/adjust *)
	sleep(60 -(curtime mod 60));
	void := BlockRefreshAdjust(1);	(* block refresh/adjust *)

	(* What are the new hand positions? *)
	NewPosition;

	(* If either hand moved, delete it by drawing it in surface color *)
	if (pos.minutehand <> opos.minutehand) then
		DrawHand(clockx, clocky, opos.minutehand, rminute, SURFCOLOR);
	if (pos.hourhand <> opos.hourhand) then
		DrawHand(clockx, clocky, opos.hourhand, rhour, SURFCOLOR);

	(* Draw the new clock face *)
	DrawFace(clockx, clocky, pos.hourhand, pos.minutehand);

	Flush(1);	(* make sure screen is updated *)
   end;
end.
