/*
 * clkvt -- an analog clock program using libvt
 *
 * compile with:
 *	cc -o clkvt -O clkvt.c -lvt
 */

#include	<vt.h>
#include	<sys/time.h>
#include	<signal.h>

#define	SCALE		1000000
#define	NTICKS		60

#define	BACKCOLOR	VT_Gray37	/* background color */
#define	SHADOWCOLOR	VT_Black	/* clock shadow color */
#define	BORDERCOLOR	VT_Black	/* clock border color */
#define	SURFCOLOR	VT_White	/* clock surface color */
#define	HANDCOLOR	VT_Black	/* clock hand color */
#define	MARKCOLOR	VT_Black	/* clock marker color */
#define	POSTBCOLOR	VT_Black	/* clock post background color */
#define	POSTFCOLOR	VT_White	/* clock post foreground color */

struct TRIG {
	long	y;	/* x and y coordinates of points on a large face */
	long	x;
}	table[NTICKS] = {
	{ 1000000,	000000},	/* 0 minutes (90 degrees) */
	{ 994522,	104529},	/* 1 minute  (84 degrees) */
	{ 978147,	207912},	/* 2 minutes (78 degrees) */
	{ 951056,	309017},	/* 3 minutes (72 degrees) */
	{ 913545,	406737},	/* 4 minutes (66 degrees) */
	{ 866025,	500000},	/* 5 minutes (60 degrees) */
	{ 809017,	587786},	/* 6 minutes (54 degrees) */
	{ 743145,	669131},	/* 7 minutes (48 degrees) */
	{ 669130,	743145},	/* 8 minutes (42 degrees) */
	{ 587785,	809017},	/* 9 minutes (36 degrees) */
	{ 500000,	866025},	/* 10 minutes (30 degrees) */
	{ 406737,	913546},	/* 11 minutes (24 degrees) */
	{ 309017,	951057},	/* 12 minutes (18 degrees) */
	{ 207912,	978148},	/* 13 minutes (12 degrees) */
	{ 104528,	994522},	/* 14 minutes (6 degrees) */
	{ 000000,	1000000},	/* 15 minutes (0 degrees) */
	{ -104531,	994522},	/* 16 minutes (354 degrees) */
	{ -207914,	978147},	/* 17 minutes (348 degrees) */
	{ -309019,	951056},	/* 18 minutes (342 degrees) */
	{ -406739,	913545},	/* 19 minutes (336 degrees) */
	{ -500002,	866024},	/* 20 minutes (330 degrees) */
	{ -587787,	809016},	/* 21 minutes (324 degrees) */
	{ -669132,	743143},	/* 22 minutes (318 degrees) */
	{ -743146,	669129},	/* 23 minutes (312 degrees) */
	{ -809018,	587784},	/* 24 minutes (306 degrees) */
	{ -866026,	499998},	/* 25 minutes (300 degrees) */
	{ -913546,	406735},	/* 26 minutes (294 degrees) */
	{ -951057,	309015},	/* 27 minutes (288 degrees) */
	{ -978148,	207910},	/* 28 minutes (282 degrees) */
	{ -994522,	104527},	/* 29 minutes (276 degrees) */
	{ -1000000,	000000},	/* 30 minutes (270 degrees) */
	{ -994522,	-104530},	/* 31 minutes (264 degrees) */
	{ -978147,	-207913},	/* 32 minutes (258 degrees) */
	{ -951056,	-309018},	/* 33 minutes (252 degrees) */
	{ -913545,	-406738},	/* 34 minutes (246 degrees) */
	{ -866025,	-500001},	/* 35 minutes (240 degrees) */
	{ -809016,	-587786},	/* 36 minutes (234 degrees) */
	{ -743144,	-669132},	/* 37 minutes (228 degrees) */
	{ -669130,	-743146},	/* 38 minutes (222 degrees) */
	{ -587784,	-809018},	/* 39 minutes (216 degrees) */
	{ -499999,	-866026},	/* 40 minutes (210 degrees) */
	{ -406735,	-913546},	/* 41 minutes (204 degrees) */
	{ -309016,	-951057},	/* 42 minutes (198 degrees) */
	{ -207911,	-978148},	/* 43 minutes (192 degrees) */
	{ -104527,	-994522},	/* 44 minutes (186 degrees) */
	{ 000000,	-1000000},	/* 45 minutes (180 degrees) */
	{ 104530,	-994522},	/* 46 minutes (174 degrees) */
	{ 207913,	-978147},	/* 47 minutes (168 degrees) */
	{ 309018,	-951056},	/* 48 minutes (162 degrees) */
	{ 406738,	-913545},	/* 49 minutes (156 degrees) */
	{ 500001,	-866025},	/* 50 minutes (150 degrees) */
	{ 587786,	-809016},	/* 51 minutes (144 degrees) */
	{ 669131,	-743144},	/* 52 minutes (138 degrees) */
	{ 743145,	-669130},	/* 53 minutes (132 degrees) */
	{ 809017,	-587785},	/* 54 minutes (126 degrees) */
	{ 866026,	-499999},	/* 55 minutes (120 degrees) */
	{ 913546,	-406736},	/* 56 minutes (114 degrees) */
	{ 951057,	-309016},	/* 57 minutes (108 degrees) */
	{ 978148,	-207911},	/* 58 minutes (102 degrees) */
	{ 994522,	-104528},	/* 59 minutes (96 degrees) */
};

long		time();
char		*ctime();
long		curtime=0;
struct tm	*tm;
		RefreshTime();
		AdjustTime();

/*
 * display list
 */

short		clockx, clocky;	/* clock position */
short		clockr;		/* clock radius */
struct {
	short	hourhand;
	short	minutehand;
}		pos, opos;	/* current and previous hand positions */
short		rminute;	/* minute hand radius */
short		rhour;		/* hour hand radius */
short		rmarker;	/* radial position of markers */
short		thickness;	/* thickness of face objects */
struct wstate	save;

cleanup()
{
    BlockRefreshAdjust(1);
    SetPosition(1, 0, 0);
    SetColor(1, save.bcolor);
    PaintRectangleInterior(1, 10000, 10000);
    SetWindowState(1, &save);
    exit(0);
}

main(argc, argv)
int	argc;
char	*argv[];
{
struct gconfig	gc;

if (GetGraphicsConfig(1, &gc) == -1) {
	printf("%s: not a window\n", argv[0]);
	exit(1);
}


/*
 * Initialize the screen.
 */

GetWindowState(1, &save);	/* mostly interested in window size */
SetLineDisc(1, TWSDISC);	/* want the graphics line discipline */
BlockRefreshAdjust(1);		/* no refresh/adjust until we initialize */
SetRefresh(1, 0, RefreshTime);	/* specify the refresh routine */
SetAdjust(1, 0, AdjustTime);	/* specify the adjust routine */
signal(SIGTERM, cleanup);
signal(SIGHUP, cleanup);
signal(SIGQUIT, cleanup);
signal(SIGINT, cleanup);
SetMouseMode(1, 0);		/* no mouse input needed */
SetBuf(1, 1000);		/* buffer the output */
SetBColor(1, VT_Gray37);
SetAddressing(1, VT_RELATIVE);	/* addressing relative to current position */


/*
 * Draw the initial clock.
 */

NewPosition();			/* set initial position and size; draw clock */
NewSize(save.width, save.height);
DrawBackground();		/* fill in the background */
DrawClock(clockx, clocky, clockr, pos.hourhand, pos.minutehand);
Flush(1);			/* flush the output so it is displayed */


/*
 * Update the clock hands every minute.
 */

while(1) {		/* time never stops */
	/* Wait until the next change in minutes. */
	BlockRefreshAdjust(0);		/* allow refresh/adjust */
	sleep(60 -(curtime % 60));
	BlockRefreshAdjust(1);		/* block refresh/adjust */

	/* What are the new hand positions? */
	NewPosition();

	/* If either hand moved, delete it by drawing it in surface color */
	if (pos.minutehand != opos.minutehand)
		DrawHand(clockx, clocky, opos.minutehand, rminute, SURFCOLOR);
	if (pos.hourhand != opos.hourhand)
		DrawHand(clockx, clocky, opos.hourhand, rhour, SURFCOLOR);

	/* Draw the new clock face */
	DrawFace(clockx, clocky, pos.hourhand, pos.minutehand);

	Flush(1);	/* make sure screen is updated */
}
}

NewPosition()
{
	/* Save the previous positions of the hands. */
	opos = pos;

	/* Get the current time. */
	curtime = time(0);

	/* Break it into components. */
	tm = localtime(&curtime);

	/* Set the new positions of the hands. */
	pos.minutehand = tm->tm_min;
	pos.hourhand = ((tm->tm_hour % 12) * 5) + (tm->tm_min / 12);
}

DrawClock(x, y, r, h, m)
short	x;	/* x position of clock */
short	y;	/* y position of clock */
short	r;	/* radius of clock */
short	h;	/* hour hand position in minutes from vertical */
short	m;	/* minute hand position in minutes from vertical */
{

	DrawShadow(x, y, r);	/* draw the shadow of the clock */
	DrawBorder(x, y, r);	/* draw the border of the clock */
	DrawSurface(x, y, r);	/* draw the surface of the clock */
	DrawFace(x, y, h, m);	/* draw the face of the clock */
}

DrawShadow(x, y, r)
short	x;	/* x coordinate of clock */
short	y;	/* y coordinate of clock */
short	r;	/* radius of clock */
{
	SetPosition(1, x+2*thickness, y+2*thickness);	/* offset the shadow */
	SetColor(1, SHADOWCOLOR);
	PaintCircleInterior(1, r);
}

DrawBorder(x, y, r)
short	x;	/* x coordinate of clock */
short	y;	/* y coordinate of clock */
short	r;	/* radius of clock */
{
	SetPosition(1, x, y);
	SetColor(1, BORDERCOLOR);
	SetThickness(1, thickness);
	PaintCircleBorder(1, r);
}

DrawSurface(x, y, r)
short	x;	/* x coordinate of clock */
short	y;	/* y coordinate of clock */
short	r;	/* radius of clock */
{
	SetPosition(1, x, y);
	SetColor(1, SURFCOLOR);
	PaintCircleInterior(1, r);
}

DrawFace(x, y, h, m)
short	x;	/* x coordinate of clock */
short	y;	/* y coordinate of clock */
short	h;	/* position of hour hand */
short	m;	/* position of minute hand */
{
	DrawHand(x, y, m, rminute, HANDCOLOR);	/* draw the minute hand */
	DrawHand(x, y, h, rhour, HANDCOLOR);	/* draw the hour hand */
	DrawPost(x, y);				/* draw the center post */
	DrawMarkers(x, y, rmarker);		/* draw the hour/minute marks */
}

DrawBackground()
{
	SetPosition(1, 0, 0);
	SetColor(1, BACKCOLOR);
	PaintRectangleInterior(1, 2*clockx, 2*clocky);
}

DrawHand(x, y, position, radius, color)
short		x;		/* x coordinate of clock */
short		y;		/* y coordinate of clock */
register
short		position;	/* radial position of hand */
short		radius;		/* length of hand */
VT_COLOR	color;		/* color of hand */
{
	short	xpoint[3];
	short	ypoint[3];

	SetPosition(1, x, y);
	SetColor(1, color);
	/* points on polygon defining the hand */
	xpoint[0] = x + ((table[(position+15)%60].x * 2*thickness)/SCALE);
	ypoint[0] = y - ((table[(position+15)%60].y * 2*thickness)/SCALE);
	xpoint[1] = x + ((table[(position+45)%60].x * 2*thickness)/SCALE);
	ypoint[1] = y - ((table[(position+45)%60].y * 2*thickness)/SCALE);
	xpoint[2] = x + ((table[position].x * radius)/SCALE);
	ypoint[2] = y - ((table[position].y * radius)/SCALE);
	SetAddressing(1, VT_ABSOLUTE);	/* need absolute here */
	PaintPolygonInterior(1, 3, xpoint, ypoint);
	SetAddressing(1, VT_RELATIVE);	/* set back to relative */
}

DrawMarkers(x, y, r)
short	x;	/* x coordinate of clock */
short	y;	/* y coordinate of clock */
short	r;	/* radius of clock */
{
	register int	i;

	SetColor(1, MARKCOLOR);
	for (i=0; i<60; i++) {
		SetPosition(1, (short)(x+(table[i].x*r)/SCALE),
				(short)(y-(table[i].y*r)/SCALE));
		if ((i % 5) == 0)
			/* large dots */
			PaintCircleInterior(1, thickness);
		else
			/* small dots */
			PaintCircleInterior(1, thickness/3);
	}
}

DrawPost(x, y)
short	x;	/* x coordinate of clock */
short	y;	/* y coordinate of clock */
{
	SetPosition(1, x, y);
	SetThickness(1, thickness);
	SetColor(1, POSTBCOLOR);
	PaintCircleBorder(1, thickness);
	SetColor(1, POSTFCOLOR);
	PaintCircleInterior(1, thickness);
}

NewSize(w, h)
short	w;	/* new width of window */
short	h;	/* new height of window */
{
	/* center the clock */
	clockx = w/2;
	clocky = h/2;
	/* radius is 45% of smaller window dimension, but at least 1 */
	clockr = (45 * ((w <= h) ? w : h)) / 100;
	if (clockr <= 0)
		clockr = 1;
	/* minute hand radius is 90% of clock radius, but at least 1 */
	if ((rminute = ((90 * clockr) / 100)) <= 0)
		rminute = 1;
	/* marker radius is at tip of minute hand */
	rmarker = rminute;
	/* hour hand radius is 60% of clock radius, but at least 1 */
	if ((rhour = ((60 * clockr) / 100)) <= 0)
		rhour = 1;
	/* thickness is 1/25th of clock radius, but at least 2 */
	if ((thickness = clockr / 25) <= 1)
		thickness = 2;
}

RefreshTime(id, x, y, w, h)
int	id;
short	x, y, w, h;
{
	struct wstate	ws;

	GetWindowState(1, &ws);			/* save current state */
	SetTemporaryClipping(1, x, y, w, h);	/* area to refresh */
	DrawBackground();			/* draw background and clock */
	Flush(1);
	DrawClock(clockx, clocky, clockr, pos.hourhand, pos.minutehand);
	SetWindowState(1, &ws);			/* restore previous state */
}

AdjustTime(id, w, h)
int	id;
short	w, h;
{
	/*
	 * Refresh will redraw the screen.  Just need to get the
	 * new size and clock position into the display list.
	 */
	 NewSize(w, h);
}
