// 'Windows CE 3.0 Programming' Source Code Samples (Prentice Hall, 2000)
// Source Code Author: Nick Grattan (nick@softwarepaths.com)
// Version 1.00

// Chapter 4: Databases. Sample Code

#include "stdafx.h"
#include "examples.h"


// *** Listing 4.1
//
// Creates a database volume on a storage card, and shows how to flush and unmount
// the database volume.

void Listing4_1()
{
	CEGUID pceguid;

	if(!CeMountDBVol(&pceguid, _T("\\MyVolume.CDB"), CREATE_NEW))
		cout << _T("Could not create database volume") << endl;
	else
		cout << _T("Database volume created") << endl;
	// now unmount the database volume
	if(CeUnmountDBVol(&pceguid))
		cout << endl << _T("Database volume unmounted") << endl;
	else
		cout << endl << _T("Database volume could not be unmounted") << endl;
}

// *** Listing 4.2
//
// Lists all mounted databases

void Listing4_2()
{
	CEGUID ceguid;
	TCHAR szVolumeName[MAX_PATH + 1];
	
	CREATE_INVALIDGUID(&ceguid);
	while(CeEnumDBVolumes(&ceguid, szVolumeName, MAX_PATH))
	{
		cout << _T("Mounted vol: ") << szVolumeName << endl;
		if(CHECK_SYSTEMGUID(&ceguid))
			cout << _T("Object Store database volume!") << endl;
	}
}


// *** Listing 4.3
//
// Creates database in object store with two indexes

const CEPROPID propCompany = MAKELONG(CEVT_LPWSTR, 100); 
const CEPROPID propCompanyID = MAKELONG(CEVT_I4, 101); 
const CEPROPID propCompanyTel = MAKELONG(CEVT_LPWSTR, 102); 
const CEPROPID propTimeStamp = MAKELONG(CEVT_BLOB, 103);

void Listing4_3()
{
	CEOID ceDB;
	CEGUID ceObjStore;
	CEDBASEINFO ceDBInfo;

	// initialize structure
	ceDBInfo.dwFlags = CEDB_VALIDNAME | CEDB_VALIDSORTSPEC | CEDB_VALIDTYPE;
	wcscpy(ceDBInfo.szDbaseName, _T("Company"));
	ceDBInfo.dwDbaseType = 19500; // arbitary database type identifier
	ceDBInfo.wNumSortOrder = 2;	// number of sort orders
	// setup two sort orders
	ceDBInfo.rgSortSpecs[0].propid = propCompany;
	ceDBInfo.rgSortSpecs[0].dwFlags = 0;	// default sort order
	ceDBInfo.rgSortSpecs[1].propid = propCompanyID;
	ceDBInfo.rgSortSpecs[1].dwFlags = 0;  // default sort order


	CREATE_SYSTEMGUID(&ceObjStore); 
	ceDB = CeCreateDatabaseEx(&ceObjStore, &ceDBInfo);
	if(ceDB == NULL)
		cout << _T("Could not create database");
	else
		cout << _T("Database created");
}

// *** Listing 4.4
//
// Opens database, specifying Company name as sort order. Returns handle to open database

HANDLE Listing4_4()
{
	CEGUID ceObjStore;
	HANDLE hDB;
	CEOID ceOidDB = 0;

	CREATE_SYSTEMGUID(&ceObjStore); 
	hDB = CeOpenDatabaseEx(&ceObjStore, 
			&ceOidDB,
			_T("Company"),	
			propCompany,	// prop.id. of sort order
			CEDB_AUTOINCREMENT,
			NULL);			// no notifications
	if(hDB == INVALID_HANDLE_VALUE)
		cout << _T("Could not open database") << endl;
	else
		cout << _T("Database Opened") << endl;
	return hDB;
}	

// *** Listing 4.5
//
// Closes database specified by object identifier

void Listing4_5(HANDLE hDB)
{
	if(!CloseHandle(hDB))
		cout << _T("Could not close database");
	else
		cout << _T("Database closed");
}

// *** Listing 4.6
//
// Deletes a property database

void Listing4_6()
{
	CEGUID ceObjStore;
	CEOID ceOidDB = 0;
	HANDLE hDB;

	CREATE_SYSTEMGUID(&ceObjStore); 

	hDB = CeOpenDatabaseEx(&ceObjStore, 
			&ceOidDB,
			_T("Company"),	
			propCompany,	// prop.id. of sort order
			CEDB_AUTOINCREMENT,
			NULL);			// no notifications
	if(hDB == INVALID_HANDLE_VALUE)
		cout << _T("Could not open database") << endl;
	else
	{
		CloseHandle(hDB);
		if(CeDeleteDatabaseEx(&ceObjStore, ceOidDB))
			cout << _T("Database deleted") << endl;
		else
			cout << _T("Database not deleted") << endl;
	}
}

// *** Listing 4.7
//
// Writing records to a property database. Includes writing a BLOB
// property described in chapter 4 'Using the CEVT_BLOB property data
// type' section.

void WriteDBRecord(HANDLE hDB, LPTSTR lpCompanyName, 
				   long lpCompanyID, LPTSTR lpCompanyTel)
{
	CEPROPVAL propval[4];
	CEOID ceoidRec;
	SYSTEMTIME sysTime;

	GetSystemTime(&sysTime);
	propval[0].propid = propCompany;
	propval[0].val.lpwstr = lpCompanyName;
	propval[1].propid = propCompanyID;
	propval[1].val.lVal = lpCompanyID;
	propval[2].propid = propCompanyTel;
	propval[2].val.lpwstr = lpCompanyTel;
	// Setup a BLOB property to write
	propval[3].propid = propTimeStamp;
	propval[3].val.blob.dwCount = sizeof(sysTime);
	propval[3].val.blob.lpb = (LPBYTE)&sysTime;

	// End of blob code
	ceoidRec = CeWriteRecordProps(hDB, 
				0, // write new record
				4, // number of properties
				propval);
	if(ceoidRec == 0)
		cout << _T("Record write failed") << endl;
	else
		cout << _T("Record written") << endl;
}

void Listing4_7()
{
	HANDLE hDB = Listing4_4(); // open database	
	if(hDB != INVALID_HANDLE_VALUE)
	{
		WriteDBRecord(hDB, _T("Company 1"), 1, _T("998-12311"));
		WriteDBRecord(hDB, _T("Company 2"), 2, _T("998-12312"));
		WriteDBRecord(hDB, _T("Company 3"), 3, _T("998-12313"));
		Listing4_5(hDB); // close database
	}
}

// *** Listing 4.8
//
// Reading records a property database

void ReadNextDBRecord(HANDLE hDB)
{
	CEOID ceoidRec;
	DWORD dwBuf;
	CEPROPVAL *props = NULL;
	unsigned short lProps;

	ceoidRec = CeReadRecordPropsEx(hDB, 
				CEDB_ALLOWREALLOC, 
				&lProps,
				NULL,
				(LPBYTE*)&props,
				&dwBuf,
				NULL);
	if(ceoidRec == 0)
		cout << _T("Could not read record") << endl;
	else
	{
		for (int i =0; i < lProps; i++)
		{
			switch (props[i].propid)
			{
			case propCompany:
				cout << _T(" Company: ") << props[i].val.lpwstr;
				break;
			case propCompanyID:
				cout << _T(" Company ID: ") << props[i].val.lVal;
				break;
			case propCompanyTel:
				cout << _T(" Company Tel: ") << props[i].val.lpwstr;
				break;
			// check BLOB timestamp data. Only display time, not date.
			case propTimeStamp:
				LPSYSTEMTIME lpSysTime = (LPSYSTEMTIME)props[i].val.blob.lpb;
				cout << _T(" Record written at:") 
						<< lpSysTime->wHour << _T(":")
						<< lpSysTime->wMinute << _T(":")
						<< lpSysTime->wSecond << _T(":")
						<< lpSysTime->wMilliseconds;
				LocalFree(lpSysTime);
				break;

			}
		}
		cout << endl;
		LocalFree(props);
	}
}

void Listing4_8()
{
	HANDLE hDB = Listing4_4(); // open database	
	if(hDB != INVALID_HANDLE_VALUE)
	{
		ReadNextDBRecord(hDB);
		ReadNextDBRecord(hDB);
		ReadNextDBRecord(hDB);
		Listing4_5(hDB); // close database
	}
}

// *** Listing 4.9
//
// Reading a single property from a property database

void ReadOneProp(HANDLE hDB)
{
	CEOID ceoidRec;
	DWORD dwBuf;
	CEPROPVAL *props = NULL;
	unsigned short lProps = 1; // # properties to read
	CEPROPID propsToRead;

	propsToRead = propCompany; // only read company name

	ceoidRec = CeReadRecordPropsEx(hDB, 
				CEDB_ALLOWREALLOC, 
				&lProps,
				&propsToRead,
				(LPBYTE*)&props,
				&dwBuf,
				NULL);
	if(ceoidRec == 0)
		cout << _T("Could not read record") << endl;
	else
	{
		cout << _T(" Company: ") << props[0].val.lpwstr << endl;
		LocalFree(props);
	}
}

void Listing4_9()
{
	HANDLE hDB = Listing4_4(); // open database	
	if(hDB != INVALID_HANDLE_VALUE)
	{
		ReadOneProp(hDB);
		ReadOneProp(hDB);
		ReadOneProp(hDB);
		Listing4_5(hDB); // close database
	}
}

// *** Listing 4.10
//
// Lists the last database in the database

void Listing4_10()
{
	HANDLE hDB = Listing4_4(); // open database	
	DWORD dwIndex;
	if(hDB != INVALID_HANDLE_VALUE)
	{
		if(CeSeekDatabase(hDB, CEDB_SEEK_END, 0, &dwIndex))
		{
			cout << _T("Record index: ") << dwIndex << endl;
			ReadOneProp(hDB);
		}
		Listing4_5(hDB); // close database
	}
}

// *** Listing 4.11
//
// Lists all records with "Company 2" as the name. Run Listing 4-7 to duplicate
// records to show this code working. 

void Listing4_11()
{
	HANDLE hDB;
	CEGUID ceObjStore;
	DWORD dwIndex;
	CEOID ceOidDB = 0;

	CREATE_SYSTEMGUID(&ceObjStore); 

	hDB = CeOpenDatabaseEx(&ceObjStore, 
			&ceOidDB,
			_T("Company"),	
			propCompany,	// prop.id. of sort order
			0,				// no auto-increment
			NULL);			// no notifications
	if(hDB != INVALID_HANDLE_VALUE)
	{
		CEPROPVAL propSeek;

		propSeek.propid = propCompany;
		propSeek.val.lpwstr = _T("Company 2");
		if(CeSeekDatabase(hDB, CEDB_SEEK_VALUEFIRSTEQUAL, 
					(DWORD)&propSeek, &dwIndex))
		{
			do
			{
				cout << _T("Record index: ") << dwIndex << endl;
				ReadOneProp(hDB);
			} while (CeSeekDatabase(hDB, CEDB_SEEK_VALUENEXTEQUAL, 
						(DWORD)&propSeek, &dwIndex));
		}
		CloseHandle(hDB);
	}
	else
		cout << _T("Could not open database") << endl;
}

// *** Listing 4.12
//
// Deletes propCompanyTel property from the first record in database

void Listing4_12()
{
	HANDLE hDB = Listing4_4(); // open database	

	if(hDB != INVALID_HANDLE_VALUE)
	{
		CEPROPVAL propDelete;
		CEOID oidRec;
		DWORD dwIndex;

		propDelete.propid = propCompanyTel;
		propDelete.wFlags = CEDB_PROPDELETE;
		propDelete.val.lpwstr = NULL;

		oidRec = CeSeekDatabase(hDB, 
			CEDB_SEEK_CURRENT, 0, &dwIndex);
		if(CeWriteRecordProps(hDB,
				oidRec, // object id of current record
				1, // number of properties to delete
				&propDelete))
			cout << _T("Property deleted") << endl;
		else
			cout << _T("Could not delete property") << endl;
		Listing4_5(hDB); // close database
	}
}

// *** Listing 4.13
//
// Deletes first record from a database

void Listing4_13()
{
	HANDLE hDB = Listing4_4(); // open database	

	if(hDB != INVALID_HANDLE_VALUE)
	{
		CEOID oidRec;
		DWORD dwIndex;

		oidRec = CeSeekDatabase(hDB, 
			CEDB_SEEK_CURRENT, 0, &dwIndex);
		if(CeDeleteRecord(hDB, oidRec))
			cout << _T("Record deleted") << endl;
		else
			cout << _T("Could not delete record") << endl;
		Listing4_5(hDB); // close database
	}
}

// *** Listing 4.14
//
// Updates telephone number for company with id '2'

void Listing4_14()
{
	HANDLE hDB;
	CEGUID ceObjStore;
	DWORD dwIndex;
	CEOID ceOidDB = 0;

	CREATE_SYSTEMGUID(&ceObjStore); 

	hDB = CeOpenDatabaseEx(&ceObjStore, 
			&ceOidDB,
			_T("Company"),	
			propCompanyID,	// prop.id. of sort order
			0,				// no auto-increment
			NULL);			// no notifications
	if(hDB != INVALID_HANDLE_VALUE) 
	{
		CEPROPVAL propSeek, propUpdate;
		CEOID ceOidRec = 0;

		propSeek.propid = propCompanyID;
		propSeek.val.lVal = 2; // company id to seek
		ceOidRec = CeSeekDatabase(hDB, CEDB_SEEK_VALUEFIRSTEQUAL, 
					(DWORD)&propSeek, &dwIndex);
		if(ceOidRec != 0)
		{
			propUpdate.propid = propCompanyTel;
			propUpdate.wFlags = 0;
			propUpdate.val.lpwstr = _T("444-99988");

			if(CeWriteRecordProps(hDB,
					ceOidRec, // object id of record to update
					1, // number of properties to update
					&propUpdate))
				cout << _T("Record updated") << endl;
			else
				cout << _T("Record not updated") << endl;
		}
		else
			cout << _T("Could not locate company id 2") << endl;
		CloseHandle(hDB);
	}
	else
		cout << _T("Could not open database") << endl;
}

// *** Listing 4.15
//
// Starts database notification

HANDLE g_hDBNotification = INVALID_HANDLE_VALUE;	// handle to open database
CENOTIFYREQUEST g_cNotifyRequest;

void Listing4_15()
{
	CEOID ceOidDB = 0;
	CEGUID ceObjStore;

	g_cNotifyRequest.dwSize = sizeof(CENOTIFYREQUEST);
	g_cNotifyRequest.hwnd = hWnd;
	g_cNotifyRequest.dwFlags = CEDB_EXNOTIFICATION;
	g_cNotifyRequest.hHeap = NULL; // use default heap
	g_cNotifyRequest.dwParam = 999; // value passed to notification

	CREATE_SYSTEMGUID(&ceObjStore); 
	g_hDBNotification  = CeOpenDatabaseEx(&ceObjStore, 
			&ceOidDB,
			_T("Company"),	
			0, 0, &g_cNotifyRequest);
	if(g_hDBNotification != INVALID_HANDLE_VALUE)
		cout << _T("Notification set!") << endl;
	else
		cout << _T("Could not open database") << endl;
}

// *** Listing 4.16
//
// See WM_DBNOTIFICATION in examples.cpp. This function stops database notification

void Listing4_16()
{
	if(g_hDBNotification != INVALID_HANDLE_VALUE)
	{
		CloseHandle(g_hDBNotification);
		g_hDBNotification = NULL;
		cout << _T("Notification ceased") << endl;
	}
}

// *** Listing 4.17
//
// Getting database information

void Listing4_17()
{
	HANDLE hDBFind;
	CEGUID ceObjStore;
	CEOID ceDBOid;

	CREATE_SYSTEMGUID(&ceObjStore); 
	hDBFind = CeFindFirstDatabaseEx(&ceObjStore, 0);

	if(hDBFind != INVALID_HANDLE_VALUE)
	{
		while((ceDBOid = CeFindNextDatabaseEx(hDBFind, &ceObjStore)) != 0)
		{
			Listing4_18(&ceObjStore, ceDBOid);
		}
		CloseHandle(hDBFind);
	}
	else
		cout << _T("Could not enumerate databases") << endl;
}

// *** Listing 4.18
//
// Lists database information for object identifier

void Listing4_18(CEGUID* pceObjStore, CEOID ceDBOid)
{
	CEOIDINFO cdbInfo;

	if(CeOidGetInfoEx(pceObjStore, ceDBOid, &cdbInfo))
	{
		if(cdbInfo.wObjType == OBJTYPE_DATABASE)
		{
			cout << _T("DB: ")    
					<< cdbInfo.infDatabase.szDbaseName;
			cout << _T(" Size: ") 
					<< cdbInfo.infDatabase.dwSize;
			cout << _T(" Recs: ") 
					<< cdbInfo.infDatabase.wNumRecords << endl;
		}
		else
			cout << _T("Not a database!") << endl;
	}
	else
		cout << _T("Could not get database information") << endl;
}


// *** Listing 4.19
//
// Rename a database

void Listing4_19()
{
	CEOID ceOidDB = 0;
	CEGUID ceObjStore;
	HANDLE hDB;
	CEDBASEINFO cdbInfo;

	// find oid of database
	CREATE_SYSTEMGUID(&ceObjStore); 
	hDB = CeOpenDatabaseEx(&ceObjStore, &ceOidDB,
			_T("Company"), 0, 0, NULL);
	if(hDB != INVALID_HANDLE_VALUE)
	{
		CloseHandle(hDB);
		cdbInfo.dwSize = sizeof(CEDBASEINFO);
		cdbInfo.dwFlags = CEDB_VALIDNAME;
		wcscpy(cdbInfo.szDbaseName, _T("Company_new"));
		if(CeSetDatabaseInfoEx(&ceObjStore, ceOidDB, &cdbInfo))
			cout << _T("Database renamed") << endl;
		else
			cout << _T("Could not rename database") << endl;
	}
	else
		cout << _T("Could not open database") << endl;
}

// *** Listing 4.20
//
// Adding or updating registry item

void Listing4_20()
{
	HKEY hKey;
	DWORD dwDisp;
	TCHAR szStr[200];
	DWORD dwVal;

	if(RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
					_T("Software\\MyCompany\\MyApplication"),
					0, NULL, 0, 0, NULL,
					&hKey, &dwDisp) != ERROR_SUCCESS)
	{
		// Warning! Key could have been created, but not opened.
		cout << _T("Could not open/create key:");
		return;
	}
	if(dwDisp == REG_CREATED_NEW_KEY)
		cout << _T("Created new key") << endl;
	else if(dwDisp == REG_OPENED_EXISTING_KEY)
		cout << _T("Opened existing key") << endl;

	wcscpy(szStr, _T("Contents of Key 'String'"));
	if(RegSetValueEx(hKey, _T("StringTest"), NULL, 
			REG_SZ, (LPBYTE)szStr, 
			(wcslen(szStr) + 1) * sizeof(TCHAR)) != ERROR_SUCCESS)
		cout << _T("Could not save key");

	dwVal = 456;
	if(RegSetValueEx(hKey, _T("DWORDTest"), NULL, 
			REG_DWORD, (LPBYTE)&dwVal, 
			sizeof(DWORD)) != ERROR_SUCCESS)
		cout << _T("Could not save key");

	wcscpy(szStr, _T("Default Value"));
	if(RegSetValueEx(hKey, NULL, NULL, // default value
			REG_SZ, (LPBYTE)szStr, 
			(wcslen(szStr) + 1) * sizeof(TCHAR)) != ERROR_SUCCESS)
		cout << _T("Could not save key");

	RegCloseKey(hKey); 
	return;
}

// *** Listing 4.21
//
// Querying a registry item

void Listing4_21()
{
	HKEY hKey;
	DWORD dwDisp, dwcbData, dwType;
	TCHAR szStr[200];
	DWORD dwVal;

	if(RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
					_T("Software\\MyCompany\\MyApplication"),
					0, NULL, 0, 0, NULL,
					&hKey, &dwDisp) != ERROR_SUCCESS)
	{
		cout << _T("Could not open/create key:");
		return;
	}
	dwcbData = sizeof(szStr) * sizeof(TCHAR);
	if(RegQueryValueEx(hKey, _T("StringTest"), 
			NULL, &dwType, (LPBYTE)szStr, &dwcbData) != 0)
		cout << _T("Could not open key") << endl;
	else
		cout << _T("StringTest:") << szStr << endl;

	dwcbData = sizeof(DWORD);
	if(RegQueryValueEx(hKey, _T("DWORDTest"), 
			NULL, &dwType, (LPBYTE)&dwVal, &dwcbData) != 0)
		cout << _T("Could not open key") << endl;
	else
		cout << _T("DWORDTest:") << dwVal << endl;

	dwcbData = sizeof(szStr) * sizeof(TCHAR);
	if(RegQueryValueEx(hKey, NULL,		// default value
			NULL, &dwType, (LPBYTE)szStr, &dwcbData) != 0)
		cout << _T("Could not open key") << endl;
	else
		cout << _T("(default):") << szStr << endl;

	RegCloseKey(hKey);
}

// *** Listing 4.22
//
// Deleting a registry value

void Listing4_22()
{
	HKEY hKey;
	DWORD dwDisp;
	if(RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
					_T("Software\\MyCompany\\MyApplication"),
					0, NULL, 0, 0, NULL,
					&hKey, &dwDisp) != ERROR_SUCCESS)
	{
		cout << _T("Could not open/create key:");
		return;
	}
	if(RegDeleteValue(hKey, _T("StringTest")) != ERROR_SUCCESS)
		cout << _T("Could not delete key") << endl;
	if(RegDeleteValue(hKey, _T("DWORDTest")) != ERROR_SUCCESS)
		cout << _T("Could not delete key") << endl;
	if(RegDeleteValue(hKey, NULL) != ERROR_SUCCESS)
		cout << _T("Could not delete key") << endl;
	RegCloseKey(hKey);
}

// *** Listing 4.23
//
// Deleting a registry Key

void Listing4_23()
{	
	if(RegDeleteKey(HKEY_LOCAL_MACHINE, 
					_T("Software\\MyCompany")) != ERROR_SUCCESS)
		cout << _T("Could not delete key");
	else
		cout << _T("Key deleted");
}

// *** Listing 4.24
//
// Enumerating a registry key

void Listing4_24()
{
	HKEY hKey;
	DWORD dwSubKeys, dwValues, dwIndex, dwValueNameLen;
	DWORD dwValueType, dwValueLen;
	TCHAR szValueName[255];
	TCHAR szValue[255];
	DWORD dwValue;

	cout << _T("Opening key HKEY_LOCAL_MACHINE\\Platform") << endl;
	if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, _T("Platform"), 0, 0,
			&hKey) != ERROR_SUCCESS)
		cout << _T("Could not open key") << endl;
	else if (RegQueryInfoKey(hKey, NULL, NULL, NULL,
					&dwSubKeys, NULL, NULL, &dwValues,
					NULL, NULL, NULL, NULL) != ERROR_SUCCESS)
	{
		cout << _T("Could not query info") << endl;
		RegCloseKey(hKey);
	}
	else
	{
		cout << _T("Key has ") 
			<< dwSubKeys << _T(" subkeys and ") 
			<< dwValues << _T(" values") <<endl;
		// now read each value
		for(dwIndex = 0; dwIndex < dwValues; dwIndex++)
		{
			// First determine name and data type.
			dwValueNameLen = sizeof(szValueName);
			RegEnumValue(hKey, dwIndex, szValueName, &dwValueNameLen,
				NULL, &dwValueType, NULL, NULL);
			cout << _T("Value Name: ") << szValueName;
			switch (dwValueType)
			{
			case REG_SZ:
				cout << _T(" String:");
				dwValueLen = sizeof(szValue);
				dwValueNameLen = sizeof(szValueName);
				RegEnumValue(hKey, dwIndex, szValueName, &dwValueNameLen,
					NULL, &dwValueType, (LPBYTE)szValue, &dwValueLen);
				cout << szValue;
				break;
			case REG_DWORD:
				cout << _T(" DWORD:");
				dwValueLen = sizeof(DWORD);
				dwValueNameLen = sizeof(szValueName);
				RegEnumValue(hKey, dwIndex, szValueName, &dwValueNameLen,
					NULL, &dwValueType, (LPBYTE)&dwValue, &dwValueLen);
				cout << dwValue;
				break;
			default:
				cout << _T(" Other");
				break;
			}
			cout << endl;
		}
		RegCloseKey(hKey);
	}
}


// *** Listing 4.25
//
// Database counter with synchronization

LONG GetNextCounterValue()
{
	LONG dwCounter;
	HKEY hKey;
	DWORD dwDisp;
	HANDLE hMutex;

	hMutex = CreateMutex(NULL, FALSE, _T("CounterMutex"));
	if(hMutex == NULL)
	{
		cout << _T("Could not create mutex");
		return -1;
	}	
	else
		WaitForSingleObject(hMutex, INFINITE);
	if(RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
			_T("Software\\MyCompany\\MyApplication"),
			0, NULL, 0, 0, NULL,
			&hKey, &dwDisp) != 0)
	{
		cout << _T("Could not open registry key");
		ReleaseMutex(hMutex);
		CloseHandle(hMutex);
		return -1;
	}
	DWORD cbData, cbType;

	cbData = sizeof(DWORD);
	if(RegQueryValueEx(hKey, _T("Counter"), NULL, &cbType, 
				(LPBYTE)&dwCounter, &cbData) != 0)
	{
		dwCounter = 0;
	}
	dwCounter++;
	if(RegSetValueEx(hKey, _T("Counter"), NULL, REG_DWORD, 
			(LPBYTE)&dwCounter, 
			sizeof(DWORD)) != 0)
	{
		cout << _T("Could not save Server key");
		ReleaseMutex(hMutex);
		CloseHandle(hMutex);
		return 0;
	}
	RegCloseKey(hKey); 
	ReleaseMutex(hMutex);
	CloseHandle(hMutex);
	return dwCounter - 1;
}

void Listing4_25()
{
	cout << _T("Next counter value:") << GetNextCounterValue() << endl;

}