// 'Windows CE 3.0 Programming' Source Code Samples (Prentice Hall, 2000)
// Source Code Author: Nick Grattan (nick@softwarepaths.com)
// Version 1.00

// Chapter 3: The Object Store and Storage. Sample Code

#include "stdafx.h"
#include "examples.h"


// *** Listing 3.1
//
// Displays the free space in the object store.

void Listing3_1()
{
	ULARGE_INTEGER ulFree, ulTotalBytes, ulTotalFree;
	if(GetDiskFreeSpaceEx(_T("\\"),  // specify root directory in Object Store
		&ulFree, &ulTotalBytes, &ulTotalFree))
	{
		cout << _T("Bytes available to caller: ") << tab <<
				ulFree.LowPart << tab << ulFree.HighPart << endl;
		cout << _T("Total number bytes: ") << tab <<
				ulTotalBytes.LowPart << tab << ulTotalBytes.HighPart << endl;
		cout << _T("Total num. free bytes: ") << tab <<
				ulTotalFree.LowPart << tab << ulTotalFree.HighPart << endl;
	}
	else
		cout << _T("Could not get free space: ") << GetLastError();
}

// *** Listing 3.2
//
// Creates the specified directory.

void Listing3_2()
{
	TCHAR szPath[MAX_PATH + 1];

	if(!GetTextResponse(_T("Enter Directory to Create:"), szPath, MAX_PATH))
		return;

	if(!CreateDirectory(szPath, 0))
		cout << _T("Could not create directory: ") << GetLastError();
}

// *** Listing 3.3
//
// Deletes the specfied directory.

void Listing3_3()
{
	TCHAR szPath[MAX_PATH + 1];

	if(!GetTextResponse(_T("Enter Directory to Remove:"), szPath, MAX_PATH))
		return;

	if(!RemoveDirectory(szPath))
		cout << _T("Could not remove directory: ") << GetLastError();
}

// *** Listing 3.4
//
// List directory contents.

void PrintFindData(WIN32_FIND_DATA *fdData)
{
	// Directory and temporary means removeable media
	if ((fdData->dwFileAttributes & FILE_ATTRIBUTE_TEMPORARY)
		&& (fdData->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) )    
	{
		cout << _T("Removeable Media: ") << fdData->cFileName << endl;
	}
	// If it's a directory, print the name
	else if(fdData->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
	{
		cout << _T("Directory: ") << fdData->cFileName << endl;
	}
	else	// its a file, print name and size
	{
		cout << fdData->cFileName;
		cout << tab << _T("(") << fdData->nFileSizeLow << _T(")") << endl;
	}
}

void ListDirectoryContents(LPTSTR lpFileMask)
{
	HANDLE hFindFile;
	WIN32_FIND_DATA fdData;
	// get first file
	hFindFile = FindFirstFile(lpFileMask, &fdData);
	if(hFindFile != INVALID_HANDLE_VALUE)
	{
		PrintFindData(&fdData);
		while(FindNextFile(hFindFile, &fdData))
		{
			PrintFindData(&fdData);
		}
		FindClose(hFindFile);
	}
	else
		cout << _T("Call to FindFirstFile failed: ") << GetLastError();
}

void Listing3_4()
{
	ListDirectoryContents(_T("\\*.*"));
}

// *** Listing 3.5
//
// Responding to a storage card being inserted or removed. Function called in response to WM_DEVICECHANGE message.

void Listing3_5(WORD wParam, DEV_BROADCAST_HDR* dbt)
{
	// Must include dbt.h
	if(wParam == DBT_DEVICEARRIVAL)
		cout << _T("Device inserted") << endl;
	else if(wParam == DBT_DEVICEREMOVECOMPLETE)
		cout << _T("Device removed") << endl;
	switch(dbt->dbch_devicetype)
	{
		case DBT_DEVTYP_OEM:
			cout << _T("OEM/IHV Device") << endl;
			break;
		case DBT_DEVTYP_VOLUME:
			cout << _T("Logical Volume") << endl;
			break;
		case DBT_DEVTYP_PORT:
			cout << _T("Serial/Parallel Port") << endl;
			break;
		case DBT_DEVTYP_NET:
			cout << _T("Network device") << endl;
			break;
	}
}

// *** Listing 3.6
//
// Enumerate all Compact Flash cards

#include <projects.h>
// link with NOTE_PRJ.LIB

void Listing3_6()
{
	HANDLE hCF;
	WIN32_FIND_DATA fndMountable;

	hCF = FindFirstFlashCard(&fndMountable);
	if(hCF == INVALID_HANDLE_VALUE)
		cout << _T("No CF Cards") << endl;
	else
	{
		do
		{
			cout << _T("CF Card: ") << fndMountable.cFileName << endl;
		} while(FindNextFlashCard(hCF, &fndMountable));
		FindClose(hCF);
	}
}

// *** Listing 3.7
//
// Listing all objects (shares and printers) on a network.

// NB: include winnetwk.h

// This function handles WNet errors

void ErrorHandler(DWORD dwErrorNum, LPTSTR s)
{
	cout << _T("Failure in: ") << s << _T(" ") << GetLastError() << endl;
}

// This function displays the information in a NETRESOURCE structure

void DisplayStruct(LPNETRESOURCE nr)
{
	cout << _T("Type: ");
	switch(nr->dwType)
	{
	case RESOURCETYPE_DISK:
		cout << _T("Disk") << endl;
		break;
	case RESOURCETYPE_PRINT:
		cout << _T("Printer") << endl;
		break;
	case RESOURCETYPE_ANY:
		cout << _T("Any") << endl;
	}
	cout << _T("Display Type: ");
	switch(nr->dwDisplayType)
	{
	case RESOURCEDISPLAYTYPE_DOMAIN:
		cout << _T("Domain") << endl;
		break;
	case RESOURCEDISPLAYTYPE_GENERIC:
		cout << _T("Generic") << endl;
		break;
	case RESOURCEDISPLAYTYPE_SERVER:
		cout << _T("Server") << endl;
		break;
	case RESOURCEDISPLAYTYPE_SHARE:
		cout << _T("Share") << endl;
	}
	if(nr->lpLocalName)
		cout << _T("Local Name: ") << nr->lpLocalName << endl;
	if(nr->lpRemoteName)
		cout << _T("Remote Name: ") << nr->lpRemoteName << endl;
	if(nr->lpComment)
		cout << _T("Comment: ") << nr->lpComment << endl;
	if(nr->lpProvider)
		cout << _T("Provider: ") << nr->lpProvider << endl;
	cout << endl;
}

// Recursive function to enumerate resources
BOOL EnumerateResources(LPNETRESOURCE nrStartingPoint)
{
	DWORD dwResult, dwResultEnum, i;
	LPNETRESOURCE lpNRBuffer;
	DWORD dwBufferSize = 16384;
	DWORD dwNumEntries = 0xFFFFFFFF;
	HANDLE hEnum;

	dwResult = WNetOpenEnum(RESOURCE_GLOBALNET, RESOURCETYPE_ANY, 
			0, nrStartingPoint, &hEnum);
	if(dwResult != NO_ERROR)
	{
		ErrorHandler(dwResult, _T("WNetOpenEnum"));
		return FALSE;
	}
	// allocate a buffer to hold resources
	lpNRBuffer = (LPNETRESOURCE) LocalAlloc(LPTR, dwBufferSize);
	// loop through all the elements in the container
	do
	{
		dwBufferSize = 16384;
		dwNumEntries = 0xFFFFFFFF;
		// Get resources
		dwResultEnum = WNetEnumResource(hEnum, &dwNumEntries, lpNRBuffer, &dwBufferSize);
		if(dwResultEnum == NO_ERROR)
		{
			// loop through each of the entries
			for(i = 0; i < dwNumEntries; i++)
			{
				DisplayStruct(&lpNRBuffer[i]);
				// if container, recursively open it
				if(lpNRBuffer[i].dwUsage & RESOURCEUSAGE_CONTAINER)
				{
					if(!EnumerateResources(&lpNRBuffer[i]))
						cout << _T("Enumeration Failed.") << endl;
				}
			}
		}
		else if(dwResultEnum != ERROR_NO_MORE_ITEMS)
		{
			ErrorHandler(dwResultEnum, _T("WNetEnumResource"));
			break;
		}
	}
	while(dwResultEnum != ERROR_NO_MORE_ITEMS);
	// Clean up
	LocalFree(lpNRBuffer);
	dwResult = WNetCloseEnum(hEnum);
	if(dwResult != NO_ERROR)
	{
		ErrorHandler(dwResult, _T("WNetCloseEnum"));
		return FALSE;
	}
	return TRUE;
}

void Listing3_7()
{
	// Start the recursion at the net level
	NETRESOURCE nr;
	TCHAR szContainer[MAX_PATH + 1];
	if(!GetTextResponse(_T("Enter Container (domain or computer) to list:"), szContainer, MAX_PATH))
		return;
	memset(&nr, 0, sizeof(nr));
	nr.lpRemoteName = szContainer;
	nr.dwUsage = RESOURCEUSAGE_CONTAINER;
	EnumerateResources(&nr);
}

// *** Listing 3.8
//
// Adds a network connection.

void Listing3_8()
{
	TCHAR szUNCPath[MAX_PATH + 1];
	TCHAR szLocalName[MAX_PATH + 1];
	NETRESOURCE nr;

	if(!GetTextResponse(_T("Enter UNC to Connect to:"), szUNCPath, MAX_PATH))
		return;
	if(!GetTextResponse(_T("Enter Local Name:"), szLocalName, MAX_PATH))
		return;
	
	nr.dwType = RESOURCETYPE_DISK;
	nr.lpRemoteName = szUNCPath;
	nr.lpLocalName = szLocalName;
	nr.lpProvider = NULL;		// Microsoft Network is only provider
	if(WNetAddConnection3(hWnd, &nr, NULL, NULL, CONNECT_UPDATE_PROFILE) != NO_ERROR)
		cout << _T("Error adding connection: ") << GetLastError() << endl;
}

// *** Listing 3.9
//
// Disconnects a network connection.

void Listing3_9()
{
	TCHAR szPath[MAX_PATH + 1];
	if(!GetTextResponse(_T("Enter UNC or Local Name to disconnect:"), szPath, MAX_PATH))
		return;
	if(WNetCancelConnection2(szPath, CONNECT_UPDATE_PROFILE, TRUE) != ERROR_SUCCESS )
		cout << _T("Error disconnecting: ") << GetLastError();
}

// *** Listing 3.10
//
// Adds a network connection using a dialog box.

void Listing3_10()
{
	CONNECTDLGSTRUCT cs;
	DWORD dwResult;
	NETRESOURCE nr;

	nr.dwType = RESOURCETYPE_DISK;
	nr.lpRemoteName = NULL;
	nr.lpLocalName = NULL;
	nr.lpProvider = NULL;

	cs.cbStructure = sizeof(cs);
	cs.hwndOwner = hWnd;
	cs.lpConnRes = &nr;	
	cs.dwFlags = 0;

	dwResult = WNetConnectionDialog1(&cs);
	if(dwResult == 0xFFFFFFFF)
		cout << _T("User cancelled") << endl;
	else if(dwResult != WN_SUCCESS)
		cout << _T("Error connecting: ") << dwResult << endl;
}

// *** Listing 3.11
//
// Disconnects a network connection using a dialog box.

void Listing3_11()
{
	DWORD dwResult;

	dwResult = WNetDisconnectDialog(hWnd, 0);
	if(dwResult == 0xFFFFFFFF)
		cout << _T("User cancelled dialog") << endl;
	else if(dwResult != NO_ERROR)
		cout << _T("Error disconnecting: ") << GetLastError();
}

// *** Listing 3.12
//
// Connects to network printer.

void Listing3_12()
{
	TCHAR szUNCPath[MAX_PATH + 1], szLocal[MAX_PATH + 1];
	NETRESOURCE nr;

	if(!GetTextResponse(_T("Enter Printer UNC to Connect to:"), szUNCPath, MAX_PATH))
		return;
	if(!GetTextResponse(_T("Enter Local name for printer:"), szLocal, MAX_PATH))
		return;
	
	nr.dwType = RESOURCETYPE_PRINT;
	nr.lpRemoteName = szUNCPath;
	nr.lpLocalName = szLocal;
	nr.lpProvider = NULL;		// Microsoft Network is only provider
	if(WNetAddConnection3(hWnd, &nr, NULL, NULL, CONNECT_UPDATE_PROFILE) != NO_ERROR)
		cout << _T("Error adding Printer connection: ") << GetLastError() << endl;
}

// *** Listing 3.13
//
// Disconnects from network printer.

void Listing3_13()
{
	DWORD dwResult;
	DISCDLGSTRUCT ds;
	
	TCHAR szUNCPath[MAX_PATH + 1];

	if(!GetTextResponse(_T("Enter Printer UNC to disconnect from:"), szUNCPath, MAX_PATH))
		return;
	ds.cbStructure = sizeof(ds);
	ds.hwndOwner = hWnd;
	ds.lpLocalName = NULL;
	ds.lpRemoteName = szUNCPath;
	ds.dwFlags = DISC_NO_FORCE ;
	dwResult = WNetDisconnectDialog1(&ds);
	if(dwResult != NO_ERROR)
		cout << _T("Error disconnecting: ") << GetLastError();
}

// *** Listing 3.14
//
// Lists security details for network connection.

void Listing3_14()
{
	DWORD dwLen = 50;
	
	TCHAR szConnection[MAX_PATH + 1];
	TCHAR szUser[51];

	if(!GetTextResponse(_T("Enter connection to list:"), szConnection, MAX_PATH))
		return;

	if(WNetGetUser(szConnection, szUser, &dwLen) != ERROR_SUCCESS)
		cout << _T("Error getting user information: ") << GetLastError() << endl; 
	else
		cout << szConnection << _T(" connected as user ") << szUser << endl;

}

// *** Listing 3.15
//
// Lists current connections

void PrintConnectionData(WIN32_FIND_DATA* lpFD)
{
	TCHAR szRemoteName[MAX_PATH + 1];
	DWORD dwSize = MAX_PATH;

	cout << _T("Connection: ") << lpFD->cFileName;
	if(WNetGetConnection(lpFD->cFileName, szRemoteName, &dwSize) == NO_ERROR)
		cout << _T(" to ") << szRemoteName << endl;
	else if(GetLastError() == ERROR_CONNECTION_UNAVAIL)
		cout << _T(" not currently connected.");
	else
		cout << _T(" Error calling WNetGetConnection ") << GetLastError() << endl;
}

void Listing3_15()
{
	HANDLE hFindFile;
	WIN32_FIND_DATA fdData; 
	// get first file
	hFindFile = FindFirstFile(_T("\\network\\*.*"), &fdData);
	if(hFindFile != INVALID_HANDLE_VALUE)
	{
		PrintConnectionData(&fdData);
		while(FindNextFile(hFindFile, &fdData))
		{
			PrintFindData(&fdData);
		}
		FindClose(hFindFile);
	}
	else if(GetLastError() == ERROR_NO_MORE_FILES)
		cout << _T("No shares");
	else
		cout << _T("Call to FindFirstFile failed: ") << GetLastError();
}



