// 'Windows CE 3.0 Programming' Source Code Samples (Prentice Hall, 2000)
// Source Code Author: Nick Grattan (nick@softwarepaths.com)
// Version 1.00


// Chapter 15: Microsoft Message Queue

#include "stdafx.h"
#include "examples.h"

// NB: Default compilation includes empty versions of the sample files so that
// the examples.exe will run on a Windows CE device that does not have MMQ
// installed. To produce a version that includes the MMQ samples uncomment the
// next #define. Note that MSMQ is not available under emulation.

#ifndef _WIN32_WCE_EMULATION
// #define USE_MMQ
#endif

#ifdef USE_MMQ

// *** Listing 15.1
//
// Open a Queue

#include <mq.h>
// Add MSMQRT.LIB to project

void DisplayOpenError(HRESULT hr)
{
	if(hr == MQ_ERROR_ACCESS_DENIED)
		cout << _T("Don't have access rights") << endl;
	else if(hr == MQ_ERROR_ILLEGAL_FORMATNAME)
		cout << _T("Illegal Format Name") << endl;
	else if(hr == MQ_ERROR_QUEUE_NOT_FOUND )
		cout << _T("Queue not found") << endl;
	else if(hr == MQ_ERROR_SERVICE_NOT_AVAILABLE )
		cout << _T("Cannot connect to queue manager") << endl;
	else if(hr == MQ_ERROR_INVALID_PARAMETER )
		cout << _T("Invalid Parameter") << endl;
	else if(hr == MQ_ERROR_SHARING_VIOLATION )
		cout << _T("Sharing violation") << endl;
	else if(hr == MQ_ERROR_UNSUPPORTED_ACCESS_MODE )
		cout << _T("Invalid access mode") << endl;
	else if(hr == MQ_ERROR_UNSUPPORTED_FORMATNAME_OPERATION )
		cout << _T("Invalid format name") << endl;
	else 
		cout << _T("Unexpected Error") << endl;
}

void Listing15_1()
{
	HRESULT hr;
	QUEUEHANDLE hq;

	TCHAR wszFormatName[256];
	DWORD dwFormatNameLength = 256;
	
	hr = MQPathNameToFormatName(_T("nickdell\\Private$\\WinCEQueue"),
                              wszFormatName,
                              &dwFormatNameLength);
	cout << wszFormatName << endl;

	hr = MQOpenQueue(wszFormatName,
			MQ_SEND_ACCESS,
			MQ_DENY_NONE,
			&hq);
	if(hr == MQ_OK)
		cout << _T("Opened queue") << endl;
	else
	{
		DisplayOpenError(hr);
		return;
	}

	DWORD cPropId = 0; 
  
	MQMSGPROPS msgprops;
	MSGPROPID aMsgPropId[4];     
	MQPROPVARIANT aMsgPropVar[4];
	HRESULT aMsgStatus[4];
  
	aMsgPropId[cPropId] = PROPID_M_LABEL;
	aMsgPropVar[cPropId].vt = VT_LPWSTR; 
	aMsgPropVar[cPropId].pwszVal = _T("Test Message");
	cPropId++;
  
	aMsgPropId[cPropId] = PROPID_M_BODY_TYPE;
	aMsgPropVar[cPropId].vt = VT_UI4;
	aMsgPropVar[cPropId].ulVal = VT_BSTR;	
	cPropId++;

	BSTR bStr = SysAllocString(_T("Body text for the message"));

  	aMsgPropId[cPropId] = PROPID_M_BODY;
	aMsgPropVar[cPropId].vt = VT_VECTOR|VT_UI1;
	aMsgPropVar[cPropId].caub.pElems = (LPBYTE)bStr;
	aMsgPropVar[cPropId].caub.cElems = SysStringByteLen(bStr);
	cPropId++;

	msgprops.cProp = cPropId;                           // Number of properties
	msgprops.aPropID = aMsgPropId;                      // Ids of properties
	msgprops.aPropVar = aMsgPropVar;                    // Values of properties
	msgprops.aStatus = aMsgStatus;                      // Error reports
   
	
	hr = MQSendMessage(hq,								// Handle to open queue
                     &msgprops,							// Properties of message
                     NULL);								// No transaction
	if (FAILED(hr))
		cout << _T("Could not send message") << endl;

	else
		cout << _T("Message queued") << endl;
	MQCloseQueue(hq);
}

// *** Listing 15.2
//
// Create Queue

void Listing15_2()
{

	DWORD cPropId = 0;                                  // Define the property counter.
  
	//Define the MQQUEUPROPS structure.
	MQQUEUEPROPS QueueProps;
	MQPROPVARIANT aQueuePropVar[2];
	QUEUEPROPID aQueuePropId[2];
	HRESULT aQueueStatus[2];
  
	HRESULT hr;                                         // Define results.
	PSECURITY_DESCRIPTOR pSecurityDescriptor=NULL;      // Security descriptor
  
  
	// Queue pathname
	LPWSTR wszPathName = _T(".\\PRIVATE$\\WinCEInQueue");
  
	// Queue label
	LPWSTR wszQueueLabel = _T("Message to be received by Windows CE Device");
  
	// Format name buffer for queue
	DWORD dwFormatNameLength = 256;               // Length of format name buffer
	WCHAR wszFormatName[256];                     // Format name buffer
  
	aQueuePropId[cPropId] = PROPID_Q_PATHNAME;          // Property ID
	aQueuePropVar[cPropId].vt = VT_LPWSTR;              // Type indicator
	aQueuePropVar[cPropId].pwszVal = wszPathName;       // Pathname of queue
	cPropId++;
  
	aQueuePropId[cPropId] = PROPID_Q_LABEL;             // Property ID
	aQueuePropVar[cPropId].vt = VT_LPWSTR;              // Type indicator
	aQueuePropVar[cPropId].pwszVal = wszQueueLabel;     // Label of queue
	cPropId++;
  
  	QueueProps.cProp = cPropId;                         // Number of properties
	QueueProps.aPropID = aQueuePropId;                  // Ids of properties
	QueueProps.aPropVar = aQueuePropVar;                // Values of properties
	QueueProps.aStatus = aQueueStatus;                  // Error reports
  
	hr = MQCreateQueue(pSecurityDescriptor,             // Security descriptor
						 &QueueProps,                     // Queue properties
						 wszFormatName,             // OUT: Format name of queue
						 &dwFormatNameLength);      // OUT: Format name length

	if(hr == MQ_OK)
		cout << wszFormatName << _T(" created") << endl;
	else if(hr == MQ_ERROR_ACCESS_DENIED )
		cout << _T("Access Denied") << endl;
	else if(hr == MQ_ERROR_ILLEGAL_PROPERTY_VALUE )
		cout << _T("Illegal Property Value") << endl;
	else if(hr == MQ_ERROR_ILLEGAL_QUEUE_PATHNAME )
		cout << _T("Illegal pathname") << endl;
	else if(hr == MQ_ERROR_ILLEGAL_SECURITY_DESCRIPTOR )
		cout << _T("Illegal security descriptor") << endl;
	else if(hr == MQ_ERROR_INSUFFICIENT_PROPERTIES )
		cout << _T("Path name not specified") << endl;
	else if(hr == MQ_ERROR_INVALID_OWNER )
		cout << _T("Invalid owner") << endl;
	else if(hr == MQ_ERROR_PROPERTY )
		cout << _T("Error in property specification") << endl;
	else if(hr == MQ_ERROR_PROPERTY_NOTALLOWED )
		cout << _T("Property not allowed when creating queue") << endl;
	else if(hr == MQ_ERROR_QUEUE_EXISTS )
		cout << _T("Queue already exists") << endl;
	else if(hr == MQ_ERROR_SERVICE_NOT_AVAILABLE )
		cout << _T("Service not available") << endl;
	else if(hr == MQ_INFORMATION_FORMATNAME_BUFFER_TOO_SMALL )
		cout << _T("Format name buffer too small") << endl;
	else if(hr == MQ_INFORMATION_PROPERTY )
		cout << _T("Succeeded, but property returned warning") << endl;
}

// *** Listing 15.3
//
// Read Message From Queue 

void DisplayReadError(HRESULT hr)
{
	if(hr == MQ_ERROR_ACCESS_DENIED)
		cout << _T("Don't have access rights") << endl;
	else if(hr == MQ_ERROR_BUFFER_OVERFLOW )
		cout << _T("Buffer Overflow") << endl;
	else if(hr == MQ_ERROR_SENDERID_BUFFER_TOO_SMALL )
		cout << _T("Sender ID Buffer too small") << endl;
	else if(hr == MQ_ERROR_SYMM_KEY_BUFFER_TOO_SMALL )
		cout << _T("Symmetric key buffer too small") << endl;
	else if(hr == MQ_ERROR_SENDER_CERT_BUFFER_TOO_SMALL )
		cout << _T("Cert buffer too small") << endl;
	else if(hr == MQ_ERROR_SIGNATURE_BUFFER_TOO_SMALL )
		cout << _T("Signature buffer too small") << endl;
	else if(hr == MQ_ERROR_PROV_NAME_BUFFER_TOO_SMALL )
		cout << _T("Provider name too small") << endl;
	else if(hr == MQ_ERROR_LABEL_BUFFER_TOO_SMALL)
		cout << _T("Label buffer too small") << endl;
	else if(hr == MQ_ERROR_FORMATNAME_BUFFER_TOO_SMALL )
		cout << _T("Format name buffer too small") << endl;
	else if(hr == MQ_ERROR_DTC_CONNECT )
		cout << _T("Cannot connect to DTC") << endl;
	else if(hr == MQ_ERROR_INSUFFICIENT_PROPERTIES )
		cout << _T("Insufficient properties") << endl;
	else if(hr == MQ_ERROR_INVALID_HANDLE )
		cout << _T("Invalid queue handle") << endl;
	else if(hr == MQ_ERROR_IO_TIMEOUT )
		cout << _T("Timeout") << endl;
	else if(hr == MQ_ERROR_MESSAGE_ALREADY_RECEIVED )
		cout << _T("Message has been removed from queue") << endl;
	else if(hr == MQ_ERROR_OPERATION_CANCELLED )
		cout << _T("Operation cancelled") << endl;
	else if(hr == MQ_ERROR_PROPERTY )
		cout << _T("Property error") << endl;
	else if(hr == MQ_ERROR_QUEUE_DELETED )
		cout << _T("Queue deleted") << endl;
	else if(hr == MQ_ERROR_ILLEGAL_CURSOR_ACTION )
		cout << _T("Illegal cursor action") << endl;
	else if(hr == MQ_ERROR_SERVICE_NOT_AVAILABLE )
		cout << _T("Service not available") << endl;
	else if(hr == MQ_ERROR_STALE_HANDLE )
		cout << _T("Stale handle") << endl;
	else if(hr == MQ_ERROR_TRANSACTION_USAGE )
		cout << _T("Transaction Error") << endl;
	else if(hr == MQ_INFORMATION_PROPERTY )
		cout << _T("Property returned information") << endl;
	else
		cout << _T("Unknown error") << endl;
}
	
void TimeToSystemTime(time_t t, LPSYSTEMTIME pst)
{
	// Note that LONGLONG is a 64-bit value
    LONGLONG ll;
	FILETIME ft;

    ll = Int32x32To64(t, 10000000) + 116444736000000000;
    ft.dwLowDateTime = (DWORD)ll;
    ft.dwHighDateTime = ll >> 32;
    FileTimeToSystemTime(&ft, pst);
} 

void Listing15_3()
{
	HRESULT hr;
	QUEUEHANDLE hq;

	TCHAR wszFormatName[256];
	DWORD dwFormatNameLength = 256;
	
	hr = MQPathNameToFormatName(_T(".\\Private$\\WinCEInQueue"),
                              wszFormatName,
                              &dwFormatNameLength);
	cout << wszFormatName << endl;

	hr = MQOpenQueue(wszFormatName,
			MQ_RECEIVE_ACCESS,
			MQ_DENY_NONE,
			&hq);
	if(hr == MQ_OK)
		cout << _T("Opened queue") << endl;
	else
	{
		DisplayOpenError(hr);
		return;
	}

	DWORD dwRecAction = MQ_ACTION_RECEIVE;            // Receive action
  
	MQMSGPROPS msgprops;
	MSGPROPID aMsgPropId[4];
	MQPROPVARIANT aMsgPropVar[4];
	HRESULT aMsgStatus[4];
	// Message body buffer
	DWORD dwBodyBufferSize = 1024;
	LPTSTR lpszBodyBuffer = new TCHAR[dwBodyBufferSize];
	DWORD cPropId = 0;
	
	aMsgPropId[cPropId] = PROPID_M_BODY_SIZE;       // Property ID
	aMsgPropVar[cPropId].vt = VT_UI4;               // Type indicator
	cPropId++;
  
	aMsgPropId[cPropId] = PROPID_M_BODY;            // Property ID
	aMsgPropVar[cPropId].vt = VT_VECTOR|VT_UI1;     // Type indicator
	aMsgPropVar[cPropId].caub.pElems = (UCHAR*)lpszBodyBuffer;
	aMsgPropVar[cPropId].caub.cElems = dwBodyBufferSize;
  	cPropId++;
	// Reading additional properties....
	aMsgPropId[cPropId] = PROPID_M_SENTTIME;
	aMsgPropVar[cPropId].vt = VT_UI4;               
	cPropId++;
	aMsgPropId[cPropId] = PROPID_M_ARRIVEDTIME;
	aMsgPropVar[cPropId].vt = VT_UI4;               
	cPropId++;

  
	msgprops.cProp = cPropId;          // Number of message properties
	msgprops.aPropID = aMsgPropId;     // Ids of message properties
	msgprops.aPropVar = aMsgPropVar;   // Values of message properties
	msgprops.aStatus  = aMsgStatus;    // Error reports
  
  	hr = MQReceiveMessage(hq,             // Queue handle
                        0,                  // Max time (msec)
                        dwRecAction,        // Receive action
                        &msgprops,          // Msg property structure
                        NULL,               // No OVERLAPPED structure
                        NULL,               // No callback function
                        NULL,               // No cursor
                        NULL                // No transaction
                        );
  
	if (FAILED(hr))
	{
		DisplayReadError(hr);
		MQCloseQueue(hq);
		delete lpszBodyBuffer;
		return;
	}
  
	if (aMsgPropVar[0].ulVal == 0)
		cout <<  _T("No message body exists.") << endl;
	else
	{
		cout << _T("The message body size is: ")  << lpszBodyBuffer << endl;
		cout << _T("The message body is: ")  << lpszBodyBuffer << endl;
		// display sent time and arrived time
		SYSTEMTIME st;

		TimeToSystemTime(aMsgPropVar[2].ulVal, &st);
		cout << _T("Sent Time: ") << st.wMonth << _T("/") << st.wDay
			<< _T("/") << st.wYear << _T(" ") << st.wHour 
			<< _T(":") << st.wMinute << _T(":") << st.wSecond << endl;

		TimeToSystemTime(aMsgPropVar[3].ulVal, &st);
		cout << _T("Receive Time: ") << st.wMonth << _T("/") << st.wDay
			<< _T("/") << st.wYear << _T(" ") << st.wHour 
			<< _T(":") << st.wMinute << _T(":") << st.wSecond << endl;

	}
	delete lpszBodyBuffer;
  
	MQCloseQueue(hq);
}

// *** Listing 15.4
//
// Using a cursor

void Listing15_4()
{
	HRESULT hr;
	QUEUEHANDLE hq;

	TCHAR wszFormatName[256];
	DWORD dwFormatNameLength = 256;
	
	hr = MQPathNameToFormatName(_T(".\\Private$\\WinCEInQueue"),
                              wszFormatName,
                              &dwFormatNameLength);
	cout << wszFormatName << endl;

	hr = MQOpenQueue(wszFormatName,
			MQ_RECEIVE_ACCESS,
			MQ_DENY_NONE,
			&hq);
	if(hr == MQ_OK)
		cout << _T("Opened queue") << endl;
	else
	{
		DisplayOpenError(hr);
		return;
	}

	DWORD dwRecAction;
  
	MQMSGPROPS msgprops;
	MSGPROPID aMsgPropId[4];
	MQPROPVARIANT aMsgPropVar[4];
	HRESULT aMsgStatus[4];
	// Message body buffer
	DWORD dwBodyBufferSize = 1024;
	TCHAR lpszBodyBuffer[1024];
	DWORD cPropId = 0;
	
	aMsgPropId[cPropId] = PROPID_M_LABEL_LEN;       // Property ID
	aMsgPropVar[cPropId].vt = VT_UI4;               // Type indicator
	aMsgPropVar[cPropId].ulVal = 1024;
	cPropId++;
  
	aMsgPropId[cPropId] = PROPID_M_LABEL;            // Property ID
	aMsgPropVar[cPropId].vt = VT_LPWSTR;			// Type indicator
	aMsgPropVar[cPropId].pwszVal  = lpszBodyBuffer;
  	cPropId++;
  
	msgprops.cProp = cPropId;          // Number of message properties
	msgprops.aPropID = aMsgPropId;     // Ids of message properties
	msgprops.aPropVar = aMsgPropVar;   // Values of message properties
	msgprops.aStatus  = aMsgStatus;    // Error reports

	HANDLE hCursor;
	hr = MQCreateCursor(hq, &hCursor);
	if(FAILED(hr))
	{
		cout << _T("Could not open cursor") << endl;
		MQCloseQueue(hq);
	}

	dwRecAction = MQ_ACTION_PEEK_CURRENT;
	while(TRUE)
	{
  		hr = MQReceiveMessage(hq,				// Queue handle
							0,                  // Max time (msec)
							dwRecAction,        // Receive action
							&msgprops,          // Msg property structure
							NULL,               // No OVERLAPPED structure
							NULL,               // No callback function
							hCursor,            // Cursor
							NULL                // No transaction
							);
	 	dwRecAction = MQ_ACTION_PEEK_NEXT;
 		if (FAILED(hr))
		{
			DisplayReadError(hr);
			MQCloseCursor(hCursor);
			MQCloseQueue(hq);
			return;
		}
		cout << _T("Label: ")  << lpszBodyBuffer << endl;
	}  
}

// *** Listing 15.5
//
// Asynchronous Message Read

void APIENTRY ReceiveCallbackRoutine(HRESULT hr, QUEUEHANDLE hSource,
			DWORD dwTimeout, DWORD dwAction, MQMSGPROPS* pMessageProps,
			LPOVERLAPPED lpOverlapped, HANDLE hCursor)
{
	if (FAILED(hr))
	{
		DisplayReadError(hr);
	}
	else
	{
		cout << _T("Async Msg Read: ")  <<
			pMessageProps->aPropVar[1].pwszVal << endl;
		MQCloseQueue(hSource);
	}
	delete pMessageProps->aPropVar[1].pwszVal;
	delete pMessageProps->aPropID;
	delete pMessageProps->aPropVar;
	delete pMessageProps->aStatus;
	delete pMessageProps;
}


void Listing15_5()
{
	HRESULT hr;

	TCHAR wszFormatName[256];
	DWORD dwFormatNameLength = 256;
	QUEUEHANDLE hq;
	
	hr = MQPathNameToFormatName(_T(".\\Private$\\WinCEInQueue"),
                              wszFormatName,
                              &dwFormatNameLength);
	cout << wszFormatName << endl;

	hr = MQOpenQueue(wszFormatName,
			MQ_RECEIVE_ACCESS,
			MQ_DENY_NONE,
			&hq);
	if(hr == MQ_OK)
		cout << _T("Opened queue") << endl;
	else
	{
		DisplayOpenError(hr);
		return;
	}

	DWORD dwRecAction;
	LPTSTR lpszLabelBuffer = new TCHAR[1024];

	MQMSGPROPS* pMsgprops = new MQMSGPROPS;
	MSGPROPID* pMsgPropId = new MSGPROPID[4];
	MQPROPVARIANT* pMsgPropVar = new MQPROPVARIANT[4];
	HRESULT* pMsgStatus = new HRESULT[4];

	DWORD cPropId = 0;
	
	pMsgPropId[cPropId] = PROPID_M_LABEL_LEN;       // Property ID
	pMsgPropVar[cPropId].vt = VT_UI4;               // Type indicator
	pMsgPropVar[cPropId].ulVal = 1024;
	cPropId++;
  
	pMsgPropId[cPropId] = PROPID_M_LABEL;            // Property ID
	pMsgPropVar[cPropId].vt = VT_LPWSTR;			// Type indicator
	pMsgPropVar[cPropId].pwszVal  = lpszLabelBuffer;
  	cPropId++;
  
	pMsgprops->cProp = cPropId;          // Number of message properties
	pMsgprops->aPropID = pMsgPropId;     // Ids of message properties
	pMsgprops->aPropVar = pMsgPropVar;   // Values of message properties
	pMsgprops->aStatus  = pMsgStatus;    // Error reports

	dwRecAction = MQ_ACTION_RECEIVE;
  	hr = MQReceiveMessage(hq,				// Queue handle
						1000 * 60 * 100,    // Max time (msec)
						dwRecAction,        // Receive action
						pMsgprops,          // Msg property structure
						NULL,               // No OVERLAPPED structure
						ReceiveCallbackRoutine, // Callback function
						NULL,            // No Cursor
						NULL                // No transaction
						);
	if(FAILED(hr))
	{
		delete pMsgPropId;
		delete pMsgPropVar;
		delete pMsgStatus;
		delete pMsgprops;
		delete lpszLabelBuffer;
		DisplayReadError(hr);
	}
}


// *** Listing 15.6
//
// Admin queue and message timeout

void DisplayMsgId(BYTE bMsgId[])
{
	for(int i = 0; i < 20; i++)
		cout << (int)bMsgId[i] << _T(" ");
}

void APIENTRY AdministrationCallbackRoutine(HRESULT hr, QUEUEHANDLE hSource,
			DWORD dwTimeout, DWORD dwAction, MQMSGPROPS* pMessageProps,
			LPOVERLAPPED lpOverlapped, HANDLE hCursor)
{
	if (FAILED(hr))
	{
		DisplayReadError(hr);
	}
	else
	{
		cout << _T("Async Admin Msg Read: ") << 
			pMessageProps->aPropVar[1].pwszVal << endl;
		DisplayMsgId(pMessageProps->aPropVar[2].caub.pElems);
		MQCloseQueue(hSource);
	}
	delete pMessageProps->aPropVar[1].pwszVal;
	delete pMessageProps->aPropVar[2].caub.pElems;
	delete pMessageProps->aPropID;
	delete pMessageProps->aPropVar;
	delete pMessageProps->aStatus;
	delete pMessageProps;
}

void CreateAdminQueue()
{
	DWORD cPropId = 0;                                  // Define the property counter.
  
	MQQUEUEPROPS QueueProps;
	MQPROPVARIANT aQueuePropVar[2];
	QUEUEPROPID aQueuePropId[2];
	HRESULT aQueueStatus[2];
  
	HRESULT hr;                                         // Define results.
  
	// Queue pathname
	LPWSTR wszPathName = _T(".\\Private$\\WinCEInQueueAdmin");
  
	// Queue label
	LPWSTR wszQueueLabel = _T("Admin Queue for WinCEInQueue");
  
	// Format name buffer for queue
	DWORD dwFormatNameLength = 256;               // Length of format name buffer
	WCHAR wszFormatName[256];                     // Format name buffer
  
	aQueuePropId[cPropId] = PROPID_Q_PATHNAME;          // Property ID
	aQueuePropVar[cPropId].vt = VT_LPWSTR;              // Type indicator
	aQueuePropVar[cPropId].pwszVal = wszPathName;       // Pathname of queue
	cPropId++;
  
	aQueuePropId[cPropId] = PROPID_Q_LABEL;             // Property ID
	aQueuePropVar[cPropId].vt = VT_LPWSTR;              // Type indicator
	aQueuePropVar[cPropId].pwszVal = wszQueueLabel;     // Label of queue
	cPropId++;
  
  	QueueProps.cProp = cPropId;                         // Number of properties
	QueueProps.aPropID = aQueuePropId;                  // Ids of properties
	QueueProps.aPropVar = aQueuePropVar;                // Values of properties
	QueueProps.aStatus = aQueueStatus;                  // Error reports
  
	hr = MQCreateQueue(NULL,             // Security descriptor
						 &QueueProps,                     // Queue properties
						 wszFormatName,             // OUT: Format name of queue
						 &dwFormatNameLength);      // OUT: Format name length

	if(hr == MQ_OK)
		cout << wszFormatName << _T(" created") << endl;
	else
		cout << _T("Could not create admin queue") << endl;
}

void InitializeAdminQueueRead()
{
	HRESULT hr;

	TCHAR wszFormatName[256];
	DWORD dwFormatNameLength = 256;
	QUEUEHANDLE hq;
	
	hr = MQPathNameToFormatName(_T(".\\Private$\\WinCEInQueueAdmin"),
                              wszFormatName,
                              &dwFormatNameLength);
	hr = MQOpenQueue(wszFormatName, MQ_RECEIVE_ACCESS, MQ_DENY_NONE, &hq);
	if(hr == MQ_OK)
		cout << _T("Opened admin queue") << endl;
	else
	{
		DisplayOpenError(hr);
		return;
	}

	DWORD dwRecAction;
  
	LPTSTR lpszAdminLabelBuffer = new TCHAR[1024];
	MQMSGPROPS* pMsgprops = new MQMSGPROPS;
	MSGPROPID* pMsgPropId = new MSGPROPID[4];
	MQPROPVARIANT* pMsgPropVar = new MQPROPVARIANT[4];
	HRESULT* pMsgStatus = new HRESULT[4];

	DWORD cPropId = 0;
	
	pMsgPropId[cPropId] = PROPID_M_LABEL_LEN;
	pMsgPropVar[cPropId].vt = VT_UI4;
	pMsgPropVar[cPropId].ulVal = 1024;
	cPropId++;
  
	pMsgPropId[cPropId] = PROPID_M_LABEL;
	pMsgPropVar[cPropId].vt = VT_LPWSTR;
	pMsgPropVar[cPropId].pwszVal  = lpszAdminLabelBuffer;
  	cPropId++;
  	
	LPBYTE lpbMsgID = new BYTE[20];
	memset(lpbMsgID, 0, 20);
	pMsgPropId[cPropId] = PROPID_M_CORRELATIONID;     
	pMsgPropVar[cPropId].vt = VT_VECTOR | VT_UI1; 
	pMsgPropVar[cPropId].caub.pElems = lpbMsgID;
	pMsgPropVar[cPropId].caub.cElems = 20;
	cPropId++;

	pMsgprops->cProp = cPropId;          // Number of message properties
	pMsgprops->aPropID = pMsgPropId;     // Ids of message properties
	pMsgprops->aPropVar = pMsgPropVar;   // Values of message properties
	pMsgprops->aStatus  = pMsgStatus;    // Error reports

	dwRecAction = MQ_ACTION_RECEIVE;
  	hr = MQReceiveMessage(hq,				// Queue handle
						60 * 1000,			// Max time (msec)
						dwRecAction,        // Receive action
						pMsgprops,          // Msg property structure
						NULL,               // No OVERLAPPED structure
						AdministrationCallbackRoutine, // Callback function
						NULL,				// No Cursor
						NULL                // No transaction
						);
	if(FAILED(hr))
	{
		delete pMsgPropId;
		delete pMsgPropVar;
		delete pMsgStatus;
		delete pMsgprops;
		delete lpszAdminLabelBuffer;
		delete lpbMsgID;
		DisplayReadError(hr);
	}
}

void Listing15_6()
{
	HRESULT hr;
	QUEUEHANDLE hq;

	TCHAR wszFormatName[256];
	DWORD dwFormatNameLength = 256;
	
	hr = MQPathNameToFormatName(_T("nickdell\\Private$\\WinCEQueue"),
                              wszFormatName,
                              &dwFormatNameLength);
	hr = MQOpenQueue(wszFormatName, MQ_SEND_ACCESS, MQ_DENY_NONE, &hq);
	if(hr == MQ_OK)
		cout << _T("Opened queue") << endl;
	else
	{
		DisplayOpenError(hr);
		return;
	}

	DWORD cPropId = 0; 
  
	MQMSGPROPS msgprops;
	MSGPROPID aMsgPropId[7];     
	MQPROPVARIANT aMsgPropVar[7];
	HRESULT aMsgStatus[7];
  
	aMsgPropId[cPropId] = PROPID_M_LABEL;
	aMsgPropVar[cPropId].vt = VT_LPWSTR; 
	aMsgPropVar[cPropId].pwszVal = _T("Test Acknowledge Message");
	cPropId++;
  
	aMsgPropId[cPropId] = PROPID_M_BODY_TYPE;
	aMsgPropVar[cPropId].vt = VT_UI4;
	aMsgPropVar[cPropId].ulVal = VT_BSTR;	
	cPropId++;

	BSTR bStr = SysAllocString(_T("Body text for the message"));

  	aMsgPropId[cPropId] = PROPID_M_BODY;
	aMsgPropVar[cPropId].vt = VT_VECTOR|VT_UI1;
	aMsgPropVar[cPropId].caub.pElems = (LPBYTE)bStr;
	aMsgPropVar[cPropId].caub.cElems = SysStringByteLen(bStr);
	cPropId++;

	TCHAR wszAdminFormatName[1024];
	DWORD dwAdminFormatNameLength = 1024;
	hr = MQPathNameToFormatName(_T(".\\Private$\\WinCEInQueueAdmin"),
                              wszAdminFormatName,
                              &dwAdminFormatNameLength);
	if (FAILED(hr))
	{
     cout << _T("Failed in MQPathNameToFormatName for administration queue");
     return;
	}
  
	aMsgPropId[cPropId] = PROPID_M_ADMIN_QUEUE;
	aMsgPropVar[cPropId].vt = VT_LPWSTR;
	aMsgPropVar[cPropId].pwszVal = wszAdminFormatName;  
	cPropId++;
  
	aMsgPropId[cPropId] = PROPID_M_ACKNOWLEDGE;
	aMsgPropVar[cPropId].vt = VT_UI1;
	aMsgPropVar[cPropId].bVal = MQMSG_ACKNOWLEDGMENT_NACK_RECEIVE;
	cPropId++;
  
	aMsgPropId[cPropId] = PROPID_M_TIME_TO_BE_RECEIVED;     
	aMsgPropVar[cPropId].vt = VT_UI4; 
	aMsgPropVar[cPropId].ulVal = 30; // seconds
	cPropId++;

	BYTE bMsgID[20];
	memset(bMsgID, 0, 20);
	aMsgPropId[cPropId] = PROPID_M_MSGID;     
	aMsgPropVar[cPropId].vt = VT_VECTOR | VT_UI1; 
	aMsgPropVar[cPropId].caub.pElems = bMsgID;
	aMsgPropVar[cPropId].caub.cElems = 20;
	cPropId++;

	msgprops.cProp = cPropId;                           // Number of properties
	msgprops.aPropID = aMsgPropId;                      // Ids of properties
	msgprops.aPropVar = aMsgPropVar;                    // Values of properties
	msgprops.aStatus = aMsgStatus;                      // Error reports
   
	
	hr = MQSendMessage(hq,								// Handle to open queue
                     &msgprops,							// Properties of message
                     NULL);								// No transaction
	if (FAILED(hr))
		cout << _T("Could not send message") << endl;

	else
	{
		DisplayMsgId(bMsgID);
		cout << endl << _T("Message queued") << endl;
	}
	MQCloseQueue(hq);
	InitializeAdminQueueRead();
}

// *** Listing 15.7
//
// Transacted Queues

void CreateTransactedQueue()
{
	DWORD cPropId = 0;                                  // Define the property counter.
  
	//Define the MQQUEUPROPS structure.
	MQQUEUEPROPS QueueProps;
	MQPROPVARIANT aQueuePropVar[3];
	QUEUEPROPID aQueuePropId[3];
	HRESULT aQueueStatus[3];
  
	HRESULT hr;                                         // Define results.
	PSECURITY_DESCRIPTOR pSecurityDescriptor=NULL;      // Security descriptor
  
	// Queue pathname
	LPWSTR wszPathName = _T(".\\PRIVATE$\\WinCETransQueue");
  
	// Queue label
	LPWSTR wszQueueLabel = _T("Transacted Queue");
  
	// Format name buffer for queue
	DWORD dwFormatNameLength = 256;               // Length of format name buffer
	WCHAR wszFormatName[256];                     // Format name buffer
  
	aQueuePropId[cPropId] = PROPID_Q_PATHNAME; 
	aQueuePropVar[cPropId].vt = VT_LPWSTR;   
	aQueuePropVar[cPropId].pwszVal = wszPathName;     
	cPropId++;
  
	aQueuePropId[cPropId] = PROPID_Q_LABEL;      
	aQueuePropVar[cPropId].vt = VT_LPWSTR;           
	aQueuePropVar[cPropId].pwszVal = wszQueueLabel;    
	cPropId++;
  
	aQueuePropId[cPropId] = PROPID_Q_TRANSACTION; 
	aQueuePropVar[cPropId].vt = VT_UI1 ;
	aQueuePropVar[cPropId].bVal = MQ_TRANSACTIONAL ;
	cPropId++;

  	QueueProps.cProp = cPropId;                         // Number of properties
	QueueProps.aPropID = aQueuePropId;                  // Ids of properties
	QueueProps.aPropVar = aQueuePropVar;                // Values of properties
	QueueProps.aStatus = aQueueStatus;                  // Error reports
  
	hr = MQCreateQueue(pSecurityDescriptor,             // Security descriptor
						 &QueueProps,                     // Queue properties
						 wszFormatName,             // OUT: Format name of queue
						 &dwFormatNameLength);      // OUT: Format name length

	if(hr == MQ_OK)
		cout << _T("Transacted queue created") << endl;
	else if(hr == MQ_OK)
		cout << wszFormatName << _T(" created") << endl;
	else if(hr == MQ_ERROR_ACCESS_DENIED )
		cout << _T("Access Denied") << endl;
	else if(hr == MQ_ERROR_ILLEGAL_PROPERTY_VALUE )
		cout << _T("Illegal Property Value") << endl;
	else if(hr == MQ_ERROR_ILLEGAL_QUEUE_PATHNAME )
		cout << _T("Illegal pathname") << endl;
	else if(hr == MQ_ERROR_ILLEGAL_SECURITY_DESCRIPTOR )
		cout << _T("Illegal security descriptor") << endl;
	else if(hr == MQ_ERROR_INSUFFICIENT_PROPERTIES )
		cout << _T("Path name not specified") << endl;
	else if(hr == MQ_ERROR_INVALID_OWNER )
		cout << _T("Invalid owner") << endl;
	else if(hr == MQ_ERROR_PROPERTY )
		cout << _T("Error in property specification") << endl;
	else if(hr == MQ_ERROR_PROPERTY_NOTALLOWED )
		cout << _T("Property not allowed when creating queue") << endl;
	else if(hr == MQ_ERROR_QUEUE_EXISTS )
		cout << _T("Queue already exists") << endl;
	else if(hr == MQ_ERROR_SERVICE_NOT_AVAILABLE )
		cout << _T("Service not available") << endl;
	else if(hr == MQ_INFORMATION_FORMATNAME_BUFFER_TOO_SMALL )
		cout << _T("Format name buffer too small") << endl;
	else if(hr == MQ_INFORMATION_PROPERTY )
		cout << _T("Succeeded, but property returned warning") << endl;
}

void Listing15_7()
{
	HRESULT hr;
	QUEUEHANDLE hq;

	TCHAR wszFormatName[256];
	DWORD dwFormatNameLength = 256;
	
	hr = MQPathNameToFormatName(_T(".\\PRIVATE$\\WinCETransQueue"),
                              wszFormatName,
                              &dwFormatNameLength);
	cout << wszFormatName << endl;
	MQDeleteQueue(wszFormatName);
	CreateTransactedQueue();
	hr = MQOpenQueue(wszFormatName,
			MQ_SEND_ACCESS,
			MQ_DENY_NONE,
			&hq);
	if(hr == MQ_OK)
		cout << _T("Opened transacted queue") << endl;
	else
	{
		DisplayOpenError(hr);
		return;
	}

	DWORD cPropId = 0; 
  
	MQMSGPROPS msgprops;
	MSGPROPID aMsgPropId[4];     
	MQPROPVARIANT aMsgPropVar[4];
	HRESULT aMsgStatus[4];
  
	aMsgPropId[cPropId] = PROPID_M_LABEL;
	aMsgPropVar[cPropId].vt = VT_LPWSTR; 
	aMsgPropVar[cPropId].pwszVal = _T("Test Transacted Message");
	cPropId++;
  
	aMsgPropId[cPropId] = PROPID_M_BODY_TYPE;
	aMsgPropVar[cPropId].vt = VT_UI4;
	aMsgPropVar[cPropId].ulVal = VT_BSTR;	
	cPropId++;

	BSTR bStr = SysAllocString(_T("Body text for the tranasacted message"));

  	aMsgPropId[cPropId] = PROPID_M_BODY;
	aMsgPropVar[cPropId].vt = VT_VECTOR|VT_UI1;
	aMsgPropVar[cPropId].caub.pElems = (LPBYTE)bStr;
	aMsgPropVar[cPropId].caub.cElems = SysStringByteLen(bStr);
	cPropId++;

	msgprops.cProp = cPropId;                           // Number of properties
	msgprops.aPropID = aMsgPropId;                      // Ids of properties
	msgprops.aPropVar = aMsgPropVar;                    // Values of properties
	msgprops.aStatus = aMsgStatus;                      // Error reports
   
	
	hr = MQSendMessage(hq,								// Handle to open queue
                     &msgprops,							// Properties of message
                     MQ_SINGLE_MESSAGE);				// Single message transaction
	if (FAILED(hr))
	{
		cout << _T("Could not send transacted message") << endl;
		DisplayReadError(hr);
	}
	else
		cout << _T("Transacted Message queued") << endl;
	MQCloseQueue(hq);
}

// *** Listing 15.8
//
// Receive transacted message 

void Listing15_8()
{
	HRESULT hr;
	QUEUEHANDLE hq;

	TCHAR wszFormatName[256];
	DWORD dwFormatNameLength = 256;
	
	hr = MQPathNameToFormatName(_T(".\\PRIVATE$\\WinCETransQueue"),
                              wszFormatName,
                              &dwFormatNameLength);
	cout << wszFormatName << endl;

	hr = MQOpenQueue(wszFormatName,
			MQ_RECEIVE_ACCESS,
			MQ_DENY_NONE,
			&hq);
	if(hr == MQ_OK)
		cout << _T("Opened queue") << endl;
	else
	{
		DisplayOpenError(hr);
		return;
	}

	DWORD dwRecAction = MQ_ACTION_RECEIVE;            // Receive action
  
	MQMSGPROPS msgprops;
	MSGPROPID aMsgPropId[4];
	MQPROPVARIANT aMsgPropVar[4];
	HRESULT aMsgStatus[4];
	// Message body buffer
	DWORD dwBodyBufferSize = 1024;
	LPTSTR lpszBodyBuffer = new TCHAR[dwBodyBufferSize];
	DWORD cPropId = 0;
	
	aMsgPropId[cPropId] = PROPID_M_BODY_SIZE;       // Property ID
	aMsgPropVar[cPropId].vt = VT_UI4;               // Type indicator
	cPropId++;
  
	aMsgPropId[cPropId] = PROPID_M_BODY;            // Property ID
	aMsgPropVar[cPropId].vt = VT_VECTOR|VT_UI1;     // Type indicator
	aMsgPropVar[cPropId].caub.pElems = (UCHAR*)lpszBodyBuffer;
	aMsgPropVar[cPropId].caub.cElems = dwBodyBufferSize;
  	cPropId++;
	// Reading additional properties....
	aMsgPropId[cPropId] = PROPID_M_SENTTIME;
	aMsgPropVar[cPropId].vt = VT_UI4;               
	cPropId++;
	aMsgPropId[cPropId] = PROPID_M_ARRIVEDTIME;
	aMsgPropVar[cPropId].vt = VT_UI4;               
	cPropId++;

  	msgprops.cProp = cPropId;          // Number of message properties
	msgprops.aPropID = aMsgPropId;     // Ids of message properties
	msgprops.aPropVar = aMsgPropVar;   // Values of message properties
	msgprops.aStatus  = aMsgStatus;    // Error reports
  
  	hr = MQReceiveMessage(hq,             // Queue handle
                        0,                  // Max time (msec)
                        dwRecAction,        // Receive action
                        &msgprops,          // Msg property structure
                        NULL,               // No OVERLAPPED structure
                        NULL,               // No callback function
                        NULL,               // No cursor
                        NULL                // No transaction
                        );
  
	if (FAILED(hr))
	{
		DisplayReadError(hr);
		MQCloseQueue(hq);
		delete lpszBodyBuffer;
		return;
	}
  
	if (aMsgPropVar[0].ulVal == 0)
		cout <<  _T("No message body exists.") << endl;
	else
	{
		cout << _T("The message body size is: ")  << lpszBodyBuffer << endl;
		cout << _T("The message body is: ")  << lpszBodyBuffer << endl;
		// display sent time and arrived time
		SYSTEMTIME st;

		TimeToSystemTime(aMsgPropVar[2].ulVal, &st);
		cout << _T("Sent Time: ") << st.wMonth << _T("/") << st.wDay
			<< _T("/") << st.wYear << _T(" ") << st.wHour 
			<< _T(":") << st.wMinute << _T(":") << st.wSecond << endl;

		TimeToSystemTime(aMsgPropVar[3].ulVal, &st);
		cout << _T("Receive Time: ") << st.wMonth << _T("/") << st.wDay
			<< _T("/") << st.wYear << _T(" ") << st.wHour 
			<< _T(":") << st.wMinute << _T(":") << st.wSecond << endl;

	}
	delete lpszBodyBuffer;
  
	MQCloseQueue(hq);
}

#else	// USE_MMQ

void Listing15_1()
{
	MessageBox(NULL, _T("Code not included in compilation. See Chapter15.cpp"), 
			_T("MMQ Samples"), MB_OK);
}

void Listing15_2()
{
	MessageBox(NULL, _T("Code not included in compilation. See Chapter15.cpp"), 
			_T("MMQ Samples"), MB_OK);
}

void Listing15_3()
{
	MessageBox(NULL, _T("Code not included in compilation. See Chapter15.cpp"), 
			_T("MMQ Samples"), MB_OK);
}

void Listing15_4()
{
	MessageBox(NULL, _T("Code not included in compilation. See Chapter15.cpp"), 
			_T("MMQ Samples"), MB_OK);
}

void Listing15_5()
{
	MessageBox(NULL, _T("Code not included in compilation. See Chapter15.cpp"), 
			_T("MMQ Samples"), MB_OK);
}

void Listing15_6()
{
	MessageBox(NULL, _T("Code not included in compilation. See Chapter15.cpp"), 
			_T("MMQ Samples"), MB_OK);
}

void Listing15_7()
{
	MessageBox(NULL, _T("Code not included in compilation. See Chapter15.cpp"), 
			_T("MMQ Samples"), MB_OK);
}

void Listing15_8()
{
	MessageBox(NULL, _T("Code not included in compilation. See Chapter15.cpp"), 
			_T("MMQ Samples"), MB_OK);
}


#endif