// 'Windows CE 3.0 Programming' Source Code Samples (Prentice Hall, 2000)
// Source Code Author: Nick Grattan (nick@softwarepaths.com)
// Version 1.00

// Chapter 11: TAPI and RAS

#include "stdafx.h"
#include "examples.h"

// *** Listing 11.1
//
// Initialize TAPI and determine number of available line devices

#include <tapi.h>

HLINEAPP g_hLineApp;

// TAPI call back function
VOID FAR PASCAL lineCallbackFunc(DWORD hDevice, DWORD dwMsg, DWORD dwCallbackInstance, 
		DWORD dwParam1, DWORD dwParam2, DWORD dwParam3 )
{
	if(dwMsg != LINE_CALLSTATE)
		return;		// only interested in LINE_CALLSTATE messages
	cout << _T("LINE_CALLSTATE: ");
	// dwParam1 is the specific LINE_CALLSTATE change occurring
	switch (dwParam1) 
		{
		case LINECALLSTATE_IDLE:
			cout << _T("Idle");
			break;

		case LINECALLSTATE_DIALTONE:
			cout << _T("Dial tone");
			break;

		case LINECALLSTATE_DIALING:
			cout << _T("Dialing");
			break;

		case LINECALLSTATE_PROCEEDING:
			cout << _T("Dialing has completed");
			break;

		case LINECALLSTATE_RINGBACK:
			cout << _T("Ring back");
			break;

		case LINECALLSTATE_CONNECTED:
			cout << _T("Connected");
			break;

		case LINECALLSTATE_BUSY:
			cout << _T("Busy");
			break;

		case LINECALLSTATE_DISCONNECTED:
			switch (dwParam2)
			{
				case LINEDISCONNECTMODE_NORMAL:
					cout << _T("Disconnected: Remote party disconnected");
					break;

				case LINEDISCONNECTMODE_UNKNOWN:
					cout << _T("Disconnected: Unknown reason");
					break;

				case LINEDISCONNECTMODE_REJECT:
					cout << _T("Disconnected: Remote Party rejected call");
					break;

				case LINEDISCONNECTMODE_BUSY:
					cout << _T("Disconnected: Remote busy");
					break;

				default:
					cout << _T("Disconnected: Other reason") << dwParam2;
					break;
				Listing11_4();	// close call and line
			}
			break;
		default:
			cout << _T("Other notification") << dwParam1;
	}
	cout << endl;
}

// initializes TAPI and returns available number of line devices
DWORD InitializeTAPI()
{
	DWORD dwReturn, dwNumLines;
	dwReturn = lineInitialize (&g_hLineApp, 
			hInst, 
			(LINECALLBACK) lineCallbackFunc, 
			_T("Examples Application"), 
			&dwNumLines);
	if(dwReturn == LINEERR_REINIT)
		cout << _T("Cannot initialize TAPI at present.")
			 << _T("Try again later") << endl;
	else if (dwReturn != 0)
		cout << _T("Error initializing TAPI: ") << dwReturn << endl;
	return dwNumLines;
}

void ShutdownTAPI()
{
	if(g_hLineApp != NULL)
	{
		lineShutdown(g_hLineApp);
		g_hLineApp = NULL;
	}
}
void Listing11_1()
{
	DWORD dwNumLines;

	dwNumLines = InitializeTAPI();
	if(dwNumLines > 0)
		cout << _T("Number of available line devices: ")
			 << dwNumLines << endl;
	else
		cout << _T("TAPI Error or no line devices present.") << endl;
	ShutdownTAPI();
}

// *** Listing 11.2
//
// List all TAPI line devices 

#define TAPI_VERSION_1_0      0x00010003
#define TAPI_VERSION_3_0      0x00030000
#define TAPI_CURRENT_VERSION  TAPI_VERSION_3_0

DWORD NegotiateTAPIVersion(DWORD dwLineId)
{
	DWORD dwReturn, dwRAPIVersion;

	if (dwReturn = lineNegotiateAPIVersion (
			g_hLineApp,                 // TAPI registration handle
			dwLineId,                   // Line device to be queried
			TAPI_VERSION_1_0,           // Least recent API version 
			TAPI_CURRENT_VERSION,       // Most recent API version 
			&dwRAPIVersion,				// Negotiated API version 
			NULL))                      // Must be NULL
	{
		cout << _T("Could not negotiate TAPI version with device") 
				<< dwLineId << endl;
		return 0;
	}
	return dwRAPIVersion;
}

void DisplayLineInfo(DWORD dwLineId)
{
	DWORD dwRAPIVersion, dwSize, dwReturn;
	LPLINEDEVCAPS lpLineDevCaps = NULL;
	LPTSTR lpszString;

	// first negotiate TAPI version
	dwRAPIVersion = NegotiateTAPIVersion(dwLineId);
	if(dwRAPIVersion == 0)
	{
		cout << _T("Could not negotiate TAPI version with device") 
				<< dwLineId << endl;
		return;
	}

	dwSize = sizeof (LINEDEVCAPS);
	// Allocate enough memory for lpLineDevCaps.
	do
	{
		if (!(lpLineDevCaps = (LPLINEDEVCAPS) 
						LocalAlloc (LPTR, dwSize)))
		{
			cout << _T("Out of memory") << endl;
			return;
		}

		lpLineDevCaps->dwTotalSize = dwSize;

		if (dwReturn = lineGetDevCaps (g_hLineApp,
					dwLineId,
					dwRAPIVersion,
					0,
					lpLineDevCaps))
		{
			cout << _T("Could not get Dev Caps") << endl;
			return;
		}

		// Stop if the allocated memory is equal to or greater than the 
		// needed memory.
		if (lpLineDevCaps->dwNeededSize <= 
						lpLineDevCaps->dwTotalSize)
			break;  

		dwSize = lpLineDevCaps->dwNeededSize;
		LocalFree (lpLineDevCaps);
		lpLineDevCaps = NULL;
    
	} while (TRUE);
	lpszString = (LPTSTR)((LPBYTE)lpLineDevCaps +
				lpLineDevCaps->dwLineNameOffset);
	// now display information
	cout << _T("Device: ") << dwLineId << _T(" ") 
			<< lpszString << endl;
	LocalFree (lpLineDevCaps);
}

void Listing11_2()
{
	DWORD dwNumLines, dw;

	if(!(dwNumLines = InitializeTAPI()))
		return;
	for(dw = 0; dw < dwNumLines; dw++)
		DisplayLineInfo(dw);

	ShutdownTAPI();
}

// *** Listing 11.3
//
// Making a call

HLINE g_hLine = NULL;
HCALL g_hCall = NULL;

void MakeCall(DWORD dwLineId, LPTSTR szPhoneNumber)
{
	DWORD dwTAPIVersion, dwReturn;
	LPLINETRANSLATEOUTPUT lpTransOutput = NULL;
    DWORD dwSizeOfTransOut = sizeof (LINETRANSLATEOUTPUT);
	TCHAR szDialablePhoneNum[TAPIMAXDESTADDRESSSIZE + 1];

	cout << _T("Dialing: ") << szPhoneNumber << endl;
	dwTAPIVersion = NegotiateTAPIVersion(dwLineId);
	if(dwTAPIVersion == 0)
		return;
	
	if (dwReturn = lineOpen(
		g_hLineApp,			// Usage handle for TAPI
		dwLineId,			// Cannot use the LINEMAPPER value
		&g_hLine,			// Line handle
		dwTAPIVersion,		// API version number
		0,					// Must set to zero for Windows CE
		0,                  // No data passed back 
		LINECALLPRIVILEGE_NONE,     // Can only make an outgoing call
		0,                  // Media mode 
		NULL))              // Must set to NULL for Windows CE
	{
		cout << _T("Could not open line: ") << dwReturn;
		return;
	}

	// Call translate address before dialing.
	do
	{
		// Allocate memory for lpTransOutput.
		if (!(lpTransOutput = (LPLINETRANSLATEOUTPUT) 
					LocalAlloc(LPTR, dwSizeOfTransOut)))
			return;

		lpTransOutput->dwTotalSize = dwSizeOfTransOut;

		if (dwReturn = lineTranslateAddress (
				g_hLineApp,         // Usage handle for TAPI
				dwLineId,			// Line device identifier 
				dwTAPIVersion,		// Highest TAPI version supported 
				szPhoneNumber,      // Address to be translated
				0,                  // Must be 0 for Windows CE
				0,                  // No associated operations 
				lpTransOutput))     // Result of the address translation
		{
			LocalFree(lpTransOutput);                                    
			return;
		}
    
		if (lpTransOutput->dwNeededSize <= lpTransOutput->dwTotalSize)
			break; 
		else
		{
			dwSizeOfTransOut = lpTransOutput->dwNeededSize;
			LocalFree (lpTransOutput);
			lpTransOutput = NULL;
		}

	} while (TRUE);
     
	// Save the translated phone number for dialing.
	wcscpy(szDialablePhoneNum, 
		(LPTSTR) ((LPBYTE) lpTransOutput + lpTransOutput->dwDialableStringOffset));

	cout << _T("Translated Number: ") << szDialablePhoneNum << endl;
	// Make the phone call. 
	/////////////////
	dwReturn = lineMakeCall(
		g_hLine,			// handle to open line
		&g_hCall,			// return handle to call
		szDialablePhoneNum,	// phone number to dial
		0,					// default country code
		NULL);		// call parameters
	if(dwReturn < 0)
		cout << _T("Could not make call") << dwReturn << endl;
	else if(dwReturn >= 0)
		cout << _T("Dialing asynchronously") << endl;
}

void Listing11_3()
{
	DWORD dwNumLines;

	if(!(dwNumLines = InitializeTAPI()))
		return;
	MakeCall(6, _T("+353 (1) 016624777"));
}

// *** Listing 11.4
//
// Shut down a call

void Listing11_4()
{
	lineDrop(g_hCall,	// call to drop
		NULL,			// no data to be sent on drop
		0);				// length of data to be sent
	lineDeallocateCall(g_hCall);
	g_hCall = NULL;
	lineClose(g_hLine);
	g_hLine = NULL;
	ShutdownTAPI();
}

// *** Listing 11.5
//
// Get Serial Communications handle for open connection, send
// and receive data.

HANDLE GetCommPort()
{
	DWORD dwSize = sizeof(VARSTRING) + 1024;
	DWORD dwReturn;

	LPVARSTRING lpVarString = (LPVARSTRING)LocalAlloc(LPTR, dwSize);
	if(lpVarString == NULL)
		return NULL;
	lpVarString->dwTotalSize = dwSize;

	dwReturn = lineGetID(g_hLine,	// handle to open line
		0,							// address ID ignored
		NULL,						// call handle ignored
		LINECALLSELECT_LINE,		// we're only passing a line handle
		lpVarString,	
		_T("comm/datamodem"));
	if(dwReturn != 0)
	{
		cout << _T("Could not get line ID") << endl;
		return NULL;
	}
	LPHANDLE lpHandle = (HANDLE*)((LPBYTE)
					lpVarString + lpVarString->dwStringOffset);
	HANDLE hComm = *lpHandle;
	cout << _T("Port handle: ") << (DWORD)hComm << endl;
	cout << _T("Communications port: ")
			<< (LPTSTR)((LPBYTE)lpVarString + lpVarString->dwStringOffset + sizeof(HANDLE))
			<< endl;
	return hComm;
}

BOOL SendAndReceive(HANDLE hComm, LPTSTR lpszSend, LPTSTR lpszReceive)
{
	DWORD dwBytesWritten, dwBytesRead;
	DWORD dwBytesToWrite;
	char szmbsSend[1024], szmbsReceive[1024];

	dwBytesToWrite = wcstombs(szmbsSend, lpszSend, 1024);
	if(!WriteFile(hComm, szmbsSend, dwBytesToWrite,
			&dwBytesWritten, NULL))
	{
		cout << _T("Could not write file: ") << GetLastError() << endl;
		return FALSE;
	}
	if(!ReadFile(hComm, szmbsReceive, 1024, &dwBytesRead, NULL))
	{
		cout << _T("Could not read file: ") << GetLastError() << endl;
		return FALSE;
	}
	lpszReceive[dwBytesRead] = '\0';
	mbstowcs(lpszReceive, szmbsReceive, 1024);
	cout << _T("Bytes Read: ") << dwBytesRead << endl;
	cout << lpszReceive << endl;
	return TRUE;
}

void Listing11_5()
{
	HANDLE hComm;
	TCHAR szReceive[1024];

	if(g_hLine == NULL)
	{
		cout << _T("No open line") << endl;
		return;
	}
	hComm = GetCommPort();
	SendAndReceive(hComm, _T("\n"), szReceive);
}

// *** Listing 11.6
//
// List RAS Phone book entries

#include <ras.h>
#include <raserror.h>

void Listing11_6()
{
	LPRASENTRYNAME lpRasEntry = NULL;
	DWORD dwRes, dwSize, dwEntries, dw;

	lpRasEntry = new RASENTRYNAME[20];
	if(lpRasEntry == NULL)
	{
		cout << _T("Out of memory") << endl;
		return;
	}
	lpRasEntry[0].dwSize = sizeof(RASENTRYNAME);
	dwSize = sizeof(RASENTRYNAME) * 20;
	dwRes = RasEnumEntries(NULL, NULL, lpRasEntry, 
					&dwSize, &dwEntries);
	if (dwRes != 0)
		cout << _T("Error getting RAS entries") << dwRes << endl;
	else
	{
		for(dw = 0; dw < dwEntries; dw++)
		{
			cout << lpRasEntry[dw].szEntryName << endl;
		}
	}
	delete[] lpRasEntry;
}

// *** Listing 11.7
//
// Make a connection using a RAS phone book entry

HRASCONN g_hRasConn = NULL;

// NB: Assumes that a RAS connection (such as ActiveSync) is not
// already open. If this is the case, RasDial returns an error 602.

void Listing11_7(HWND hWnd)
{
	RASDIALPARAMS rasDialParams;
	DWORD dwRes;
	BOOL bPassword;

	rasDialParams.dwSize = sizeof(RASDIALPARAMS);
	// Change "SPL" to your RAS entry name
	wcscpy(rasDialParams.szEntryName, _T("SPL"));
	dwRes = RasGetEntryDialParams(NULL, &rasDialParams, &bPassword);
	if(dwRes != 0)
	{
		cout << _T("Error getting Dial Params:") << dwRes << endl;
		return;
	}
	if(!bPassword)
		cout << _T("Password not returned") << endl;
	dwRes = RasDial(NULL, NULL, &rasDialParams, 
			0xFFFFFFFF, hWnd, &g_hRasConn);
	if(dwRes != 0)
		cout << _T("Error dialing RAS: ") << dwRes << endl;
}

// This function is called from the message processing function
// for the windows with the hWnd handle passed to RasDial.
// See code in Examples.cpp relating to the WM_RASDIALEVENT message

void RasDialEvent(HWND hWnd, WPARAM wParam, LPARAM lParam)
{
	if(wParam == RASCS_OpenPort)
		cout << _T("Opening Port") << endl;
	else if(wParam == RASCS_PortOpened)
		cout << _T("Port Opened") << endl;
	else if(wParam == RASCS_ConnectDevice)
		cout << _T("Connecting to device") << endl;
	else if(wParam == RASCS_DeviceConnected)
		cout << _T("Connected") << endl;
	else if(wParam == RASCS_Authenticated)
		cout << _T("Authenticated") << endl;
	else if(wParam == RASCS_DeviceConnected)
		cout << _T("Connected") << endl;
	else if(wParam == RASCS_AllDevicesConnected)
		cout << _T("All devices connected") << endl;
	else if(wParam == RASCS_Authenticate)
		cout << _T("Waiting for authentication") << endl;
	else if(wParam == RASCS_AuthAck)
		cout << _T("Authentication acknowledged") << endl;
	else if(wParam == RASCS_Disconnected)
		cout << _T("Disconnected") << endl;
}

// *** Listing 11.8
//
// Disconnect a RAS connection

// NB: Call is not disconnected automatically when application
// terminates.

void Listing11_8()
{
	if(g_hRasConn != NULL)
	{
		RasHangUp(g_hRasConn);
		g_hRasConn = NULL;
	}
	else
		cout << _T("Not connected") << endl;
}


// *** Listing 11.9
//
// Test if there's already a RAS Connection

void Listing11_9()
{
	RASCONN rsconn[10];
	DWORD dwcb, dwConnections;
	RASCONNSTATUS rasStatus;

	dwcb = sizeof(rsconn);
	rsconn[0].dwSize = sizeof(RASCONN);
	if(RasEnumConnections(rsconn, &dwcb, &dwConnections) == 0)
	{
		if(dwConnections == 0 || rsconn[0].hrasconn == NULL)
		{
			cout << _T("No current connections") << endl;
			return;
		}
		// Find the current status of the RAS connection
		// Note there will only ever be one connection
		rasStatus.dwSize = sizeof(rasStatus);
		if(RasGetConnectStatus(rsconn[0].hrasconn, &rasStatus) != 0)
		{
			cout << _T("Could not get RAS connection status")  << endl;
			return;
		}
		if(rasStatus.rasconnstate != RASCS_Connected)
		{
			cout << _T("Not connected") << endl;
			return;
		}
		// has connection state changed since last call?
		cout << _T("Current connection to: ")
				<< rsconn[0].szEntryName;
	}
	else
		cout << _T("Could not enumerate RAS connections") << endl;
}