/****************************************************************************
 File: xlate.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved

 $Revision:   1.13  $
   $Author:   gmills  $
     $Date:   24 Feb 1992 16:55:56  $

 This file contains part of the API definition for clsXtract.  For
 the remainder see xtract.h.

 clsXtract inherits from clsObject.
 Implements basic translation functions for converting pen input, in the
 form of strokes, to gestures or text characters.
 
 Translators are objects that use pattern recognition techniques to convert
 pen input to gestures or text characters.  There are three stages to the
 translation process: initialization, control (stroke collection and
 recognition), and notification (data output).

 Since the translation object may preprocess input data as it is received,
 initialization messages should be sent before any strokes are added to the
 object.  Initialization messages establish the rules for translation.

 Control messages are used by the client to communicate specific information
 regarding the state of the translation as it pertains to the input stroke
 stream.

 Notification messages are used by the translation object to notify the
 client as to the current state of the translation process.
 
 For historical reasons messages and data types relating to translation
 are defined in terms of two class names:  clsXlate and clsXtract.
 Conceptually, clsXlate is an abstract class (a class with no default
 behavior, i.e. no methods) and clsXtract is a subclass of clsXlate which
 implements methods for a large number of messages.  As implemented,
 however, there is no such class as clsXlate in PenPoint 1.0.  When
 PenPoint boots, clsXlate is not installed in the class hierarchy,
 and clsXtract is installed as a subclass of clsObject.

 The clsXtract/clsXlate does not implement enough behavior to be used
 directly as a translator.  Rather translation objects should be created
 as instances of one of the following subclasses:
	clsXGesture		for gestures
	clsXText		for letters with minimal language support
	clsXWord		for letters as part of normal American English
	clsXTeach		for letters when the application is to train the
					recognition engine. (It is not possible to train gestures)

 See Also
	xtract.h, xgesture.h, xtext.h, xword.h, xteach.h

****************************************************************************/

#ifndef XLATE_INCLUDED
#define XLATE_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif
#ifndef GEO_INCLUDED
#include <geo.h>
#endif
#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef XLIST_INCLUDED
#include <xlist.h>
#endif
#ifndef SPELL_INCLUDED
#include <spell.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Internal Constants ****/

/*
 The following are used globally by the translation object.
*/
#define xltCharWordTerminator ('\0')  // standard string terminator
#define xltCharSpace           (' ')  // character code for space
#define xltCharDotlessI       (0x80)  // character code for dotless i (private)
#define xltCharDotlessJ       (0x81)  // character code for dotless j (private)
#define xltCharUnknownDefault (0x15)  // default "meatball" for unrecognized char

#define xltMaxWordLength		(32)	// buffer size for word translations

typedef struct POINT {
   S16 x, y;
} POINT, * P_POINT;	// internal representation of a digitizer point

/****  Status Values  ****/
/*
 The translation object may return the following status values.
*/
#define stsXlateBufferOverflow	MakeStatus(clsXlate, 1)
#define stsXlateBadProtoFile	MakeStatus(clsXlate, 2)
#define stsXlateBadTransFile	MakeStatus(clsXlate, 3)
#define stsXlateBadTrigramFile	MakeStatus(clsXlate, 4)
#define stsXlateInputTruncated	MakeNonErr(clsXlate, 1)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Creation Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/*
 Characteristics of the insertion pad.
*/
typedef struct XLATE_METRICS {
	U16 lineCount;		// number of lines (0 = indeterminate)
	U16 charCount;		// number of character columns (0 = indeterminate)
	SIZE32 charBox;		// size of character box (height and width)
	S32 baselineOffset;	// baseline offset to bottom of char box (if charCount != 0)
	XY32 origin;		// origin of insertion pad in digitizer coordinates
} XLATE_METRICS, *P_XLATE_METRICS;

/*
 When "case smarts" are turned on (i.e. xltSmartCaseDisable hwx flag is OFF), 
 the translation object will ignore the case in which the user wrote
 the input and will instead figure out the correct capitalization
 based on the settings in XLATE_CASE_METRICS.
 XLATE_CASE_TYPE tells the type of capitalization rules which the
 translation string should be made to obey.  "No rules" means
 make everything lower case.
*/
typedef enum XLATE_CASE_TYPE {
	xcmNone,	// Don't capitalize anything, force it all to lower case
	xcmSentence,// Capitalize first letter of each sentence, etc
	xcmField	// Capitalize as per XLATE_CASE_METRICS.context.field
} XLATE_CASE_TYPE, * P_XLATE_CASE_TYPE;

/*
 If the writer is a mixed case writer, then he/she is presumed to write
 both upper case and lower case shapes.  An AllCapsWriter, on the other
 hand, will only write upper case shapes, never lower case shapes.
 This knowledge can help the shape recognizer by limiting the number
 of alternatives it has to choose from.
 This does not mean, however, that the translation will be all upper
 case, for it is the job of "case smarts" to convert the translation
 to the correct case.
*/
typedef	enum XLATE_CASE_WRITER {
	xcmMixedCaseWriter,		// Writer writes both upper and lower case shapes
	xcmAllCapsWriter,		// Writer writes in all upper case shapes
} XLATE_CASE_WRITER, * P_XLATE_CASE_WRITER;

typedef enum XLATE_CASE_FIELD {
	xcmOneInitialCapField,	// capitalize first letter in the field
	xcmAllInitialCapsField,	// capitalize first letter in each 'word'
	xcmAllCapsField,		// captialize all letters in the field
} XLATE_CASE_FIELD, * P_XLATE_CASE_FIELD;

typedef struct XLATE_CASE_METRICS {
	XLATE_CASE_TYPE		type;		// type of rule to use
	XLATE_CASE_WRITER	writer;		// type of input to expect
	union {
		SPELL_CASE_CONTEXT sentence;// specific rules if type is xcmSentence
		XLATE_CASE_FIELD field;		// specific rules if type is xcmField
	} context;
} XLATE_CASE_METRICS, * P_XLATE_CASE_METRICS;

typedef struct XLATE_NEW_ONLY {
	U32					hwxFlags;			// xlate rules (see msgXlateSetFlags)
	U16					charConstraints;	// constrained char set flags
	XLATE_METRICS		metrics;			// insertion pad parameters
	P_UNKNOWN			pTemplate;			// compiled XTemplate; pNull if none.
	XLATE_CASE_METRICS	xlateCaseMetrics;	// case post-processing controls.
} XLATE_NEW_ONLY, *P_XLATE_NEW_ONLY;

typedef struct XLATE_NEW {
	OBJECT_NEW_ONLY		object;
	XLATE_NEW_ONLY		xlate;
} XLATE_NEW, *P_XLATE_NEW;


/****************************************************************************
 msgNewDefaults: takes P_XLATE_NEW, returns STATUS
	category: class message
	Initializes the XLATE_NEW structure to default values.

 The default values are 0 for everything.

 This message should, of course, be sent to one of the subclasses of
 clsXtract, not to clsXlate, since clsXlate is a fiction, and not to
 clsXtract, since clsXtract does not implement the complete behavior
 needed to do translation.
*/


/****************************************************************************
 msgNew: takes P_XLATE_NEW, returns STATUS
	category: class message
	Creates a new translation object.

 This message should, of course, be sent to one of the subclasses of
 clsXtract, not to clsXlate, since clsXlate is a fiction, and not to
 clsXtract, since clsXtract does not implement the complete behavior
 needed to do translation.
*/


/****************************************************************************
 msgFree: takes P_NULL, returns STATUS
	Destroys a translation object.

 This message should be sent to the object you wish to destroy.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Initialization Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* 
 The following messages control various settings and modes which 
 govern the way translation is carried out.  These messages must
 all be received by the translator BEFORE any strokes are received
 by it, since translators are allowed to begin translating "in the
 background", (i.e. before the input is complete).
*/

/****************************************************************************
 msgXlateModeSet: takes XLATE_MODE, returns STATUS
	Sets the mode (i.e. character/code type) of a translation object.

 The translation object can be configured to processes a variety of
 character/code types.  The mode flag determines the type of character
 set and default behavior for the object.
*/
#define msgXlateModeSet					MakeMsg(clsXlate, 5)
 
typedef enum {
	xlCharacter,	// obsolete
	xlText,			// use default text rules (ASCII)
	xlNumber,		// obsolete
	xlGesture,		// use default gesture rules
	xlGeometric		// obsolete
} XLATE_MODE, *P_XLATE_MODE;


/****************************************************************************
 msgXlateModeGet: takes P_XLATE_MODE, returns STATUS
	Gets the mode of a translation object.

 The mode was set either at msgNew time or by msgXlateModeSet.
*/
#define msgXlateModeGet					MakeMsg(clsXlate, 10)


/****************************************************************************
 msgXlateMetricsSet: takes P_XLATE_METRICS, returns STATUS
	Tells translator the dimensions and layout of the writing area.

 In order to assist the writer and the recognition system, an insertion pad
 can display guidelines, or "character boxes", that direct the writer in
 targeting.  When character boxes are used, the XLATE_METRICS are used to
 communicate the physical box information to the translation object.  The
 translator can use this information (when available) to decide how to
 group the strokes into characters.

 Most internal processes key off the charCount field.  If charCount is 0,
 the translation object assumes that there are no boxes.  In that case it
 will default to a heuristic algorithm that combines information from the
 shape matching and context processing to estimate the writing baseline and
 character spacing.

 (As an aside, the translation object does not use baseline
 information when charCount is 0.  I.e. lineCount is ignored in that case.)

 If charCount > 0, the translation object uses lineCount and charCount to
 calculate the number of boxes in the insertion pad.  A combination of
 the charBox height and width and the x and y coordinates of the origin are
 used to define the physical bounds of each box.  The translation object
 then uses this to determine character segmentation.

*/
#define msgXlateMetricsSet				MakeMsg(clsXlate, 8)


/****************************************************************************
 msgXlateMetricsGet: takes P_XLATE_METRICS, returns STATUS
	Gets metrics of a translation object.

 The metrics were set in response to either msgNew or msgXlateMetricsSet.
*/
#define msgXlateMetricsGet				MakeMsg(clsXlate, 16)


/****************************************************************************
 msgXlateStringSet: takes P_XLATE_STRING, returns STATUS
	Sets the current textual context for a translation object.

 The following structure is used to communicate currently displayed text
 in the insertion pad.  It is only applicable when using boxed insertion
 pads.  The existing textual information must be registered if the
 translation object is using any string-based knowledge source (such as
 the dictionary or a template) where positional information within the
 string is crucial for proper recognition.

 It is possible to allow only a portion of the displayed string to
 be in the insertion pad (and hence, editable).  To allow for this,
 startIndex represents the first editable character's position in the
 string, and endIndex represents the last editable characters's position
 in the string.  If the entire string is editable, set startIndex = 0 and
 endIndex = string length.
 
*/
#define msgXlateStringSet				MakeMsg(clsXlate, 12)

typedef struct XLATE_STRING {
	P_CHAR pCurrentText;	// pointer to current text string
	U16 length;				// string length
	S16 startIndex;			// index of first editable character
	S16 endIndex;			// index of last editable character
} XLATE_STRING, *P_XLATE_STRING;


/****************************************************************************
 msgXlateSetFlags: takes U32, returns STATUS
	Sets the translation flags.

 The translation flags (hwxFlags) govern which of the various scoring rules
 will be applied in choosing the best translation.  They include built-in
 language rules, choice of assisting knowledge sources (speller, templates),
 and postprocessing rules, such as sentence-level case correction.

 Built-in Rules:
 The translation object can be directed to use various default language 
 rules to assist recognition.  When a flag is turned on, the translator
 will show a preference for translations which obey the rule associated
 with that flag.  For example if xltCaseEnable is on, the translator will 
 show a preference for words that are either all lower case, all upper case
 or all lower case except the first letter.

 Knowledge Source Controls:
 The translation object can be directed to use spelling and/or
 template information in order to assist recognition.  Each of these
 knowledge sources, when it is turned on, has a choice of four modes
 of operation:
	Enable, Enable+Veto, Enable+Propose and Enable+Veto+Propose.
 The Enable flag must be ON in all four cases.  This enables the use
 of the knowledge source and causes the translator to show a
 preference for words which conform to the source (i.e. are in the
 dictionary or match the template).  If the Veto flag is also on, then
 the translator will ONLY consider translations which conform to the
 source and will reject all translations which do not.  If the Propose
 flag is also on, it allows the translator to change some letters if
 it will result in a translation which conforms to the knowledge
 source even if the raw shape matcher did not suggest those letters.
 
 Post-processing Rules:
 The translation object can apply post-processing rules to assist
 error-checking and proofing (spell correction).  The only processing
 that is currently implemented is the "smart case" capability.
 This capability calls for the translator to use linguistic rules to
 correct the capitalization of the translation.  This correction is
 always applied unless it is disabled by turning the smartCaseDisable
 flag on.  
*/
#define msgXlateSetFlags				MakeMsg(clsXlate, 14)

// Built-in Rules
#define xltSegmentVeto			flag0	// allow one and only one char per box
#define xltCaseEnable			flag8	// prefer standard rules of capitalization
#define xltAlphaNumericEnable	flag9	// prefer standard grouping of letters and digits
#define xltPunctuationEnable	flag10	// prefer standard use of punctuation
#define xltVerticalEnable		flag14	// take height and vertical position of chars into account
#define xltSpaceDisable		flag15	// ignore spaces (translate as one string)
#define xltConnectedEnable		flag1	// currently not implemented

// Knowledge Source Controls
#define xltSpellingEnable		flag2	// use dictionary, prefer dictionary words
#define xltSpellingVeto		flag3	// disallow non-dictionary words
#define xltSpellingPropose		flag4	// propose from dictionary when stuck
#define xTemplateEnable		flag5	// use xTemplate, prefer template words
#define xTemplateVeto		flag6	// disallow words not matching template
#define xTemplatePropose	flag7	// propose from template when stuck

// Post-processing Rules
#define xltProofEnable			flag11	// currently not implemented
#define xltAbbrEnable			flag12	// currently not implemented
#define xltExpansionEnable		flag13	// currently not implemented
#define xltSmartCaseDisable    flag16	// DON'T correct the capitalization
 
// Not currently implemented
#define hwxGeoPolylines		flag24	// currently not implemented
#define hwxGeoSingleLines	flag25	// currently not implemented
#define hwxGeoLinesAlways	flag26	// currently not implemented


/****************************************************************************
 msgXlateGetFlags: takes P_U32, returns STATUS
	Gets the translation flags of an object.

 The translation flags are also called the hwxFlags.
*/
#define msgXlateGetFlags				MakeMsg(clsXlate, 17)


/****************************************************************************
 msgXlateFlagsClear: takes U32, returns STATUS
	Clears the given set of translation flags.

 Performs the operation
//{
	hwxFlags &= ~pArgs;
//}
 thus turning OFF all flags which are ON in pArgs and leaving unchanged
 those flags which are OFF in pArgs.
*/
#define msgXlateFlagsClear				MakeMsg(clsXlate, 15)


/****************************************************************************
 msgXlateCharConstrainsSet: takes P_U16, returns STATUS
	Sets the character constraints of a translation object.

 Character constraints impose limits on the shapes that the writer is
 allowed to write.  Setting the flag when appropriate may improve
 translation accuracy or performance since the shape matcher will know
 that it does not need to consider certain shapes as possibilities.

 For example, a numeric-only translator can be constructed by setting all
 of the disable flags except for xltDisableNumerals.

 Note that character constraints do not restrict the case of the 
 translation string if "case smarts" are on.  For example, case smarts
 may force the translation to be all lower case letters even if the
 xltDisableLowerCase charConstraint flag is set.
*/
#define msgXlateCharConstraintsSet		MakeMsg(clsXlate, 11)

#define xltDisableUpperCase	flag0	// disallow A thru Z
#define xltDisableLowerCase	flag1	// disallow a thru z
#define xltDisableNumerals		flag2	// disallow 0 thru 9
#define xltDisableCommonPunct	flag3	// disallow .,'!?;:%$#+-*()"=/
#define xltDisableOtherPunct	flag4	// disallow all other punctuation


/****************************************************************************
 msgXlateCharConstrainsGet: takes P_U16, returns STATUS
	Gets the character constraints of a translation object.

 The charConstraints were set in response to either msgNew or
 msgXlateCharConstraintsSet
*/
#define msgXlateCharConstraintsGet		MakeMsg(clsXlate, 18)


/****************************************************************************
 msgXlateTemplateGet: takes PP_UNKNOWN, returns STATUS
	Gets the template for a translation object.

 Will return in *pArgs a pointer to the compiled template currently in
 effect for the translator.
*/
#define msgXlateTemplateGet				MakeMsg(clsXlate, 13)


/****************************************************************************
 msgXlateTemplateSet: takes P_UNKNOWN, returns STATUS
	Sets the template for a translation object.

 The pArg should be a pointer to the "compiled" template created by
 calling the function  XTemplateCompile() defined in xtemplt.h
*/
#define msgXlateTemplateSet				MakeMsg(clsXlate, 9)


/****************************************************************************
 msgXlateCharMemorySet: takes P_CHARACTER_MEMORY, returns STATUS
	Sets the current Character memory for character box mode.

 In "boxed" mode (which typically is used when editing a short string),
 the translation object can accept a list of characters already attempted
 in this position.  This is used to allow ambiguous character shapes to
 be translated differently on overwrite.

 For example, a writer attempting to enter a lower case "L" may want to
 avoid repeatedly entering a straight vertical stroke and receiving a
 numeral "1" as the translation.  The character memory feature allows a
 client that keeps track of previously overwritten text to pass this
 information to the translation object.  The translation object will then
 disallow any character in the "already tried" string.

 This feature is implemented only for single character entries.  The
 Position field refers to the position of the character in the XLATE_STRING
 pCurrentText string.  Setting character memory for more than one position
 for a single translation will result in the character memory being 
 ignored in all positions.
*/
#define msgXlateCharMemorySet			MakeMsg(clsXlate, 22)

typedef struct CHARACTER_MEMORY {
	U16 position;						// position in the string
	P_CHAR usedCharacters;			// list of characters already used
} CHARACTER_MEMORY, *P_CHARACTER_MEMORY;


/****************************************************************************
 msgXlateCharMemoryGet: takes P_CHARACTER_MEMORY, returns STATUS
	Gets the current Character memory for character box mode.

 This message is intended for use by subclasses.
*/
#define msgXlateCharMemoryGet			MakeMsg(clsXlate, 27)


/****************************************************************************
 msgXlateSetXlateCaseMetrics: takes P_XLATE_CASE_METRICS, returns STATUS
	Sets the "smart case" metrics.

 The translation object can be directed to use Case (capitalization)
 heuristics above and beyond the basic xltCaseEnable heuristics set in the
 xlate flags.  These rules are communicated via the XLATE_CASE_METRICS
 structure.  They are applied in a post-processing pass by the translator,
 whereas the hwxFlags are applied during the initial search for a good
 translation.

 These rules set expectations for input (writer style) as well as output
 format.  The writer (CASE_WRITER) field prepares the system for the type of 
 input, allowing either mixed case or all upper case input.  The type
 (CASE_TYPE) field sets the style of heuristics.  The context field sets the
 specific rules to implement.

 See spell.h for definitions for SPELL_CASE_CONTEXT.
*/
#define msgXlateSetXlateCaseMetrics		MakeMsg(clsXlate, 26)


/****************************************************************************
 msgXlateGetXlateCaseMetrics: takes P_XLATE_CASE_METRICS, returns STATUS
    Gets the "smart case" metrics.

 Returns the values that were set either at msgNew time or by
 msgXlateSetXlateCaseMetrics.
*/
#define msgXlateGetXlateCaseMetrics		MakeMsg(clsXlate, 25)


/****************************************************************************
 msgXlateGetHistoryTemplate: takes PP_UNKNOWN, returns STATUS
	Gets the current alternate Translation Template.

 There is no behavior of class xlate associated with the history template
 other than to respond to the Set and Get messages.  It may used by the
 client to implement a "history" or cache mechanism, allowing the system
 to "remember" things previously translated.
*/
#define msgXlateGetHistoryTemplate		MakeMsg(clsXlate, 23)


/****************************************************************************
 msgXlateSetHistoryTemplate: takes P_UNKNOWN, returns STATUS
	Sets the current alternate Translation Template.

*/
#define msgXlateSetHistoryTemplate		MakeMsg(clsXlate, 24)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Control Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgXlateComplete: takes NULL, returns STATUS
	Initiates completion of translation after input is complete.

 Obsolete.  See msgXtractComplete in xtract.h.

 Not to be confused with msgXlateCompleted (see below).
*/
#define msgXlateComplete				MakeMsg(clsXlate, 3)


/*
 Other control messages are defined in xtract.h.  In general, the 
 client does not need to play an active role in sending or receiving
 control messages.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Notification Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgXlateData: takes P_XLATE_DATA, returns STATUS
	Allows a client to read the results from a translation object.

 The client reads the translation results from the translation object
 via this message.

 The translation object fills in the clients xlist data with the output
 data.  The specific xlist type is dependent upon the specific translation
 class.  Please refer to xlist.h for the information on each translation
 class.

 The output data is only available upon completion of the translation
 process.  Partial data cannot be read before the client has received
 the completion notification message (msgXlateCompleted) from the
 translation object (see below).

 The output data is a read-once function.  That is, you cannot send
 msgXlateData twice to the same translator. All translation object internal
 resources pertaining to the translated data are freed during the reading
 process.

 This message must be sent to an instance of one of the subclasses of
 clsXtract, such as clsXText or clsXGesture.  The clsXtract itself does not
 implement any behavior for this message.
*/
#define msgXlateData					MakeMsg(clsXlate, 2)

typedef struct XLATE_DATA {
	OS_HEAP_ID heap;        // In: heap to allocate structures
	struct XLIST *pXList;   // Out: pointer to return info
} XLATE_DATA, *P_XLATE_DATA;

typedef struct XLATE_BDATA {		//
	RECT32	box;					// bounding information
	S32		baseline;				// baseline offset
} XLATE_BDATA, *P_XLATE_BDATA;

typedef struct WORD_ENTRY {			// structure for a word
   S16	score;						// confidence factor
   CHAR	string[xltMaxWordLength];		// word
} WORD_ENTRY, *P_WORD_ENTRY;

typedef struct WORD_LIST {			// structure for a list of words
   RECT32		bound;				// bounding information
   U16			count;				// number of words in list
   WORD_ENTRY	word[1];			// variable length array of words
} WORD_LIST, *P_WORD_LIST;


/****************************************************************************
 msgXlateCompleted: takes OBJECT, returns STATUS
	Notification to client that the translation has been completed.

 This notification is sent by the translation object to its observers
 to inform them that translation is completed.  Upon receiving this
 message the client should send msgXlateData (see above) back to the
 translator to read the output.

 The pArgs is the id of the translator.
*/
#define msgXlateCompleted				MakeMsg(clsXlate, 128)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Private Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgXlateWordTranslate: takes P_XLATE_WORD_DATA, returns STATUS
	Translates the next word in the current line.

 This message is only for internal use by clsXtract itself.  Developers 
 should not send this message.
*/
#define msgXlateWordTranslate			MakeMsg(clsXlate, 19)

typedef struct XLATE_WORD_DATA {
	P_UNKNOWN pLine;
	P_UNKNOWN pWord;
	P_UNKNOWN pContext;
	U16 contextId;
} XLATE_WORD_DATA, *P_XLATE_WORD_DATA;

#endif
