/****************************************************************************
 txtview.h

 Copyright 1992, GO Corporation, All Rights Reserved.
	
 $Revision:   1.214  $
   $Author:   msapsfor  $
     $Date:   24 Feb 1992 13:00:50  $

 This file contains the API definition for clsTextView and clsTextIP.

 clsTextView inherits from clsView.

 clsTextView implements the user interface of a text editor.  It uses an
 instance of clsText (or one of its subclasses) to hold its data.

 clsTextIP inherits from clsIP.

 clsTextIP is a specialization of clsIP used by a Text Views.

 The functions described in this file are contained in TEXT.LIB. 
****************************************************************************/
 
/**** Introduction ****/
/*
 An instance of clsTextView (or textView) provides a user interface which
 presents text data to the user and lets the user edit that data.

 Every textView has an associated data object of clsText (or a subclass of
 clsText).  This object is referred to as textData.
*/


/**** Painting Model ****/
/*
 A textView displays the textData as a series of non-overlapping,
 exhaustively tiling, horizontal display lines.  With the possible
 exception of space below the last line, there is no area between lines
 that does not belong to any line.  Characters are laid out left to right
 with lines running from top to bottom.  
 
 When first created, the textView positions the first line of textData at
 the top of itself.  Subsequent user or client actions (e.g. scrolling) can
 position some other line to the top of the window.  However, the top line
 is always completely visible unless the view is too small to allow this. 
 The last visible line, in contrast, may be clipped at the bottom.

 Even though a textView is a descendant subclass of clsBorder, clsTextView
 ignores all clsBorder functionality relating to display of the view's
 background and border.
*/


/**** Deferred Repaint ****/
/*
 A textView uses a "delayed repair" model in which several changes to the
 textData may be made before the visible display lines are repainted. For
 certain operations  (e.g. selection change), such a delay can be
 misleading to the user and the individual operations provide a way to
 override the normal delay.  If no override is available within a message's
 arguments, msgTextViewRepair can be used.
*/


/**** Word Wrap ****/
/*
 By default, a textView displays each line beginning at the left edge of
 its window and "word wraps" at the right edge.  That is, if a word would
 be clipped by the right edge of the window, it is instead placed at the
 beginning of the next line.  By modifying paragraph margin attributes the
 line can be adjusted to have uninked margins in which no character is
 displayed.  

 Word wrap can be turned off by setting the textView's style (see
 msgTextViewSetStyle). When off, a line breaks only when a "hard break"
 character (such as teNewLine or teNewParagraph) is encountered.  As a
 result, a significant portion of many lines may be invisible to the user.
*/

 
/**** Embedded Objects ****/
/*
 Other objects can be embedded within a textView (see msgTextViewAddIP and
 msgTextViewEmbed).  (All embedded instances of some subclasses of
 clsEmbeddedWin.)

 A textView handles an embedded object as if it is a "very large"
 character.

 The textView's displayed lines are always as tall as the tallest character
 or embedded object in the line.  Therefore the presence of a large
 embedded object causes the containing line to be quite tall.  (Not all
 embedded objects are large.  For instance, closed application icons and
 reference buttons are only slightly larger than typical text.)

 The baseline of the line containing embedded objects is determined, in
 part, by the embedded object's response to msgWinGetBaseline.  (See
 win.h.)
*/


/**** Text IPs ****/
/*
 An instance of clsTextIP (or textIP) implements two special features that
 are useful to textViews.
 
 The first is size management.  An embedded textIP tracks the width of its
 parent window.  When the parent's width changes, an embedded textIP
 modifies its own width so that it fits within and completely fills the
 parent window (in the horizontal direction).
 
 The second is special filtering of text going from the IP into a textView.
 A textIP filters translated data from its superclass (clsIP) before
 passing its data onto its client (typically a textView).  Two kinds of
 filtering are performed:  paragraph break insertion and space correction. 
 A textIP inserts paragraph breaks based on how many blank lines there are
 between scribbles on an IP.  textIP also filters out unnecessary spaces
 between words and adds spaces after a sentence-ending character such as a
 period or question-mark.
*/


/**** Limitations ****/
/*
 textView is not WYSIWYG: although it will closely match font sizes and
 line breaks and spacing on a printer, it is based on a "make the printer
 match the screen" model that has enough variability that clients requiring
 WYSIWYG will find unacceptable (e.g., an overlaying mark-up layer).

 textViews do not support multiple views of a single data object.  Thus
 each textView is the unique view for its textData object.  This
 restriction is not checked by clsTextView.

 Although TV_NEW_ONLY has a "dc" field, there are so many restrictions on
 its use in PenPoint 1.0 that the field should always be left at the default
 value of Nil(OBJECT).  In addition, changing the units or scale used
 by the view-allocated "dc" is forbidden.  This prevents "magnifying glass"
 and "pan in or out" effects from being used with a textView.
*/

#ifndef TXTVIEW_INCLUDED
#define TXTVIEW_INCLUDED $Revision:   1.214  $ 

#ifndef						TXTDATA_INCLUDED
#include <txtData.h>		// For TEXT_INDEX
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Types and Constants							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT TEXT_VIEW;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Message Arguments							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Text View Style ****/

/*
 * The prefix "TV" indicates that an identifier is related to "TextView."
 *
 * The prefix "tvs" indicates that an identifier is related to "text view
 * style."
*/

typedef struct TV_STYLE {
	U16		flags;				// One of the values below
	S8		magnification;		// when tvsFormatForPrint is not on, this
								// value (in points) is added to the 
								// character font sizes.
	U8		showSpecial;		// 0: show no special characters.
								// 1: undefined -- do not use.
								// 2: undefined -- do not use.
								// 3: show all special characters.
								// (6 spare bits)
	OBJECT	printer;			// Not implemented.  Should be null.
} TV_STYLE, *P_TV_STYLE;

/* 
 * Use these flags in the flags field of TV_STYLE:
 *
 *    tvsEmbedOnlyComponents:	can only embed components. Cannot embed apps
 *    tvsEmbedOnlyIPs:			can only embed subclasses of clsIP. Can 
 *								embed no other objects.
 *    tvsFormatForPrinter:		printer preview.  style.magnification is 
 *								ignored.
 *    tvsQuietWarning:			don't display warning notes to user
 *    tvsQuietError:			don't display error notes to user 
 *    tvsQuiet:					both tvsQuietWarning and tvsQuietError
 *    tvsReadOnlyChars:			characters are read-only;  user cannot add, 
 *								remove or replace characters.
 *    tvsReadOnlyAttrs:			attributes are read-only;  user cannot change
 *								any attribute information.
 *    tvsReadOnly:				both tvsReadOnlyChars and tvsReadOnlyAttrs
 *    tvsWordWrap:				break display line by wrapping words that 
 *								don't fit at the right edge of the view.
*/
#define tvsEmbedOnlyComponents	flag0
#define tvsEmbedOnlyIPs			(tvsEmbedOnlyComponents|flag1)
#define tvsFormatForPrinter		flag2
#define tvsQuietWarning			flag3
#define tvsQuietError			flag4
#define tvsQuiet				(tvsQuietWarning|tvsQuietError)
#define tvsReadOnlyChars		flag5
#define tvsReadOnlyAttrs		flag6
#define tvsReadOnly				(tvsReadOnlyChars|tvsReadOnlyAttrs)
#define tvsWordWrap				flag7
#define tvsSpare1				flag8							// Reserved
#define tvsSpare2				flag9							// Reserved
#define tvsSpare3				(flag10|flag11|flag12|flag13)	// Reserved
#define tvsSpare4				flag14							// Reserved
#define tvsSpare5				flag15							// Reserved


/**** Embedding ****/
/*
 * TV_EMBED_METRICS describes where and how to embed an object. The client
 * either specifies the object to embed, or sets the embedded field to Nil
 * and lets the text view create a new object based on the flags field.  In
 * the latter case, the UID of the newly created object is passed back in the
 * embedded field.
*/
typedef struct TV_EMBED_METRICS {
	TEXT_INDEX		pos;			// In: embedded object is inserted 
									// just before this position.
	U16				flags;			// One of the values below
	OBJECT			embedded;		// In-Out: the UID of the embedded object
} TV_EMBED_METRICS, *P_TV_EMBED_METRICS;


/*
 * Use these in the flags field of a TV_EMBED_METRICS.
*/
#define tvEmbedAnnotate		flag0	// Not implemented
#define tvEmbedFloat		flag1	// Make the embeddee floating
#define tvEmbedReplace		flag2	// The IP's contents replace the
									// character following the IP.

//REFGEN BEGINIGNORE
/*
 * These two flags are used internally to remember some IP translator
 * settings.  Sometimes the settings are temporarily overridden;  their
 * original values are recorded in these two flags.  (See xlate.h for
 * information the translator flags.)
*/
#define tvEmbedSpellWasOn	flag3	// Private.  For internal use only.  The
									// IP's translator had xltSpellingEnable on.
#define tvEmbedCaseWasOn	flag4	// Private.  For internal use only. The
									// IP's translator had xltSmartCaseDisable 
									// off.
//REFGEN ENDIGNORE

/*
 * Use this in the flags field of a TV_EMBED_METRICS.
*/
#define tvEmbedAddMargin	flag5	// Leave small between previous line
									// and the IP.

//REFGEN BEGINIGNORE
/* Private.  For internal use only. */
#define tvEmbedSpecial		(tvEmbedAnnotate|tvEmbedFloat|tvEmbedReplace| \
							 tvEmbedSpellWasOn|tvEmbedCaseWasOn|tvEmbedAddMargin)
//REFGEN ENDIGNORE

/*
 * Use these in the flags field of a TV_EMBED_METRICS when using the struct
 * as the pArgs to msgTextViewAddIP.
*/
#define tvEmbedAtEnd		flag8	// IP should be last char of data.
#define tvEmbedPara			flag9	// IP is a paragraph pad
#define tvEmbedOneChar		flag10	// IP is only 1-char
#define tvEmbedPreload		flag11	// preload the selection into the IP

#define tvEmbedDisplayType	(flag13|flag14|flag15)	// Obsolete.


/**** Resolution ****/

/*
 * The prefix "tvr" indicates that an identifier is related to "text view
 * resolve."
*/

/* 
 The values for the xRegion and yRegion fields of a TV_RESOLVE struct are
 illustrated here.  The values are of the form (xRegion, yRegion).
//{
             |              | 
    (-1,1)   |     (0,1)    |  (1,1)
             |              | 
          ---+--------------+---
             |              | 
             |  Line's ink  | 
    (-1,0)   |     (0,0)    |  (1,0)
             |              | 
          ---+--------------+---
             |              | 
    (-1,-1)  |     (0,-1)   |  (1,-1)
             |              | 
//}
*/

/*
 * The fields of this structure are described in more detail in the comments
 * for msgTextViewResolveXY.
*/
typedef struct TV_RESOLVE {
	XY32			xy;			// In:Out: Units are LWC
	U16				flags;		// One of the values below
	TEXT_INDEX		pos;		// Out: Pos of char containing xy, or
								// maxTEXT_INDEX if no such char
	TEXT_INDEX		lineStart;	// Out: Pos of first char on line
								// containing xy, or maxTEXT_INDEX
								// if no line contains xy.
	S8				xRegion;	// Out: Region x was in.  See diagram.
	S8				yRegion;	// Out: Region y was in.  See diagram.
	TEXT_INDEX		selects;	// Out: Pos of char "selected" by xy
	XY32			offset;		// Out: Offset to prev/next char's ink
	P_UNKNOWN		spares[4];	// Reserved.
} TV_RESOLVE, *P_TV_RESOLVE;


/* 
 * Use these flags in the flags field of TV_RESOLVE.  Note that they are
 * not completely orthogonal; in particular, only one of [tvrSelFirst,
 * tvrSelLPO and tvrBalance] should be enabled at once, similarly for
 * [tvrPrevChar and tvrNextChar].
 *  tvrSelFirst:	causes TV_RESOLVE.selects to be <= TV_RESOLVE.pos (i.e.,
 *					the "selected" character is at or before the character
 *					"hit" by TV_RESOLVE.xy.)
 *  tvrSelLPO:		causes TV_RESOLVE.selects to be >= TV_RESOLVE.pos
 *			 		(i.e., the "selected" character is after the character
 *			 		"hit" by TV_RESOLVE.xy, unless the line contains only
 *			 		one character in which case
 *			 		TV_RESOLVE.selects == TV_RESOLVE.pos.)
 *  tvrBalance:		has the effect of tvrSelFirst or tvrSelLPO, depending on
 *					which edge of the character "hit" by TV_RESOLVE.xy is
 *					closest to TV_RESOLVE.xy.x.
 *  tvrSelWord:		causes the "selection" behavior specified by any of
 *					the previous three flags to occur for the "word"
 *					containing the character "hit" by TV_RESOLVE.xy.x.
 *  tvrPrevChar:	normally TV_RESOLVE.offset.x is 0 upon return.	Enabling
 *					tvrPrevChar causes TV_RESOLVE.offset.x to contain the
 *					amount that TV_RESOLVE.xy.x exceeds the x coordinate of
 *					the lower-left corner of the character specified by
 *					TV_RESOLVE.pos (i.e., the distance past the previous
 *					character's right edge).
 *  tvrNextChar:	normally TV_RESOLVE.offset.x is 0 upon return.	Enabling
 * 					tvrNextChar causes TV_RESOLVE.offset.x to contain the
 *					amount that TV_RESOLVE.xy.x falls short of the x
 *					coordinate of the lower-right corner of the character
 *					specified by TV_RESOLVE.pos (i.e., the distance before
 *					the next character's left edge).
 *  tvrPastEOL:		normally a line contains only those character positions
 *					for the characters displayed on the line. tvrPastEOL
 *					permits TV_RESOLVE.selects to return with the TEXT_INDEX
 *					of the first character of the following line if the
 *					specified TV_RESOLVE.xy.x is to the right of the last
 *					character in the line.
 *  tvrNLIfPastEOL:	when disabled, if TV_RESOLVE.xy.x is to the right of
 *					the last character in a line with a hard line break
 *					(e.g., teNewLine or teNewParagraph) and at least one
 *					other character, TV_RESOLVE.selects specifies the
 *					character immediately before the hard line break.
 *					When enabled, if tvrPastEOL is also enabled and would
 *					have caused TV_RESOLVE.selects to be after the hard line
 *					break, tvrNLIfPastEOL will override and cause
 *					TV_RESOLVE.selects to specify the break character instead.
*/
#define tvrSelFirst		flag0
#define tvrSelLPO		flag1
#define tvrSelWord		flag5
#define tvrPrevChar		flag2
#define tvrNextChar		flag3
#define tvrBalance		flag4
#define tvrPastEOL		flag6
#define tvrNLIfPastEOL	flag7


/**** Selection ****/

/*
 * The prefix "tvs" indicates that an identifier is related to "text view 
 * select."
*/

/*
 * The fields of this structure are described in more detail in the comments
 * for msgTextViewSetSelection.
*/
typedef struct TV_SELECT {
	TEXT_INDEX		first;		// lpoTEXT_INDEX means "clear selection"
	TEXT_INDEX		length;		// 0 results in an 0 length selection
	U16				flags;		// either 0 or wsSynchRepaint (see win.h)
	ATOM			level;		// Obsolete.  Don't use.
} TV_SELECT, *P_TV_SELECT;


//REFGEN BEGINIGNORE
/* Use these in the flags field of a TV_SELECT. */
#define tvsUseLevel			flag0	// Obsolete.  Don't use.
#define tvsSuppressNotify	flag1	// Private.  For internal use only. Don't
									// interact with theSelectionManager.
//REFGEN ENDIGNORE


/**** Scrolling ****/

/*
 * The prefix "ts" indicates that an identifier is related to "text view
 * scroll."
*/

typedef struct TV_SCROLL {
	TEXT_INDEX		pos;		// Position to scroll to
	U32				flags;		// One of the values below
} TV_SCROLL, *P_TV_SCROLL;

/* 
 * Use these in the flags field of a TV_SCROLL.
 *    tsAlignAtTop:			scroll so that pArgs->pos is "near the top."
 *							See tsAlignEdge.
 *    tsAlignAtBottom:		scroll so that pArgs->pos is "near the
 *                          bottom." See tsAlignEdge.
 *    tsAlignAtCenter:		scroll so that pArgs->pos is in the center
 *                          displayed line
 *    tsAlignEdge:			If set, and tsAlignAtTop or tsAlignAtBottom is
 *                          set, this flag forces the line containing
 *                          pArgs->pos to be the exact edge.  If this
 *                          flag is off, and tsAlignAtTop
 *                          tsAlignAtBottom is set, the textView tries
 *                          to leave an extra line or two between the
 *                          line containing pArgs->pos and the view's
 *                          edge.
 *    tsIffInvisible:		If set, the textView scrolls only if pArgs->pos 
 *							is not already visible.  If not set, the 
 *                          textView scrolls even if pArgs->pos is visible.
 *    textNoScrollNotify:	By default, the scrollbar(s) for the view
 *                          are notified (via a msgWinSend of
 *                          msgScrollbarUpdate) that they should update
 *                          after a msgTextViewScroll.  If this flag is
 *                          set, the notification is not sent.
*/
#define tsAlignAtTop		0L
#define tsAlignAtBottom		1L
#define tsAlignAtCenter		2L
#define tsAlignEdge			((U32)flag2)
#define tsIffInvisible		((U32)flag3)
#define textNoScrollNotify	((U32)flag15)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages Defined by Other Classes					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNewDefaults		takes P_TV_NEW, returns STATUS
	category: class message
	Initializes the NEW structure.

 Zeros out pNew->tv and sets:
//{
	tv.style.flags			= tvsWordWrap;
	tv.flags				= tvFillWithIP;

	win.flags.style			|= wsGrowBottom | wsSendFile | 
							   wsSendGeometry | wsCaptureGeometry;
	win.flags.style			&= ~(wsSendLayout | wsCaptureLayout);
	win.flags.input			|= inputMoveDown | inputMoveDelta | 
							   inputHoldTimeout | inputOutProx | 
							   inputTip | inputEnter | inputExit;
	view.createDataObject	= true;
	gWin.helpId				= tagTextView;
//}
*/


/****************************************************************************
 msgNew		takes P_TV_NEW, returns STATUS
	category: class message
	Creates a new instance of clsTextView.

 If pArgs->view.createDataObject is true, then the textView creates a Text
 data object (clsText; see txtdata.h) and sets the view's data object
 If pArgs->tv.dc is NULL the textView creates a DC for its exclusive use.
*/


/****************************************************************************
 msgGWinXList	takes P_XLIST, returns STATUS
	Defined in gwin.h.

 In response to this message, a textView typically performs some editing
 operation on its associated data object.  A textView can process both
 "vanilla" xlists as described in xlist.h or text-specific xlists as
 txtxlist.h.

 Here's how a textView responds to each xlist element:
	xtBounds:		remembers the bounds of a gesture element
	xtGesture:		processes the gesture
	xtText:			inserts the text
	xtObject:		embeds the object
	xtCharAttrs:	modifies the character attributes of the specified
					characters 
	xtParaAttrs:	modifies the attributes of the specified paragraphs 
	xtTabs:			modifies the tabs of the specified paragraphs 
	xtCharPos:		sets the insertion point for text to the specified
					character position
*/


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Private									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* These #defines must be on multiple lines to avoid breaking gensyms. */

#define TVMakeMsg(i) \
			MakeMsg(clsTextView, i)
// Next free: 11; recycled: 7;

// Reserved for Option Sheet (FRIENDS ONLY): [64..96)
#define TVPSMakeMsg(i) \
			TVMakeMsg(64+i)
// Next free: 2; recycled: none

//REFGEN ENDIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgTextViewAddIP		takes P_TV_EMBED_METRICS, returns STATUS
	Adds an insertion pad to the textView.

 The client must set all of the fields of pArgs as described in the
 discussion of TV_EMBED_METRICS.
*/
#define	msgTextViewAddIP		TVMakeMsg(0)


/****************************************************************************
 msgTextViewCheck		takes P_UNKNOWN, returns STATUS
	A textView performs a self-consistency check.
	
 This message is only available in the debugging version of text.dll.
 The only currently defined value for pArgs is zero.

 Return Value
	stsOK:		no problems detected
	< stsOK:	problems detected
*/
#define	msgTextViewCheck		TVMakeMsg(5)


/****************************************************************************
 msgTextViewEmbed		takes P_TV_EMBED_METRICS, returns STATUS
	Embeds an object in the textView.  Makes associated changes in text data.

 The client must set all of the fields of pArgs as described in the
 discussion of TV_EMBED_METRICS.
*/
#define	msgTextViewEmbed		TVMakeMsg(1)


/****************************************************************************
 msgTextViewGetEmbedMetrics	takes P_TV_EMBED_METRICS, returns STATUS
	Passes back the textView-specific metrics for an embedded object.

 The client need only fill in pArgs->embedded.
*/
#define	msgTextViewGetEmbedMetrics		TVMakeMsg(2)


/****************************************************************************
 msgTextViewRepair			takes pNull, returns stsOK
	Forces a delayed paint operation to take place immediately.

 Use with caution, as overuse of this message significantly degrades 
 performance.
*/
#define	msgTextViewRepair		TVMakeMsg(3)


/****************************************************************************
 msgTextViewResolveXY		takes P_TV_RESOLVE, returns STATUS
	Given an point in LWC space, passes back the character at (or near) 
	the point.

 pArgs->flags control exactly which character is "selected", and how
 much information is provided by the message.

 Clients can also use this message to "reverse resolve" as follows.  If
 both pArgs->xy.x and pArgs->xy.y are maxS32, then the textView sets
 pArgs->xy to the coordinates of the lower left corner of the character at
 pArgs->pos.

 Warning:  The response to this message always updates pArgs->xy to reflect
 information about the line either containing (or near) the original xy (or
 pos).

 "LWC" is short for Local Window Coordinates.  See win.h for more
 information.

 Return Value
	stsBadParam:	if no line's y extents include pArgs->xy.y
	stsNoMatch:		if a containing line exists but it has no character
					under pArgs->xy.x; of if reverse resolve of a character
					not contained in any display line
*/
#define	msgTextViewResolveXY	TVMakeMsg(4)


/****************************************************************************
 msgTextViewScroll			takes P_TV_SCROLL, returns stsOK
	Repositions displayed text within the textView.

 The client must set the fields of pArgs as described in the discussion of
 TV_SCROLL.
*/
#define	msgTextViewScroll		TVMakeMsg(6)


/****************************************************************************
 msgTextViewGetStyle		takes P_TV_STYLE, returns stsOK
	Passes back a textView's style.
*/
#define	msgTextViewGetStyle		TVMakeMsg(8)


/****************************************************************************
 msgTextViewSetSelection	takes P_TV_SELECT, returns stsOK
	Selects one or more characters displayed by the textView.

 The fields of pArgs are used as follows:
    first:		The first character to select.  The value lpoTEXT_INDEX
                means that cause the selection to be cleared.
    length:		Number of characters to select.  The value 0 results in a 
                zero-length I-Bean selection.
    flags:		if this field is wsSynchRepaint (defined in win.h) the
				textView repaint immediately.  Otherwise this field 
				must be zero.

 While handling this message,  the textView becomes the selection owner
 unless pArgs->first is lpoTEXT_INDEX, in which case the text view ensures
 that it is NOT the selection owner.
*/
#define	msgTextViewSetSelection	TVMakeMsg(9)


/****************************************************************************
 msgTextViewSetStyle		takes P_TV_STYLE, returns stsOK
	Sets a textView's style.

 pArgs->printer should be set to Nil(OBJECT).
*/
#define	msgTextViewSetStyle		TVMakeMsg(10)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Definitions for msgNew							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef NO_NEW
#ifndef txtViewNewFields

#ifndef						VIEW_INCLUDED
#include <view.h>
#endif

/* See comment with msgNew and msgNewDefaults for more information. */
typedef struct TV_NEW_ONLY {
	U16				flags;			// One of the values below
	OBJECT			dc;
	TV_STYLE		style;
} TV_NEW_ONLY, *P_TV_NEW_ONLY;

/* Use this in the flags field of a TV_NEW_ONLY. */
#define tvFillWithIP	flag0


#define txtViewNewFields	\
    viewNewFields			\
	TV_NEW_ONLY				tv;

typedef struct TV_NEW {
	txtViewNewFields
} TV_NEW, *P_TV_NEW;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Utility Functions						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 TextCreateTextScrollWin	returns STATUS
	Utility function that creates a textView (with a data object) 
	placed inside a scroll window.  (See swin.h.)

 Clients often need a "vanilla" textView inside a vanilla scrollWin.  This
 function does just that.  Clients can modify the created objects after the
 creation if this function doesn't do quite the right thing.  Client who
 need more control over the creation should probably create the objects
 manually.

 The pNew parameter should be null or should point at an already
 initialized NEW struct.  If it is null, then the function creates a
 default instance of clsTextView.

 Because the view is created with formatForPrinter FALSE, the scrollWin's
 expandChildWidth is set to true.  This causes the scrollWin to manage the
 width of the textView.

 Here's a simplified indication of how the scrollWin is created:
//{
	ObjectCall(msgNewDefaults, clsScrollWin, &sn)
	sn.scrollWin.clientWin					= <the text view>
	sn.scrollWin.style.vertScrollbar		= true;
	sn.scrollWin.style.autoVertScrollbar	= false;
	sn.scrollWin.style.expandChildWidth		= true;
	sn.scrollWin.style.expandChildHeight	= true;
	sn.scrollWin.style.contractChildWidth	= true;
	sn.scrollWin.style.contractChildHeight	= true;
	sn.scrollWin.style.vertClient			= swClientWin;
	sn.scrollWin.style.horizClient			= swClientScrollWin;
	sn.win.flags.input						|= inputHoldTimeout;
	sn.scrollWin.style.forward				= swForwardGesture;
	if (<creating on screen>) {
	    sn.border.style.leftMargin = bsMarginMedium;
	    sn.border.style.rightMargin = bsMarginMedium;
	    sn.border.style.topMargin = bsMarginMedium;
	} else {
	    sn.border.style.leftMargin = bsMarginNone;
	    sn.border.style.rightMargin = bsMarginNone;
	    sn.border.style.topMargin = bsMarginNone;
	}
	ObjectCall(msgNew, clsScrollWin, &sn);
	*scrollWin = sn.object.uid;		
//}

 Warning: When printing, the scrollWin and textView are probably restored,
 not created anew.  Therefore the client needs to go in and set the
 scrollWin's margins to 0.
*/
STATUS EXPORTED
TextCreateTextScrollWin(
	P_TV_NEW	pNew,
	P_OBJECT	scrollWin);		// Out:


#endif // txtViewNewFields
#endif // NO_NEW

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *									TextIP								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef struct TEXTIP_METRICS {
	U16		flags;					// Reserved.
} TEXTIP_METRICS, *P_TEXTIP_METRICS,
  TEXTIP_NEW_ONLY, *P_TEXTIP_NEW_ONLY;


/****************************************************************************
 msgNewDefaults		takes P_TEXTIP_NEW, returns STATUS
	category: class message
	Initializes the NEW struct.

 In response to this message, clsTextIP does the following:
//{
	pArgs->win.flags.style			|= wsSendGeometry | wsSendFile | 
								       wsShrinkWrapHeight;
	pArgs->ip.rows 					= 5;
	pArgs->ip.lines					= 5;
//}

 If the user input pad style preference is Boxed:
//{
	pArgs->ip.style.displayType		= ipsCharBox;
	pArgs->ip.style.delayed			= 1;
//}

 If the user input pad style preference is Ruled:
//{
	pArgs->ip.style.displayType		= ipsRuledLines;
//}

 If the user input pad style preference is RuledAndBoxed:
//{
	pArgs->ip.style.displayType		= ipsRuledLines;
	pArgs->ip.style.ruledToBoxed	= true;
//}
*/


/****************************************************************************
 msgNew		takes P_TEXTIP_NEW, returns STATUS
	category: class message
	Creates a new instance of clsTextIP.
*/


/****************************************************************************
 msgTextIPGetMetrics		takes P_TEXTIP_METRICS, returns stsOK
	Passes back a textIP's metrics.
*/
#define msgTextIPGetMetrics			MakeMsg(clsTextIP, 1)


/****************************************************************************
 msgTextIPSetMetrics		takes P_TEXTIP_METRICS, returns stsOK
	Sets a textIP's metrics.
*/
#define msgTextIPSetMetrics			MakeMsg(clsTextIP, 2)


#ifndef NO_NEW
#ifndef textIPNewFields

#ifndef						INSERT_INCLUDED
#include <insert.h>
#endif

#define textIPNewFields		\
	ipNewFields				\
	TEXTIP_NEW_ONLY			textIP;

typedef struct TEXTIP_NEW {
	textIPNewFields
} TEXTIP_NEW, *P_TEXTIP_NEW;

#endif // textIPNewFields
#endif // NO_NEW


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *									Private								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define stsTVApplyWithoutSelection	MakeStatus(clsTextView, 0)
#define stsTVTabsOverlap			MakeStatus(clsTextView, 1)
#define stsTxtOverwriteLT			MakeStatus(clsTextView, 255)

STATUS GLOBAL InitClsTextView(void);

/****************************************************************************
 msgTVApplyPropSheet		takes pNull, returns STATUS
	Private message.  Do not send.
*/
#define	msgTVApplyPropSheet		TVPSMakeMsg(0)


/****************************************************************************
 msgTVCancelPropSheet		takes pNull, returns STATUS
	Private message.  Do not send.
*/
#define	msgTVCancelPropSheet	TVPSMakeMsg(1)

//REFGEN ENDIGNORE

#endif	// ifndef TXTVIEW_INCLUDED

