/****************************************************************************
 File: ts.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.12  $
   $Author:   kguzik  $
     $Date:   24 Feb 1992 19:35:14  $

 This file contains the API definition for clsTable.
 clsTable inherits from clsObject.

 clsTable provides a general-purpose table mechanism with random and
 sequential access. The table allows clients to create, destroy, modify,
 and access the table and its data using a row and column metaphor. Data
 for the table is stored in a table file, whose lifetime can be independent
 to that of the table object.

 Tables are two dimensional arrays consisting of a fixed number of columns
 and a variable number of rows. Each column can contain data of a single
 data type such as a U32, a variable length string, a fixed sized byte
 field, date and time, etc.

 The number of and types of these columns are defined when the table is
 created. Once that table has been created, these parameters cannot be
 changed.

 Clients access rows in the table using a TBL_ROW_POS data structure. The
 value for this row position is returned to the client when a row is added
 to the table. All messages for manipulating data in the table require this
 value to specify an individual row.

 Clients address columns using their position in the TBL_COL_DESC array which
 the client provides in the TBL_CREATE data structure during msgNew.

 The table is an observable object and clients choosing to be observers will
 receive notification when data in the table changes or a row has been added
 to or removed from the table.

****************************************************************************/

#ifndef TS_INCLUDED
#define TS_INCLUDED

#include <clsmgr.h>
#include <fs.h>
#include <resfile.h>

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Status Codes								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Status values return by messages to clsTable.
*/
#define stsTBLRefCountNotZero 		MakeStatus( clsTable, 1 )
#define stsTBLColNameNotFound 		MakeStatus( clsTable, 2 )
#define stsTBLStrBufTooSmall		MakeStatus( clsTable, 3 )
#define stsTBLBadNewFlags			MakeStatus( clsTable, 4 )
#define stsTBLEndOfTable			MakeStatus( clsTable, 5 )
#define stsTBLInvalidSortColValue	MakeStatus( clsTable, 7 )
#define stsTBLCorruptedIndex		MakeStatus( clsTable, 8 )
#define stsTBLColNotIndexed			MakeStatus( clsTable, 9 )
#define stsTBLContainsIndexedCols	MakeStatus( clsTable, 10 )

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Common macros and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Class Declaration
*/
#define clsTable 				MakeWKN(2003,1,wknGlobal)

/*
 Object Declarations
*/
typedef OBJECT	TABLE;
typedef OBJECT	TBLOBJ;
typedef TBLOBJ	*P_TBLOBJ;

/*
 Table Parameter Definitions
*/
#define TBL_MAXCOLNAMELEN		nameBufLength
#define TBL_MAXTBLNAMELEN		nameBufLength
#define TBL_MAXROWCOUNT			0x2000				// 8192 entries

/*
 Table Row Definitions
*/
typedef RES_ID	TBL_ROW_POS,		*P_TBL_ROW_POS;		// Absolute TS Row Key
typedef U16		TBL_ROW_NUM,		*P_TBL_ROW_NUM;		// Position relative TS Row Key
typedef U16		TBL_ROW_COUNT,  	*P_TBL_ROW_COUNT;
typedef U16		TBL_ROW_LENGTH, 	*P_TBL_ROW_LENGTH;
typedef S32		TBL_ROW_OFFSET, 	*P_TBL_ROW_OFFSET;
typedef S16		TBL_REF_COUNT,  	*P_TBL_REF_COUNT;

/*
 Table Data Type Definitions
*/
typedef P_U8					P_ROW_BUFFER, *PP_ROW_BUFFER;
typedef P_UNKNOWN 				P_TBL_COL_DATA_HOLDER;

/*
 Column Index Declarations
*/
typedef U16	TBL_COL_INX_TYPE,	*P_TBL_COL_INX_TYPE;
typedef U16	TBL_COL_COUNT,   	*P_TBL_COL_COUNT;
typedef U16 TBL_COL_LENGTH,		*P_TBL_COL_LENGTH;
typedef U32	TBL_COL_OFFSET,  	*P_TBL_COL_OFFSET;

/*
 Column Descriptor Definitions
*/
typedef enum TBL_TYPES {
	tsChar			= 0, // fixed length byte array of case sensitive chars
	tsCaseChar		= 1, // fixed length byte array of case insensitive chars
	tsU16			= 2, // unsigned 16 bit integer
	tsU32			= 3, // unsigned 32 bit integer
	tsFP			= 4, // double precision floating point
	tsDate			= 5, // date field (system compressed time format)
	tsString		= 6, // null-terminated variable length ascii string (case sensitive)
	tsCaseString	= 7, // same as tsString but is case insensitive
	tsByteArray		= 8, // variable length byte array, contained in unsigned chars
	tsUUID			= 9, // UUID struct.
	tsLastType = tsUUID 
} TBL_TYPES;

typedef struct TBL_COL_DESC {
	CHAR				name[TBL_MAXCOLNAMELEN];	// Column name
	TBL_TYPES			type;						// Column type
	TBL_COL_LENGTH		length;						// Column data length
	TBL_COL_INX_TYPE	repeatFactor;				// # of times to repeat the column
	TBL_COL_OFFSET		offset;						// Column offset in the row
	BOOLEAN				sorted;						// Is the column sorted?
} TBL_COL_DESC, *P_TBL_COL_DESC;

/*
 Variable Length Data Buffer Definition
*/
typedef struct TBL_STRING {
	U16		strLen;	// In/Out: length of string or byte array column data
	U16		strMax;	// In:     length of string or byte array buffer
	P_CHAR	pStr;	// In:     pointer to client buffer.
} TBL_STRING, *P_TBL_STRING;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Class Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_TBL_NEW, returns STATUS
	category: class message
	Creates a new table object. 

 This message creates a new table file or opens an existing file.

 The table name is an optional field. The locator and colDescAry fields
 must be valid and colCount must be non zero or this message returns
 stsBadParam.

 Return Value
	stsTBLBadNewFlags:	TBL_EXIST flags were invalid.
	stsBadParam:		locator or colDescAry fields are invalid. colCount
						is 0.
*/

typedef enum TBL_FREE_BEHAVE {
	tsFreeNoDeleteFile    	= 0,		// Free only the object, not the file
	tsFreeDeleteFile	    = flag0,	// Destroy the file when freed
	tsFreeWhenNoClients	 	= flag1,	// Free when # clients accessing is 0
	tsFreeNoObservers		= flag2,	// Free when # of observers is 0
	tsFreeNoCompact			= flag3,	// Don't compact the table when freed
	tsFreeDefault 			= tsFreeNoDeleteFile
} TBL_FREE_BEHAVE, *P_TBL_FREE_BEHAVE;
  
typedef enum TBL_EXIST {
	// Same values as FS_EXIST_MODE
	tsExistOpen 		 	= 0,			// Open an existing table
	tsExistGenError  		= 1,			// Return error if table exists
	tsExistGenUnique 		= 2, 			// Create table with a unique name
	tsNoExistCreate  		= MakeU16(0,0),	// Create a new table
	tsNoExistGenError     	= MakeU16(0,1),	// Return error if no table exists
	tsExistDefault		    = tsExistOpen | tsNoExistCreate
} TBL_EXIST, *P_TBL_EXIST;

typedef struct TBL_CREATE {
	TBL_COL_COUNT	colCount;	  // number of columns
	P_TBL_COL_DESC	colDescAry;	  // TBL_COL_DESC array
} TBL_CREATE, *P_TBL_CREATE;

typedef struct TBL_NEW_ONLY {
	CHAR				name[TBL_MAXTBLNAMELEN];	// Table name
	FS_LOCATOR	   		locator;					// Table file
	TBL_EXIST			exist;						// Table exist behavior
	TBL_CREATE			create;						// Column specifications
  	TBL_FREE_BEHAVE		freeBehavior;				// Table free behavior
	BOOLEAN				createSemaphore;			// Provide semaphore?
} TBL_NEW_ONLY, *P_TBL_NEW_ONLY;

#define	tableNewFields		\
	objectNewFields 		\
	TBL_NEW_ONLY	table;

typedef struct TBL_NEW {
	tableNewFields
} TBL_NEW, *P_TBL_NEW;

/****************************************************************************
 msgNewDefaults				takes P_TBL_NEW, returns STATUS
	category: class message
	Initializes the TBL_NEW structure to default values.

 Zeroes out pNew->table and sets:
 //{
	pNew->table.name[0]				= '\0';
	pNew->table.locator.uid			= objNull;
	pNew->table.locator.pPath		= pNull;
	pNew->table.exist				= tsExistDefault;
	pNew->table.create.colCount		= 0;
	pNew->table.create.colDescAry	= pNull;
	pNew->table.freeBehavior		= tsFreeDefault;
	pNew->table.createSemaphore		= false;
 //}
*/

/****************************************************************************
 msgDestroy		takes OBJ_KEY, returns STATUS
	category: class message
	Destroys an existing table object.

 This message destroys the table object and frees the table files if the
 object was created with the tsFreeDeleteFile flag specified.

 The table file will not be destroyed regardless of whether tsFreeDeleteFile
 was specified if there are still accessors to the table. Only the object
 will be freed.

 Return Value
	stsTBLRefCountNotZero:	The number of accessors of the table is not
							zero. The table file will not be destroyed.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Object Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Table Row Addition and Deletion Messages  ****/

/****************************************************************************
 msgTBLAddRow	takes P_TBL_ROW_POS, returns STATUS
	Adds a row/record with no data to the table server object.

 The row position (TBL_ROW_POS) for the new row is passed back. The row
 position is the key to access data in the row or to delete the row.
*/
#define msgTBLAddRow				MakeMsg(clsTable, 1)

/****************************************************************************
 msgTBLDeleteRow	takes P_TBL_ROW_POS, returns STATUS
	Deletes the specified row.

 Rows are deleted from the table at the completion of this call. The row's
 TBL_ROW_POS is no longer valid after the row has been deleted.

 Return Value
	stsTBLEndOfTable:	TBL_ROW_POS value was not found in the table.
*/
#define msgTBLDeleteRow				MakeMsg(clsTable, 5)

/****  Table Data Messages  ****/

/****************************************************************************
 msgTBLColGetData	takes P_TBL_COL_GET_SET_DATA, returns STATUS
	Passes back the data for the specified row and column.

 tblColData is of type P_TBL_STRING if the column type is tsString, 
 tsCaseString, or tsByteArray.

 The client is responsible for allocating storage for the tblStr.pStr
 buffer. If the buffer is too small to accomodate the requested data, the
 table will return stsTBLStrBufTooSmall and pass back the truncated data
 and the actual length of the data in tblStr.strLen.

 Return Value
	stsTBLStrBufTooSmall:	Returned if column type is tsString, tsCaseString
							or tsByteArray and tblStr.strMax is less than the
							actual data length. The data is truncated and the
							length is returned in tblStr.strLen.
	stsTBLEndOfTable:		TBL_ROW_POS value was not found in the table.

*/
#define msgTBLColGetData			MakeMsg(clsTable, 13)

typedef struct TBL_COL_GET_SET_DATA {
	TBL_ROW_POS				tblRowPos;		// In: Table row position
	TBL_COL_INX_TYPE		colNumber;		// In: Column number
	P_TBL_COL_DATA_HOLDER 	tblColData;		// Out: Column data
} TBL_COL_GET_SET_DATA, *P_TBL_COL_GET_SET_DATA;

/****************************************************************************
 msgTBLColSetData	takes P_TBL_COL_GET_SET_DATA, returns STATUS
	Sets the data for the specified row and column.

 tblColData is of type P_TBL_STRING if the column type is tsString, 
 tsCaseString, or tsByteArray. Clients are responsible for setting the
 strLen field of the TBL_STRING argument for all column types.

 Return Value
	stsTBLEndOfTable:		TBL_ROW_POS value was not found in the table.
*/
#define msgTBLColSetData			MakeMsg(clsTable, 14)

/****************************************************************************
 msgTBLRowGetData	takes P_TBL_GET_SET_ROW, returns STATUS
	Gets the contents of an entire row.

 Not valid for tables containing variable length columns.

 The client is responsible for providing storage for the pRowData buffer.
 The length of a table row can be obtained using msgTBLGetRowLength.

 Return Value
	stsTBLEndOfTable:			TBL_ROW_POS value was not found in the table.
	stsTBLContainsIndexedCols:	Table contains variable length columns.

 See Also
	msgTBLGetRowLength
*/
#define msgTBLRowGetData						MakeMsg(clsTable, 15)

typedef struct TBL_GET_SET_ROW {
	TBL_ROW_POS		tblRowPos;	// In:  Which row
	P_UNKNOWN		pRowData;	// Out: Row data
} TBL_GET_SET_ROW, *P_TBL_GET_SET_ROW;

/****************************************************************************
 msgTBLRowSetData	takes P_TBL_GET_SET_ROW, returns STATUS
	Sets the contents of an entire row.

 Not valid for tables containing variable length columns.

 Return Value
	stsTBLEndOfTable:			TBL_ROW_POS value was not found in the table.
	stsTBLContainsIndexedCols:	Table contains variable length columns.

 See Also
	msgTBLGetRowLength
*/
#define msgTBLRowSetData			MakeMsg(clsTable, 16)

/****  Table Information Messages  ****/

/****************************************************************************
 msgTBLGetInfo	takes P_TBL_HEADER, returns STATUS
	Gets the table header information.

 See Also
	msgTBLGetColCount,
	msgTBLGetColDesc,
	msgTBLGetRowCount,
	msgTBLGetRowLength
*/
#define msgTBLGetInfo				MakeMsg(clsTable, 10)

typedef struct TBL_HEADER {  
	TBL_COL_COUNT	colCount;  				// number of columns in table
	CHAR			name[TBL_MAXTBLNAMELEN];// non-file table reference
	TBL_ROW_COUNT	nRows;					// how many rows in table
	TBL_ROW_LENGTH	rowLength;				// row buffer length
	TBL_ROW_POS		firstRow;				// position of first row in table
	TBL_ROW_POS		currentRow;				// position of current row in table
	TBL_ROW_POS		lastRow;				// position of last row in table
	TBL_REF_COUNT	refCount;				// number of active clients.
} TBL_HEADER, *P_TBL_HEADER, **PP_TBL_HEADER;

/****************************************************************************
 msgTBLGetColCount	takes P_TBL_COL_COUNT, returns STATUS
	Gets the number of columns in the table.
*/
#define msgTBLGetColCount			MakeMsg(clsTable, 7)

/****************************************************************************
 msgTBLGetColDesc	takes P_TBL_GET_COL_DESC, returns STATUS
	Passes back the column description for the specified column.
*/
#define msgTBLGetColDesc			MakeMsg(clsTable, 2)

typedef struct TBL_GET_COL_DESC {

	TBL_COL_INX_TYPE	colInx;		// In: column number
	TBL_COL_DESC 		colDesc;	// Out: column decription

} TBL_GET_COL_DESC, *P_TBL_GET_COL_DESC;

/****************************************************************************
 msgTBLGetRowCount	takes P_TBL_ROW_COUNT, returns STATUS
	Gets the current number of rows in the table.
*/
#define msgTBLGetRowCount			MakeMsg(clsTable, 6)

/****************************************************************************
 msgTBLGetRowLength	takes P_TBL_ROW_LENGTH, returns STATUS
	Gets the length (in bytes) of the specified row.

 The row length indicates the total width of all columns for each row
 in the table. This information is useful when getting and setting row
 data.

 See Also
	msgTBLRowGetData
	msgTBLRowSetData
*/
#define msgTBLGetRowLength			MakeMsg(clsTable, 8)

/****************************************************************************
 msgTBLGetState	takes P_TBL_GET_STATE, returns STATUS
	Gets the current state of a specified row.

 The state of a row in the table indicates its general positioning
 within the table.

 Return Value
	stsTBLEndOfTable:	TBL_ROW_POS value was not found in the table.
*/
#define msgTBLGetState				MakeMsg(clsTable, 11)

typedef enum TBL_STATE {
	tsBegin		= 0, // rowPos is the first row
	tsEnd		= 1, // rowPos is the last row
	tsPosition	= 2	 // rowPos is not first or last
} TBL_STATE, *P_TBL_STATE;

typedef struct TBL_GET_STATE {
	TBL_STATE 	tblState;	// Out:	State of the specified row
	TBL_ROW_POS	tblRowPos;	// In: 	Row position of the specified row.
} TBL_GET_STATE, *P_TBL_GET_STATE;

/****  Table Access Messages  ****/

/****************************************************************************
 msgTBLBeginAccess	takes P_TBL_BEGIN_ACCESS, returns STATUS
	Initiates table access by a client on this table. 

 Passes back the row length of the first row. Adds the sender to the table's
 observer list.
*/
#define msgTBLBeginAccess			MakeMsg(clsTable, 17)

typedef struct TBL_BEGIN_ACCESS {
	OBJECT			sender;		// In: sender's id IFF wants to be observer
	TBL_ROW_LENGTH	rowLength;	// Out: Length of the first row
} TBL_BEGIN_ACCESS, *P_TBL_BEGIN_ACCESS;

/****************************************************************************
 msgTBLEndAccess	takes P_TBL_END_ACCESS, returns STATUS
	Ends client access to the table.

 Removes sender from the observer list.
*/
#define msgTBLEndAccess				MakeMsg(clsTable, 18)
 
typedef struct TBL_END_ACCESS {
		OBJECT	sender;		// In: Sender's uid
} TBL_END_ACCESS, *P_TBL_END_ACCESS;

/****************************************************************************
 msgTBLSemaClear	takes nothing, returns STATUS
	Releases the table's semaphore.

 The next client currently waiting on the table semaphore will unblock
 when this messages completes.
*/
#define msgTBLSemaClear				MakeMsg(clsTable, 23)

/****************************************************************************
 msgTBLSemaRequest	takes nothing, returns STATUS
	Requests access to the table's semaphore.

 Waits on the table semaphore if another client already has access. Provides
 exclusive access of the table semaphore to the sender when it returns.

 Semaphore access has no timeout.
*/
#define msgTBLSemaRequest			MakeMsg(clsTable, 22)

/****  Table Search Messages  ****/

/****************************************************************************
 msgTBLFindFirst	takes P_TBL_FIND_ROW, returns STATUS;
	Finds the first record that meets the search specification.

 Passes back the TBL_ROW_POS and TBL_ROW_NUM of the row.

 srchSpec.pConstOperand is of type P_TBL_STRING if the column type is 
 tsString, tsCaseString, or tsByteArray. The length of the string/array 
 used in the search is decalred in the strLen field of the TBL_STRING struct.
 Clients are responsible for setting this field to the appropriate length
 for columns of type tsString, tsCaseString, and tsByteArray.

 srchSpec.pConstOperand is ignored if srchSpec.relOp is tsAlwaysTrue. 
 
 Currently, tsSubstring searches are always case sensitive regardless of
 the column type.

 Return Value
	stsTBLEndOfTable:			No data was found matching the search spec.
	stsTBLInvalidSortColValue:	sortCol is not a valid column value.
*/
#define msgTBLFindFirst				MakeMsg(clsTable, 3)

typedef enum TBL_BOOL_OP {
	tsEql			= 0,	// Match if operands are equal
	tsEqual			= 1,	// Match if operands are equal
	tsLess			= 2,	// Match if opnd1 < opnd2
	tsGreater		= 3,	// Match if opnd1 > opnd2
	tsGreaterEqual	= 4,	// Match if opnd1 <= opnd2
	tsLessEqual		= 5,	// Match if opnd1 >= opnd2
	tsNotEqual		= 6,	// Match if the operands do not match
	tsSubstring		= 7,	// Match if opnd1 is an exact substring of opnd2
	tsStartsWith	= 8,	// Match if opnd1 starts with opnd2
	tsAlwaysTrue	= 9		// Match the first (or next) row
} TBL_BOOL_OP, *P_TBL_BOOL_OP;

typedef struct TBL_SEARCH_SPEC {
	TBL_COL_INX_TYPE			colOperand;		// In: Which column
	TBL_BOOL_OP					relOp;			// In: Operation
	P_TBL_COL_DATA_HOLDER		pConstOperand;	// In: Value to search against
} TBL_SEARCH_SPEC, *P_TBL_SEARCH_SPEC;

typedef struct TBL_FIND_ROW {
	TBL_ROW_POS			rowPos;		// In:Out - current table position
	TBL_ROW_NUM			rowNum;		// Out: indexed column row number
	TBL_SEARCH_SPEC		srchSpec;	// In: search query
	TBL_COL_INX_TYPE	sortCol;	// In: which column sort to use (if any)
	P_ROW_BUFFER		pRowBuffer; // In: pointer to client's buffer space
} TBL_FIND_ROW, *P_TBL_FIND_ROW;

/****************************************************************************
 msgTBLFindNext	takes P_TBL_FIND_ROW, returns STATUS
	Find the next record following the specified TBL_ROW_POS that meets 
	the search specification.

 Passes back the TBL_ROW_POS and TBL_ROW_NUM of the row.

 srchSpec.pConstOperand is of type P_TBL_STRING if the column type is 
 tsString, tsCaseString, or tsByteArray. The length of the string/array 
 used in the search is decalred in the strLen field of the TBL_STRING struct.
 Clients are responsible for setting this field to the appropriate length
 for columns of type tsString, tsCaseString, and tsByteArray.

 srchSpec.pConstOperand is ignored if srchSpec.relOp is tsAlwaysTrue.

 If srchSpec.colOperand is an unsorted column, then the order of the rows
 searched is random.

 Return Value
	stsTBLEndOfTable:			No data was found matching the search spec,
								or rowPos is was not found in the table.
	stsTBLInvalidSortColValue:	sortCol is not a valid column value.
*/
#define msgTBLFindNext				MakeMsg(clsTable, 4)

/****  Table Utility Messages  ****/

/****************************************************************************
 msgTBLFindColNum	takes P_TBL_COL_NUM_FIND, returns STATUS
	Passes back the column number for the specifed column name.

 Return Value
	stsTBLColNameNotFound:	A column with the specified name does not exist.
*/
#define msgTBLFindColNum			MakeMsg(clsTable, 12)

typedef struct TBL_COL_NUM_FIND {
	P_CHAR				name;		// In: Column name
	TBL_COL_INX_TYPE	number;		// Out: Column number
} TBL_COL_NUM_FIND, *P_TBL_COL_NUM_FIND;

/****************************************************************************
 msgTBLCompact	takes nothing, returns STATUS
	Compacts the table without closing it.

 This message allows clients to compact a table on demand. Compaction frees
 up any storage associated with previously deleted rows and compacts the
 table to its minimum file size. Ordinarily, a table is compacted
 automatically when the last client accessing the table closes it unless
 specifically prevented by specifying tsFreeNoCompact during msgNew.
*/
#define msgTBLCompact				MakeMsg(clsTable, 24)

/****************************************************************************
 msgTBLRowNumToRowPos	takes P_TBL_CONVERT_ROW_NUM, returns STATUS
	Converts a TBL_ROW_NUM to its corresponding TBL_ROW_POS for the
	specified column.

 This message is defined only for sorted columns. Unsorted columns do not
 have a defined order.

 Return Value
	stsTBLEndOfTable:		rowNum is larger than the number of rows in the table.
	stsTBLColNotIndexed:	The specified column is not sorted.
*/
#define msgTBLRowNumToRowPos		MakeMsg(clsTable, 28)

typedef struct TBL_CONVERT_ROW_NUM {
	TBL_ROW_POS			rowPos;		// Out:	- Table row pos.
	TBL_ROW_NUM			rowNum;		// In:	- Index row number.
	TBL_COL_INX_TYPE	colNum;		// In:	- Indexed (sorted) column number.
} TBL_CONVERT_ROW_NUM, *P_TBL_CONVERT_ROW_NUM;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Observer Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgTBLRowAdded	takes P_TBL_ROW_POS, returns STATUS
	category: observer notification
	Sent to observers indicating that a row has been added.

 A pointer to the newly added TBL_ROW_POS is sent as an argument.  
*/
#define msgTBLRowAdded				MakeMsg(clsTable, 19)

/****************************************************************************
 msgTBLRowDeleted	takes nothing, returns STATUS
	category: observer notification
	Sent to observers indicating that a row has been deleted.
*/
#define msgTBLRowDeleted			MakeMsg(clsTable, 20)

/****************************************************************************
 msgTBLRowChanged	takes P_TBL_ROW_POS, returns STATUS
	category: observer notification
	Sent to observers indicating that row data has been changed.

 A pointer to the changed TBL_ROW_POS is sent as an argument.  
*/
#define msgTBLRowChanged			MakeMsg(clsTable, 21)

#endif
