/****************************************************************************
 File: system.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.68  $
   $Author:   sisaac  $
     $Date:   19 Mar 1992 11:46:16  $

 This file contains the API definition for clsSystem.
 
 clsSystem inherits from clsObject.
 Provides information about the system.

 There is a single instance of clsSystem, theSystem. You send all 
 clsSystem messages to theSystem.

 theSystem manages PenPoint booting. If you need to know when PenPoint
 booting reaches a certain stage or is complete then you can observe
 theSystem and recieve msgBootStateChanged.	You can also send
 msgSysGetBootState to find out what stage booting is currently at.

//{
	PenPoint Booting Sequence

	Cold Boot 								Warm Boot
	----------------------------------	 	---------
	Kernel									Kernel
	System Dlls Loaded (boot.dlc)	 		System Dll Upgrade 
	System Apps Installed (sysapp.ini)		System Dlls	reinitialized
	Initial App Installed 					Instance 0's/DLLMain()s rerun
	Bookshelf Created						App Upgrade 
	Services Installed (service.ini)		Services Upgrade 
	Apps Installed (app.ini)				Run Initial App
	Run Initial App							Boot Complete
	Boot Complete
//} 

 This header file defines constants for all the interesting PenPoint
 filesystem locations that you might be tempted to hard-code. Use these
 defines instead; for example, to set a string to the location where
 PenPoint applications live, use:
//{
	strcpy(pFoo, sysBaseDir "\\" sysInstallableAppDir);
//}

 PenPoint defines "live" areas for documents on volumes. The live area is
 where the volume's bookshelf is. Use msgSysGetLiveRoot to access the live
 area on a volume.
****************************************************************************/
#ifndef SYSTEM_INCLUDED
#define SYSTEM_INCLUDED

#ifndef APPDIR_INCLUDED
#include <appdir.h>
#endif

#ifndef APPMGR_INCLUDED
#include <appmgr.h>
#endif

#ifndef UUID_INCLUDED
#include <uuid.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						System Debugging Flags							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 System debug flag is 'B', values are:
//{
	1	 = Enable active doc cache tracing
	2	 = Install items from theSelectedVolume at warm boot
	4	 = Go into debugger when stdmsg functions are called
	8	 = Enable serial port option sheet testing
	800	 = Enable showing of the RAM (theSelectedVolume) Volume
//}
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  		Common #defines and typedefs  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* penpoint.res is invalid. This is checked during cold and warm boot. */
#define  stsSysInvalidSystemResFile			MakeStatus(clsSystem, 1)

/* Penpoint base directory. */
#define sysBaseDir					"PENPOINT"

/* Filesystem locations off the base Penpoint directory. */
#define	sysInstallableFontDir		"FONT"
#define	sysInstallablePrefDir		"PREFS"
#define sysInstallableHWXProtDir	"HWXPROT"
#define sysInstallableGestureDir	"GESTURE"
#define sysInstallablePDictDir		"PDICT"
#define sysInstallableAppDir		"APP"
#define sysInstallableServiceDir	"SERVICE"
#define sysBootDir					"BOOT"
#define sysQuickInstall				"QINSTALL"

#define sysRuntimeRootDir			"SYS"

/* Filesystem locations off the runtime root. */
#define	sysSysAppFile				"SYSAPP.INI"
#define	sysAppFile					"APP.INI"
#define	sysSysServiceFile 			"SYSSERV.INI"
#define	sysServiceFile				"SERVICE.INI"
#define sysCopyFile					"SYSCOPY.INI"
#define	sysResFile					"PENPOINT.RES"
#define	sysMILResFile				"MIL.RES"
#define sysLiveRoot					"Bookshelf"
#define sysLoaderDir				"LOADER"

/* Default initial app (in penpoint\boot\app). */
#define	sysDefaultInitialApp		"Bookshelf"

/* Boot type. */
typedef enum SYS_BOOT_TYPE {
	sysWarmBoot					= 1,
	sysColdBoot					= 2
} SYS_BOOT_TYPE, *P_SYS_BOOT_TYPE;

/* Boot progess. */
typedef enum SYS_BOOT_PROGRESS {
	sysKernelComplete			= 1,
	sysSystemDllsComplete		= 2,
	sysSystemAppsInstalled		= 3,
	sysInitialAppInstalled		= 4,
	sysBookshelfItemsCreated	= 5,
	sysServicesInstalled		= 6,
	sysAppsInstalled			= 7,
	sysInitialAppRunning		= 8,	
	sysBootComplete				= 9	
} SYS_BOOT_PROGRESS, *P_SYS_BOOT_PROGRESS;

/* Boot state. */
typedef struct SYS_BOOT_STATE {
	BOOLEAN				booted;				// Has booting totally completed?
	SYS_BOOT_PROGRESS	progress;			// Where are we in the boot cycle?
	SYS_BOOT_TYPE		type;				// Boot type; warm or cold.
	CLASS				initialAppClass;	// Class of the initial app.
} SYS_BOOT_STATE, *P_SYS_BOOT_STATE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  				  	Messages  		  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_SYS_NEW, returns STATUS
	category: class message
	Used by PenPoint to create well-known uid theSystem.

 This message should never be called by anybody else.
*/

typedef struct SYS_NEW_ONLY {
	U32					unused1;
	U32					unused2;
	U32					unused3;
	U32					unused4;
} SYS_NEW_ONLY, *P_SYS_NEW_ONLY;

#define	systemNewFields					\
	objectNewFields			  			\
	SYS_NEW_ONLY   		system;

typedef struct SYS_NEW {
	systemNewFields
} SYS_NEW, *P_SYS_NEW;


/****************************************************************************
 msgSysGetBootState				takes P_SYS_BOOT_STATE, returns STATUS
	What stage of booting is the system in?
 
 This message allows callers to determine the current state of system 
 booting.

 See Also 
	msgSysBootStateChanged			Observer message sent at each stage.
*/
#define msgSysGetBootState		 				MakeMsg(clsSystem, 1)


/****************************************************************************
 msgSysGetRuntimeRoot			takes P_OBJECT, returns STATUS
	Returns a dir handle onto the root of the Penpoint runtime area.
 
 Penpoint maintains all of its runtime information in one area of the 
 filesystem on the "selected" volume (theSelectedVolume). This message 
 returns a directory handle onto the root of this area.

 NOTE: Caller must free the handle when finished.
*/
#define msgSysGetRuntimeRoot	 				MakeMsg(clsSystem, 2)


/****************************************************************************
 msgSysGetLiveRoot				takes P_SYS_GET_LIVE_ROOT, returns STATUS
	Returns an appDir handle onto the root of a volume's live document area.
 
 Live Penpoint documents (those that can be activated) are stored within
 the live area of a volume. This message returns the root of the live area
 for a given volume.

 pArgs->volHandle is a filesystem handle onto the volume in question. This
 handle can be on any location of the volume. You can also use the root
 directory handle for a volume. Use theSelectedVolume if you want to get
 the live area within the filesystem that Penpoint stores its on-machine
 documents in.

 NOTE: Caller must free the pArgs->liveHandle when finished.

 Return Value
	stsFSNodeNotFound:		No live root on this volume.
*/
#define msgSysGetLiveRoot		 				MakeMsg(clsSystem, 3)

typedef struct SYS_GET_LIVE_ROOT {
	OBJECT				volHandle;	// Handle onto volume in question.
	OBJECT				liveRoot;	// Out: appDir handle to live root on 
									//  the volume.
} SYS_GET_LIVE_ROOT, *P_SYS_GET_LIVE_ROOT;


/****************************************************************************
 msgSysIsHandleLive		takes P_SYS_IS_HANDLE_LIVE, returns STATUS
	Determines if a filesystem handle is within the live document area.

 Penpoint maintains live documents within a particular point in the
 directory heirarchy of each volume. This message determines whether a
 filesystem handle is within the live area of its volume.

 Return Value
	stsFSNodeNotFound:		No live root on the handle's volume.
*/
#define msgSysIsHandleLive		 				MakeMsg(clsSystem, 4)

typedef struct SYS_IS_HANDLE_LIVE {
	OBJECT				handle;		// Handle onto the node in question.
	BOOLEAN				live;		// Out: Is it in the live area?
} SYS_IS_HANDLE_LIVE, *P_SYS_IS_HANDLE_LIVE;


/****************************************************************************
 msgSysCreateLiveRoot	takes P_SYS_CREATE_LIVE_ROOT, returns STATUS
	Create a new live root on a volume.

 Penpoint maintains live documents within a particular point in the
 directory heirarchy of each volume. This message creates a new live root
 on a volume if one doesn't already exist. If the live root already
 exists it creates an instance of the app over whatever is there
 currently. Use msgSysGetLiveRoot if you want to check for an existing
 live root.
*/
#define msgSysCreateLiveRoot	 				MakeMsg(clsSystem, 5)

typedef struct SYS_CREATE_LIVE_ROOT {
	OBJECT				volHandle;	// Handle onto volume in question.
	CLASS				rootClass;	// Class of app which should run on the
									// live root directory.
} SYS_CREATE_LIVE_ROOT, *P_SYS_CREATE_LIVE_ROOT;


/****************************************************************************
 msgSysGetVersion				takes P_U16, returns STATUS
	Returns the system version number.
 
 This message allows callers to determine the current PenPoint system
 version number.
*/
#define msgSysGetVersion			 			MakeMsg(clsSystem, 6)


/****************************************************************************
 msgSysGetSecurityObject			takes P_OBJECT, returns STATUS
 	Gets the current security object.

 Returns objNull if there is no current security object.
*/
#define msgSysGetSecurityObject 	 	 		MakeMsg(clsSystem, 31)


/****************************************************************************
 msgSysSetSecurityObject	takes P_SYS_SET_SECURITY_OBJECT, returns STATUS
 	Sets the current security object.

 If a security object already exists then it is destroyed, using the
 key specified in the arguments. If it refuses to be destroyed then
 the new security object will not be set.

 The security object will be sent msgSysPowerOn and msgSysPowerOff when
 the power goes on and off.  At shutdown, msgSysPowerOff is sent to the
 security object after msgSysPowerOff is sent to power button observers 
 and after msgAppSave is sent to applications.  At power up, msgSysPowerOn 
 is sent to the security object before msgSysPowerOn is sent to power 
 button observers.

 msgSysPowerOn and msgSysPowerOff are sent when the machine is suspended/
 resumed, or shutdown and swap-booted. However, these messages are not
 sent when a warm-boot occurs. A warm-boot destroys all processes and 
 objects. The service or application that owns the security object will
 be restarted in the warm-boot case. Security objects must handle
 the warm-boot case. For example, if the security object is created by
 the app monitor, the app monitor will receive msgAppInit when the 
 application is first installed and msgRestore on all warm-boots. 

 At power down, anything painted on the screen by the security object
 will not appear immediately, but will appear on the screen when it is
 restored at power on time.  If the security object wishes to display
 a window on top of all other windows, it should observe theSystem for
 msgBootStateChanged to determine when booting is complete.
 
 At power on, the security object may choose to veto the powering on of 
 the system by sending msgPMSetPowerState to thePowerMgr to turn off power.

 Return Value
	stsProtectionViolation:	 old security object refused to be destroyed.
*/
#define msgSysSetSecurityObject 	 	 		MakeMsg(clsSystem, 32)

typedef struct SYS_SET_SECURITY_OBJECT {
	OBJECT				securityObject;	// New security object.
	OBJ_KEY				oldKey;			// Object key for old security
										// object.
} SYS_SET_SECURITY_OBJECT, *P_SYS_SET_SECURITY_OBJECT;


/****************************************************************************
 msgSysGetCorrectiveServiceLevel	takes P_STRING, returns STATUS
 	Gets the corrective service level.

 The corrective service level is a string of up to maxNameLength 
 characters.
*/
#define msgSysGetCorrectiveServiceLevel	   		MakeMsg(clsSystem, 33)


/****************************************************************************
 msgSysSetCorrectiveServiceLevel	takes P_STRING, returns STATUS
 	Sets the corrective service level.

 The corrective service level is a string of up to maxNameLength 
 characters.
*/
#define msgSysSetCorrectiveServiceLevel	   		MakeMsg(clsSystem, 34)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  			Notification Messages	  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSysBootStateChanged		   	takes P_SYS_BOOT_STATE, returns STATUS
	category: observer notification
	The system has reached another stage of booting.

 This message is sent to all observers of theSystem whenever another
 stage of booting is attained. If you are just interested in whether the
 system has completed booting or not, look at the pArgs->booted boolean.
*/
#define msgSysBootStateChanged 					MakeMsg(clsSystem, 10)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Private																   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Name of file in \penpoint that signifies the disk is a golden master */
#define sysGoldenMaster						"GOLDEN"

/* Class of initial application. Stored on the root of the runtime. */
/* Existence signifies that a warm boot is happening. */
#define sysAttrInitialAppClass				FSMakeFix32Attr(clsSystem, 0)

/* Corrective service level. */
#define sysAttrCorrectiveServiceLevel		FSMakeStrAttr(clsSystem, 1)

/* Memory status. */
#define sysAttrNonSwapAtBoot				FSMakeFix32Attr(clsSystem, 2)
#define sysAttrSystemAtBoot					FSMakeFix32Attr(clsSystem, 3)
#define sysAttrSwapFileSizeAtBoot	 		FSMakeFix32Attr(clsSystem, 4)
#define sysAttrSwapFileFreeAtBoot	 		FSMakeFix32Attr(clsSystem, 5)

/* Dialog tags */
#define tagSysActiveDocs				   	MakeDialogTag(clsSystem, 0)
#define tagSysLastDoc					   	MakeDialogTag(clsSystem, 1)
#define tagSysDeinstallAndWarmBoot 		   	MakeDialogTag(clsSystem, 2)

/****************************************************************************
 msgSysUpdateBootProgress 	takes P_SYS_UPDATE_BOOT_PROGRESS, returns STATUS
	Update the state of system booting.

 This message is sent by various internal components as each stage of the 
 boot is reached.
*/
#define msgSysUpdateBootProgress					MakeMsg(clsSystem, 20)

typedef struct SYS_UPDATE_BOOT_PROGRESS {
	SYS_BOOT_PROGRESS	progress;
	P_ARGS				pArgs;
} SYS_UPDATE_BOOT_PROGRESS, *P_SYS_UPDATE_BOOT_PROGRESS;


/****************************************************************************
 msgSysGetDocAttr			takes P_SYS_GET_SET_DOC_ATTR, returns STATUS
	Gets the uid of an active document.

 Return Value
	stsNoMatch	 	Attribute set for this document not found.
*/
#define msgSysGetDocAttr	 						MakeMsg(clsSystem, 21)

// Document's cached attributes. 
typedef struct SYS_DOC_ATTRS {
	UUID				uuid;	// doc's uuid. Must *always* be specified 
								//   by caller.
	OBJECT				uid;	// In: (SetDocAttr) Out: (GetDocAttr) doc's 
								//   uid.
	U8					unused[40]; // MUST be set to 0 by caller.
} SYS_DOC_ATTRS, *P_SYS_DOC_ATTRS;

typedef struct SYS_GET_SET_DOC_ATTR {
	FS_LOCATOR			docLocation;   // (SetDocAttr only) Filesystem 
									   // location of doc. Must	not use
									   // an absolute path here.
	SYS_DOC_ATTRS		docAttrs;
} SYS_GET_SET_DOC_ATTR, *P_SYS_GET_SET_DOC_ATTR;


/****************************************************************************
 msgSysSetDocAttr			takes P_SYS_GET_SET_DOC_ATTR, returns STATUS
	Set runtime attributes for an active document.

*/
#define msgSysSetDocAttr	 				   		MakeMsg(clsSystem, 22)


/****************************************************************************
 msgSysGetActiveDocs		takes P_SYS_GET_ACTIVE_DOCS, returns STATUS
	Gets the uuids of all the active docs (including app monitors).

 Caller MUST OSHeapBlockFree *ppUUID when done.

*/
#define msgSysGetActiveDocs		 					MakeMsg(clsSystem, 23)

typedef struct SYS_GET_ACTIVE_DOCS {
	P_UUID				pUUID;  // Out: pointer to UUID array. Allocated by
								//   theSystem.	Must be freed by caller.
	U16					num;	// Out: Number of active docs.
} SYS_GET_ACTIVE_DOCS, *P_SYS_GET_ACTIVE_DOCS;


/****************************************************************************
 msgSysPowerOff					takes void, returns STATUS
 	Checkpoints all active apps (including app monitors).
	
*/
#define msgSysPowerOff				 	 	 		MakeMsg(clsSystem, 40)


/****************************************************************************
 msgSysPowerOn					takes void, returns STATUS
 	The power has come back on.
	
*/
#define msgSysPowerOn				 	 	 		MakeMsg(clsSystem, 41)


/****************************************************************************
 msgSysPowerOffDevice			takes U16, returns STATUS
	Tells a MIL service that power is about to go off.

 pArgs is the logical id of the device. If pArgs is set to maxU16 then all 
 devices will be informed that the power is going off.
*/
#define msgSysPowerOffDevice	 		 	 		MakeMsg(clsSystem, 42)


/****************************************************************************
 msgSysPowerOnDevice			takes U16, returns STATUS
	Tells a MIL service that power has just come on.

 pArgs is the logical id of the device. If pArgs is set to maxU16 then all 
 all devices will be informed that the power has just come on.

*/
#define msgSysPowerOnDevice			 	 	 		MakeMsg(clsSystem, 43)


/****************************************************************************
 msgSysLastDoc					takes P_UUID, returns STATUS
	Is this the last document of its kind in Accessories and Stationery?

 pArgs is the uuid of the document. Returns stsOK if this is the
 last document; stsFailed if it is not.

 Clients should call StdMsg(tagSysLastDoc, pDocName) if this is the last
 doc.
*/
#define msgSysLastDoc			 	 		 		MakeMsg(clsSystem, 44)


/****************************************************************************
 msgSysDeinstallAndWarmBoot			takes IM_HANDLE, returns STATUS
	Private message for update application.	

 Allows the update application to deinstall itself and have the system
 warm-boot.	This message must be posted.

 Asks the user if he wants to warm boot at this time.
*/
#define msgSysDeinstallAndWarmBoot	 		 		MakeMsg(clsSystem, 45)


/****************************************************************************
 msgSysMemoryStatus			takes P_SYS_MEMORY_STATUS, returns STATUS
	Returns memory and disk information for the memory status card.

 This interface should only be used by the memory status card in Settings.
*/
#define msgSysMemoryStatus		 	 			MakeMsg(clsSystem, 7)

typedef struct SYS_MEMORY_STATUS {
	U32					nonSwapAtBoot;
	U32					systemAtBoot;
	U32					swapFileSizeAtBoot;
	U32					swapFileFreeAtBoot;
	U32					installedSoftware;
	U32					totalBytes;
	U32					freeBytes;
	U8					reserved[16];
} SYS_MEMORY_STATUS, *P_SYS_MEMORY_STATUS;


/****************************************************************************
 msgSysShowMemoryUsage			takes P_STRING, returns STATUS
	Prints out current system-wide memory usage.

 Argument is a text string to display.
*/
#define msgSysShowMemoryUsage		 	 		MakeMsg(clsSystem, 8)


/****************************************************************************
 msgSysShowMemoryUsageDelta			takes P_STRING, returns STATUS
	Prints out system memory usage delta since last ShowMemoryUsage 
    or ShowMemoryUsageDelta.

 Argument is text string to display.
*/
#define msgSysShowMemoryUsageDelta	 	 		MakeMsg(clsSystem, 9)


/****************************************************************************
 msgSysShowTaskMemoryUsage			takes P_STRING, returns STATUS
	Prints out memory usage for OSThisTask().

 Argument is a text string to display.
*/
#define msgSysShowTaskMemoryUsage 	 	 		MakeMsg(clsSystem, 30)


/****************************************************************************
 msgSysMove					takes P_SYS_MOVE_COPY, returns STATUS
	Performs an appMgrMove with source and destination locators.

 The appMgr move/copy messages requires a destination object, sending
 msgEmbeddedWinGetDest to find out the actual destination location. This
 requires clients to keep track of the destination location in its
 instance data, a somewhat cumbersome approach. This message allows a
 client to simply make one call.
*/
#define msgSysMove						 		MakeMsg(clsSystem, 35)

typedef struct SYS_MOVE_COPY {
	OBJECT					caller;				// Caller, who will recieve
												// msgXXX
	APP_MGR_MOVE_COPY_STYLE	style;				// Move/copy style.
	FS_LOCATOR				source;				// Source document location.
	FS_LOCATOR				dest;				// Destination doc location.
	CHAR					name[nameBufLength];// in:out New doc name;
	BOOLEAN					renumber;			// true=update global sequence #s.
} SYS_MOVE_COPY, *P_SYS_MOVE_COPY;


/****************************************************************************
 msgSysCopy				takes P_SYS_MOVE_COPY, returns STATUS
	Performs an appMgrCopy with source and destination locators.

 The appMgr move/copy messages requires a destination object, sending
 msgEmbeddedWinGetDest to find out the actual destination location. This
 requires clients to keep track of the destination location in its
 instance data, a somewhat cumbersome approach. This message allows a
 client to simply make one call.
*/
#define msgSysCopy						 		MakeMsg(clsSystem, 36)

// Next Message: 37

#endif	// SYSTEM_INCLUDED


