/****************************************************************************
 File: swin.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.62  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 14:47:02  $

 This file contains the API definition for clsScrollWin.

 clsScrollWin inherits from clsBorder.
 A scrollWin positions, sizes, and displays a client window (optionally
 part of a "deck" of other child windows) together with optional scrollbars,
 and can scroll the client window by repositioning it.
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsScrollWin debugging flag is '%'.  Defined values are:

    flag6  (0x0040):    layout
*/

#ifndef SWIN_INCLUDED
#define SWIN_INCLUDED

												#ifndef SBAR_INCLUDED
#include <sbar.h>
												#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT SCROLL_WIN;

/****  Client styles for scrollbar client  ****/
#define	swClientScrollWin		0	// scrollWin is the scrollbar client
#define	swClientWin				1	// clientWin is the scrollbar client
#define	swClientOther			2	// scrollWin will not set the client
//								3	// unused (reserved)

/****  Alignment styles  ****/
#define	swAlignLeft				0				// left-justified
#define	swAlignCenter			1				// centered
#define	swAlignRight			2				// right-justified
#define	swAlignTop				swAlignLeft		// top-justified
#define	swAlignBottom			swAlignRight  	// bottom-justified
#define	swAlignSelf				3				// clientWin will align itself

/****  Forward styles  ****/
#define	swForwardNone			0				// don't forward anything
#define	swForwardGesture		1				// forward msgGWinGesture
#define	swForwardXList			2				// forward msgGWinXList

typedef struct SCROLL_WIN_STYLE {
	U16	vertScrollbar		: 1,	// vertical scrollbar on/off
		horizScrollbar		: 1,	// horizontal scrollbar on/off
		autoVertScrollbar	: 1,	// vert scrollbar on/off based on clientWin
		autoHorizScrollbar 	: 1,	// horiz scrollbar on/off based on clientWin
		maskScrollbars		: 1,	// mask out vertScrollbar and horizScrollbar
		expandChildWidth 	: 1,	// expand the child's width to avail width
		expandChildHeight 	: 1,	// expand the child's height to avail height
		contractChildWidth 	: 1,	// contract the child's width to avail width
		contractChildHeight : 1,	// contract the child's height to avail height
   		getDelta			: 1,	// send msgScrollWinProvideDelta to client
		getSize				: 1,	// send msgScrollWinProvideSize
		wideVertScrollbar	: 1,	// make the vertical scrollbar wide
		wideHorizScrollbar	: 1,	// make the horizontal scrollbar wide
		forward				: 2,	// what to forward from margins to clientWin
		maskAll				: 1;	// mask out maskScrollbars

	U16	xAlignment		: 2,	// x Alignment if innerWin wider than clientWin
		yAlignment			: 2,	// y Alignment if innerWin taller than clientWin
		vertClient			: 2,	// choice of vertical sb client
		horizClient			: 2,	// choice of horizontal sb client
      xAlignRigorous    : 1,    // use xAlignment continuously
      yAlignRigorous    : 1,    // use yAlignment continuously
      private1          : 1,  // private
		spare1				: 5;	// unused (reserved)

	U16	spare2				: 16;	// unused (reserved)
} SCROLL_WIN_STYLE, *P_SCROLL_WIN_STYLE;

/*
 Default SCROLL_WIN_STYLE:
//{
 	getDelta			= false
 	vertScrollbar		= false
 	horizScrollbar		= false
 	autoVertScrollbar	= true
 	autoHorizScrollbar	= true
 	expandChildWidth	= false
 	expandChildHeight	= false
 	vertClient			= swClientScrollWin
 	horizClient			= swClientScrollWin
 	xAlignment			= swAlignLeft
 	yAlignment			= swAlignTop
 	getSize				= false
 	contractChildWidth	= false
 	contractChildHeight	= false
 	forward				= swForwardNone
 	maskScrollbars		= false
 	xAlignRigorous		= true
 	yAlignRigorous		= true
//}
*/

/*
 The x- and yAlignment styles are used primarily when the innerWin is
 wider/taller than the clientWin.  However, they are also used when
 a clientWin that is wider/taller than the innerWin is changing size.
 In this case, the innerWin alters the origin to compensate for the size
 change so that the appropriate edge of the clientWin is held fixed
 (either by doing the math itself or sending out msgScrollWinAlign if the
 alignment is set to swAlignSelf).  An example: a top-aligned clientWin
 of height 100 in an innerWin of height 50 is growing by 20.  The innerWin
 would subtract 20 from the clientWin's new origin.y.

 Clients can disable the adjustments that occur in the second case
 (clientWin is wider/taller than the innerWin) by setting the appropriate
 x- or yAlignRigorous flag to false.
 */

typedef struct SCROLL_WIN_METRICS {
	SCROLL_WIN_STYLE	style;				// style bits
	OBJECT				client;			    // for msgScrollWinProvideDelta
	WIN					clientWin; 		   	// current window to scroll
	U16					colDelta, rowDelta;	// metrics in device units
	U32					spare1;				// unused (reserved)
	U32					spare2;				// unused (reserved)
} SCROLL_WIN_METRICS, *P_SCROLL_WIN_METRICS;

typedef struct SCROLL_WIN_DELTA {
	SCROLL_WIN			scrollWin;	// in:  requesting scroll win
	SCROLLBAR_ACTION	action;		// in:  action to resolve
	S32					offset; 	// in:  current or new offset
	RECT32				viewRect;	// in/out:  viewable portion of clientWin
	S32					lineCoord;	// in:  line coordinate, if any
	U32					spare;		// unused (reserved)
} SCROLL_WIN_DELTA, *P_SCROLL_WIN_DELTA;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_SCROLL_WIN_NEW, returns STATUS
	category: class message
	Creates a scrollWin.

 The fields you commonly set are:
	pArgs->scrollWin.style:			appropriate style values
	pArgs->scrollWin.clientWin:		a window to scroll
*/

typedef SCROLL_WIN_METRICS	SCROLL_WIN_NEW_ONLY, *P_SCROLL_WIN_NEW_ONLY;

#define	scrollWinNewFields	\
	borderNewFields			\
	SCROLL_WIN_NEW_ONLY		scrollWin;

typedef struct SCROLL_WIN_NEW {
	scrollWinNewFields
} SCROLL_WIN_NEW, *P_SCROLL_WIN_NEW;


/****************************************************************************
 msgNewDefaults				takes P_SCROLL_WIN_NEW, returns STATUS
	category: class message
	Initializes the SCROLL_WIN_NEW structure to default values.

 Zeroes out pArgs->scrollWin and sets:
//{
	pArgs->win.flags.style |= wsSendFile | wsClipChildren | wsClipSiblings;
	pArgs->win.flags.style &= ~wsParentClip;

	pArgs->scrollWin.style.autoVertScrollbar = true;
	pArgs->scrollWin.style.autoHorizScrollbar = true;
	pArgs->scrollWin.style.xAlignRigorous =
	pArgs->scrollWin.style.yAlignRigorous = true;
	pArgs->scrollWin.colDelta = 10;
	pArgs->scrollWin.rowDelta = 10;
//}
*/


/****************************************************************************
 msgScrollWinGetStyle		takes P_SCROLL_WIN_STYLE, returns STATUS
	Passes back the current style values.
*/
#define	msgScrollWinGetStyle		MakeMsg(clsScrollWin, 13)


/****************************************************************************
 msgScrollWinSetStyle		takes P_SCROLL_WIN_STYLE, returns STATUS
	Sets the style values.

 The scrollWin self-sends msgWinSetLayoutDirty(true).  It is the caller's
 responsibility to re-layout the scrollWin.
*/
#define	msgScrollWinSetStyle		MakeMsg(clsScrollWin, 14)


/****************************************************************************
 msgScrollWinGetMetrics		takes P_SCROLL_WIN_METRICS, returns STATUS
	Passes back the metrics.
*/
#define	msgScrollWinGetMetrics		MakeMsg(clsScrollWin, 1) 


/****************************************************************************
 msgScrollWinSetMetrics		takes P_SCROLL_WIN_METRICS, returns STATUS
	Sets the metrics.
*/
#define	msgScrollWinSetMetrics		MakeMsg(clsScrollWin, 2) 


/****************************************************************************
 msgScrollWinGetClientWin		takes P_WIN, returns STATUS
	Passes back the current clientWin.

 The current clientWin is the last window to be shown using
 msgScrollWinShowClientWin.
*/
#define	msgScrollWinGetClientWin 	MakeMsg(clsScrollWin, 3) 


/****************************************************************************
 msgScrollWinShowClientWin		takes WIN, returns STATUS
	Sets the current clientWin; the specified window is be made visible.

 If P_ARGS is not a child of the scrollWin's inner window,
 msgScrollWinAddClientWin is self-sent followed by msgWinLayout.
*/
#define	msgScrollWinShowClientWin	MakeMsg(clsScrollWin, 4) 


/****************************************************************************
 msgScrollWinAddClientWin		takes WIN, returns STATUS
	Adds another clientWin, inserting the specified window as a child of 
	the scrollWin's inner window.

 The specified window is set to be invisible (window flag wsVisible off).
*/
#define	msgScrollWinAddClientWin	MakeMsg(clsScrollWin, 11)


/****************************************************************************
 msgScrollWinRemoveClientWin		takes WIN, returns STATUS
	Extracts the specified child window from the scrollWin.
*/
#define	msgScrollWinRemoveClientWin	MakeMsg(clsScrollWin, 12)


/****************************************************************************
 msgScrollWinGetVertScrollbar		takes P_WIN, returns STATUS
	Passes back the vertical scrollbar.
*/
#define	msgScrollWinGetVertScrollbar	MakeMsg(clsScrollWin, 6)


/****************************************************************************
 msgScrollWinGetHorizScrollbar		takes P_WIN, returns STATUS
	Passes back the horizontal scrollbar.
*/
#define	msgScrollWinGetHorizScrollbar	MakeMsg(clsScrollWin, 7)


/****************************************************************************
 msgScrollWinGetInnerWin		takes P_WIN, returns STATUS
	Passes back the inner window of the scrollWin.
*/
#define	msgScrollWinGetInnerWin			MakeMsg(clsScrollWin, 9)


/****************************************************************************
 msgScrollWinProvideDelta		takes P_SCROLL_WIN_DELTA, returns STATUS
	category: descendant/client responsibility
	Self-sent when style.getDelta is set to true so that descendant or
	client can normalize the scroll if desired.

 clsScrollWin responds by forwarding this message to the current clientWin.
 If you receive this message, you can send msgScrollWinGetDefaultDelta
 to pArgs->scrollWin to fill out pArgs with the default scrollWin response.
*/
#define	msgScrollWinProvideDelta		MakeMsg(clsScrollWin, 5)


/****************************************************************************
 msgScrollWinProvideSize		takes P_SCROLL_WIN_SIZE, returns STATUS
	category: descendant/client responsibility
	Self-sent to determine bubble location and size.

 clsScrollWin responds by forwarding to the current clientWin
 (if style.getSize is true), or sending msgWinGetDesiredSize to the
 current clientWin (if style.getSize is false).  In the latter case
 if there is no current clientWin, clsScrollWin uses 0 for docSize.

 A clientWin responding to msgScrollWinProvideSize should fill out
 pArgs->viewSize and pArgs->docSize.
*/
#define	msgScrollWinProvideSize			MakeMsg(clsScrollWin, 10)

typedef struct SCROLL_WIN_SIZE {
	SCROLL_WIN			scrollWin;	// in:  requesting scroll win
	SIZE32				viewSize;	// out: desired view size (device units)
	SIZE32				docSize;	// out: logical doc size (device units)
	U32					spare;		// unused (reserved)
} SCROLL_WIN_SIZE, *P_SCROLL_WIN_SIZE;


/****************************************************************************
 msgScrollWinCheckScrollbars	takes P_BOOLEAN, returns STATUS
	Determines whether the on/off state of either scrollbar needs to change
	and passes back the result.

 Clients wishing to fix up the states should dirty the layout of the 
 scrollWin and then send msgWinLayout.
*/
#define	msgScrollWinCheckScrollbars		MakeMsg(clsScrollWin, 15)


/****************************************************************************
 msgScrollWinAlign		        takes P_SCROLL_WIN_ALIGN, returns STATUS
	category: client responsibility
 	Sent to client when style.xAlignment or style.yAlignment is swAlignSelf.

 clsScrollWin sends this message to the scrollWin's client or clientWin
 if the client is objNull.  This message is sent when the child window
 changes size or the scrollWin inner window changes size.  See the comment
 after "Default SCROLL_WIN_STYLE" for a further description of alignment.
*/
#define	msgScrollWinAlign		MakeMsg(clsScrollWin, 16)

typedef struct SCROLL_WIN_ALIGN {
	SCROLL_WIN			scrollWin;	// in:  requesting scroll win
    BOOLEAN             alignX;     // in:  x dimension (false for y)
	RECT32				innerRect;	// in/out:  rect of innerWin, device units
	RECT32				clientRect;	// in/out:  rect of clientWin, device units
	U32					spare;		// unused (reserved)
} SCROLL_WIN_ALIGN, *P_SCROLL_WIN_ALIGN;


/****************************************************************************
 msgScrollWinGetDefaultDelta        takes P_SCROLL_WIN_DELTA, returns STATUS
 	Compute the default response to msgScrollWinProvideDelta.

 You can send this message to a scrollWin to compute the default scroll
 values for a given scrolling action.

 See Also
 	msgScrollWinProvideDelta
*/
#define	msgScrollWinGetDefaultDelta     MakeMsg(clsScrollWin, 17)


/****************************************************************************
 msgScrollWinRefreshSize            takes P_SIZE32, returns STATUS
    Informs the receiver that msgScrollWinProvideSize would now return
    different size.

 A client would send this to a scrollWin when the scrollWin has style.getSize
 set true and the client would now return different size in response to
 msgScrollWinProvideSize.  The client passes the current size, and the
 scrollWin sends msgScrollWinProvideSize to the client (or the clientWin
 if that's null), then adjust the positions as if the clientWin had changed
 size.

 clsScrollWin just returns stsOK if style.getSize is false.
*/
#define	msgScrollWinRefreshSize     MakeMsg(clsScrollWin, 18)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 clsScrollWin responds by filing away all its state, including any child
 windows that have wsSendFile turned on (wsSendFile is the default for
 clsBorder and its descendents).
*/


/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 clsScrollWin responds by restoring all of its state, including the child
 windows that were filed with the last msgSave.
*/


/****************************************************************************
 msgWinLayoutSelf takes P_WIN_METRICS, returns STATUS
    Tell a window to layout its children.

 The scrollWin first gets the dimensions of its current clientWin by
 using msgWinGetDesiredSize.  If there is no current clientWin, the
 scrollWin uses a width and height of 0 in its computations.

 If the scrollWin did not shrinkwrap around the current clientWin,
 then the expandChild* and contractChild* styles come into play.
 If the clientWin's width is less than the width of the scrollWin's inner
 window (a direct child of the scrollWin that serves as a clipping window)
 and expandChildWidth is true, then the clientWin's width is expanded
 to fit.  If the clientWin's width is greater than the inner window's and
 contractChildWidth is true, then the clientWin's width is reduced
 to fit.  These rules hold for the height as well.

 Finally, if the clientWin's (possibly modified) width is still less than
 the inner window's, then the xAlignment style is used to place the
 clientWin within the inner window.  This is also done in the vertical
 direction using yAlignment.

 The scrollWin adds or remove a vertical scrollBar as necessary if
 style.autoVertScrollbar is on, and the same is done for the horizontal
 direction (when both style.maskScrollbars and style.maskAll are off).
*/


/****************************************************************************
 msgWinSetFlags takes P_WIN_METRICS, returns STATUS
    Sets the window flags.

 clsScrollWin responds by first propagating the shrinkwrap values to
 the inner window, then calling its ancestor.
*/


/****************************************************************************
 msgWinSend 	takes WIN_SEND, returns STATUS
    Sends a message up a window ancestry chain.

 clsScrollWin responds by checking to see if pArgs->msg is msgScrollbarUpdate.
 If so, the scrollWin sends msgScrollbarUpdate to both of its
 scrollbars and then return stsOK.  If not, then the scrollWin just
 calls its ancestor.
*/


/****************************************************************************
 msgGWinGesture		takes P_GWIN_GESTURE, returns STATUS
	Self-sent to process the gesture.

 If there is a current clientWin and style.forward is swForwardGesture,
 then the pArgs are transformed to the clientWin, and the clientWin is sent
 msgGWinGesture.  The scrollWin returns the resulting status
 to the caller.

 Otherwise, the scrollWin compares the pArgs->msg with the state of
 the corresponding scrollbar.  If the message is not a scrolling gesture,
 then the scrollWin returns stsMessageIgnored.  If it is a vertical
 scrolling gesture and the vertical scrollbar is not active, then the
 scrollWin returns stsOK.  Finally, if the message is a vertical scrolling
 gesture and the vertical scrollbar is active, the scrollWin transforms
 the pArgs to the scrollbar's space and return the result of sending
 msgGWinGesture to the scrollbar (unless the msgGWinGesture originated
 with the scrollbar, i.e. pArgs->uid == the scrollbar -- in this case
 the scrollWin returns stsMessageIgnored).  This processing is also done
 for horizontal scrolling gestures and the horizontal scrollbar.

 The above processing also is done whenever the scrollWin's inner window
 receives msgGWinGesture.

 Return Value
	stsOK:				a scrolling gesture would have had to be sent to
						an inactive scrollbar.
	stsMessageIgnored:	not a scrolling gesture, or message originated
						with the scrollbar to which msgGWinGesture would
						be sent.
*/


/****************************************************************************
 msgGWinForwardedGesture	takes P_GWIN_GESTURE, returns STATUS
	Message recieved when object is forwarded a gesture.

 The scrollWin compares the pArgs->msg with the state of the
 corresponding scrollbar.  If the message is not a scrolling gesture,
 then the scrollWin returns stsMessageIgnored.  If it is a vertical
 scrolling gesture and the vertical scrollbar is not active, then the
 scrollWin returns stsOK.  Finally, if the message is a vertical scrolling
 gesture and the vertical scrollbar is active, the scrollWin transforms
 the pArgs to the scrollbar's space and return the result of sending
 msgGWinGesture to the scrollbar (unless the msgGWinGesture originated
 with the scrollbar, i.e. pArgs->uid == the scrollbar -- in this case
 the scrollWin returns stsMessageIgnored).  This processing is also done
 for horizontal scrolling gestures and the horizontal scrollbar.

 The above processing also is done whenever the scrollWin's inner window
 receives msgGWinGesture.

 Return Value
	stsOK:				a scrolling gesture would have had to be sent to
						an inactive scrollbar.
	stsMessageIgnored:	not a scrolling gesture, or message originated
						with the scrollbar to which msgGWinGesture would
						be sent.
*/


/****************************************************************************
 msgGWinXList	takes P_XLIST, returns STATUS
	Call back to announce gesture translation completed.

 If there is no current clientWin, or style.forward is not swForwardXList,
 then the scrollWin just calls its ancestor.

 Otherwise, the scrollWin transforms the pArgs to the clientWin and
 return the result of sending msgGWinXList to the clientWin.

 The above processing also is done whenever the scrollWin's inner window
 receives msgGWinXList.
*/


/****************************************************************************
 msgScrollbarVertScroll		takes P_SCROLLBAR_SCROLL, returns STATUS
	category: client responsibility
	Client should perform vertical scroll.

 Responding to this message is one of the key functions that scrollWins
 provide.  Since the default scrollWin style.vertClient and .horizClient
 are both swClientScrollWin, it is usually the case that the scrollbars
 send their scrolling messages to the scrollWin.

 If there is no current clientWin, or the pArgs->action is sbEndScroll,
 the scrollWin just returns stsOK.

 If style.getDelta is true, the scrollWin sends msgScrollWinProvideDelta
 to the client (if that is non-null) or the clientWin (if the client was null).
 Otherwise, the scrollWin uses metrics.rowDelta for the sbLine* actions,
 and the inner window's height - metrics.rowDelta for the sbPage* actions.

 Once the scrollWin has determined the amount to scroll, it sends
 msgWinDelta to the clientWin.
*/


/****************************************************************************
 msgScrollbarHorizScroll		takes P_SCROLLBAR_SCROLL, returns STATUS
	category: client responsibility
	Client should perform horizontal scroll.

 Responding to this message is one of the key functions that scrollWins
 provide.  Since the default scrollWin style.vertClient and .horizClient
 are both swClientScrollWin, it is usually the case that the scrollbars
 send their scrolling messages to the scrollWin.

 If there is no current clientWin, or the pArgs->action is sbEndScroll,
 the scrollWin just returns stsOK.

 If style.getDelta is true, the scrollWin sends msgScrollWinProvideDelta
 to the client (if that is non-null) or the clientWin (if the client was null).
 Otherwise, the scrollWin uses metrics.colDelta for the sbLine* actions,
 and the inner window's width - metrics.colDelta for the sbPage* actions.

 Once the scrollWin has determined the amount to scroll, it sends
 msgWinDelta to the clientWin.
*/

/****************************************************************************
 msgScrollbarProvideVertInfo		takes P_SCROLLBAR_PROVIDE, returns STATUS
	category: client responsibility
	Client should provide the document and view info.

 clsScrollWin responds by filling out the pArgs fields.  It sets the
 viewLength to the height of the inner window.  If there is a current
 clientWin, then the scrollWin sets the docLength to the height
 of the clientWin, and the offset to the difference between the top of the
 clientWin and the top of the inner window.  If there is no
 current clientWin, the scrollWin sets both docLength and offset to 0.
*/

/****************************************************************************
 msgScrollbarProvideHorizInfo		takes P_SCROLLBAR_PROVIDE, returns STATUS
	category: client responsibility
	Client should provide the document and view info.

 clsScrollWin responds by filling out the pArgs fields.  It sets the
 viewLength to the width of the inner window.  If there is a current
 clientWin, then the scrollWin sets the docLength to the width
 of the clientWin, and the offset to the negative of the clientWin's x.
 If there is no current clientWin, the scrollWin sets both docLength
 and offset to 0.
*/


/****************************************************************************
 msgEmbeddedWinShowChild	takes P_EMBEDDED_WIN_SHOW_CHILD, returns STATUS
 	Display a given area of an embeddedWin to the user

 clsScrollWin responds by sending messages to the vertical and/or horizontal
 scrollbars to scroll the client window area into view.
*/

#endif	// SWIN_INCLUDED
