/****************************************************************************
 File: servmgr.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.61  $
   $Author:   sisaac  $
     $Date:   02 Mar 1992 10:10:00  $

 This file contains the API definition for clsServiceMgr.
 clsServiceMgr inherits from clsInstallMgr.
 Provides access to a category of PenPoint service instances.

****************************************************************************/

/****  1. Introduction  ****/
/*
 A service manager represents a category of services in PenPoint. Service
 managers have well-known ids so they can be globally accessed. PenPoint
 creates several service managers by default. They are:

	theModems:					Modems.
	thePrinters:				Printers.
	thePrinterDevices:			Devices that a printer can talk to.
	theSendableServices:		All services that interface to the 
								Send Manager. See sendserv.h.
	theTransportHandlers:		Transport level network protocol handlers.
	theLinkHandlers: 			Link level network protocol stacks.
	theHWXEngines:				Installable handwriting engines.
	theMILDevices:				All MIL services (device drivers).
	theParallelDevices:			Parallel port devices.
	theSerialDevices:			Serial port devices.
	theHighSpeedPacketHandlers:	High performance packet drivers.
	theOutboxServices:			All outbox services.
	theInboxServices:			All inbox services.
	theDatabases:				All PIA databases.

 Additional service managers can be created on the fly by third parties
 or by GO.

 All of the service instances in a given category are on that service
 manager. All the instances on a service manager support the same API, so 
 they can be used interchangeably.

 Each service instance on a service manager is identified with a unique
 string name. For example, there might be three printers on thePrinters:
 "MyLaserJet", "MarketingPrinter1", and "LittleDotMatrix". 

 You can find a particular service instance or enumerate all the instances 
 that are available. You can observe a service manager and be informed when 
 a new instance is added or an existing one	goes away.

 Once you know which service instance you want to use you must open it
 in order to gain access. This returns the uid of the service. You can 
 then send messages directly to the service object. You must close the
 service instance after you are done using it.

*/
/****  2. Basic Service Manager Usage  ****/
/*
 The simplest use of a service manager is to access a known service instance
 on the manager. Here's an example:
//{
 	SM_ACCESS		access;
	SM_RELEASE		release;

 	access.pServiceName = "Service Instance Name";
 	access.caller = self;
 	ObjCallRet(msgSMAccess, aServiceManager, &access, s);
 	// access.service can now be sent messages.
 	...
 	// When you are done with the service, release it.
 	release.caller = self;
 	release.service = access.service;
	release.handle = access.handle;
 	ObjCallRet(msgSMRelease, aServiceManager, &release, s);
//}
 Some service instances allow the client to specify pArgs. You must	
 initialize the pArgs with msgSMAccessDefaults for these. For example:
//{
 	access.pServiceName = "Service Instance Name";
 	access.caller = self;
 	access.pArgs = &args;
 	ObjCallRet(msgSMAccessDefaults, aServiceManager, &access, s);
 	args.foo = ...;
 	ObjCallRet(msgSMAccess, aServiceManager, &access, s);
//}
*/
/****  3. Advanced Service Manager Usage  ****/
/*
 Accessing a service instance is actually composed of several steps. 
 msgSMAccess and msgSMRelease performs all of them at once; more 
 sophisticated users might find situations where they need to control 
 the intermediary steps	themselves.

 Each service instance has a 32 bit "handle" associated with it in 
 addition to its name. This handle is a convenient shorthand for referencing
 a service instance. Most service manager messages use handles. 
 Note that a handle is not a permanent id; it is dynamically generated
 when a service instance is first added to a service manager, and 
 regenerated whenever PenPoint is rebooted.	Handles should never be filed.

 Enumerating all of the service instances on a service manager is done by
 getting a list of all the handles and going through the list. For example,
 here's some code that gets all the names of all the service instances on
 a manager list:
//{
	OBJECT				list;
	LIST_ENTRY			le;
	IM_GET_SET_NAME		getName;

	ObjectCall(msgIMGetList, aServiceManager, &list);
	ObjectCall(msgListNumItems, list, &n);
	for (le.position = 0; le.position < n; le.position++) {
		ObjectCall(msgListGetItem, list, &le);
		getName.handle = (OBJECT) le.item;
		getName.pName = pName;
		ObjectCall(msgIMGetName, aServiceManager, &getName);
		// le.item is the handle, pName contains the name.
	}
	ObjCallWarn(msgDestroy, list, pNull);
//}
 If you know the name of a service, you can get its handle with msgIMFind:
//{
	find.pName = "Service Instance Name";
	ObjectCall(msgIMFind, aServiceManager, &find);
	serviceInstanceHandle = find.handle;
//}
 The next step in accessing a service instance is binding. Binding tells
 a service instance that you are interested in it. After you have bound
 to a service you will get messages from that service telling you about
 changes in its state, such as when it becomes connected or disconnected.
//{
	bind.handle = serviceInstanceHandle;
	bind.caller = self;
	ObjectCall(msgSMBind, aServiceManager, &bind);
//}
 Next you become the owner of the service instance. Ownership gives you the
 right to open the instance. It is the mechanism used to ensure that only
 one client is using a exclusive access device (such as a serial port) at
 a time. Some services are non-exclusive access (such as network devices).
 Setting owner is a no-op for these. 

 The owner protocol informs the both the new and old owners that an 
 ownership change is being proposed. Either of them	can veto the change. 
 The service instance can also veto the change.

 The owner of a service can be set to objNull to signify no owner. You
 should do this when you want to give up ownership of a service instance.

 Here is an example of requesting an owner change:
//{
	setOwner.owner = newOwner;
	setOwner.handle = serviceInstanceHandle;
	ObjectCall(msgSMSetOwner, aServiceManager, &setOwner);
//}
 Now you can open the service. An open request can optionally take pArgs. 
 The format of the pArgs is service-specific. However, all the service 
 instances on a particular service manager have the same pArgs format. 
 The pArgs must be set to defaults with msgSMOpenDefaults.
 
 A service that has open instances cannot be deinstalled. An open service 
 instance cannot have its owner changed. Here is an example of opening a 
 service instance:
//{
	open.caller = self;
	open.handle = serviceInstanceHandle;
	open.pArgs = &openArgs;
	ObjectCall(msgSMOpenDefaults, aServiceManager, &open);
	ObjectCall(msgSMOpen, aServiceManager, &open);
	// open.service contains the service object at this point
//}
 Clients should close a service instance when they have completed using	it:
//{
	close.caller = self;
	close.handle = serviceInstanceHandle;
	close.service = open.service;
	close.pArgs = pNull;
	ObjectCall(msgSMClose, aServiceManager, &close);
//}
 Clients should unbind from a service instance when they are no longer 
 interested in it.
//{
	unBind.handle = serviceInstanceHandle;
	unBind.caller = self;
	ObjectCall(msgSMUnbind, aServiceManager, &unBind);
//}
*/
/****  4. Additional Service Manager Functionality	****/
/*
 Adding yourself as an observer of a service manager will cause all 
 notification messages from the service manager and all the service instances
 on the service manager to go to you. These messages include:

	msgIMInstalled:			A new service has been added to the service 
							manager.
	msgIMDeinstalled:		A service has been removed from the service
							manager.
	msgIMInUseChanged:		A service has been opened or closed.
	msgIMModifiedChanged:	A service has modified its state node.
	msgSMConnectedChanged:	A service has become connected or disconnected.
	msgSMOwnerChanged:		The owner of a service has changed.

 Plus, any service instance can send service-specific notification messages 
 via msgSvcPropagateMsg (see service.h). All observer messages include the
 handle of the service instance being affected and the uid of the service 
 manager.

 Sometimes a client needs to access a service object without becoming the
 owner, or need to override the open checks. This can be done, but it
 must be done with care. msgSMQueryLock and msgSMQuery can be used to
 do this.

 QueryLocking a service returns the service uid without opening it. However,
 the call will fail if the service is exclusive-open and currently open.
 Also, a query lock will lock out other opens until the query lock is
 released. msgSMQueryUnlock must be sent to release the query lock.

 msgSMQuery is just like msgSMQueryLock, except no open check is made. 
 
 Service managers automatically clean up if an object that owns or opens
 a service instance terminates before releaseing the service instance.

 There is a well-known list object, theServiceManagers, that is a list of 
 all the service managers in the system. You can observe this list and get 
 notification when a service manager is	added and removed.

*/
/****  5. Creating New Service Managers	 ****/
/*
 As stated above, PenPoint defines several default service managers. 
 You can create additional service managers if you desire. 

 PenPoint will automatically create a service manager if a service instance
 tries to add itself to a service manager and the service manager doesn't
 exist. This allows services to be arbitrarily installed and deinstalled
 without having to worry about who creates and frees the service manager.
****************************************************************************/
#ifndef SERVMGR_INCLUDED
#define SERVMGR_INCLUDED

#ifndef SERVICE_INCLUDED
#include <service.h>
#endif

#ifndef INSTLMGR_INCLUDED
#include <instlmgr.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  				 Core Messages  						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_SM_NEW, returns STATUS
	category: class message
	Creates a new service manager.
 
 Clients (other than those who are creating their own service managers) do
 not call this message. The well-known service managers are created by the
 system at cold-boot time.
*/

typedef struct SM_NEW_ONLY {
	BOOLEAN					autoDestroy;  // Have the service manager be owned
										  // by a system process, and have it
										  // destroy itself when the number of
										  // service instances on it goes to 0.

	BOOLEAN					noChecks;	  // Turn off error checking, client
										  // tracking and binding; a service
										  // on this list cannot be a target.
										  // This improves performance but
										  // is dangerous. Experts only!
	U32						unused2;
	U32						unused3;
	U32						unused4;
} SM_NEW_ONLY, *P_SM_NEW_ONLY;

#define	serviceManagerNewFields	\
	installMgrNewFields			\
	SM_NEW_ONLY				sm;

typedef struct SM_NEW {
	serviceManagerNewFields	
} SM_NEW, *P_SM_NEW;


/****************************************************************************
 msgNewDefaults					takes P_SM_NEW, returns STATUS
	category: class message
    Initializes the SM_NEW structure to default values.

 Sets
	installMgr.style.createInitial = false;
	installMgr.style.copyOnInstall = false;
	installMgr.style.addToGlobalList = false;
	installMgr.style.createIcon = false;
	sm.autoDestroy = false;
	sm.noChecks = false;
*/

/****************************************************************************
 msgDump					takes OBJ_KEY, returns STATUS
	Prints out the services known by this service manager and their state.

 clsServiceManager provides an elaborate response to msgDump. This is very 
 useful	for debugging services!
*/


/****************************************************************************
 msgSMAccess		  	takes P_SM_ACCESS, returns STATUS
	 Accesses a service instance, given its name.

 This is a convenience message that performs the sequence most
 clients do to access a service.

 This message performs a find, bind, setOwner, and open for the
 specified service. 

 Note: This	message cannot be used when you want to provide pArgs to
 a service. 

 Return Value
	stsNoMatch:			Item not found.
	stsSvcLocked:		Someone has this exclusive-open service query locked.
	stsSvcNotOwner:		Someone else is the owner of this owner-checked 
						service.
	stsSvcAlreadyOpen:	Someone already has this exclusive-open service open.
	Service-Specific Error Returns.

 See Also
	msgIMFind
	msgSMBind
	msgSMSetOwner
	msgSMOpen
*/
#define msgSMAccess							MakeMsg(clsServiceMgr, 43)

typedef struct SM_ACCESS {
	P_STRING			pServiceName; // Service name.
	OBJECT				caller;		  // Object making this call, 
									  //  typically self.
	P_ARGS				pArgs;		  // Use this if service requires pArgs.
									  //  Send msgSMAccessDefaults first.
	OBJECT				handle;		  // Out: Service handle.
	OBJECT				service;	  // Out: Service instance.
} SM_ACCESS, *P_SM_ACCESS;


/****************************************************************************
 msgSMAccessDefaults		  	takes P_SM_ACCESS, returns STATUS
	 Sets pArgs defaults for msgSMAccess.

 This message should be used if the service you wish to access takes
 pArgs. This message sets up the defaults for the pArgs.

 Return Value
	stsNoMatch:			Item not found.

 See Also
	msgSMAccess
*/
#define msgSMAccessDefaults					MakeMsg(clsServiceMgr, 45)


/****************************************************************************
 msgSMRelease		  	takes P_SM_RELEASE, returns STATUS
	 Releases a service instance.

 Call this message when you are finished using a service. 

 This is a convenience message that performs the sequence most
 clients do when they are finished with a service.

 This message performs a close, sets the owner to objNull, and unbinds.

 Return Value
	stsFailed:			Service is not open by the caller.
	Service-Specific Error Returns.

 See Also
	msgSMClose
	msgSMSetOwner
	msgSMUnBind
*/
#define msgSMRelease						MakeMsg(clsServiceMgr, 44)

typedef struct SM_RELEASE {
	OBJECT				caller;		  // Object making this call, 
									  //  typically self.
	OBJECT				handle;		  // Service handle.
	OBJECT				service;	  // Service instance.
} SM_RELEASE, *P_SM_RELEASE;


/****************************************************************************
 msgSMBind					  	takes P_SM_BIND, returns STATUS
	Binds to a service.
	
 The caller is made an observer of this service. Service manager 
 notification messages will be sent to the caller. 

 The caller is added to the bind list of the service instance.

 Sends msgSvcBindRequested to the service being bound to. The service
 has the right to refuse the bind. The service-specific error return that
 indicates a refusal is passed back to the client. 

 Return Value
  	stsBadObject:  	Caller is not an object.
  	stsBadAncestor:	Caller has invalid ancestor.
	Service-Specific Error Returns.

 See Also
	msgSvcBindRequested	(service.h)
	msgSvcGetBindList   (service.h)	
*/
#define msgSMBind			 					MakeMsg(clsServiceMgr, 1)

typedef struct SM_BIND {
	IM_HANDLE			handle;		// Service handle to bind to.
	OBJECT				caller;		// Object making this call.
} SM_BIND, *P_SM_BIND;


/****************************************************************************
 msgSMUnbind 				  	takes P_SM_BIND, returns STATUS
	Unbinds from a service.

 This removes the caller as an observer of the handle and removes the caller 
 from the service instance's bind list.

 Note: Clients *must* first close a service before unbinding from it.

 The service manager will automatically send msgSMUnbind for all services
 that are bound to a client when that client object is freed. This
 means that you must not send msgSMUnbind from your msgFree routine; the
 object freed notification occurs before your msgFree routine is entered.

 Sends msgSvcUnbindRequested to the service being unbound from.
 
 Return Value
	stsFailed:			Service is not bound by the caller.
*/
#define msgSMUnbind			 					MakeMsg(clsServiceMgr, 2)


/****************************************************************************
 msgSMGetOwner				takes P_SM_GET_OWNER, returns STATUS
	Gets the current owner of a service.

*/
#define msgSMGetOwner		 			   	MakeMsg(clsServiceMgr, 31)

typedef struct SM_GET_OWNER {
	IM_HANDLE			handle;		// Handle of item to get owner on.
	OBJECT				owner;		// Out: current owner.
} SM_GET_OWNER, *P_SM_GET_OWNER;


/****************************************************************************
 msgSMSetOwner				takes P_SM_SET_OWNER, returns STATUS
	Sets a new service owner.

 Old and new owners (whether they are clients or other services) will recieve
 service messages which allow them to veto the ownership change and informs
 them that the change has taken effect. The message sequence is as follows:

	1.:	msgSvcOwnerAquireRequested is sent to the new owner. The new owner can
		veto the owner change by returning a status of anything other
	    than stsOK or stsNotUnderstood. msgSMSetOwner returns with the abort 
		status. 

	2.:	msgSvcOwnerReleaseRequested is sent to the old owner. The old owner 
		can	veto the owner change by returning a status of anything other
	    than stsOK or stsNotUnderstood. If the old owner agrees to the 
		ownership change it must immediately close the service if it is 
		open.
 
	3.:	msgSvcChangeOwnerRequested is sent to the service. This informs the 
		service	that ownership is going to be changed and allows it to
    	veto. By default the services will veto the change if they are open.

	4.:	msgSvcOwnerReleased is sent to the old owner.

	5.:	msgSvcOwnerAquired is sent to the new owner.

	6.:	msgSMOwnerChanged is sent to everyone who is bound to the service
		or observing a service manager that the service is on.

 Return Value
	stsBadObject:  			New owner is not an object.
	stsBadAncestor:			New owner has invalid ancestor.
	stsSvcInUse:			Service is open.

 See Also
	service.h, for definition of msgSvc... messages.
*/
#define msgSMSetOwner		 			   	MakeMsg(clsServiceMgr, 11)

typedef struct SM_SET_OWNER {
	IM_HANDLE			handle;		// Handle of item to set owner on.
	OBJECT				owner;		// New owner.
} SM_SET_OWNER, *P_SM_SET_OWNER;


/****************************************************************************
 msgSMOpen					  	takes P_SM_OPEN_CLOSE, returns STATUS
	Opens a service, given its handle. 

 Clients should do this only when they are ready to transfer data to
 the service, and should leave the service open for as little time as 
 possible. 

 A bind is automatically performed if the client is not yet bound.

 The caller is added to the open list of the service instance.

 Sends msgSvcOpenRequested to the service being opened. The service
 has the right to refuse the open. The service-specific error return that
 indicates a refusal is passed back to the client. 

 Return Value
	stsBadObject:  		Caller is not an object.
	stsBadAncestor:		Caller has invalid ancestor.
	stsSvcNotBound:		Caller is not bound to the service.
	stsSvcLocked:		Someone has this exclusive-open service query locked.
	stsSvcNotOwner:		Someone else is the owner of this owner-checked 
						service.
	stsSvcAlreadyOpen:	Someone already has this exclusive-open service open.
	Service-Specific Error Returns	

 See Also
	msgSMBind
	msgSvcOpenRequested 	(service.h)
	msgSvcGetOpenList   	(service.h)	
	msgSvcGetOpenObjectList	(service.h)	
*/
#define msgSMOpen			 					MakeMsg(clsServiceMgr, 4)

typedef struct SM_OPEN_CLOSE {
	IM_HANDLE			handle;		// Handle of service to open.	   
	OBJECT				caller;		// Object making this call.
	P_ARGS				pArgs;		// Service-specific open parameters.
	OBJECT				service;	// In: (SMClose) Out: (SMOpen) Service 
									//   object.
} SM_OPEN_CLOSE, *P_SM_OPEN_CLOSE;


/****************************************************************************
 msgSMOpenDefaults		 	  	takes P_SM_OPEN_CLOSE, returns STATUS
	Initializes SMOpen pArgs to default value.

 See Also
	msgSMOpen
	msgSvcOpenDefaultsRequested (service.h)
*/
#define msgSMOpenDefaults	 					MakeMsg(clsServiceMgr, 34)


/****************************************************************************
 msgSMClose					takes P_SM_OPEN_CLOSE, returns STATUS
	Close an open service.

 The caller is removed from the open list of the service instance.

 Clients should send msgSMClose as soon as they are finished actively
 transfering data. Clients *must* first close a service before unbinding 
 from it.

 The service manager will automatically send msgSMClose for all services
 that are held open by a client when that client object is freed. This
 means that you must not send msgSMClose from your msgFree routine; the
 object freed notification occurs before your msgFree routine is entered.

 Sends msgSvcCloseRequested to the service being opened.
 
 Return Value
	stsFailed:			Service instance is not open by the caller.
	
*/
#define msgSMClose			 					MakeMsg(clsServiceMgr, 5)


/****************************************************************************
 msgSMQueryLock				   	takes P_SM_QUERY_LOCK, returns STATUS
	Gets the uid of a service and locks out any opens. 

 This message is similar to msgSMOpen, in that it returns a service object,
 given a handle. However, it is not seen as an open by the service.

 This message is meant for non-data transfer access to a service, for 
 example, generating a service's option card.

 The sender of this message does *not* have to be the owner of the 
 service.

 This message will fail if the service instance is exclusive open and 
 currently in use (open). If this message succeeds then all opens will 
 fail until msgSMQueryUnlock is sent.

 This message will return the real uid of the service instance in the case
 of a multi-user service.

 Return Value
	stsSvcLocked:			Service instance is already query locked.
	stsSvcInUse:			Service instance is open.

*/
#define msgSMQueryLock		 	   			MakeMsg(clsServiceMgr, 8)

typedef struct SM_QUERY_LOCK {
	IM_HANDLE			handle;		// Handle of service instance to query.	   
	OBJECT				service;	// Out: Service object.
} SM_QUERY_LOCK, *P_SM_QUERY_LOCK;


/****************************************************************************
 msgSMQueryUnlock				   	takes P_SM_QUERY_UNLOCK, returns STATUS
	Unlocks a service that was locked via msgSMQueryLock.

*/
#define msgSMQueryUnlock					MakeMsg(clsServiceMgr, 9)

typedef struct SM_QUERY_UNLOCK {
	IM_HANDLE			handle;		// Handle of service instance to unlock.
} SM_QUERY_UNLOCK, *P_SM_QUERY_UNLOCK;


/****************************************************************************
 msgSMQuery				    takes P_SM_QUERY_LOCK, returns STATUS
	Gets the uid of a service.

 This message gets the uid of a service instance. It must be used very 
 carefully. It bypasses all checking mechanisms, so the caller can get into 
 trouble if he subsequently sends messages to the service that are not 
 expected. Use msgSMQueryLock instead of msgSMQuery if at all possible.
*/
#define msgSMQuery			 	   			MakeMsg(clsServiceMgr, 33)


/****************************************************************************
 msgSMGetCharacteristics    takes P_SM_GET_CHARACTERISTICS, returns STATUS
	Gets the characteristics of the specified service instance.

 Characterstics are service-specific properties of a particular service.
 For example, modem services might pass back whether Fax is supported,
 maximum baud rate, etc. All the services on a particular service manager
 return the same characterstics set.

 Callers should first send this message with pArgs->len set to 0. This
 will return the size of the actual characterisitics buffer. Callers should
 then allocate this space and make the call again with pArgs->len set to 
 this size. pArgs->len can be less than the actual size, in which case only
 the number of bytes specified by pArgs->len is returned.

*/
#define msgSMGetCharacteristics		  		MakeMsg(clsServiceMgr, 42)

typedef struct SM_GET_CHARACTERISTICS {
	IM_HANDLE			handle;	  // Handle of item to get characteristics of.
	P_UNKNOWN			pBuf; 	  // Out through Ptr: Characterisitics buffer.
	U16					len;	  // In/Out: Buffer size. If 0 then the 
								  // actual size is returned.
} SM_GET_CHARACTERISTICS, *P_SM_GET_CHARACTERISTICS;


/****************************************************************************
 msgSMSave				   	takes P_SM_SAVE, returns STATUS
	Saves a service instance to a specified external location.

 The pArgs specify the parent directory that the service instance will save
 itself into. Note that the service instance's current target is also saved.
 When the service instance is reloaded it will try and bind to this target.

 See Also
	msgSvcClassLoadInstance			load a service instance from arbitrary
									location on disk (service.h).
*/
#define msgSMSave			 	   			MakeMsg(clsServiceMgr, 36)

typedef struct SM_SAVE {
	IM_HANDLE			handle;		// Handle of service instance to save.
	BOOLEAN				reserved;	// Reserved.
	FS_FLAT_LOCATOR		flat;		// Location to save to.
} SM_SAVE, *P_SM_SAVE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 					   Auxiliary Messages 						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSMFindHandle			  	takes P_SM_FIND_HANDLE, returns STATUS
	Finds a handle, given a service instance uid. 

 This message allows you to find the handle of a service if you know its
 uid.

 Return Value
	stsNoMatch:			Service not found on this service manager list.
*/
#define msgSMFindHandle		 				MakeMsg(clsServiceMgr, 10)

typedef struct SM_FIND_HANDLE {
	OBJECT				service;	// Service object to look for.
	IM_HANDLE			handle;		// Out: resulting handle.
} SM_FIND_HANDLE, *P_SM_FIND_HANDLE;


/****************************************************************************
 msgSMSetOwnerNoVeto			takes P_SM_SET_OWNER, returns STATUS
	Sets a new service owner without giving owners veto power.

 This message is the same as msgSMSetOwner, except the old owner and new 
 owners do not get the chance to veto. msgSvcReleaseRequest and 
 msgSvcAquireRequest are not sent. This message does the following:

	1.:	The open status of the service is checked. If it is open (in use) the
	    SetOwner fails, with a return status of stsSvcInUse.

	2.:	msgSvcChangeOwnerRequested is sent to the service. This informs the 
		service	that ownership is going to be changed and allows it to
    	veto the owner change by returning anything other than stsOK or 
		stsNotUnderstood. msgSMSetOwner returns with the abort status. 

	3.:	msgSMOwnerChanged is sent to everyone who is bound to the service
		or observing a service manager that the service is on.

	4.:	msgSvcOwnerReleased is sent to the old owner.

	5.:	msgSvcOwnerAquired is sent to the new owner.

 Return Value
	stsSvcInUse:			Service is open.
*/
#define msgSMSetOwnerNoVeto	 			   	MakeMsg(clsServiceMgr, 30)


/****************************************************************************
 msgSMGetState				takes P_SM_GET_STATE, returns STATUS
	Gets the state of a service.

 This message provides service state. There is some additional state
 (in use, modified) that is gotten via msgIMGetState. See instlmgr.h
 for details.
*/
#define msgSMGetState				  		MakeMsg(clsServiceMgr, 12)

typedef struct SM_GET_STATE {
	IM_HANDLE			handle;		// In:  Handle of service to get state on.
	BOOLEAN				connected;	// Out: Is service connected?
	OBJECT				owner;		// Out: My owner, if any.
	OBJECT				owned;		// Out: The service that I own, if any.
	U8					reserved[24];
} SM_GET_STATE, *P_SM_GET_STATE;


/****************************************************************************
 msgSMGetClassMetrics			takes P_SM_GET_CLASS_METRICS, returns STATUS
	Gets the service's class metrics. 

 This message passes back information about the service class. See service.h
 for a definition of SVC_CLASS_METRICS.
*/
#define msgSMGetClassMetrics		  		MakeMsg(clsServiceMgr, 13)

typedef struct SM_GET_CLASS_METRICS {
	IM_HANDLE			handle;		// Handle of item to get class metrics on.
	SVC_CLASS_METRICS	metrics;	
} SM_GET_CLASS_METRICS, *P_SM_GET_CLASS_METRICS;


/****************************************************************************
 msgIMDeinstall					takes P_IM_DEINSTALL, returns STATUS
	Remove and free a service instance.

 This will remove the specified service instance from all the service 
 managers that it is on, destroy its state file, and free it. 

 Note that a service is initially created by sending msgNew to the service 
 class.	Services automatically add themselves to service manager.
 Do not use msgIMInstall for this purpose; msgIMInstall should NEVER be
 used by clients.

 Causes observer message msgIMDeinstalled to be propogated to all objects
 that are bound to the service instance and to the service managers.

 This message causes msgSvcDeinstallRequested to be sent to the service
 instance. The instance can veto the deinstall at this point; if it does 
 then the return value from msgIMDeinstall is the status that the service 
 instance used to veto the deinstall. 

 See Also
	msgSvcDeinstallRequested
	msgSvcDeinstallVetoed
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 						Notification Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSMConnectedChanged		 	takes P_SM_CONNECTED_NOTIFY, returns STATUS
	category: observer notification
	A service's connection state changed.

*/
#define msgSMConnectedChanged 	   			MakeMsg(clsServiceMgr, 20)

typedef struct SM_CONNECTED_NOTIFY {
	OBJECT				manager;	// manager that sent notification
	IM_HANDLE			handle;		// handle to service
	BOOLEAN				connected;	// new connect state
} SM_CONNECTED_NOTIFY, *P_SM_CONNECTED_NOTIFY;


/****************************************************************************
 msgSMOwnerChanged		 	takes P_SM_OWNER_NOTIFY, returns STATUS
	category: observer notification
	A service's owner has changed.

*/
#define msgSMOwnerChanged 	   				MakeMsg(clsServiceMgr, 21)

typedef struct SM_OWNER_NOTIFY {
	OBJECT				manager;	// manager that sent notification
	IM_HANDLE			handle;		// handle to service
	OBJECT				oldOwner;	// old owner
	OBJECT				owner;		// new owner
} SM_OWNER_NOTIFY, *P_SM_OWNER_NOTIFY;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Private																   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgSMRemoveReference			takes P_SM_REMOVE_REF, returns STATUS
	Remove a service from a service manager without destroying the service.

 This message is for experts only. It is used internally by services also.
 Most clients should use msgIMDeinstall, which removes a service from
 all of the service managers it is on and destroys it.

 Causes observer message msgIMDeinstalled to be sent to the service
 manager.
*/
#define msgSMRemoveReference		  		MakeMsg(clsServiceMgr, 14)

typedef struct SM_REMOVE_REF {
	IM_HANDLE			handle;		// handle to remove.
} SM_REMOVE_REF, *P_SM_REMOVE_REF;


// Next msg: 46

/* Backwards compatibility */
typedef SM_OPEN_CLOSE	SM_OPEN, *P_SM_OPEN, SM_CLOSE, *P_SM_CLOSE;


#endif	// SERVMGR_INCLUDED


