/****************************************************************************
 File: service.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.66  $
   $Author:   sisaac  $
     $Date:   17 Mar 1992 15:57:58  $

 This file contains the API definition for clsService.
 clsService inherits from clsStream.
 Provides default behavior for services.
****************************************************************************/

/****  1. Introduction  ****/
/*
 All non-application functionality under Penpoint is expressed as a service. 
 If what you want to do does not fit the application model (documents created
 via Stationery or Accessories, subclass of clsApp, etc) then it should be a 
 service. Some examples of services are: device drivers, inbox/outbox 
 transfer agents such as fax and e-mail, network protocol stacks, 
 and device drivers.

 Service instances are automatically organized onto service managers. 
 A service manager represents a category of service, such as Printers or
 Serial Devices. All of the service instances in a given category are
 can be used interchangeably; that is, they all support the API that is
 required to be in that category. Clients access service instances via 
 service managers. See servmgr.h for details.

 Each service instance has a text name, which is how it is uniquely
 identified. Clients use this name to identify a service instance on
 a service manager. A service instance's name is specified at msgNew
 time. Names must be unique for all services on the same service manager, 
 and all services of the same class.

 There are two exclusivity models for services: services that require 
 exclusive access by a single client, and services that allow multiple 
 clients simultaneous access. Services provides default behavior for
 arbitrating ownership of exclusive access services.

 Multiple access services can either be shared (each client gets back the 
 uid of the service when they open the service) or multi-user (each client
 gets back a different object when they open the service). 

 Service instances can optionally maintain state. By default each service 
 instance has a node in the filesystem. clsService will automatically 
 recreate service instances from their state files when PenPoint is rebooted. 
 Also, service instances can be saved and restored from external disks by 
 moving	their state nodes on and off the machine.
 
 A service instance can have an optional "target". A target is some
 other service instance. If a service has a target, the service
 superclass takes care of remembering what the target points at. Typically, 
 data flows from one service instance to next, going down the target chain.
 Control information, such as when a physical device is becomes connected,
 flows up the target chain.

 A service is implemented as an installable DLL. Service instances are 
 either	created in the DLLMain() of the service DLL, created dynamically
 after the service has been installed, or created from pre-configured
 instance state nodes when the service is first installed. 
 See servimgr.h for a description of how services are installed and 
 deinstalled, and how a service is organized on disk.

*/
/****  2. Writing A Basic Service  ****/
/*
 A minimal service that does not save state or use a target must handle
 just one superclass message: msgNewDefaults. There are four fields
 which need to be filled in:

 	pArgs->svc.style.exclusiveOpen - Is this an exclusive access service?
	pArgs->svc.style.openClass 	   - Is this a multi-user service? 
	pArgs->svc.pManagerList		   - List of service managers to add to.
	pArgs->svc.numManagers		   - Number of managers on the list.

 Project BASICSVC is a template for a minimal service. Use it as a guide.

*/
/****  3. Writing a Service That Saves State  ****/
/*
 clsService maintains an open handle on a service's state node. By default
 the state node type is a file and the open handle is an instance of
 clsFileHandle. Both of these things can be overridden in your msgNewDefaults
 handler.

 Services must decide for themselves when they need to update their state
 node. They should always maintain enough state to be able to survive a
 reboot. There is no explicit Save/Restore messages for	services; A SERVICE 
 MUST UPDATE ITS STATE NODE WHENEVER ITS STATE CHANGES.	

 When its time to save state, self-send msgSvcGetHandle to get your state
 node handle. Self-send msgSvcSetModified when you complete updating
 state. These are the only messages that you will need to use for this
 type of service.

 Service instances will be automatically recreated when a warm boot occurs. 
 The msgNew arguments to clsService include the locator of the state node.
 Service instances must check to see if this node is non-empty then a 
 warm boot is happening, and the service must recreate itself from the
 state node. 

 State nodes can be copied out to disk, then reloaded the next time the
 service class is installed, or reloaded one at time. clsService will 
 automatically create a service instance for each state node at this
 time using the same mechanism as warm boot recovery. There is no difference
 between warm boot recovery and creation from a pre-configured state
 node copied in at installation time, as far as the service is concerned.

*/
/****  4. Writing a Service That Has A Target  ****/
/*
 Services can also bind and open other services. In fact, this is such
 a common situation that clsService provides lots of support for this.
 Each service can have a target, which refers to some other service.
 When the service is first created the default behavior is to attempt to 
 bind to the target. clsService will automatically open the 
 target when the service is opened if the autoOpen style bit is true.

 A service becomes a client of its target. All client observer notifications
 and ownership messages from a service's target are sent to the service.

 A service's target is usually set at msgNew time, and can be changed 
 anytime after with msgSvcSetTarget. msgSvcGetTarget gets a service 
 instance's target.

 Typically a service will open its target when a client opens it, using
 msgSvcOpenTarget. msgSvcCloseTarget should be used to close the target.

 Services also support the notion of being connected. Most hardware services
 can detect whether their hardware is connected or disconnected. Each service 
 has a state bit which says whether it is connected or not. When the hardware
 changes connection state the service sends msgSVCSetConnected to itself, 
 which notifies everyone who is bound to that service. 

 Non-hardware services automatically change their connection state when
 their targets change connection state. Thus, connection state propogates
 up from the hardware to all services that are bound to that hardware.

 A hardware service for a device that cannot auto-detect connection is 
 always in the connected state.

 Project TESTSVC provides a template for a service that deals with 
 a target.

*/
/****  5. Advanced Features	 ****/
/*
 Services that can provide both global and service instance option cards.
 A global option sheet sets configuration information for the entire service.
 It is invoked when the user calls for options of a service from the
 Service card of the Installer. Services can add additional cards to the
 global option sheet. 

 Service instance option sheets allow the user to set the configuration of
 particular instance. For example, the serial service provides a card which
 allows the user to set baud rate, parity, etc. Services should update their
 state node when the user applies a change to the option sheet.
 There is no default service instance option card.

 Services should respond to the standard option sheet protocol
 (msgOptionAddCards, msgOptionRefreshCard, etc) if they wish to provide 
 option cards. See option.h for details. The option sheet messages are 
 either sent as class messages for global options or normal instance 
 messages for instance options. 

 A service's configuration information can also be queried and set 
 programmatically via msgSvcGetMetrics and msgSvcSetMetrics. A service must
 be able to respond to these messages at any time, and should update its
 state node when its metrics are changed. The Get/SetMetrics messages are
 generic; they allow a client to save and restore metrics independently
 of the size or contents of the metrics. This allows a client to have 
 absolutely no knowledge of the internals of a service. The client can
 ask the user to set configuration options, then save and restore these
 configuration options via the generic Get/Set messages.

 Service instances can have icons associated with them in the same fashion
 as documents. Create icons using tagAppIconBitmap and tagAppSmallIconBitmap
 and put them in the service resource file. This is done in the same manner
 as applications.
*/
/****  6. Services and Tasking  ****/
/*
 A service, just like any other object, is owned by some task. However, all
 services must be callable from outside the owning task (objCapCall is always
 true for service instances). Service authors must take this into account. 
 Services must either use explicitly-created global heaps or instance data; 
 never store data in a local heap or the shared process heap.

 If the service is not exclusive access or multi-user, anyone who has the
 service open can call the service at anytime, even while someone else is 
 in the middle of another call. Use semaphores to protect access where 
 appropriate.

 You must also make sure that the a service's owning task will remain 
 active for the real lifetime of the service. For instance, if a service is 
 created via some transient user interface task such as a document or a tool,
 then the service instance will become invalid when that tool is shut down.

 An alternative to keeping the creating task around for the	lifetime of the
 service instance is to use msgObjectNew to create the service instance
 under another task. A very good task to create instances under is the
 main task of the service. The service resource file handle is available
 for use with msgObjectNew. Use msgSvcGetClassMetrics to get this handle
 (metrics.resFile). Send msgObjectNew to this handle. Note that msgObjectNew
 must be sent, not called. Remember, any pointers in the msgObjectNew pArgs 
 must be in global memory.

*/
/****  5. Recovering From Unexpected Client Termination  ****/
/*
 Service instances automatically detect if a client terminates unexpectedly;
 that is, if a client terminates while it is bound to the service instance
 or owns it. msgSvcClientDestroyedEarly is sent to the service instance
 when this condition is detected. Subclasses that maintain per-client state 
 can handle this message and perform cleanup. By default the service is
 closed and unbound from the terminating object.

*/
/****  6. Sample DLLMain Routine  ****/
/*
 You must let the service framework know about a service by sending
 msgSvcClassInitService to your service class. Here's an example:

//{
	STATUS EXPORTED DLLMain(void)
	{
		SVC_INIT_SERVICE	initService;
		STATUS				s;

		StsRet(ClsTestServiceInit(), s);
		memset(initService.spare, 0, sizeof(initService.spare));
		initService.autoCreate = true;			
		initService.serviceType = 0;
		initService.initServiceFlags = 0;
		ObjCallRet(msgSvcClassInitService, clsTestService, &initService, s);

		return stsOK;

	} // DllMain
//}

****************************************************************************/
#ifndef SERVICE_INCLUDED
#define SERVICE_INCLUDED

#ifndef STREAM_INCLUDED
#include <stream.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  		Common #defines and typedefs  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Service Status Codes  ****/

/* 
 An exclusive-open service is already open by someone else                
 (msgSvcOpenRequested), or a service's target is already open             
 (msgSvcOpenTarget).                
*/                                      
#define  stsSvcAlreadyOpen					MakeStatus(clsService, 1)

/* A service tried to open its target but the target manager field is null. */
#define  stsSvcNoTarget						MakeStatus(clsService, 2)

/* 
 A service tried to open its target but the target service doesn't exist  
 or the target's service manager hasn't shown up yet.                     
*/
#define  stsSvcTargetNotBound				MakeStatus(clsService, 3)

/* 
 An autoMsgPassing service tried to pass a message to its target, but		
 the target was not open.                                                 
*/
#define  stsSvcTargetNotOpen 				MakeStatus(clsService, 4)

/* 
 An attempt was made to change ownership, queryLock, or deinstall			
 an open service.      
*/                                                   
#define  stsSvcInUse		 				MakeStatus(clsService, 5)

/* Someone who wasn't the owner of a checkOwner service tried to open it. */
#define  stsSvcNotOwner		 				MakeStatus(clsService, 6)

/* Someone tried to open or queryLock a service that is queryLocked. */
#define  stsSvcLocked		 				MakeStatus(clsService, 7)

/* Problem following the target chain during msgSvcAutoDetectingHardware. */
#define  stsSvcValidConnectStyleNotFound	MakeStatus(clsService, 8)

/* A deinstallation is in process. No new clients can be accepted. */
#define  stsSvcDeinstallInProcess			MakeStatus(clsService, 10)

/* A service of this name already exists and refuses to terminate. */
#define  stsSvcAlreadyExists				MakeStatus(clsService, 11)

/* 
 A service was created with style.waitForTarget set to false and the 		
 target wasn't found at msgNew or msgSvcSetTarget time.                    
*/
#define  stsSvcTargetNotFound				MakeStatus(clsService, 12)

/**** Target ****/
/* A target references another service.	*/
typedef struct SVC_TARGET {
		OBJECT		 		manager;			  
		U8			 		pName[nameBufLength];
		U8					spare[12];
} SVC_TARGET, *P_SVC_TARGET;

/**** Service Class Metrics ****/
/* 
 Passed back by msgSvcGetClassMetrics. Also used in clsServiceInstallMgr  
 (servimgr.h) and clsServiceMgr (servmgr.h).		
*/						
typedef struct SVC_CLASS_METRICS {
	CLASS					serviceClass;  	 // The class of this service.
	U8						pClassName[nameBufLength]; // Service class name.
	U32						type;  			 // See svctypes.h.
	U8						pTypeName[nameBufLength];  // Service type name.
	OS_PROG_HANDLE			progHandle;		 // Service dll program handle.
	U32						initServiceFlags;// As specified in 
											 //  msgSvcClassInitService.
	OBJECT					resFile;		 // Handle to service res file.
											 //  Can be objNull if not
											 //  full environment and 
											 //  service.res is empty.
	OBJECT					serviceDir;		 // Dir handle to service global
											 //  directory.
	OBJECT					privateServiceMgr;//Private service mgr, if the
											 //  svcCreatePrivateServiceMgr
											 //  flag is set.
	U32						reserved1;
	U32						reserved2;
	U32						reserved3;
	U32						reserved4;
	U32						reserved5;
} SVC_CLASS_METRICS, *P_SVC_CLASS_METRICS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  			  Auxiliary Messages 		 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* See servmisc.h for less commonly used (but important!) service messages */
#ifndef SERVMISC_INCLUDED
#include <servmisc.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  			  	Creation Messages 		 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSvcClassInitService		takes P_SVC_INIT_SERVICE, returns STATUS
	category: class message
	Initializes the service class.

 You must send this message to the service class immediately
 after it has been created.
*/
#define msgSvcClassInitService					MakeMsg(clsService, 56)

/****  initServiceFlags  ****/
/* 
 Don't show this service in the installer. User can't configure or 		
 deinstall the service if this flag is set                                
*/
#define svcNoShow  					((U32) flag0)  

/* 
 Automatically pop up the global service option card the first time		
 this service is installed.                                               
*/
#define svcPopupOptions  			((U32) flag1)  

/* 
 Don't copy in the state files from the INST directory when the service	
 is installed.                                                            
*/
#define svcNoLoadInstances 			((U32) flag2)  

/* 
 Create a private service manager for instances of this class. All 		
 instances of this class will automatically be added to the private		
 service manager. See SVC_CLASS_METRICS for uid of the private			
 service manager.                                                         
*/
#define svcCreatePrivateServiceMgr 	((U32) flag3)  

/* 
 Generate a complete process environment in the DLLMain() process. Right	
 now this means creating theProcessResList. Also, a service resource		
 file handle will be created even if the service resource file is			
 empty. Note that a complete process environment takes up significant		
 memory. Only turn this on if you need it.                                
*/
#define svcFullEnvironment		 	((U32) flag4)  
   
typedef struct SVC_INIT_SERVICE {
	BOOLEAN					autoCreate;  // Create an instance for each state 
										 // node at install and warm boot times.
	U32						serviceType; // Global service type. See 
										 // svctypes.h. Usually set to 0.
	U32						initServiceFlags; // Or-in InitService flags.
	U8						spare[12];
} SVC_INIT_SERVICE, *P_SVC_INIT_SERVICE;


/****************************************************************************
 msgNew				takes P_SVC_NEW, returns STATUS
	category: class message
	Creates a new service object. 

 Callers send msgNew to create a new service instance. The instance will add
 itself to one or more service managers. Clients should access the service
 instance via the service manager API after msgNew.

 Superclass behavior includes associating the service with it's node in 
 the filesystem, adding it to the specified service managers, and attempting
 to bind to a target service. If style.waitForTarget is false and the 
 target isn't found then stsSvcTargetNotFound is returned.
 
 The following parameters are usually set by the caller of msgNew:
	-: pServiceName
	-: target 

 The following parameters are usually set by the subclass of clsService
 in msgNewDefaults (after the ancestor call):
	-: style (including openClass)
	-: pManagerList
	-: numManagers

 If a subclass wants to change the handleClass, fsNew, or fsNewExtra
 parameters it must also execute the following in its msgNewDefaults 
 method, after sending msgNewDefaults to ancestor:
//{
	pNew->svc.handleClass = myFSHandleClass;
	ObjCallOK(msgNewDefaults, pNew->svc.handleClass, &(pNew->svc.fsNew), s);
//}
 Most services will not need to do this.
 
 If a service with the same name as the new service already exists on any
 relevant service manager, the old service will be destroyed and the new
 service will replace it. However, if any of the old services veto the
 termination then the new service will not be created and an error status
 (stsSvcAlreadyExists) is returned.

 Return Value
	stsNoMatch:				Target not found and style.waitForTarget is false.
	stsSvcAlreadyExists:	Service of this name already exists and can't be
								terminated.
	stsBadParam:			Illegal target type.
*/

/**** style.connectStyle ****/
#define	svcAutoDetect 	 0	// Can auto-detect hardware connect/disconnect.
#define	svcNoAutoDetect	 1	// Can't do hardware auto-detect.
#define	svcFollowTarget	 2	// Connect state follows target's connect state.

typedef struct SVC_STYLE {
	U16	waitForTarget   : 1,	// OK if target doesn't exist; wait for it 
								//   to show up. 
		exclusiveOpen   : 1,	// Allow only one open or QueryLock at a time.
		reserved1	  	: 1,	// Reserved.
		autoOwnTarget	: 1,	// Set this service to be the owner of its 
								//   target when it receives 
								//   msgSvcChangeOwnerRequested.
		autoOpen		: 1,	// Open/close our target when we are
								//   opened/closed.
	 	autoMsgPass		: 1,	// Forward all messages that are not  
								//   clsObject, clsService or clsOption
								//   messages to target.
		checkOwner		: 1,	// Only allow the owner to open us;
								//   return stsNotOwner if opener is wrong.
		autoOption		: 1,	// Forward all option sheet messages to
								//   target. If the target is exclusive open
								//   and checkOwner, then only forward if
								//   target is owned by this service instance. 
		connectStyle	: 2,	// Connect detect abilities.
		reserved2		: 6;	// Reserved.
	CLASS openClass;			// Class used to create object returned from
								//   msgSMOpen. Can be objNull to return the
								//   service instance object itself.
	U16	spare1;
	U16	spare2;
} SVC_STYLE, *P_SVC_STYLE;						 

typedef struct SVC_NEW_ONLY {
	SVC_TARGET				target;			// Initial target. target.manager
											//   can be objNull for no target.
	P_STRING				pServiceName;	// Name of instance.
	SVC_STYLE				style;			// Overall style.
	CLASS					handleClass;	// Class of service's node handle.
	FS_NEW					fsNew;			// NewArgs for handle, filled in
											//   at msgNewDefault time.
	U32						fsNewExtra[25];	// Extra fsNew space.
	P_UID					pManagerList;	// List of service managers that
											//   self should be added to.
	U16						numManagers;	// Number of uids in manager list.
	U32						unused1;
	U32						unused2;
	U32						unused3;
	U32						unused4;
} SVC_NEW_ONLY, *P_SVC_NEW_ONLY;

#define	serviceNewFields	\
	streamNewFields			\
	SVC_NEW_ONLY			svc;

typedef struct SVC_NEW {
	serviceNewFields	
} SVC_NEW, *P_SVC_NEW;


/****************************************************************************
 msgNewDefaults		takes P_SVC_NEW, returns STATUS
	category: class message
    Initializes the SVC_NEW structure to default values.

 Sets	
	object.cap |= objCapCall;	 // Client must not override this in msgNew
	svc.target.manager = objNull;
	strcpy(pNew->svc.target.pName, "");
	svc.pServiceName = pNull;
	svc.style.waitForTarget = true;
	svc.style.exclusiveOpen = false;
	svc.style.autoOwnTarget = true;
	svc.style.autoOpen = false;
	svc.style.autoMsgPass = false;
	svc.style.checkOwner = false;
	svc.style.autoOption = false;
	svc.style.connectStyle = svcFollowTarget;
	svc.style.openClass = objNull;
	svc.handleClass = clsFileHandle;
	ObjCallOK(msgNewDefaults, pNew->svc.handleClass, \
		&(svc.fsNew), s);
	svc.fsNew.fs.exist = fsExistOpen | fsNoExistCreate;
	svc.pManagerList = pNull;
	svc.numManagers = 0;
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *            				State File Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSvcGetHandle					  	takes P_OBJECT, returns STATUS
	Returns a handle to the service's state node.

 Every service instance has an open handle to its state node. Use this 
 message when you want to update the contents of your state node.

 NOTE: This handle must NOT be freed, closed, or changed. 
*/
#define msgSvcGetHandle				 			MakeMsg(clsService, 12)


/****************************************************************************
 msgSvcGetModified	  	takes P_SVC_GET_SET_MODIFIED, returns STATUS
	Gets the modified state of this service.

*/
#define msgSvcGetModified	 			 		MakeMsg(clsService, 36)

typedef struct SVC_GET_SET_MODIFIED {
	BOOLEAN					modified;	// modified state
} SVC_GET_SET_MODIFIED, *P_SVC_GET_SET_MODIFIED;


/****************************************************************************
 msgSvcSetModified    	takes P_SVC_GET_SET_MODIFIED, returns STATUS
	Sets modified state of self.

 Service subclasses must send this message with pArgs->modified set to
 true whenever they change their state file.  

 Propogates msgIMModifiedChanged to everyone who has bound to this 
 service and is an observer of all service managers that this service is on.

 See Also
	msgIMModifiedChanged (instlmgr.h)
*/
#define msgSvcSetModified	 			   		MakeMsg(clsService, 20)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 					 Target Messages 							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSvcOpenTarget		  	takes P_SVC_OPEN_CLOSE_TARGET, returns STATUS
	Attain access to the target service for data transfer. 

 This call should be made when the service is ready to actually transfer data 
 to its target. It will cause msgSMOpen to be sent to the target's service 
 manager. The target service instance can refuse the subsequent 
 msgSvcOpenRequested request if it wants. The target service should be kept 
 open for the minimum time possible. 

 This message is sent automatically if newArgs.style.autoOpen is
 true. Note that pArgs is set to pNull in this case.

 Return Value
	stsFailed:			target.type is not svcTypeService.
	stsSvcNoTarget:		target.manager is null. 
	stsSvcNotBound:		service is still waiting to bind to its target.
	stsSvcAlreadyOpen:	target is already open.
	errors from msgSMOpen
	target service-specific errors

 See Also
	msgSMOpen (servmgr.h)
*/
#define msgSvcOpenTarget		  				MakeMsg(clsService, 13)

typedef struct SVC_OPEN_CLOSE_TARGET {
	P_ARGS					pArgs;		// Open or close parameters.
} SVC_OPEN_CLOSE_TARGET, *P_SVC_OPEN_CLOSE_TARGET;

/* Backwards compatibility */
typedef SVC_OPEN_CLOSE_TARGET	SVC_OPEN_TARGET, *P_SVC_OPEN_TARGET;


/****************************************************************************
 msgSvcCloseTarget 		 	takes P_SVC_OPEN_CLOSE_TARGET, returns STATUS
	Give up data transfer access to the target service. 

 This will cause msgSMClose to be sent to the target's service manager,
 resulting in msgSVCCloseRequested being sent to the target.

 This message is sent automatically if newArgs.style.autoOpen is
 true. Note that pArgs is set to pNull in this case.

 Return Value
	stsFailed:			target.type is not svcTypeService.

 See Also
	msgSMClose (servmgr.h)
*/
#define msgSvcCloseTarget						MakeMsg(clsService, 14)


/****************************************************************************
 msgSvcGetTarget   		 	takes P_SVC_GET_TARGET, returns STATUS
	Returns current target. 

 target contains the target that was specified at msgNew time or by the
 last msgSvcSetTarget.

 targetHandle contains the service manager handle onto our target if we 
 have bound with the target, or objNull if we haven't yet bound.
 
 targetService is the actual service object if the target has been
 opened, objNull if it isn't open.
*/
#define msgSvcGetTarget				 			MakeMsg(clsService, 15)

typedef struct SVC_GET_TARGET {
	SVC_TARGET	   			target;		  // Out: target
	OBJECT					targetHandle; // Out: handle to target, if bound
	OBJECT					targetService;// Out: target service, if open
} SVC_GET_TARGET, *P_SVC_GET_TARGET;


/****************************************************************************
 msgSvcSetTarget	  	 	takes P_SVC_SET_TARGET, returns STATUS
	Change our target. 

 Closes the old target (if it is open), unbinds the old target (if it is
 bound) and attempts to bind with the new target. style.waitForTarget
 specifies whether we will wait for the target to show up if it does not
 exist.

 Causes msgSvcTargetChanged to be sent.

 Return Value
	stsNoMatch:			new target doesn't exist and style.waitForTarget
						is false.
*/
#define msgSvcSetTarget					  		MakeMsg(clsService, 16)

typedef struct SVC_SET_TARGET {
	SVC_TARGET				target;
} SVC_SET_TARGET, *P_SVC_SET_TARGET;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 					 Connection Messages 						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSvcGetConnected	  	takes P_SVC_GET_SET_CONNECTED, returns STATUS
	Gets the connected state of this service.

*/
#define msgSvcGetConnected	 			 		MakeMsg(clsService, 19)

typedef struct SVC_GET_SET_CONNECTED {
	BOOLEAN					connected;	// connect state
} SVC_GET_SET_CONNECTED, *P_SVC_GET_SET_CONNECTED;


/****************************************************************************
 msgSvcSetConnected    	takes P_SVC_GET_SET_CONNECTED, returns STATUS
	Sets connection state of self.

 This message should only be used by auto-detecting services that interface 
 directly to hardware when they have determined that their connection state 
 has changed. 

 Propogates msgSMConnectedChanged to everyone who has bound to this 
 service and is an observer of all service managers that this service is on.

 If a binding service's connectStyle is svcFollowTarget, then it's 
 connected state will mirror that of its target. This is will be the case
 for most services, and is how the connect state propogates up the target
 links.

 See Also
	msgSMConnectedChanged (servmgr.h)
*/
#define msgSvcSetConnected	 			   		MakeMsg(clsService, 35)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	   						Client Access Messages		  				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSvcBindRequested   			 	takes P_SVC_BIND, returns STATUS
	Client asked to bind to this service. 
	
 A client sent msgSMBind to a service manager. The service can refuse the
 request by returning stsFailed. The default superclass behavior is to 
 return stsOK. 

 The service manager maintains a list of all the objects that have bound to
 this service instance. The caller is added to this list if this message 
 returns stsOK. This list is available via msgSvcGetBindList. 

 Subclasses usually let ancestor handle this message. This message must
 always be passed to ancestor.
*/
#define msgSvcBindRequested						MakeMsg(clsService, 2)

typedef struct SVC_BIND {
	OBJECT					caller;		// Object making the request.
	OBJECT					manager;	// Service manager the request is
										//   being made through.
} SVC_BIND, *P_SVC_BIND;


/****************************************************************************
 msgSvcUnbindRequested 			 	takes P_SVC_BIND, returns STATUS
	Client asked to unbind from this service. 

 A client sent msgSMUnbind to a service manager or a client who was bound
 to the service was destroyed.

 The service cannot veto this request. The caller is removed from the 
 service instance's bind list before this message is sent.

 Subclasses usually let ancestor handle this message. This message must
 be passed to ancestor.
*/
#define msgSvcUnbindRequested			 		MakeMsg(clsService, 3)


/****************************************************************************
 msgSvcOpenRequested   			 	takes P_SVC_OPEN_CLOSE, returns STATUS
	Client asked to open this service. 

 A client sent msgSMOpen to a service manager. The service instance can 
 refuse the open request by returning stsFailed.

 The service manager maintains a list of all the objects that have opened
 this service instance. The caller is added to this list if this message 
 returns stsOK. This list is available via msgSvcGetOpenList. 

 The service instance is marked in use when one or more clients have it open. 
 A service that has instances that are in use cannot be deinstalled.

 If the style.exclusiveOpen is true then only one client can have the service
 open at a time. If style.checkOwner is true then the owner of the service is
 the only one that can open the service. Errors are returned to the client
 if these conditions aren't true; see servmgr.h for details.

 Subclasses usually do some	processing, then pass this message to superclass.
 This message must be passed to ancestor.
*/
#define msgSvcOpenRequested						MakeMsg(clsService, 4)

typedef struct SVC_OPEN_CLOSE {
	OBJECT					caller;		// Object making the request.
	OBJECT					manager;	// Service manager the request is
										//   being made through.
	P_ARGS					pArgs;		// Service-specific open or close
										//   parameters.
	OBJECT					service;	// Out (msgSvcOpen): In (msgSvcClose):
										//   uid of open handle or service.
} SVC_OPEN_CLOSE, *P_SVC_OPEN_CLOSE;


/****************************************************************************
 msgSvcOpenDefaultsRequested	 	takes P_SVC_OPEN_CLOSE, returns STATUS
	Client wants open pArgs initialized.

 A client sent msgSMOpenDefaults to a service manager. 
*/
#define msgSvcOpenDefaultsRequested					MakeMsg(clsService, 9)


/****************************************************************************
 msgSvcCloseRequested  		 	takes P_SVC_OPEN_CLOSE, returns STATUS
	Client asked to close this service. 

 A client has send msgSMClosed to a service manager or a client who had the
 service open was destroyed. The service cannot veto this request; it must
 perform any cleanup required at this time. The caller is removed from the 
 open list before this message is sent.

 Subclasses usually do some	processing, then pass this message to superclass.
 This message must be passed to ancestor.
*/
#define msgSvcCloseRequested					MakeMsg(clsService, 5)


/****************************************************************************
 msgSvcQueryLockRequested  		 	takes pNull, returns STATUS
	Client asked to QueryLock this service. 

 A client has sent msgSMQueryLock to a service manager. QueryLocking a 
 service lets the client get access to the service without opening it. 
 However, if style.exclusiveOpen is true then the QueryLock counts as an
 open as far as allowing only one open at a time.
 
 Subclasses usually let ancestor handle this message. This message must
 be passed to ancestor.
*/
#define msgSvcQueryLockRequested	  			MakeMsg(clsService, 6)


/****************************************************************************
 msgSvcQueryUnlockRequested  		 	takes pNull, returns STATUS
	Client asked to QueryUnlock this service. 

 A client has sent msgSMQueryUnlock to a service manager. This releases
 a previous QueryLock. 
 
 Subclasses usually let ancestor handle this message. This message must
 be passed to ancestor.
*/
#define msgSvcQueryUnlockRequested	  			MakeMsg(clsService, 7)


/****************************************************************************
 msgSvcCharactersticsRequested 	 	takes P_SVC_CHARACTERISTICS, returns STATUS
	Client asked to get characteristics of this service.

 A client sent msgSMGetCharacteristics to a service manager. The service 
 will return service-specific characteristics via pArgs->pBuf. pArgs->len
 specifies the maximum size of the client's buffer. If pArgs->len is 0 
 then the service should return the actual size of its characteristics
 in pArgs->len and not pass back any data.
*/
#define msgSvcCharacteristicsRequested			MakeMsg(clsService, 54)

typedef struct SVC_CHARACTERISTICS {
	OBJECT				handle;	  // Handle of item to get characteristics of.
	P_UNKNOWN			pBuf; 	  // Out through Ptr: Characterisitics buffer.
	U16					len;	  // In/Out: Buffer size. If 0 then the 
								  // actual size is returned.
} SVC_CHARACTERISTICS, *P_SVC_CHARACTERISTICS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *      	      	   			     Tags								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#define tagServiceClassOptionSheet				MakeTag(clsService, 1) 
#define tagServiceFirstTime						MakeTag(clsService, 2) 

// Next message up: 59


// Obsolete, here for backwards compatibility.
STATUS EXPORTED InitService(
	P_STRING				pReserved1,  // Set this to pNull.
	CLASS					serviceClass,// class id.
	BOOLEAN					autoCreate,  // Create an instance for each state 
										 // node at install and warm boot times.
	U32						serviceType, // Global service type. See 
										 // svctypes.h. Usually set to 0.
	U32						initServiceFlags, // Or-in InitService flags.
	U32						reserved2,	 // Set this to 0
	U32						reserved3);	 // Set this to 0

#endif	// SERVICE_INCLUDED

											
