/****************************************************************************
 File: option.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.75  $
   $Author:   cmeyer  $
     $Date:   27 Mar 1992 15:57:44  $

 This file contains the API for clsOption.

 clsOption inherits from clsFrame. 
 Provides the standard looks, behavior, and protocol of option sheets. 

 An option sheet is a special kind of frame that you can use to display
 the properties of a selected object. If the selected object has several
 different sets of properties, then the option sheet will have several
 windows stacked in it like a deck of cards. Each of these windows is 
 called an option card. For more information on option cards, please see 
 clsOptionTable (in opttable.h).

 The user navigates between the option cards with a popup choice, which
 is available on the title line of the option sheet. The popup choice
 contains a clsTabButton for each option card. The typical PenPoint 
 developer does not need to know about how option sheets use clsTabButton, 
 but feel free to take a look at it (in tbutton.h).

 Although clsOption provides a rich API, most PenPoint developers need 
 to understand only the following:

 Messages sent by a client to an option sheet: 
	msgOptionAddCard
	msgOptionAddLastCard
        
 Messages sent to a sheet's client by an option sheet: 
	msgOptionClosed
	msgOptionProvideTopCard
        
 Messages sent to a card's client by an option sheet: 
	msgOptionProvideCardWin
	msgOptionApplyCard
	msgOptionRefreshCard
	msgOptionApplicableCard
        
 Messages self-sent by a client to create an option sheet: 
	msgOptionCreateSheet
	msgOptionAddCards
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsOption debugging flag is '%'.  Defined values are:

    flag8  (0x0100):    general
*/

#ifndef OPTION_INCLUDED
#define OPTION_INCLUDED

												#ifndef FRAME_INCLUDED
#include <frame.h>
												#endif
												#ifndef TKTABLE_INCLUDED
#include <tktable.h>
												#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define tagOptionApplyButton	  			MakeTag(clsOption, 1)
#define tagOptionApplyAndCloseButton		MakeTag(clsOption, 2)
#define tagOptionCloseButton				MakeTag(clsOption, 3)

#define hlpOptionApplyButton	  			tagOptionApplyButton
#define hlpOptionApplyAndCloseButton		tagOptionApplyAndCloseButton
#define hlpOptionCloseButton	  			tagOptionCloseButton

typedef OBJECT	OPTION;

/****  Sheet Modality Style  ****/
/* 
 The sheet modality style specifies whether the card is modal, and if so,
 whether system-modal or application-modal.
*/
#define osModalNone		0
#define osModalApp		1
#define osModalSystem	2

/****  Card Navigation Style  ****/
/* 
 The card navigation style specifies how the user can move between 
 option cards. GO recommends that you use a popup choice. 
*/

#define osNavPopup		0		// popup choice in the title bar
#define osNavTabBar		1		// tab buttons in the tab bar
//						2		// unused (reserved)
//						3		// unused (reserved)

typedef struct OPTION_STYLE {
	U16	senseSelection	: 1,	// observe theSelectionManager
		modality		: 2,	// whether modal, and what type
		cardNav			: 2,	// card navigation style
		getCards		: 1,	// true => enable msgOptionAddCards protocol
		needCards		: 1,	// true => current list of cards is invalid
		needTopCard		: 1,	// true => current top card is invalid
		hideNav			: 1,	// true => hide card navigation
		spare1			: 7;	// unused (reserved)

	U16	spare2			: 16;	// unused (reserved)
} OPTION_STYLE, *P_OPTION_STYLE;
	
/*
 Default OPTION_STYLE:
//{
     senseSelection = true
     modality       = osModalNone
     cardNav        = osNavPopup
     getCards		= false
 	 needCards		= true
 	 needTopCard	= true
 	 hideNav		= false
//}
*/

typedef struct OPTION_CARD {
	OPTION	option;   		// out: option sheet sending the msg.
	U32   	tag;      		// in: tag for tab
	WIN   	win;      		// in: card window or objNull
	P_CHAR	pName;    		// in: card name
	U16   	nameLen;  		// in: max. len for pName (for msgOptionGetCardAndName)
	OBJECT	client;   		// in: for msgOptionRefreshCard, etc.
	U32		clientData[2];	// in: arbitrary client data
	U32		spare1;			// unused (reserved)
	U32		spare2;			// unused (reserved)
} OPTION_CARD, *P_OPTION_CARD;

typedef struct OPTION_TAG {
	OPTION  option;
	TAG     tag;
} OPTION_TAG, *P_OPTION_TAG;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew			takes P_OPTION_NEW, returns STATUS
	category: class message
	Creates an option sheet.

 If pArgs->option.style.cardNav is osNavPopup, clsOption will create an
 instance of clsTkTable with a label and a popupChoice in it as the frame's
 title bar.  The label string will be set to the frame's title string.
 The popup choice will contain a choice for each card in the option sheet.
*/

typedef struct OPTION_NEW_ONLY {
	OPTION_STYLE  		style;				// overall style
	P_TK_TABLE_ENTRY	pCmdBarEntries;		// optional override
	U32					spare1;				// unused (reserved)
	U32					spare2;				// unused (reserved)
} OPTION_NEW_ONLY, *P_OPTION_NEW_ONLY;

/*
 If pCmdBarEntries is not null, then it should be the address of a 
 null-terminated array of entries.  It is used to create a custom command
 bar rather than the usual Apply and Apply&Close buttons.  The client of 
 this custom command bar is set to the frame's client.
*/

#define	optionNewFields	\
	frameNewFields		\
	OPTION_NEW_ONLY		option;

typedef struct OPTION_NEW {
	optionNewFields
} OPTION_NEW, *P_OPTION_NEW;


/****************************************************************************
 msgNewDefaults			takes P_OPTION_NEW, returns STATUS
	category: class message
	Initializes the OPTION_NEW structure to default values.

 	Zeroes out pArgs->option and sets:
//{
	pArgs->win.flags.style |= wsSendGeometry | wsSaveUnder;

	pArgs->embeddedWin.style.selection = ewSelectPreserve;

	pArgs->border.style.shadow = bsShadowThickGray;
	pArgs->border.style.resize = bsResizeBottom;
	pArgs->border.style.drag = bsDragDown;
	pArgs->border.style.backgroundInk = bsInkGray33;
	pArgs->border.style.edge = bsEdgeAll;
	pArgs->border.style.leftMargin = bsMarginMedium;
	pArgs->border.style.rightMargin = bsMarginMedium;
	pArgs->border.style.bottomMargin = bsMarginMedium;
	pArgs->border.style.topMargin = bsMarginLarge;

	pArgs->frame.style.clipBoard = true;
	pArgs->frame.style.closeBox = false;
	pArgs->frame.style.zoomable = false;
	pArgs->frame.style.cmdBar = true;

	pArgs->option.style.senseSelection = true;
	pArgs->option.style.needCards = true;
	pArgs->option.style.needTopCard	= true;
	pArgs->option.style.cardNav = osNavPopup;
//}
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 The option sheet saves its style and the tag of the current top card.  
 This tag is used as the default value for the top card when 
 msgOptionProvideTopCard is next sent (e.g., after the option sheet 
 is restored and inserted in the window tree).

 Saving an option sheet causes msgSave to be sent to each of the
 option card's tab buttons. If a card's client is OSThisApp(), its
 tab button records and saves this fact. Otherwise, the client is
 not saved.
*/

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 The option sheet restores its instance data and sets the following:

	style.needTopCard = true;
	style.needCards = true;

 If the restored frame has a command bar, msgTkTableSetClient is
 sent to it to force its client to be the option sheet.

 If style.getCards and style.senseSelection are true, the option sheet
 is set up to observe theSelectionManager.

 Restoring an option sheet causes msgRestore to be sent to each of the
 option card's tab buttons. If a card's client was OSThisApp(), its
 tab button sets the client to the new value for OSThisApp(). 
 Other cards have their client set to objNull.
*/


/****************************************************************************
 msgOptionGetStyle				takes P_OPTION_STYLE, returns STATUS
	Passes back the style of the option sheet.

 Most clients do not need to deal with this message.
*/
#define msgOptionGetStyle		MakeMsg(clsOption, 1)


/****************************************************************************
 msgOptionSetStyle				takes P_OPTION_STYLE, returns STATUS
	Sets the style of the option sheet.

 Note that changing style.cardNav is not supported.

 Most clients do not need to deal with this message.
*/
#define msgOptionSetStyle		MakeMsg(clsOption, 2)


/****************************************************************************
 msgOptionGetNeedCards		takes P_BOOLEAN, returns STATUS
	Passes back the value of style.needCards.

 Most clients do not need to deal with this message.
*/
#define msgOptionGetNeedCards	MakeMsg(clsOption, 34)

/****************************************************************************
 msgOptionSetNeedCards		takes BOOLEAN, returns STATUS
	Sets style.needCards.

 If style.needCards and style.getCards are true, the option sheet  
 self-sends msgOptionGetCards when the current cards are needed.

 Most clients do not need to deal with this message.
*/
#define msgOptionSetNeedCards	MakeMsg(clsOption, 35)

/****************************************************************************
 msgOptionGetCard				takes P_OPTION_CARD, returns STATUS
	Passes back some information about a card in the option sheet.

 In parameters:
 	tag:		tag of the card to get.

 Out parameters:
 	win:		uid of the card.
 	client:		client of the card.

 Will return stsBadParam if a card matching the passed tag was not found
 in the option sheet.

 Most clients do not need to deal with this message.
*/
#define msgOptionGetCard		MakeMsg(clsOption, 3)


/****************************************************************************
 msgOptionGetTopCard			takes P_OPTION_CARD, returns STATUS
	Passes back some information about the top card in the option sheet.

 Out parameters:
 	tag:		tag of the top card.
 	win:		uid of the card.
 	client:		client of the card.

 If there is no top card, the option sheet sets all of the out parameters 
 to null.

 Most clients do not need to deal with this message.
*/
#define msgOptionGetTopCard		MakeMsg(clsOption, 25)


/****************************************************************************
 msgOptionGetCardAndName		takes P_OPTION_CARD, returns STATUS
	Passes back some information about a card in the option sheet.

 In parameters:
	tag:		tag of the card to get.
	pName:		pointer to a buffer in which to put the card's name.
	nameLen:	size of pName buffer in bytes (if 0, pName is ignored).

 Out parameters:
	win:		uid of the card.
	client:		client of the card.
	pName:		buffer is filled in with the first nameLen bytes of
				the name of the card (if 0 was not passed for nameLen).

 Will return stsBadParam if a card matching the passed tag was not found
 in the option sheet.

 Most clients do not need to deal with this message.
*/
#define msgOptionGetCardAndName		MakeMsg(clsOption, 20)


/****************************************************************************
 msgOptionEnumCards			takes P_OPTION_ENUM, returns STATUS
	Enumerates the tags of the cards in the option sheet.

 This message is sent to enumerate all of the cards that have been
 added to the option sheet.  Typical usage is shown below.

//{
	TAG			cards[10];
	OPTION_ENUM	oe;
  	
  	oe.max		= 10;		// we have space for 10 card tags
	oe.count	= maxU16;	// we want all the card tags
	oe.pTag		= cards;	// our tag buffer
	oe.next		= 0;		// first call to msgOptionEnumCards
	oe.flags	= 0;		// unused for now

	ObjCallRet(msgOptionEnumCards, sheet, &oe, s);
	// oe.pTag[0 .. oe.count] is the array of card tags
	// ...
	// free any allocated memory when finished with the tags
	if (oe.pTag != cards)
		StsWarn(OSHeapBlockFree(oe.pTag));
//}	


 Most clients do not need to deal with this message.
*/
#define msgOptionEnumCards		MakeMsg(clsOption, 33)

typedef struct OPTION_ENUM {
	U16		max,	// in = size of pTags[] array
			count;	// in = # to return in array
					// if count > max then memory may be allocated
					// out = # of valid entries in array
                                
	P_TAG	pTag;	// in = ptr to array of card tags
					// out = if memory was allocated
					// client should free the memory using OSHeapBlockFree()
                                
	U16		next;	// in  = 0 to start at beginning 
					//       OR previous out value to pick up
					//       where we left off
					// out = where we left off
	U32		flags;	// in  = various flags (must be 0 for now)
	U32		spare;	// unused (reserved)
} OPTION_ENUM, *P_OPTION_ENUM;

/****************************************************************************
 msgOptionSetCard	takes P_OPTION_CARD, returns STATUS
	Changes some of the information of a card in the option sheet.

 In parameters:
	tag:		tag of the card to set.
	client:		client for the card.
	win:		window for the card.
	pName:		pointer to a buffer holding a new name, or pNull to keep
				the old name.

 The option sheet changes the various parameters of the specified card.
 To avoid changing the name of the card, set pArgs->pName to pNull.

 Most clients do not need to deal with this message.
*/
#define msgOptionSetCard		MakeMsg(clsOption, 4)


/****************************************************************************
 msgOptionAddCard				takes P_OPTION_CARD, returns STATUS
	Adds a card to the option sheet.

 In parameters:
 	tag:		tag of the card to set.
 	pName:		pointer to a buffer holding the card's name.
 	win:		window for the card.
 	client:		client for the card.
 	clientData:	any client data you want stored with the card.

 If the card specified by pArgs->tag has already been added to the option
 sheet, the following is done:
  	- if pArgs->win is objNull, the window for the card is unchanged.
	- otherwise, the current window for the card is destroyed and replaced 
	  by pArgs->win.
	- if pArgs->pName is not pNull, the new name is used.
	- the card client is replaced by pArgs->client.

 Note that the card's tag is also used as the helpId of the tab button
 representing the card (in the popup choice card navigation menu or the tab
 bar).  The caller should insure that quick help exists for the card with
 the card's tag as the helpId.

 Most clients send this message to add a card to an option sheet (if there
 is more than one card). 

 See Also 
	msgOptionAddLastCard
*/
#define msgOptionAddCard		MakeMsg(clsOption, 5)


/****************************************************************************
 msgOptionAddLastCard				takes P_OPTION_CARD, returns STATUS
	Adds the last card of a group to the option sheet.

 This is the same as msgOptionAddCard, except that the menu button for
 this card has a line break after it.

 Most clients send this message to add the last card to an option sheet.
 
 See Also
  	msgOptionAddCard
*/
#define msgOptionAddLastCard		MakeMsg(clsOption, 29)


/****************************************************************************
 msgOptionAddFirstCard				takes P_OPTION_CARD, returns STATUS
	Adds the first card of a group to the option sheet.

 This is the same as msgOptionAddCard, except that the menu button for
 this card has a line break before it.

 Most clients don't need to send this message.

 See Also
  	msgOptionAddCard
*/
#define msgOptionAddFirstCard		MakeMsg(clsOption, 42)


/****************************************************************************
 msgOptionAddAndInsertCard		takes P_OPTION_CARD, returns STATUS
	Adds a card to the option sheet and inserts it into the sheet.

 This message is handled exactly as in msgOptionAddCard, including the
 case in which pArgs->tag has already been added to the sheet.

 Normally, msgOptionAddCard does not actually insert the card's window
 into the option sheet's window tree.  msgOptionAddAndInsertCard
 does insert the window.

 Most clients do not need to deal with this message.

 See Also
  	msgOptionAddCard
*/
#define msgOptionAddAndInsertCard	MakeMsg(clsOption, 17)


/****************************************************************************
 msgOptionRemoveCard			takes P_OPTION_CARD, returns STATUS
	Removes a card from an option sheet and destroys that card.

 The option sheet removes and destroys the specified card. It also
 removes the window for the card, but does not destroy the window. 

 In parameters:
	tag:		tag of card to remove.

 Will return stsBadParam if a card matching the passed tag was not found
 in the option sheet.

 Most clients do not need to deal with this message.

 See Also
	msgOptionExtractCard
*/
#define msgOptionRemoveCard		MakeMsg(clsOption, 6)


/****************************************************************************
 msgOptionExtractCard			takes P_OPTION_CARD, returns STATUS
	Extracts a card's window from an option sheet.

 The option sheet extracts the card's window, but does not destroy it.
 Note that the tab button for the card remains, with its win set to objNull.

 In parameters:
	tag:		tag of card to extract.

 Out parameters:
	win:		win of extracted card.

 Will return stsBadParam if a card matching the passed tag was not found
 in the option sheet.

 Most clients do not need to deal with this message.

 See Also
	msgOptionRemoveCard
*/
#define msgOptionExtractCard		MakeMsg(clsOption, 19)


/****************************************************************************
 msgOptionShowCard				takes P_OPTION_CARD, returns STATUS
	Causes the specified card to be displayed as the current card.
    
 The option sheet sends msgOptionRefreshCard to the card.

 In parameters:
	tag:		tag of card to show.

 Out parameters:
	win:		uid of card.
	client:		client of card.

 Will return stsBadParam if a card matching the passed tag was not found
 in the option sheet.

 Most clients do not need to deal with this message.
*/
#define msgOptionShowCard		MakeMsg(clsOption, 14)

/****************************************************************************
 msgOptionShowCardAndSheet			takes TAG, returns STATUS
	Causes the specified card to be displayed as the current card.

 The sheet is shown if it is not currently shown.

 The option sheet self-sends msgOptionShowCard(OPTION_CARD.tag = pArgs),
 followed by msgOptionShowSheet.

 Most clients do not need to deal with this message.

 See Also
 	msgOptionShowCard
	msgOptionShowSheet
*/
#define msgOptionShowCardAndSheet		MakeMsg(clsOption, 44)

/****************************************************************************
 msgOptionShowTopCard			takes nothing, returns STATUS
 	Shows the client-defined top card.

 The option sheet sends msgOptionProvideTopCard to its client
 with the following OPTION_CARD parameters:

  	option	= uid of the option sheet
	tag 	= tag of the current top card
	win		= win of the current top card
	pName	= pNull
	nameLen	= 0
	client	= client of the current top card

 The option sheet then shows the new top card specified by OPTION_CARD.tag 
 by self-sending msgOptionShowCard.

 Most clients do not need to deal with this message.
*/
#define msgOptionShowTopCard		MakeMsg(clsOption, 30)


/****************************************************************************
 msgOptionGetCards			takes nothing, returns STATUS
 	Gets the cards from the option sheet's client

 If style.getCards is false, this message is ignored.
 Otherwise, the option sheet sends msgOptionAddCards to its
 client with the following OPTION_TAG parameters:

  	option	= uid of the option sheet
	tag 	= tag of the option sheet

 Most clients do not need to deal with this message.
*/
#define msgOptionGetCards		MakeMsg(clsOption, 32)


/****************************************************************************
 msgOptionApply					takes nothing, returns STATUS
	Tell the option sheet to initiate the Apply protocol.

 This message is sent by the sheet's Apply button.
 The option sheet sends msgOptionApplyCard to the top card.

 Most clients do not need to deal with this message.
*/
#define msgOptionApply			MakeMsg(clsOption, 8)


/****************************************************************************
 msgOptionApplyAndClose			takes nothing, returns STATUS
	Tell an option sheet to run the Apply protocol and then close itself.

 This message is sent by the sheet's Apply&Close button.
 The option sheet:
	sends msgOptionApplyCard to the top card in the sheet, and
	sends msgOptionClosed to the sheet's client.

 Most clients do not need to deal with this message.
*/
#define msgOptionApplyAndClose	MakeMsg(clsOption, 9)


/****************************************************************************
 msgOptionRefresh				takes nothing, returns STATUS
	Tells an option sheet to refresh its card settings.

 This is sent to an option sheet by the default application code
 when it receives a forwarded "check" gesture.

 If the apply buttons in the command bar are grayed out (i.e., the top
 card is not applicable), nothing is done, and stsOK is returned.

 Otherwise, the option sheet sends msgOptionRefreshCard to its top card.
 It then marks the other cards as needing to be refreshed when shown.

 Most clients do not need to deal with this message.
*/
#define msgOptionRefresh		MakeMsg(clsOption, 21)


/****************************************************************************
 msgOptionApplicable			takes P_BOOLEAN, returns STATUS
	Tells an option sheet to ask the top card if it is applicable.

 The option sheet sends msgOptionApplicableCard to its top card.  It then
 marks the other cards as needing to be sent msgOptionApplicableCard 
 when shown.

 If the top card is not applicable, the command bar buttons are grayed out.

 If pArgs is not pNull, true is passed back if the top card is
 applicable; otherwise false is passed back.

 Most clients do not need to deal with this message.

*/
#define msgOptionApplicable		MakeMsg(clsOption, 37)


/****************************************************************************
 msgOptionDirty					takes nothing, returns STATUS
	Tells an option sheet to ask the top card to dirty its controls.

 The option sheet sends msgOptionDirtyCard to its top card.  It then marks
 the other cards as needing to be sent msgOptionDirtyCard when shown.

 Most clients do not need to deal with this message.
*/
#define msgOptionDirty			MakeMsg(clsOption, 38)


/****************************************************************************
 msgOptionClean			takes nothing, returns STATUS
	Tells an option sheet to ask the top card to clean its controls.

 The option sheet sends msgOptionCleanCard to its top card.  The other
 cards are NOT cleaned.

 Most clients do not need to deal with this message.
*/
#define msgOptionClean		MakeMsg(clsOption, 39)


/****************************************************************************
 msgOptionToggleDirty 		takes nothing, returns STATUS
	Tells an option sheet to toggle the dirty/clean state of the cards.

 The option sheet sends msgOptionProvideCardDirty to the top card's client
 to determine the dirty/clean state of the top card.  If the client
 responds with stsNotUnderstood, the option sheet sends msgBorderGetDirty
 to the top card's window to determine the dirty/clean state.

 If the top card is clean, msgOptionDirty is then self-sent; otherwise
 msgOptionClean is self-sent.

 Most clients do not need to deal with this message.
*/
#define msgOptionToggleDirty			MakeMsg(clsOption, 40)


/****************************************************************************
 msgOptionClose					takes nothing, returns STATUS
	Tells an option sheet to close itself.

 When a sheet receives msgOptionClose, it sends msgOptionClosed
 to the sheet's client.

 A sheet self-sends msgOptionClose when it receives msgFrameClose.

 Most clients do not need to deal with this message.
*/
#define msgOptionClose			MakeMsg(clsOption, 10)


/****************************************************************************
 msgOptionGetCardMenu			takes P_MENU, returns STATUS
	Passes back the card navigation menu.

 A copy of the popup card navigation menu is passed back.
 The option sheet returns objNull if style.cardNav is not osNavPopup.

 The menu buttons in the navigation menu have the option sheet as their 
 client, msgOptionShowCardAndSheet as their message, and the appropriate 
 card tag as their data.  This causes the sheet being displayed and 
 the appropriate card being turned to when the user taps on a menu button.

 The caller must send msgOptionCardMenuDone when finished with the menu.

 Most clients do not need to deal with this message.

 See Also
 	msgOptionShowCardAndSheet
	msgOptionCardMenuDone
*/
#define msgOptionGetCardMenu		MakeMsg(clsOption, 26)

/****************************************************************************
 msgOptionCardMenuDone		takes MENU, returns STATUS
	Indicates the caller is finished with the card menu.

 This message should be sent to an option sheet when the card menu retrieved
 via msgOptionGetCardMenu is no longer needed.

 Most clients do not need to deal with this message.
*/
#define msgOptionCardMenuDone		MakeMsg(clsOption, 27)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Messages Option Sheets send to each card's client		   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgOptionShowSheet			takes P_OPTION_TAG, returns STATUS
	category: client responsibility
	Asks the client of the option sheet to show the option sheet.

 This message is sent by the option sheet when the user taps on a menu button 
 in the card menu and the option sheet is not inserted in the window tree.

 The client should respond by inserting the option sheet into the window
 tree.
*/
#define msgOptionShowSheet		MakeMsg(clsOption, 28)


/****************************************************************************
 msgOptionProvideCardWin			takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Asks the client of the card to provide the window for the card.

 This message is sent by the option sheet when a card is about to be
 shown, and the window for the card is objNull.

 The card client should set pArgs->win to the desired card window.

 Most clients need to override and handle this message.
*/
#define msgOptionProvideCardWin		MakeMsg(clsOption, 18)

/****************************************************************************
 msgOptionProvideTopCard			takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Asks the client of the option sheet to provide the tag for the top card.

 This message is sent by the option sheet when the top card must be shown.
 This can be in response to msgOptionShowTopCard or when the option sheet
 is first inserted.

 The option sheet sends msgOptionProvideTopCard to its client
 with the following OPTION_CARD parameters:

  	option	= uid of the option sheet
	tag 	= tag of the current top card
	win		= win of the current top card
	pName	= pNull
	nameLen	= 0
	client	= client of the current top card

 The option sheet's client should set pArgs->tag to the tag for the desired
 top card.

 Note that only pArgs->tag is used as an out parameter; other changes to
 pArgs are ignored.

 See Also
 	msgOptionShowTopCard
*/
#define msgOptionProvideTopCard		MakeMsg(clsOption, 31)

/****************************************************************************
 msgOptionProvideCardDirty			takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Asks the client of the card to provide the dirtiness of the card window.

 This message is sent by the option sheet in response to msgOptionToggleDirty.

 The card's client should return stsOK if the card is dirty, stsRequestDenied
 if clean.

 Most clients do not need to deal with this message.
*/
#define msgOptionProvideCardDirty		MakeMsg(clsOption, 41)

/****************************************************************************
 msgOptionApplyCard				takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	This is sent to a card's client when the card should apply its settings.

 With this message, an option option sheet tells a card to apply its
 settings to the selection.  This is sent whenever the user chooses
 Apply or Apply&Close on the option sheet.

 Most clients need to override and handle this message.

 Here is the typical sequence of steps a card client should take in response:

 Run through every control in the card and for each one 1) check to
 see if it's dirty, and if it is 2) validate it if necessary.  If any
 control has an invalid value, return stsFailed from the handler for
 msgOptionApplyCard.  (This step can be omitted if there's no way any
 control could have an invalid value.)

 Again make a pass through every control in the card.  If a control is
 dirty, apply its value.

 Finally, clean all the controls in the card.  This can usually be done
 by sending msgControlSetDirty(false) to the card window.  Note that
 most "command sheets" should have their control's CONTROL_STYLE.showDirty
 set false, and so this final step should be omitted.
*/
#define msgOptionApplyCard 		MakeMsg(clsOption, 12)


/****************************************************************************
 msgOptionRefreshCard			takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Tells a card's client to refresh its settings from the current selection.

 This is sent to a card's client when the option sheet has received
 msgOptionRefresh.  The client should refresh the card's settings 
 from the current selection.

 Most clients need to override and handle this message.
*/
#define msgOptionRefreshCard	MsgNoError(MakeMsg(clsOption, 11))


/****************************************************************************
 msgOptionApplicableCard		takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Finds out if a card is applicable to the current selection.

 The card's client should respond by returning stsOK if the card
 can be applied to the current selection, stsFailed if not.

 Most clients need to override and handle this message.
*/
#define msgOptionApplicableCard		MakeMsg(clsOption, 22)


/****************************************************************************
 msgOptionDirtyCard				takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Sent to a card's client when the card should dirty all its controls.

 This is sent when the user changes the selection while an option
 sheet is up.  It is needed so that if the card is applied to
 the new selection, every property on the card is applied,
 not just those changed by the user since the last apply.

 The usual scenario is for the card window to inherit from clsBorder,
 whose instances respond to msgBorderSetDirty by forwarding that message
 on to their immediate children.  Card clients may elect NOT to 
 respond to msgOptionDirtyCard--if the option sheet code gets back
 stsNotUnderstood, then it will send msgBorderSetDirty(true) 
 to the card window.

 Most clients do not need to deal with this message.
*/
#define msgOptionDirtyCard			MakeMsg(clsOption, 23)


/****************************************************************************
 msgOptionCleanCard				takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Sent to a card's client when the card should clean all its controls.

 This is sent after msgOptionApplyCard is sent.

 The usual scenario is for the card window to inherit from clsBorder,
 whose instances respond to msgBorderSetDirty by forwarding that message
 on to their immediate children.  Card clients may elect to NOT
 respond to msgOptionCleanCard--if the option sheet code gets back
 stsNotUnderstood, then it will send msgBorderSetDirty(false) to the card
 window.

 Most clients do not need to deal with this message.
*/
#define msgOptionCleanCard			MakeMsg(clsOption, 36)


/****************************************************************************
 msgOptionUpdateCard			takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Sent to a card's client every time the card is about to be shown.

 Most clients do not need to respond to this message.  It is intended
 for those circumstances where one card has dependencies on the state
 of another, and would need to look at that other card before being
 (re)displayed to the user.

 See Also
  	msgOptionRetireCard
*/
#define msgOptionUpdateCard			MsgNoError(MakeMsg(clsOption, 24))


/****************************************************************************
 msgOptionRetireCard			takes P_OPTION_CARD, returns STATUS
	category: client responsibility
	Sent to a card's client every time the current shown card is hidden.

 Most clients do not need to respond to this message.  It is intended
 for those circumstances where one card builds a context (e.g., allocates
 resources) when shown, and needs to destroy the context when the card
 is no longer shown.  This can happen when another card is turned to
 or when the option sheet is extracted or destroyed.

 See Also
  	msgOptionUpdateCard
*/
#define msgOptionRetireCard			MsgNoError(MakeMsg(clsOption, 43))


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Messages Option Sheets send to their frame's client		   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgOptionClosed				takes OPTION, returns STATUS
	category: client responsibility
	This is sent to an option sheet's client when the sheet is closed.

 The client should respond by using msgAppRemoveFloatingWin to take down
 the option sheet, then optionally destroying the sheet with msgDestroy.
*/
#define msgOptionClosed			MakeMsg(clsOption, 13)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages sheet clients should self-send				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgOptionCreateSheet			takes P_OPTION_TAG, returns STATUS
	category: descendant responsibility
	A message sent by convention by clients creating option sheets.

 When you need to create an option sheet, you should self-send this
 this message, rather than directly creating a sheet. By following
 this convention, subclasses can modify the sheet or supply a different 
 one (which would have to behave the same as the original).

 When self-sending this message, the client should fill in the 'tag'
 of the option sheet desired (if applicable) or some other identifying
 value (some clients may create different kinds of option sheets).  
 The client should also zero out the 'option' field of the
 OPTION_TAG struct.

 In msgOptionCreateSheet, a client creates an EMPTY option sheet 
 and fills in the 'option' field with the uid of the sheet.  Subclasses
 handle this message by calling the ancestor's handler and then either
 modifying the sheet or supplying a new one (and destroying any non-
 null sheet already in the 'option' field).
*/
#define msgOptionCreateSheet	MakeMsg(clsOption, 16)


/****************************************************************************
 msgOptionAddCards				takes P_OPTION_TAG, returns STATUS
	category: descendant responsibility
	A message to be sent by convention by clients creating option sheets.

 This message embodies the second step of creating an option sheet.
 Just like msgOptionCreateSheet, msgOptionAddCards is self-sent by a client
 to fill in a sheet with some cards, and to allow subclasses of the client
 to modify cards or add different ones.

 if style.getCards is true, the option sheet sends this message to the frame's
 client as follows:
 	- when the sheet is first inserted into the window tree
	- if style.cardNav is osNavPopup, when the card navigation menu is needed 
	  after the selection has changed.
*/
#define msgOptionAddCards		MakeMsg(clsOption, 15)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgContentsButtonGoto		takes TAG, returns STATUS
	category: client notification
	Default message sent when the user taps on a menu button.
 
 This is also sent to the client when the managed button is hit.

 The option sheet responds by self-sending msgOptionShowCard with 
 the following OPTION_CARD parameter:
  	tag	= pArgs;
*/

/****************************************************************************
 msgOptionBookProvideContents		takes P_WIN, returns STATUS
	Receiver passes back a window representing its contents.

 The option sheet responds by creating an instance of clsContentsTable with
 one clsContentsButton child for each card in the option sheet.  Cards
 which themselves respond to msgOptionBookProvideContents are represented
 by cbSection style contents buttons.
*/

#endif	// OPTION_INCLUDED
