/****************************************************************************
 File: notepapr.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.37  $
   $Author:   bschwart  $
     $Date:   28 Feb 1992 09:32:32  $

 This file contains the API definition for clsNotePaper.

 clsNotePaper inherits from clsView.

 NotePaper is the view class for PenPoint's ink-management or note-taking
 building block.  Most of the code for the MiniNote application actually
 resides in the building block.  Other classes of the building block are
 clsNPData (the data class), clsNPItem (the generic data item),
 clsNPScribbleItem (the ink data item), clsNPTextItem (the text data item),
 and clsGestureMargin (the subclass of clsScrollWin that implements
 MiniNote's gesture margin).												   
		   
 NotePaper provides standard PenPoint functionality including embedding,
 undo, move/copy, import, export, option sheets, and marks.  (Supporting
 marks means that search and replace, spell, proof, and reference buttons are
 all supported.)

 NotePaper displays (and alters) the contents of an NPData object.
 For PenPoint 1.0, NotePaper keeps all of the items in its data object in
 a coordinate system with (0,0) its upper-left corner.  As a result, all
 the items in the data object have a negative y coordinate.  This means that
 as the NotePaper window grows in width and height, its contents remain
 relative to the top-left corner of the page.

 A sample applications (called npapp or "NotePaper App") demonstrating the
 use of the ink building block is included in the SDK.  The ink building
 block is distributed as part of the SDK as a distributed DLL.  The
 DLL and all resources used by the ink building block are included in the
 SDK in the DLL\NOTEPAPR directory.  The resources in that directory include:
//{				
	notepaper.res:		contains all resources used by NotePaper
	paper.res:			contains the 8 bitmaps representing paper styles
	pen.res:			contains the 4 bitmaps representing pen styles
	strings.rc:			contains the source for quick help, error text,
						and undo strings
//}


****************************************************************************/
#ifndef NOTEPAPR_INCLUDED
#define NOTEPAPR_INCLUDED

#ifndef VIEW_INCLUDED
#include <view.h>
#endif
#ifndef SYSFONT_INCLUDED
#include <sysfont.h>
#endif
#ifndef ITOGGLE_INCLUDED
#include <itoggle.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Types and Constants                                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define clsNotePaper		MakeGlobalWKN(2567,1)

#define stsNotePaperNoHit			MakeWarning(clsNotePaper, 0)
#define stsNotePaperTreatAsInk		MakeWarning(clsNotePaper, 1)

Enum16(NP_PAPER_STYLE) {

	npPaperRuled				= 0,
	npPaperRuledLeftMargin		= 1,
	npPaperRuledCenterMargin	= 2,
	npPaperRuledLegalMargin		= 7,

	npPaperBlank				= 3,
	npPaperLeftMargin			= 4,
	npPaperCenterMargin			= 6,
	npPaperGrid					= 5,

};

typedef struct NOTE_NP_PAPER_STYLE {

	U16		bEditMode		: 1,	// writing/ink vs. gesture/edit mode
			bAutoGrow		: 1,	// auto grow height as user enters data?
			bWidthOpts		: 1,	// include page widths in option sheet
			bHideTopRule	: 1,	// don't paint the top ruling line for
									// the npPaperRuledxxx paper style
			bVirtualHeight	: 1,	// if set, NotePaper grows itself into
									// a long thin window and responds to
									// scroll win messages
			reserved		: 11;	// always set to 0
	U16		reserved1;

} NOTE_PAPER_STYLE, *P_NOTE_PAPER_STYLE;

typedef struct NOTE_PAPER_METRICS {

	NOTE_PAPER_STYLE	style;
	SYSDC_FONT_SPEC		paperFont;		// defines the font for the paper
	NP_PAPER_STYLE		paperStyle;		// one of the NP_PAPER_STYLE values
	COORD16				lineSpacing;	// (in points) determines font size and
										// vertical spacing
	U8					penStyle;		// use the NPPenStyle() macro

} NOTE_PAPER_METRICS, * P_NOTE_PAPER_METRICS;

/* NOTE: in NPPenStyle, color is one of: bsInkBlack, bsInkGrayXX, or bsInkWhite */
/* NOTE: in NPPenStyle, weight is one of:  1 = bold, 0 = normal                 */
#define NPPenStyle(color, weight)	((color & 0x7) | ((weight & 0x1) << 3))
#define NPPenColor(style)			(style & 0x7)
#define NPPenWeight(style)		((style & 0x8) >> 3)

/* The following definitions are included for convenience only. */
#define npPenFineBlack		NPPenStyle(bsInkBlack, 0)
#define npPenFineGray		NPPenStyle(bsInkGray50, 0)
#define npPenBoldBlack		NPPenStyle(bsInkBlack, 1)
#define npPenBoldGray		NPPenStyle(bsInkGray50, 1)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Messages                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Next up: none; Recycle: 11-51 53 58-101 103 106 120-127 */

/****************************************************************************
 msgNewDefaults				takes P_NOTE_PAPER_NEW, returns STATUS
	Initialize pArgs.

 Zeroes out pArgs->notePaper and sets:
//{
	pArgs->notePaper.style.bEditMode		= false;
	pArgs->notePaper.style.bAutoGrow		= false;
	pArgs->notePaper.style.bWidthOpts		= false;
	pArgs->notePaper.style.bHideTopRule		= false;
	pArgs->notePaper.style.bVirtualHeight	= false;
	pArgs->notePaper.paperStyle				= npPaperRuled;
	pArgs->notePaper.paperFont				= current user font preference
	pArgs->notePaper.penStyle				= NPPenStyle(bsInkBlack, 1);
	pArgs->notePaper.lineSpacing			= 24;	// 24 point
	pArgs->view.createDataObject			= true;
//}

 Various gWin and win flags are set and should only be modified
 by the fearless!

//{
	pArgs->gWin.style.gestureEnable	= true;
	pArgs->gWin.style.gestureForward= true;
	pArgs->win.flags.input &= ~inputInkThrough;
	pArgs->win.flags.input |= inputInk;
	pArgs->win.flags.style |= wsSendGeometry;
	pArgs->win.flags.style |= wsGrowBottom;
	pArgs->win.flags.style |= wsGrowRight;
	pArgs->win.flags.style |= wsCaptureGeometry;
//}

*/

typedef struct {

	NOTE_PAPER_STYLE	style;			// as in NOTE_PAPER_METRICS
	NP_PAPER_STYLE		paperStyle;		// as in NOTE_PAPER_METRICS
	SYSDC_FONT_SPEC		paperFont;		// as in NOTE_PAPER_METRICS
	COORD16				lineSpacing;	// as in NOTE_PAPER_METRICS
	U8					penStyle;		// as in NOTE_PAPER_METRICS

	S32					spares[6];

} NOTE_PAPER_NEW_ONLY, *P_NOTE_PAPER_NEW_ONLY;

#define	notePaperNewFields		\
	viewNewFields			\
	NOTE_PAPER_NEW_ONLY	notePaper;

typedef struct {

	notePaperNewFields

} NOTE_PAPER_NEW, *P_NOTE_PAPER_NEW;

/****************************************************************************
 msgNotePaperGetMetrics				takes P_NOTE_PAPER_METRICS, returns STATUS
	Passes back receiver's metrics.
*/
#define msgNotePaperGetMetrics		MakeMsg(clsNotePaper, 101)

/****************************************************************************
 msgNotePaperGetDcInfo				takes P_NOTE_PAPER_DC_INFO, returns STATUS
	Passes back the drawing contexts used by receiver.
*/
#define msgNotePaperGetDcInfo		MakeMsg(clsNotePaper, 4)

typedef struct {

	U32			units;		// currently, msgDcUnitsTwips
	OBJECT		dc;			// transformed dc in "units"
	OBJECT		dcPen;		// transformed dc in pen units
	U32			reserved[4];

} NOTE_PAPER_DC_INFO, *P_NOTE_PAPER_DC_INFO;

/****************************************************************************
 msgNotePaperGetSelType				takes P_NOTE_PAPER_SEL_TYPE, returns STATUS
	Passes back information about the types of items selected in receiver.
*/
#define msgNotePaperGetSelType		MakeMsg(clsNotePaper, 116)
typedef struct NOTE_PAPER_SEL_TYPE {

	BOOLEAN	bScribble;		// selection contains a scribble
	BOOLEAN	bTranslated;	// selection contains untranslatable text
	BOOLEAN	bReserved1;
	BOOLEAN	bReserved2;

} NOTE_PAPER_SEL_TYPE, * P_NOTE_PAPER_SEL_TYPE;

/****************************************************************************
 msgNotePaperSetEditMode			takes BOOLEAN, returns STATUS
	Sets receiver to either gesture/edit (true) or writing/ink (false) mode.
*/
#define msgNotePaperSetEditMode		MakeMsg(clsNotePaper, 102)

/****************************************************************************
 msgNotePaperSetPaperAndPen			takes P_NOTE_PAPER_METRICS, returns STATUS
	Sets paperStyle, lineSpacing, penColor, and penWeight.

 This message does not affect the pen style for selected items.
*/
#define msgNotePaperSetPaperAndPen	MakeMsg(clsNotePaper, 104)

/****************************************************************************
 msgNotePaperSetPenStyle			takes U32, returns STATUS
	Sets the pen style for selected items as well as the default for new items.
*/
#define msgNotePaperSetPenStyle		MakeMsg(clsNotePaper, 109)

/****************************************************************************
 msgNotePaperGetPenStyle			takes U32, returns STATUS
	Gets the pen style for selected items (or the default if nothing selected).
*/
#define msgNotePaperGetPenStyle		MakeMsg(clsNotePaper, 112)

/****************************************************************************
 msgNotePaperSetStyle				takes P_NOTE_PAPER_STYLE, returns STATUS
	Sets the receiver's style values.
*/
#define msgNotePaperSetStyle		MakeMsg(clsNotePaper, 2)

/****************************************************************************
 msgNotePaperGetStyle				takes P_NOTE_PAPER_STYLE, returns STATUS
	Passes back the receiver's style values.
*/
#define msgNotePaperGetStyle		MakeMsg(clsNotePaper, 3)

/****************************************************************************
 msgNotePaperTranslate				takes P_NULL, returns STATUS
	Translates untranslated scribbles in the selection.
*/
#define msgNotePaperTranslate		MakeMsg(clsNotePaper, 113)

/****************************************************************************
 msgNotePaperUntranslate			takes P_NULL, returns STATUS
	Untranslates translated scribbles in the selection.
*/
#define msgNotePaperUntranslate		MakeMsg(clsNotePaper, 114)

/****************************************************************************
 msgNotePaperEdit					takes P_NULL, returns STATUS
	Edits text and translates and edits scribbles in the selection.
*/
#define msgNotePaperEdit			MakeMsg(clsNotePaper, 115)

/****************************************************************************
 msgNotePaperTidy					takes P_NULL, returns STATUS
	Tidies the selection by normalizing the spacing of items each line.

 The inter-item spacing is determined by sending msgNPItemGetWordSpacing
 to each item to be tidied.
*/
#define msgNotePaperTidy			MakeMsg(clsNotePaper, 105)

/****************************************************************************
 msgNotePaperCenter					takes P_NULL, returns STATUS
	Centers the entire selection.

 The selection is centered on the page as a whole, not line by line.
*/
#define msgNotePaperCenter			MakeMsg(clsNotePaper, 107)

/****************************************************************************
 msgNotePaperAlign					takes U32, returns STATUS
	Aligns the selection according to pArgs.

 Alignment takes place relative to the bounding box of the selection.
*/
#define msgNotePaperAlign			MakeMsg(clsNotePaper, 108)

#define npAlignLeft	1
#define npAlignRight	2

/****************************************************************************
 msgNotePaperMerge					takes P_NULL, returns STATUS
	Joins scribbles and text in the selection.

 Consecutive scribble items are combined into a single scribble item.
 Adjacent text items are combined into a single text item.  Any subclass of
 clsNPItem that can respond to msgNPItemCanJoin and msgNPItemJoin can
 determine its own merging behavior.
*/
#define msgNotePaperMerge			MakeMsg(clsNotePaper, 110)

/****************************************************************************
 msgNotePaperSplit					takes P_NULL, returns STATUS
	Splits scribbles and text.

 First msgNotePaperSplitAsWords is self-sent.  If stsRequestDenied
 is returned, then msgNotePaperSplitAsAtoms is self-sent.
*/
#define msgNotePaperSplit			MakeMsg(clsNotePaper, 111)

/****************************************************************************
 msgNotePaperAddMenus				takes OBJECT, returns STATUS
	Modifies the passed in menu bar and appends standard NotePaper menus.
*/
#define msgNotePaperAddMenus		MakeMsg(clsNotePaper, 117)

/****************************************************************************
 msgNotePaperAddModeCtrl			takes OBJECT, returns STATUS
	Adds the standard NotePaper mode icon to the passed in menu bar.
*/
#define msgNotePaperAddModeCtrl		MakeMsg(clsNotePaper, 118)

/****************************************************************************
 msgNotePaperClear			takes pNull, returns STATUS
	Deletes all items in receiver.
*/
#define msgNotePaperClear			MakeMsg(clsNotePaper, 119)

/****************************************************************************
 msgNotePaperClearSel		takes pNull, returns STATUS
	Deletes all selected items in receiver.
*/
#define msgNotePaperClearSel		MakeMsg(clsNotePaper, 11)

/****************************************************************************
 msgNotePaperInsertLine				takes P_NULL, returns STATUS
	Inserts a blank line above the selection.
*/
#define msgNotePaperInsertLine	MakeMsg(clsNotePaper, 5)

/****************************************************************************
 msgNotePaperSelectRect		takes P_RECT32, returns STATUS
	Selects items within rect in the receiver's data.

 Return Value
	stsNotePaperNoHit:	Returned if nothing selected.
*/
#define msgNotePaperSelectRect		MakeMsg(clsNotePaper, 1)

/****************************************************************************
 msgNotePaperSelectLine		takes P_RECT32, returns STATUS
	Selects items whose baselines intersect rect in the receiver's data.

 Return Value
	stsNotePaperNoHit:	Returned if nothing selected.
*/
#define msgNotePaperSelectLine		MakeMsg(clsNotePaper, 6)

/****************************************************************************
 msgNotePaperDeselectLine		takes P_RECT32, returns STATUS
	Deselects items whose baselines intersect rect in the receiver's data.

 Return Value
	stsNotePaperNoHit:	Returned if nothing deselected.
*/
#define msgNotePaperDeselectLine	MakeMsg(clsNotePaper, 7)

/****************************************************************************
 msgNotePaperDeleteLine		takes P_RECT32, returns STATUS
	Deletes items whose baselines intersect rect in the view's data.

 Return Value
	stsNotePaperNoHit:	Returned if nothing deleted.
*/
#define msgNotePaperDeleteLine		MakeMsg(clsNotePaper, 8)

/****************************************************************************
 msgNotePaperScribble			takes OBJECT, returns STATUS
	Handles scribble (including creating and insert object into view's data).

 The passed scribble's origin should be relative to the lower-left
 corner of the receiver.
*/
#define msgNotePaperScribble	MakeMsg(clsNotePaper, 9)

/****************************************************************************
 msgGWinGesture		takes P_GWIN_GESTURE, returns STATUS
	Self-sent to process the gesture.

 The standard behavior of this gesture is defined in gwin.h.  In addition,
 subclasses can return stsNotePaperTreatAsInk if they want the gesture to
 be treated as ink.  In that case, an instance of clsNPScribbleItem will
 be created from the gesture's strokes.

 clsNotePaper's response to the various gestures is described in the
 MiniNote quick reference card.  In gesture mode, gesture can be made
 anywhere in the window.  However, any unrecognized gesture of more
 than two strokes will be treated as ink.
 In writing mode, most drawing is treated as ink (unless it is drawn
 over the selection).  However, the following gestures are allowed even
 in writing mode:
//{
	xgsScratchOut:			delete items
	xgsPigtailVert:			delete items
	xgs2Tap:				select item (if over an item)
	xgs3Tap:				select line
	xgsPlus:				toggle item (if over an item)
	xgsTapHold:				begin area selection
	xgsCircleCrossOut:		undo
	xgsDblCircle:			create reference button
	xgsUpCaretDot:			insert date/time
	xgsDblUpCaret:			embed stationery
	xgsHorzCounterFlick:	toggle mode
	xgsVertCounterFlick:	toggle application borders
//}

 Return Value
	stsNotePaperTreatAsInk:	The gesture should be treated as ink.

 See Also
	gwin.h
*/
#define msgGWinGesture				MakeMsg(clsGWin, 2)


/****************************************************************************
 msgAppSelectAll					takes P_NULL, returns STATUS
	Selects all items in the view.

 See Also
	app.h
*/

/****************************************************************************
 msgSelDelete						takes P_NULL, returns STATUS
	Deletes selected items in the view.

 Close the space that the selection occupies if an entire line or lines
 is selected and this message does is not sent within a move/copy episode.

 See Also
	sel.h
*/

/****************************************************************************
 msgOptionAddCards				takes P_OPTION_TAG, returns STATUS
	Creates and adds the Pen and Paper option sheets.


 This message is usually send to the NotePaper instance by the app framework
 if the instance holds the selection, is the client win of the app's main win,
 or is the client win of a scroll win that is the app's main win.  However, to
 force NotePaper's option sheets to appear in the "Option" menu in other
 circumstances, this message should be forwarded to the NotePaper instance
 by the application if pArgs->tag is tagAppDocOptSheet.

 See Also
	app.h
	option.h
*/

/****************************************************************************
 msgImportQuery     takes P_IMPORT_QUERY, returns STATUS
	category: class message
	Indicates whether or not passed in file can be imported.

	NotePaper will respond positively to this message if the first 5
	byte of the file are printable ASCII characters.

 See Also
	import.h
*/

/****************************************************************************
 msgImport  takes P_IMPORT_DOC, returns STATUS
	Imports the passed in file.

	After the file is imported, receiver's length is grown to accommodate
	the imported text.  If receiver's width is zero, it is grown to six
	inches wide.

 See Also
	import.h
*/

/****************************************************************************
 msgExportGetFormats    takes P_EXPORT_LIST, returns STATUS
	Passes back list of formats that can be exported.

 See Also
	export.h
*/

/****************************************************************************
 msgExport              takes P_EXPORT_DOC, returns STATUS
	Writes an ASCII version of receiver's data to the passed in file.

	A translated text version of each scribble item is written out.

 See Also
	export.h
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Private Messages                                                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNotePaperUpdateSel		takes NULL, returns STATUS
	Takes or releases the selection as is appropriate.

 This message is used internally within NotePaper.  Clients do not need
 to send it unless the affect the selection by sending msgNPItemSelect to
 individual items in the view's data object.
*/
#define msgNotePaperUpdateSel	MakeMsg(clsNotePaper, 52)

/****************************************************************************
 msgNotePaperSplitAsAtoms		takes NULL, returns STATUS
	Split the selected item to its constituent pieces.

 Scribble items convert each stroke into a scribble item.
 Text items do nothing.
*/
#define msgNotePaperSplitAsAtoms		MakeMsg(clsNotePaper, 54)

/****************************************************************************
 msgNotePaperSplitAsWords	takes NULL, returns STATUS
	Split the selected scribbles and text into words.
*/
#define msgNotePaperSplitAsWords	MakeMsg(clsNotePaper, 55)

/****************************************************************************
 msgNotePaperGrowHeightTo		takes COORD32, returns STATUS
	Grow the height of the view to at least COORD32.
*/
#define msgNotePaperGrowHeightTo		MakeMsg(clsNotePaper, 56)

/****************************************************************************
 msgNotePaperGrowHeightBy		takes COORD32, returns STATUS
	Grow the height of the view by COORD32.
*/
#define msgNotePaperGrowHeightBy		MakeMsg(clsNotePaper, 57)



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Quick help and window tags                                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Tags used in the UI of NotePaper's option sheets, menus, and quick help.

	Next up: 37; Recycle: 2
	Tag values 100-120 are reserved for pen and paper styles.
	Tag values 200-255 are reserved for private window tags.
*/

/****  Mode icons ****/
/*
 Mode icons (tags from itoggle.h)
 The bitmaps corresponding to the two tags below are found in theSystemResFile.
*/
#define tagNotePaperWriteIcon		tagIconToggleOff
#define tagNotePaperEditIcon		tagIconToggleOn

/* Quick help tag for mode icons */
#define tagNotePaperModeIcon		MakeTag(clsNotePaper, 1)

/****  Windows  ****/

/* Quick help tags for the main view and for the gesture margin. */
#define tagNotePaper				MakeTag(clsNotePaper, 4)
#define tagNotePaperMargin			MakeTag(clsNotePaper, 5)

/****  Edit Menu  ****/
#define tagNotePaperTranslate		MakeTag(clsNotePaper, 6)
#define tagNotePaperEdit			MakeTag(clsNotePaper, 7)
#define tagNotePaperClear			MakeTag(clsNotePaper, 34)
#define tagNotePaperInsertLine		MakeTag(clsNotePaper, 35)

/****  Pen Menu  ****/
#define tagPenMenu					MakeTag(clsNotePaper, 3)
#define tagPenFineBlack				MakeTag(clsNotePaper, 110)
#define tagPenBoldBlack				MakeTag(clsNotePaper, 111)
#define tagPenFineGray				MakeTag(clsNotePaper, 112)
#define tagPenBoldGray				MakeTag(clsNotePaper, 113)

/****  Arrange Menu  ****/
#define tagArrangeMenu				MakeTag(clsNotePaper, 8)
#define tagNotePaperTidy			MakeTag(clsNotePaper, 9)
#define tagNotePaperCenter			MakeTag(clsNotePaper, 10)
#define tagNotePaperAlignLeft		MakeTag(clsNotePaper, 11)
#define tagNotePaperAlignRight		MakeTag(clsNotePaper, 12)
#define tagNotePaperMerge			MakeTag(clsNotePaper, 13)
#define tagNotePaperSplitAsWords	MakeTag(clsNotePaper, 14)
#define tagNotePaperSplit			MakeTag(clsNotePaper, 15)

/****  Paper Option Card  ****/
/*
 NOTE:
 For TagPaperStyle(n), tag n is a value in the NP_PAPER_STYLE enumeration
 For NPPaperStyleFromTag converts a tag to a paper style.
*/

#define tagPaperCard				MakeTag(clsNotePaper, 16)

#define tagPaperStyleLabel			MakeTag(clsNotePaper, 17)
#define tagPaperStyle				MakeTag(clsNotePaper, 18)
#define TagPaperStyle(n)			MakeTag(clsNotePaper, 100 + n)
#define NPPaperStyleFromTag(t)		(TagNum(t) - 100)

#define tagLineSpacingLabel			MakeTag(clsNotePaper, 19)
#define tagLineSpacing				MakeTag(clsNotePaper, 20)
#define tagLineOtherRuling			MakeTag(clsNotePaper, 21)
#define tagLineOtherValue			MakeTag(clsNotePaper, 22)

#define tagPaperWidthLabel			MakeTag(clsNotePaper, 23)
#define tagPaperWidth				MakeTag(clsNotePaper, 24)
#define tagPaperFitScreen			MakeTag(clsNotePaper, 25)
#define tagPaperFitPrinter			MakeTag(clsNotePaper, 26)
#define tagPaperOtherWidth			MakeTag(clsNotePaper, 27)
#define tagPaperOtherValue			MakeTag(clsNotePaper, 28)

#define tagPaperFontLabel			MakeTag(clsNotePaper, 29)
#define tagPaperFont				MakeTag(clsNotePaper, 30)

/****  Pen Option Card  ****/
#define tagPenCard					MakeTag(clsNotePaper, 31)

#define tagPenStyleLabel			MakeTag(clsNotePaper, 32)
#define tagPenStyle					MakeTag(clsNotePaper, 33)
/* tagPenFineBlack (same value as in the pen menu) */
/* tagPenBoldBlack (same value as in the pen menu) */
/* tagPenFineGray  (same value as in the pen menu) */
/* tagPenBoldGray  (same value as in the pen menu) */

/****  Insertion Pad  ****/

#define tagNotePaperSkip			MakeTag(clsNotePaper, 36)

/****  Standard Error Resource Tags  ****/
#define stsNotePaperPageWidth		MakeStatus(clsNotePaper, 2)

/****  Undo Resource Tags  ****/
#define tagNPUndoWriting			MakeTag(clsNotePaper, 1)
#define tagNPUndoDeletion			MakeTag(clsNotePaper, 2)

#endif
