/****************************************************************************
 File: list.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.19  $
   $Author:   mlentczn  $
     $Date:   24 Feb 1992 10:24:18  $

 This file contains the API definition for clsList.

 clsList inherits from clsObject.

 Lists are a simple ordered collections of items.
****************************************************************************/

#ifndef LIST_INCLUDED
#define LIST_INCLUDED

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						 Common #defines and typedefs					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT LIST, *P_LIST;

typedef P_UNKNOWN LIST_ITEM, *P_LIST_ITEM;

/*
 LIST_ENTRY is used in many messages.  In general, the fields are treated
 as follows:

    -:  position.  An item's location.  Locations are zero-based. The first
        item is 0 and the last item is number of items - 1.  When used as
        an In parameter, position specifies the position of the item to
        operate on.  For adding operations, maxU16 means beyond the last
        item. For other operations, maxU16 means the last item in the list.
        Values beyond the size of the list but less than maxU16 are not
        recommended.  When used as an Out parameter, position contains the
        actual position of the item.  maxU16 is never passed back even if
        passed in.

    -:  item.  When used as an In parameter, item identifies the item to
        operate on.  If the same item added to the list more than once,
        then all operations work only the first appearance of the item.
        When used as an Out parameter, item contains the item operated on.
*/
typedef struct LIST_ENTRY {

	U16			position;
	LIST_ITEM	item;

} LIST_ENTRY, *P_LIST_ENTRY;


typedef struct LIST_NOTIFY {

  MESSAGE       	msg;			// In:  message to send/post
  P_ARGS        	pArgs;			// In:  pArgs for message
  SIZEOF        	lenSend;		// In:  length of pArgs

} LIST_NOTIFY, * P_LIST_NOTIFY;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Status Codes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define stsListFull						MakeStatus(clsList, 1)
#define stsListEmpty					MakeStatus(clsList, 2)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages Defined by Other Classes					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew	takes P_LIST_NEW, returns STATUS
	category: class message
	Creates a new empty list.
	
 If the heap specified in pArgs->object.heap is null, the process heap is
 used.
*/

typedef struct LIST_STYLE {
	U16			reserved:16;
} LIST_STYLE, * P_LIST_STYLE;


/* List filing behavior. */
typedef enum LIST_FILE_MODE {
	listFileItemsAsData,			// File list items as U32 data.
	listFileItemsAsObjects,			// Treat list items as objects.  Save
									// them with msgResPutObject and restore
									// them with msgResGetObject.
	listDoNotFileItems				// Don't file list items.		
} LIST_FILE_MODE, *P_LIST_FILE_MODE;


typedef struct LIST_NEW_ONLY {
	LIST_STYLE		style;
	LIST_FILE_MODE	fileMode;
	U32				reserved[4];	// Reserved
} LIST_NEW_ONLY, *P_LIST_NEW_ONLY;


#define	listNewFields		\
	objectNewFields	 		\
	LIST_NEW_ONLY	 		list;


typedef struct LIST_NEW {
	listNewFields
} LIST_NEW, *P_LIST_NEW;


/****************************************************************************
 msgNewDefaults	takes P_LIST_NEW, returns STATUS
	category: class message
	Initializes the LIST_NEW structure to default values.

 Zeroes out pNew->list and sets:
//{
	pArgs->list.fileMode = listFileItemsAsObjects
//}
*/


/****************************************************************************
 msgSave		takes P_OBJ_SAVE, returns STATUS
	Defined in clsmgr.h

 In response to this message, the list saves itself.  Then, based on the
 list's fileMode, it may save the item information.  See the commentary with
 the type LIST_FILE_MODE for more information.
*/


/****************************************************************************
 msgRestore		takes P_OBJ_RESTORE, returns STATUS
	Defined in clsmgr.h

 In response to this message, the list restores itself.  Then, based on the
 list's fileMode, it may restore the items information.  See the commentary
 with the type LIST_FILE_MODE for more information.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						List Manipulation Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgListFree		takes P_LIST_FREE, returns STATUS
	Frees a list according to mode.

 In response to this message, the list destroys itself AND all of its
 items.

 Use msgDestroy to destroy the list without affecting the list's items.
 For both messages, observers are sent msgListNotifyEmpty.
*/
#define msgListFree 					MakeMsg(clsList, 1)


typedef enum LIST_FREE_MODE {
	listFreeItemsAsData,		// Ignore the item's value.  Simply destroy
								// the list itself.  Equivalent to sending
								// msgDestroy to the list.
	listFreeItemsAsObjects,		// Treat items as objects.  Send each item
								// msgDestroy Nil(key) before destroying
								// the list itself.  Any errors are ignored.
	listDoNotFreeItems			// Obsolete.  Do not use.
} LIST_FREE_MODE, *P_LIST_FREE_MODE;


typedef struct LIST_FREE {
	OBJ_KEY			key;		// Key for freeing the list object.
	LIST_FREE_MODE	mode;
} LIST_FREE, *P_LIST_FREE;


/****************************************************************************
 msgListAddItem		takes LIST_ITEM, returns STATUS
	Adds an item to the end of a list.

 Observers are sent msgListNotifyAddition.
*/
#define msgListAddItem 					MakeMsg(clsList, 2)


/****************************************************************************
 msgListAddItemAt	takes P_LIST_ENTRY, returns STATUS
	Adds an item to a list at pArgs->position.

 If the list is empty, pArgs->position is treated as if it were 0.  If
 pArgs->position is maxU16, the item is inserted at the end of the list.  
 
 If necessary, list items move to make room for the new item.

 Observers are sent msgListNotifyAddition.

 Return Value
	stsOK:		item added.  pArgs->position contains the actual position of 
				the new item.
*/
#define msgListAddItemAt				MakeMsg(clsList, 10)


/****************************************************************************
 msgListRemoveItem	takes LIST_ITEM, returns STATUS
	The list searches for pArgs in the list and removes the item if found.

 If the argument is in the list more than once, only the first instance of
 it is removed.

 Observers are sent msgListNotifyDeletion.

 Return Value
	stsListEmpty:		the list was empty
	stsNoMatch:			item was not found
*/
#define msgListRemoveItem 				MakeMsg(clsList, 11)


/****************************************************************************
 msgListRemoveItemAt	takes P_LIST_ENTRY, returns STATUS
	Removes the item in the list at pArgs->position.

 Observers are sent msgListNotifyDeletion.

 Return Value
	stsListEmpty:		the list was empty
	stsOK:				item removed.  pArgs->position contains the position 
						of the removed item.
*/
#define msgListRemoveItemAt				MakeMsg(clsList, 3)


/****************************************************************************
 msgListReplaceItem		takes P_LIST_ENTRY, returns STATUS
	Replaces the item in the list at pArgs->position.

 If pArgs->position is maxU16, the last item in the list is replaced.

 Observers are sent msgListNotifyReplacement.

 Return Value
	stsListEmpty:		the list was empty
	stsOK:				item was replaced.  pArgs->item contains the old item
						and pArgs->position contains its old position.
*/
#define msgListReplaceItem 				MakeMsg(clsList, 4)


/****************************************************************************
 msgListGetItem		takes P_LIST_ENTRY, returns STATUS
	Gets the item in the list at pArgs->position.

 If pArgs->position is maxU16, the last item in the list is returned.

 Return Value
	stsListEmpty:		the list was empty.
	stsOK:				item found.  pArgs->position contains the position
						of the item.
*/
#define msgListGetItem 					MakeMsg(clsList, 5)


/****************************************************************************
 msgListFindItem	takes P_LIST_ENTRY, returns STATUS
	Searches for pArgs->item in the list.

 Return Value
	stsNoMatch:		item was not found.
	stsOK:			item was found.  pArgs->position contains the position of
					the item.
*/
#define msgListFindItem 				MakeMsg(clsList, 6)


/****************************************************************************
 msgListNumItems	takes P_U16, returns STATUS
	Passes back the number of items in a list.
*/
#define msgListNumItems 				MakeMsg(clsList, 7)


/****************************************************************************
 msgListRemoveItems 	takes no arguments, returns STATUS
	Removes all of the items in a list.

 The list's items are not affected in any way.
				   
 Observers are sent msgListNotifyEmpty.
*/
#define msgListRemoveItems	 			MakeMsg(clsList, 8)


/****************************************************************************
 msgListEnumItems 	takes P_LIST_ENUM, returns STATUS
	Enumerates the items in a list.

 This copies successive items from the list into an array.  There are two
 approaches a client can use:

 1) Let the list do all the work in one call.  The list allocates an array
 of items which is passed back in pArgs->pItems.  You must free this array
 when you are done with a call to OSHeapBlockFree.  LIST_ENUM Should be
 filled in as follows:

    max:		On input, should be 0.  On output, will be the the number of
                items in the allocated block.
    count:		On input, should be maxU16.  On output will be the same as
                max.
    pItems:		On input, should be null.  On output, will be the pointer to
                the allocated block.
    pNext:		On input, should be null.

 2) Go through the items, a chunk at a time.  Repeatedly call
 msgListEnumItems with the same LIST_ENUM structure and processes
 successive groups of items. The call that returns stsEndOfData indicates
 that the enumeration is finished (there are no more items to process). 
 LIST_ENUM is used as follows:

    max:		On input and output, the number of items your block can hold
    count:		On input, the same as max.  On output, will be the number
                of items returned in block. (This will be less than max the
                last time through.)
    pItems:		On input, a pointer to a block that can hold at least max
                items.
    pNext:		On input for first call, should be null.  Do not modify
                thereafter.

 Return Value
    stsEndOfData:	There are no more items to enumerate (list may be
                    empty).  When stsEndOfData is returned, pArgs->count is
                    zero.  If you passed in pItems as null and max
                    as 0, the block may not have been allocated.  Check
                    pItems for nil and free it if it isn't.
*/
#define msgListEnumItems	 			MakeMsg(clsList, 9)

typedef struct LIST_ENUM {

	U16			max;
	U16			count;
	P_LIST_ITEM	pItems;
	P_UNKNOWN	pNext;

} LIST_ENUM, * P_LIST_ENUM;


/****************************************************************************
 msgListGetHeap 	takes P_OS_HEAP_ID, returns STATUS
	Passes back the heap used by the list.
*/
#define msgListGetHeap		 			MakeMsg(clsList, 12)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					      Forwarding Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * clsList responds to these messages by sending the specified message to
 * each item in the list in turn.  clsList ignores the values returned by
 * sending this message and always returns stsOK.
*/

/****************************************************************************
 msgListCall 	takes P_LIST_NOTIFY, returns STATUS
	Sends a message to each object in the list using ObjectCall.
*/
#define msgListCall			 			MakeMsg(clsList, 13)


/****************************************************************************
 msgListSend 	takes P_LIST_NOTIFY, returns STATUS
	Sends a message to each object in the list using ObjectSend.
*/
#define msgListSend			 			MakeMsg(clsList, 14)


/****************************************************************************
 msgListPost 	takes P_LIST_NOTIFY, returns STATUS
	Sends a message to each object in the list using ObjectPost.
*/
#define msgListPost			 			MakeMsg(clsList, 15)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					      Observer Notifications						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * A list uses msgPostObservers to deliver all of its notification messages.
 * (See clsmgr.h for more information.)
*/

/****************************************************************************
 msgListNotifyAddition           takes P_LIST_NOTIFY_ADDITION, returns STATUS
	Notifies observers that an item has been added to the list.
*/

typedef struct LIST_NOTIFY_ADDITION {
	LIST				list;		// the affected list
	LIST_ITEM			listItem;	// the affected list item
	U16					count;		// new number of entries
	U8					reserved[40];
} LIST_NOTIFY_ADDITION, * P_LIST_NOTIFY_ADDITION;

#define msgListNotifyAddition				MakeMsg ( clsList, 16 )


/****************************************************************************
 msgListNotifyDeletion		takes P_LIST_NOTIFY_DELETION, returns STATUS
	Notifies observers that an item has been deleted from the list.
*/

typedef struct LIST_NOTIFY_DELETION {
	LIST				list;		// the affected list
	LIST_ITEM			listItem;	// the affected list item
	U16					count;		// new number of entries
	U8					reserved[40];
} LIST_NOTIFY_DELETION, * P_LIST_NOTIFY_DELETION;

#define msgListNotifyDeletion					MakeMsg ( clsList, 17 )


/****************************************************************************
 msgListNotifyReplacement	takes P_LIST_NOTIFY_REPLACEMENT, returns STATUS
	Notifies observers that an item in the list has been replaced.
*/

typedef struct LIST_NOTIFY_REPLACEMENT {
	LIST				list;			// the affected list
	LIST_ITEM			newListItem;	// the new list item
	LIST_ITEM			oldListItem;	// the replaced list item
	U16					index;			// index of replace item
	U8					reserved[40];
} LIST_NOTIFY_REPLACEMENT, * P_LIST_NOTIFY_REPLACEMENT;

#define msgListNotifyReplacement				MakeMsg ( clsList, 18 )


/****************************************************************************
 msgListNotifyEmpty		takes P_LIST_NOTIFY_EMPTY, returns STATUS
	Notifies observers that a list is now empty.
*/

typedef struct LIST_NOTIFY_EMPTY {
	LIST				list;			// the affected list
	U8					reserved[40];
} LIST_NOTIFY_EMPTY, * P_LIST_NOTIFY_EMPTY;

#define msgListNotifyEmpty						MakeMsg ( clsList, 19 )

#endif	// LIST_INCLUDED
