/****************************************************************************
 insert.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.19  $
   $Author:   kbentley  $
	 $Date:   16 Mar 1992 13:21:42  $

 This file contains the API definition for clsIP (Insertion Pads).

 clsIP inherits from clsCustomLayout.
****************************************************************************/

/**** Introduction ****/
/*
 IPs provide a convenient and standard mechanism for getting handwritten
 input from a user.  "IP" is an abbreviation for "Insertion Pad."

 IPs support several different visual styles -- character boxes, ruled
 lines, or blank writing areas and different optional behaviors. IPs use
 a translator to recognize handwriting if necessary.
*/


/**** Typical Uses and Settings ****/
/*
 This section describes the most common uses and settings for the various
 types of IPs.

 Character Box IPs:

    -:	Their new.ip.style.displayType is ipsCharBox.

    -:	Character Box IPs are typically used to edit or insert simple
		strings of text such as a person's name or a document name.
 
 Ruled Line IPs:

    -:	Their new.ip.style.displayType is ipsRuledLines.

    -:	Ruled Line IPs are typically used when the handwriting preference is
        Ruled.

    -:	When the preference is Ruled/Boxed, then the IP's style.ruledToBoxed
        and style.boxedToRuled fields are used to control the
		transmogrification between styles.  It is the responsibility of
		the IP user to examine the preferences and determine if these fields
		should be set.

 Blank IPs:

    -:	Blank IPs are typically used to collect and display simple scribbles
        (perhaps a signature).
 
    -:	Their new.ip.style.displayType is ipsBlank. Their
        new.ip.style.buttonType is typically ipsNoButton, as they never do
        translation.
		
    -:	They do not display ruled lines in the sPaper created by default,
        nor do they allow scribble editing (see spaper.h). 

    -:	They turn off borders when printing, allowing them to be robustly
        embedded inside a document. 
*/


/**** Quick Start ****/
/*
 A typical IP client does the following:
    -:	The client creates an IP in one of three styles described above.
    -:	The client then adds itself as an observer of the IP and handles
        msgIPDataAvailable.
    -:	The msgIPDataAvailable handler uses msgIPGetXlateString to extract
        the string and then processes the string in some application
        specific manner.

 The client should also handle either msgIPCancelled or msgFreePending so
 that the client can free any allocated data when the IP is destroyed.
*/


/**** IP Components ****/
/*
 An IP is constructed from several pieces.  Most clients and subclasses
 don't need to know anything about these details, but advanced clients and
 subclasses might.

 The main writing area of an IP is either a field or an sPaper.  An
 ipsCharBox IP contains a field (an instance of clsField);  ipsRuledLines
 IP's contain sPaper, as do ipsBlank/ipsSignature.  IP's which have
 style.ruledToBoxed or style.boxedToRuled set switch between a field and
 an sPaper.  The IP is an observer of the sPaper or field.  The sPaper
 or field has an associated translator.

 If style.buttonType is ipsBottomButtons or ipsTopButtons, then the IP also
 contains a command bar with three buttons.  The IP is the client of all of
 the buttons in the command bar.

 Technically clients and subclasses can modify these components directly,
 but this is not recommended.  If these components are modified directly, 
 extreme care must be taken -- current and future implementations of IP may
 make assumptions which can be violated by making some types of changes to
 the components.
*/


/**** Client and Observers ****/
/*
 There are two different paths for objects to receive "notification"
 messages from an IP.

 If an IP's client is non-null, then the IP sends the following messages to
 the IP's client.  If the client is null, then the IP sends the messages to
 the IP's observers.  Self is the value of pArgs for all of these messages.

	-:	msgIPCancelled
	-:	msgIPClear
	-:	msgIPDataAvailable
	-:	msgIPCopied
	-:	msgIPTransmogrified
*/


/**** IPs and Translators ****/
/*
 The sPaper or field component of an IP (whichever exists) has a translator
 which performs handwriting recognition.

 The creator of the IP may specify this translator in two ways:

    -:	A translator object may be passed to msgNew.  Do this by setting 
        new.ip.style.xlateType to ipXlateObject and new.ip.xlate.translator
        to the translator object's uid.

    -:	An (optionally null) translation template may be passed to msgNew. 
        Do this by setting new.ip.style.xlateType to ipXlateTemplate and
        new.ip.xlate.pTemplate to the address of the template.  If the
        template is non-null, the IP compiles the template.  Then the IP
        creates a translator (of clsXText; see xtext.h).  This translator
        is created with the passed-in template if the template is non-null.

 An IP with style.charOnly sets the translator to recognize single
 characters.

 The translation information (the translator object and the digested 
 template) are destroyed when the IP handles msgFree.

 See msgIPSetTranslator for additional information.
*/


/**** IP Destruction ****/
/*
 As a convenience, an IP will optionally self destruct after providing its
 data or if the IP is cancelled.  To get this behavior, set the IP's
 style.freeAfter to true.

 The automatic destruction occurs during an IP's default response to the 
 following messages:

	-:	msgIPGetXlateData
	-:	msgIPGetXlateString
	-:	msgIPCancelled
*/


/**** Transmogrification ****/
/*
 One of PenPoint's standard handwriting styles is called Ruled and Boxed.

 When writing in this style, the following steps are taken:  (1) the user
 writes into a ruled line (sPaper) IP and hits OK.  (2) the handwriting is
 translated.  (3) the ruled writing area is replaced by a combed field. 
 (4) the user makes any corrections in the field and presses OK again.  (5)
 the data is made available to the application and (6) either the IP is
 destroyed or the combed field is replaced with a ruled line sPaper ready
 for additional input.

 The term "Transmogrification" describes the switching of writing area
 types and the moving of the data from the ruled lines to the field.

 This transmogrification can happen in response to several messages,
 including msgIPClear, msgIPGetXlateData and msgIPXlateCompleted.

 During transmogrification, the IP's style.displayType is changed.  Also,
 the unnecessary components are destroyed and new ones created.  The
 translator associated with the sPaper or field (whichever exists) is moved
 to the newly created sPaper or field (whichever didn't exist).

 The ruledToBoxed and boxedToRuled fields in an IP's style determine when 
 transmogrification happens:

 ruledToBoxed

    -:	If style.ruledToBoxed is true, then a ipsRuledLines IP
        transmogrifies into a ipsCharBox IP when translation occurs.  
 
    -:	Clients typically set style.ruledToBoxed to true if the
        prInputPadStyle preference is RuledAndBoxed.

 boxedToRuled

    -:	If style.boxedToRuled is true, then a ipsCharBox IPs transmogrifies
        into a ipsRuledLines IP when data is retrieved via
        msgIPGetXlateData or msgIPGetXlateString.
 
    -:	Clients typically set style.boxedToRuled to true only if (1) the
        prInputPadStyle preference is RuledAndBoxed and (2) the IP is to be
        used multiple times before it is freed.
*/

/**** IPs and Preferences ****/
/*
 This section describes the preferences that an IP considers and when it
 considers them.  It also describes the preferences a client might consider
 when determining an IP's style.  (See prefs.h for general information on
 preferences.)

 When handling msgNew, msgIPSetStyle, and when transmogrifying, an IP uses
 the user's preferred value for Character Box Height, Character Box Width
 and Line Height.  The IP does NOT observe these preferences so changes in
 their value won't affect an existing IP unless its style changes or the IP
 is transmogrified.

 Clients may want to consider the following preferences when managing an IP
 and its translator.  (A client may want to only check the preference when
 creating the IP.  Alternatively, a client may want to observe
 theSystemPreferences and respond to changes.)  Note that this is only one
 possibility -- many clients will (correctly) chose to ignore the
 preferences or map from the preferences to IP characteristics differently.

 prInputPadStyle:
	
    -:	If this is prInputPadStyleRuledAndBoxed, the client would set an
        IP's style.displayType to ipsRuledLines and style.ruledToBoxed to
		true and possibly style.boxedToRuled to true.  This causes an IP
		to transmogrify between ipsRuledLines and ipsCharBox display types.
		(See the section "Transmogrification" for details.)

    -:	If this is prInputPadStyleRuled, the client would set an IP's
        style.displayType to ipsRuledLines and style.ruledToBoxed and
		style.boxedToRuled to false.

    -:	If this is prInputPadStyleBoxed, the client would set an IP's
        style.diplayType to ipsCharBox and style.boxedToRuled and
		style.ruledToBoxed to false.

 prWritingStyle:

    -:	Clients may want to let this preference affect the translation
        information they send with msgNew or the translator set with
        msgIPSetTranslator.
*/


/**** Single Character IPs ****/
/*
 clsIP has specific support for single character IPs.  Setting
 style.charOnly to true enables this support.  Usually if charOnly is true,
 then style.buttonType is ipsProxButton, style.takeGrab is true, and the
 client floats the IP rather than embedding it.

 Setting charOnly to true causes the IP to automatically set the number of
 rows and columns to 1.  It also prepares the translator to expect only a
 single character.
*/

/**** Debugging Flags ****/
/*
 IP's debugging flag set is 'h.'  Defined flags are:

	0001:  Show general information about IP operations.
	0002:  Show information about IP translation operations.
	0004:  Show information about IP layout and size operations.
*/

#ifndef INSERT_INCLUDED
#define INSERT_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef WIN_INCLUDED
#include <win.h>
#endif

#ifndef CLAYOUT_INCLUDED
#include <clayout.h>
#endif

// Next Up:	25	Recycled:	1, 2, 11, 12, 13

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT IP;


/****  Display Types  ****/
/*
 * Use one of these values in an IP's style.displayType.  This field defines
 * the type of the IP.
 *
 * See the section "Typical Uses and Settings" for more information.
*/
#define ipsRuledLines		0   // standard ruled lines;  contains sPaper
#define ipsCharBox			1   // character box editing;  contains field
#define ipsBlank			3   // signature pad;  contains sPaper
#define ipsSignature		3	// same as ipsBlank
#define ipsCharBoxButtons   4   // Obsolete
/*		ipsEditBox				Obsolete	*/

/****  Translator types  ****/
/*
 * Use one of these values in an IP's style.xlateType.  This field defines
 * whether new.ip.xlate contains a template or a translator object.  See the
 * section "IPs and Translators" for more information.
*/
#define ipXlateObject		0 	// pNew->xlate.translator is a translator
#define	ipXlateTemplate 	1	// pNew->xlate.pTemplate is an &XTEMPLATE


/****  Space Collapse Rules  ****/
/*
 * Use one of these values in an IP's style.spaceCollapse.  For ipsCharBox
 * IPs, this field defines how spaces are treated in text strings retrieved
 * from an IP via msgIPGetXlateData or msgIPGetXlateString.
 *
 * ipsSpaceSpace causes multiple spaces at the end of a line to be replaced
 * with a single space.  ipsSpaceNewLine causes an entire line's worth of
 * spaces to be replaced with a single newline character.  ipsSpaceAsIs
 * causes spaces to be returned literally.
*/
#define ipsSpaceAsIs	0	// WYSIWYG
#define ipsSpaceSpace	1	// Multiple spaces at end of line become 1 space
#define ipsSpaceNewLine	2	// Single line of spaces becomes a newline


/****  Button Types  ****/
/*
 * Use one of these values in an IP's style.buttonType.  This field defines
 * the type of buttons an IP contains.
 *
 * ipsNoButton is typically used with displayType of ipsBlank.  ipsProxButton
 * is valid only with ipsRuledLines.  This value cause translation to occur
 * on out of proximity events.  ipsBottomButtons and ipsTopButtons create a
 * command bar at the top or bottom containing an OK, Cancel, and Clear
 * Button.
*/
#define ipsNoButton			0	// No button
#define ipsProxButton		3   // Proximity translation for ipsRuledLines
#define ipsBottomButtons	6	// Command bar buttons on bottom
#define ipsTopButtons		7	// Command bar buttons on top


/****  Modality Behavior  ****/
/*
 * Use one of these values in an IP's style.modal.  When style.takeGrab is
 * true, style.modal defines the result of a pen tap outside of the IP.
 * The term takeGrab is somewhat misleading.  The IP actually creates a
 * modal filter to handle input.
*/
#define ipsNoMode			0	// Nothing happens on pen tap outside
#define ipsTranslateMode	1	// Translation happens on pen tap outside
#define ipsCancelMode		2	// Cancel happens if pen tap outside


/****  IP Style  ****/
typedef struct IP_STYLE {

	U16 displayType:	3,	// display type
		buttonType:	 	3,	// button type
    	freeAfter:	  	1,	// See the section "IP Destruction."
		clientReplace:  1,	// Unused
		xlateType:	  	1,	// See the section "IPs and Translators."
							// Describes what pNew->ip.xlate contains.  
		charOnly:	   	1,	// See the section "Single Character IPs."
		modal:			2,	// If style.takeGrab is true, describes modal 
							// IP's behavior.
		takeGrab:		1,	// Makes the IP modal.  Modal behavior is
							// defined by style.modal.
		reserved1:		1,  // Reserved
		takeFocus:		1,  // IP becomes the input target when created.
		delayed:		1;  // For ipsCharBox IPs, turns on the field
							// component's delayed behavior.
	U16 spaceCollapse:	3,	// Rule for collapsing spaces when 
							// extracting information from ipsCharBox IP.
		embeddedLook:	1,	// Set to true to look good when embedded;
							// false to look good when floating.  Affects
							// an IP's handling of msgNew and msgIPSetStyle.
    	reserved2:		1,	// Reserved
		ruledToBoxed:	1,	// See the "Transmogrification" and "IPs and
							// Preferences" sections.  
		boxedToRuled:	1,  // See the "Transmogrification" and "IPs and 
							// Preferences" sections.  
    	clientIsThisApp:1,	// Private
    	reserved:		8;  // Reserved

} IP_STYLE, *P_IP_STYLE;


/****  Component Tags  ****/
/*
 * The components of an IP have the following window tags.  See the section
 * "IP Components" for more information.
*/
#define tagIPSPaper			MakeTag(clsIP, 1)	// ipsRuledLines and ipsBlank
												// IP's sPaper
#define tagIPField			MakeTag(clsIP, 2)	// ipsCharBox IP's field
#define tagIPButton			MakeTag(clsIP, 3)	// "OK" button 
#define tagIPButtonClear	MakeTag(clsIP, 4)	// "Clear" button
#define tagIPButtonCancel   MakeTag(clsIP, 5)	// "Cancel" button
#define tagIPCommandBar		MakeTag(clsIP, 6)	// command bar


/****  Quick Help Tags ****/
/*
 * In most cases an IP component's window tag and quick help are identical.
 * But tagIPSignatureSPaper is the quick help tag for ipsBlank IP's sPaper and
 * tagIPSingleChar is the quick help tag of an IP with style.charOnly true.
*/
#define tagIPSignatureSPaper	MakeTag(clsIP, 7)
#define tagIPSingleChar			MakeTag(clsIP, 8)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages											   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef NO_NEW
#ifndef ipNewFields

/****************************************************************************
 msgNew		takes IP_NEW, returns STATUS
	Creates a new IP.

 In response to msgNew, clsIP creates the necessary components for the IP. 
 This may include an instance of clsField, clsSPaper, or clsCommandBar. The
 various components are initialized according to the new.ip.style settings.
 
 The internal sPaper or field requires a translator.  If xlateType is 
 ipXlateObject, pNew->ip.xlate.translator is used as the translator object.
 If xlateType is ipXlateTemplate, then pNew->ip.xlate.pTemplate is compiled
 and allocated at msgNew, and freed when the component is destroyed.  See
 the section "IPs and Translators" for more information.

 border.style.bottomMargin is always bsMarginMedium.

 Finally, based on embeddedLook, msgNew changes the border style of the IP
 and the border and margin styles of the internal components to make the IP
 look good when embedded (embeddedLook true) or when floating (embeddedLook
 false).

 Defaults changed if embeddedLook is false:
//{
	border.style.borderInk = bsInkGray66;
	border.style.leftMargin = bsMarginMedium;
	border.style.rightMargin = bsMarginMedium;
	border.style.topMargin = bsMarginMedium;
	border.style.backgroundInk = bsInkGray33;
	border.style.shadow = bsShadowThickGray;
	win.flags.style |= wsSaveUnder;
//}

 Defaults changed if embeddedLook is true:
//{
	border.style.borderInk = bsInkGray33;
	border.style.leftMargin = bsMarginNone;
	border.style.rightMargin = bsMarginNone;
	border.style.topMargin = bsMarginNone;
	border.style.backgroundInk = bsInkWhite;
	border.style.shadow = bsShadowThickWhite;
	win.flags.style &= ~wsSaveUnder;
//}
*/


//
// Translation information.  Notice that this is a union type.  See the
// section "IPs and Translators" for more information.
//
typedef union IP_XLATE {

	OBJECT		translator;		// xlateType = 0, clsXlate object
	P_UNKNOWN 	pTemplate;		// xlateType = 1, P_XTEMPLATE

} IP_XLATE, *P_IP_XLATE;


typedef struct IP_NEW_ONLY {

	IP_STYLE	style;		// IP style
	IP_XLATE	xlate;		// See the section "IPs and Translators."
							// Translation information for the IP.
	U16			lineHeight;	// Unused
	OBJECT		client;		// Client for notification messages.
							// See the section "Client and Observers."
	P_CHAR		pString;	// Initial string for ipsCharBox IP's field.
	U8			rows,cols;	// Number of rows and cols in IP. Can
							// be zero if shrinkWrap is on.
	U16			lines;		// Unused
    U16			xIndex;		// Unused
    U32			reserved1;
    U16			maxLen;		// Max string length IP can return. 
    						// 0 means no limit.
    U32			reserved;

} IP_NEW_ONLY, *P_IP_NEW_ONLY;

#define ipNewFields			\
	customLayoutNewFields   \
	IP_NEW_ONLY ip;

typedef struct IP_NEW {

	ipNewFields

} IP_NEW, *P_IP_NEW;

#endif // ipNewFields
#endif // NO_NEW


/****************************************************************************
 msgNewDefaults		takes P_IP_NEW, returns STATUS
	Initializes the IP_NEW structure to default values.

 When handling msgNew, certain border.style values are changed depending on
 the value of ip.embeddedLook.  See msgNew for details.

 Zeros out pNew->ip and sets:
//{
    ip.style.displayType = ipsRuledLines;
    ip.style.buttonType = ipsBottomButtons;
    ip.style.modal = ipsNoMode;
	ip.style.delayed = true;
	ip.maxLen = maxU16;

    border.style.edge = bsEdgeAll;
    border.style.resize = bsResizeCorner;
	border.style.drag = bsDragDown;
	border.style.top = bsTopDrag;

	customLayout.style.limitToRootWin = true;
    win.flags.input |=
			(inputTip | inputChar | inputMultiChar | inputAutoTerm | \
			inputInProx | inputEnter | inputMoveUp | inputMoveDelta);
	win.flags.style |= wsSendGeometry;
	embeddedWin.style.moveable = false;
	embeddedWin.style.copyable = false;
//}
*/


/****************************************************************************
 msgIPGetStyle		takes P_IP_STYLE, returns STATUS
	Passes back the style of the IP.
*/
#define msgIPGetStyle			MakeMsg(clsIP, 5)


/****************************************************************************
 msgIPSetStyle		takes P_IP_STYLE, returns STATUS
	Changes the style of the IP.

 Clients use this message to change the style settings of an IP.  Also, an
 IP self sends this message to perform transmogrification.  

 In response to this message, an IP destroys obsolete components and
 creates new necessary ones.  For example, changing from ipsCharBox to
 ipsRuledLines destroys the field component and creates an sPaper
 component.

 If an sPaper is being destroyed and a field being created, or vice versa,
 the IP extracts the translator information from the component about to be
 destroyed and moves it into the newly created one.

 This message dirties the layout the IP.  
 
 This method does not change the IP's embeddedLook, xlateType, takeGrab, or
 takeFocus.
*/
#define msgIPSetStyle			MakeMsg(clsIP, 6)


/****************************************************************************
 msgIPGetTranslator		takes P_OBJECT, returns STATUS
	Passes back the translator for the IP.

 Passes back the translator for the IP, regardless of how it was created.
 An ipsBlank or ipsRuledLines IP passes back the translator used by the
 sPaper component.  An ipsCharBox IP passes back the translator used by the
 field component.

 See the section "IPs and Translators" for more information.
*/
#define msgIPGetTranslator		MakeMsg(clsIP, 7)


/****************************************************************************
 msgIPSetTranslator		takes P_OBJECT, returns STATUS
	Sets the translator for the IP.

 Use this message to set an IP's translator.  

 In response to this message, a ipsCharBox IP sets its field's translator. 
 Other IPs sets their sPaper's translator.  All IPs change their
 style.xlateType to ipXlateObject.  
 
 IMPORTANT:  An IP does NOT free the current translation information in
 response to this message.  The client must free this translation
 information. See the section "IPs and Translators" for more information.
*/
#define msgIPSetTranslator		MakeMsg(clsIP, 20)


/****************************************************************************
 msgIPGetClient		takes P_OBJECT, returns STATUS
	Passes back the IP's client object in *pArgs.

 See the section "Client and Observers" for more information.

 See Also
	msgIPSetClient
*/
#define msgIPGetClient			MakeMsg(clsIP, 14)


/****************************************************************************
 msgIPSetClient		takes P_OBJECT, returns STATUS
	Makes pArgs the IP's client.

 See the section "Client and Observers" for more information.

 See Also
	msgIPGetClient
*/
#define msgIPSetClient			MakeMsg(clsIP, 15)


/****************************************************************************
 msgIPSetString		takes P_CHAR, returns STATUS
	Sets a ipsCharBox IP's string.

 Use this message to initialize or change the contents of a ipsCharBox IP.

 For ipsCharBox IPs, the default response to this message is to set the
 field component's string and to re-layout the IP.  For other types of IPs,
 the default response is to return stsOK.

 See the section "IP Components" for more information.
*/
#define msgIPSetString			MakeMsg(clsIP, 10)


/****************************************************************************
 msgIPTranslate		takes nothing, returns STATUS
	Translates scribbles in an IP.  

 When pressed, the "OK" button of an IP's command bar sends this message to
 the IP.  Also, a client can send this message to cause an IP to translate
 any scribbles.  An IP also self sends this message (1) in response to
 msgGWinForwardedKey and (2) when a modal IP terminates the mode
 (style.takeGrab is true, style.modal is ipsTranslateMode, and the pen taps
 outside of the IP).

 The IP's response to this message is as follows.

    -:	ipsRuledLines and ipsBlank IPs send msgSPaperComplete to the IP's
        sPaper component.  (The sPaper in turn sends 
        msgSPaperXlateCompleted back to the IP;  see the comments on 
        msgSPaperXlateCompleted for IP's response.)
 
    -:	ipsCharBox IPs with style.delayed false self send
        msgIPDataAvailable.

    -:	ipsCharBox IPs with style.delayed true and untranslated scribbles in
        the field first translate the scribbles and then self send
        msgIPCopied.

    -:	ipsCharBox IPs with style.delayed true and no untranslated scribbles
        in the field self send msgIPDataAvailable.

 pArgs must be 0.

 See Also
	msgSPaperXlateCompleted
*/
#define msgIPTranslate			MakeMsg(clsIP, 3)


/****************************************************************************
 msgIPCancelled		takes OBJECT, returns STATUS
	Cancels an IP.  Also sent to notify observer/client about the cancel.

 When pressed, the "Cancel" button of an IP's command bar sends this
 message to the IP.  A client can also send this message to cause an IP to
 cancel itself. Also, msgIPCancelled is sent to an IP's observers/client to
 notify them about the cancelling.

 msgIPCancelled is also self sent if a modal IP has a style.modal value of 
 ipsCancelMode and the modal IP is terminated (probably by a pen tap
 outside the IP).

 The IP's response to msgIPCancelled is a follows.  First the IP clears the
 component (field or sPaper) of any data it contains.  Next, if the IP's
 style.freeAfter is true, the IP extracts itself from the window hierarchy
 and posts msgDestroy to itself.  Finally, it sends msgIPCancelled to
 observers/client to inform them of the cancellation.

 See the sections "IP Destruction" and "Client and Observers" for additional
 information.

 See Also
	msgIPClear
*/
#define msgIPCancelled			MakeMsg(clsIP, 18)


/****************************************************************************
 msgIPClear		takes OBJECT, returns STATUS
	Clears an IP's contents.  Also sent to notify observers/client about
	the clearing.

 When pressed, the "Clear" button of an IP's command bar sends this message
 to the IP.  A client can also send this message to cause an IP to clear
 its contents.  Also, msgIPClear is sent to an IP's observers/client to
 notify them about the clearing.

 An IP's response to msgIPClear is as follows.  If the IP has an sPaper
 component (ipsRuledLines or ipsBlank IP), then msgSPaperClear is sent to
 the sPaper.  If the IP has a field component, and style.ruledToBoxed is
 false, then msgFieldClear is sent to the field.  If the IP has a field and
 style.ruledToBoxed is true, then the IP transmogrifies itself to have an
 sPaper.  Finally, msgIPClear is sent to the IP's observers/client.

 See the sections "IP Components," "Client and Observers" and
 "Transmogrification" for additional information.

 See Also
	msgIPCancelled
	msgSPaperClear (spaper.h)
	msgFieldClear (field.h)
*/
#define msgIPClear				MakeMsg(clsIP, 23)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Observer/Client Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgIPCopied		takes OBJECT, returns STATUS
	Notifies observer/client that newly translated data has been copied
	into a ipsCharBox IP's field.

 See the section "Client and Observers" for additional information.
*/
#define msgIPCopied   			MakeMsg(clsIP, 19)


/****************************************************************************
 msgIPDataAvailable		takes OBJECT, returns STATUS
	Notifies observers/client that the IP has translated data available.

 Observers/clients can respond to this message by sending msgIPGetXlateData
 or msgIPGetXlateString to get the translated data.

 See the section "Client and Observers" for additional information.

 See Also
	msgIPTranslate
*/
#define msgIPDataAvailable		MakeMsg(clsIP, 16)


/****************************************************************************
 msgIPTransmogrified		takes OBJECT, returns STATUS
	Notifies observers/client that the IP has been transmogrified.

 See the sections "Transmogrification" and "Client and Observers" for
 additional information.

 See Also
	msgIPTranslate
	msgIPGetXlateData
	msgIPGetXlateString
	msgIPSetStyle
*/
#define msgIPTransmogrified		MakeMsg(clsIP, 24)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Data Retrieval Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgIPGetXlateData		takes P_IP_XLATE_DATA, returns STATUS
	Passes back translated data in xlist form.

 The default response to msgIPGetXlateData is as follows.

 An xlist is created in pArgs->heap (or osProcessHeapId if pArgs->heap is
 null.)  Then the xlist is filled in as follows.

    -:	An ipsCharBox IP's xlist contains an xtBounds followed by an xtText
        element.  The IP's field is cleared (using msgFieldClear; see
        field.h).  (The bounds is artificially constructed.)

    -:	An ipsRuledLines IP's xlist contains the xlist returned by sending
        msgSPaperGetXlateData (see spaper.h) to the sPaper component of the
        IP.

    -:	This message should not be sent to a ipsBlank IP because no
        translation is ever performed by this type of IP.

 If the IP's style.freeAfter is true, then the IP self destructs;  see the
 section "IP Destruction" for details.

 If self is a ipsCharBox IP and style.boxedToRuled is true, then the IP
 transmogrifies into a ipsRuledLines IP.  See the "Transmogrification"
 section.

 If self is a ipsCharBox IP, then the space collapse rules defined in
 style.spaceCollapse are applied to the xtText element in the xlist.
 
 IMPORTANT:  The sender of msgIPGetXlateData must free the returned xlist
 and elements in the xlist.  (See xlist.h in general and XListFree() in
 particular.)
 
 See Also
	msgIPTransmogrified
	msgIPSetStyle
	spaper.h
	xlist.h
*/

#define msgIPGetXlateData		MakeMsg(clsIP, 4)

typedef struct IP_XLATE_DATA {

	OS_HEAP_ID heap;		// In:	heap for xlist allocation.
	P_UNKNOWN pXList;		// Out:	pointer to resulting xlist.

} IP_XLATE_DATA, *P_IP_XLATE_DATA;


/****************************************************************************
 msgIPGetXlateString		takes P_IP_STRING, returns STATUS
	Passes back translated data in string form.

 In response to this message, an IP passes back its translated contents as
 a simple string form.

 Clients should use this message rather msgIPGetXlateData if a simple
 string is needed.  Clients should use msgIPGetXlateData if the additional
 information contained in an xlist is needed.

 If pArgs->len is maxU16, the IP allocates the necessary string memory from
 the process heap.  The sender of msgIPGetXlateString must free this
 memory.

 The returned pArgs->pString is "clipped" to pArgs->maxLen.  The actual
 number of characters returned is returned in pArgs->len. 
 
 Note:  The handler of this message first  self sends msgIPGetXlateData to
 get an xlist and then converts the data xlist to a string.  See the
 comments regarding msgIPGetXlateData for information on the IP's self
 destruction and transmogrification.
 
 See Also
	msgIPGetXlateData
*/

typedef struct IP_STRING {

	U16 len;			// In-Out: length of buffer pointed to by pString
	P_CHAR pString;		// In-Out: buffer pointer

} IP_STRING, *P_IP_STRING;

#define msgIPGetXlateString		MakeMsg(clsIP, 17)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFree	takes P_OBJ_KEY, returns STATUS
	Defined in clsmgr.h.

 The IP sends msgFree to its components.  It then frees any translation
 information passed into msgNew.

 See the section "IPs and Translators" for more information.
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Defined in clsmgr.h.

 The IP saves all necessary state and uses the  window hierarchy filing
 mechanism to save any components.

 If the IP's client is OSThisApp(), this is remembered.  See msgRestore for
 more information.
*/


/****************************************************************************
 msgRestore		takes P_OBJ_RESTORE, returns STATUS
	Defined in clsmgr.h.

 clsIP restores self and uses the window hierarchy filing mechanism to
 restore any components.  clsIP then re-establishes the necessary
 connections between self and each component.

 If the IP's client was OSThisApp() when saved, then the IP's client
 becomes OSThisApp(); otherwise the client becomes to objNull.

 See Also
	win.h
*/


/****************************************************************************
 msgSetOwner	takes P_OBJ_OWNER, returns STATUS
	Defined in clsmgr.h.

 An IP lets its superclasses respond to this message and then sends
 msgSetOwner to its components.

 See the section "IP Components" for more information.
*/


/****************************************************************************
 msgSPaperXlateCompleted		takes OBJECT, returns STATUS
	Defined in spaper.h.

 Only sophisticated subclasses might want to handle this message.  An IP
 with an sPaper component (ipsRuledLines and ipsBlank) receives this message
 from the sPaper when the sPaper has completed translation.

 If style.ruledToBoxed is false, this message simply self sends
 msgIPDataAvailable.  Otherwise the IP tries to transmogrify itself, using
 the following steps:
	-:	The translated string is extracted from the sPaper component.
    -:	If the string is empty, the IP self sends msgIPDataAvailable and
        gives up the effort to transmogrify.
    -:	The IP transmogrifies itself.

 In both cases, the sPaper component (if it still exists) is cleared.

 See the "Transmogrification" section.

 See Also
	msgIPTranslate
*/


/****************************************************************************
 msgWinStartPage	takes nothing, returns STATUS
	Defined in win.h.

 Only sophisticated subclasses might want to handle this message.  This
 message informs an IP that it exists on a printer and that printing is
 about to commence.

 If the IP is not ipsBlank, an IP's default response is to return stsOK.
 Otherwise, the IP turns off all of its own margins and all of the borders
 and ruling lines for the sPaper component.  This causes the IP to
 print only the scribbles, which is particularly appropriate when an IP
 has been used to capture and hold a signature.
*/


/****************************************************************************
 msgGWinForwardedKey	takes P_INPUT_EVENT, returns STATUS
	Defined in gwin.h.

 Only sophisticated subclasses might want to handle this message.  A child
 window may send this message when the child receives a keyboard input
 event that it doesn't want to handle.

 If the key's keyCode is uKeyReturn (see key.h), the IP self sends
 msgIPTranslate.  Otherwise it returns stsRequestForward.

 Sent when a component (field) forwards a key.  An IP containing a field
 component that forwards the Return key causes msgIPTranslate to be self
 sent, as if the "OK" button was pressed.


 See Also
	msgIPTranslate
	key.h
	input.h
*/


/****************************************************************************
 msgInputTargetActivated	takes OBJECT, returns STATUS
	Defined in input.h.

 Only sophisticated subclasses might want to handle this message.  The
 input system sends this message to an object to inform an object that it
 is no the input target.

 In response to this message, an IP remembers the previous input target. 
 If the IP is a ipsCharBox IP, it makes the IP's field the input target.

 The IP restores the previous input target as part of its response to
 msgWinExtracted.
*/


/****************************************************************************
 msgTrackProvideMetrics		takes P_TRACK_METRICS, returns STATUS
	Defined in track.h.

 Only sophisticated subclasses might want to handle this message.

 clsIP is a descendant of clsBorder.  Unless turned off by a subclass, an
 IP is resizeable by the user.  When clsBorder creates a resize object and
 its associated tracker, it first self sends msgTrackProvideMetrics to
 allow itself to modify the parameters of the tracker.

 In response to this message, an IP does the following:

    -:	If the tracker is not a resizing tracker, the IP simply returns
        stsOK.  

    -:	The IP remembers the original client of the tracker so that the IP
        can forward tracker-related messages onto that original client.  It
        then makes itself be the client of the tracker.

    -:	If the IP has a command bar (style.buttonType is ipsBottomButtons or
        ipsTopButtons), then pArgs->style.draw is set to tsDrawCmdBarRect
        and pArgs->cmdBarH is set appropriately.

    -:	The tracker's minimum size constraints are adjusted so that the IP
        can get no smaller than the scribbles that are in the IP's field or
        sPaper.  This prevents scribbles from being covered.

    -:	The IP the makes itself the client of the tracker so that the IP
        receives msgTrackUpdate and msgTrackDone.

 See Also
	msgTrackUpdate
	msgTrackDone
*/


/****************************************************************************
 msgTrackUpdate		takes P_TRACK_METRICS, returns STATUS
	Defined in track.h.

 Only sophisticated subclasses might want to handle this message.

 The default response to this message is to forward the message to the
 tracker's original client, as remembered in msgTrackProvideMetrics.

 See Also
	msgTrackProvideMetrics
*/


/****************************************************************************
 msgTrackDone		takes P_TRACK_METRICS, returns STATUS
	Defined in track.h.

 Only sophisticated subclasses might want to handle this message.

 ipsBlank IPs can be resized to any size.  Otherwise the default response
 to this message is to force the new size of the IP to fit nicely around
 whole rows and columns (in ipsCharBox IPs) or lines (in ipsRuledLines
 IPs).  Then the message is forwarded to the tracker's original client, as
 remembered in msgTrackProvideMetrics.

 See Also
	msgTrackProvideMetrics
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Obsolete											   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define stsIPNotSupported	MakeStatus(clsIP, 1) // Obsolete
#define stsIPBadMode		MakeStatus(clsIP, 2) // Obsolete

#endif
