/****************************************************************************
 File: fsutil.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.44  $
   $Author:   sisaac  $
     $Date:   04 Mar 1992 13:55:36  $

 This file contains filesystem attribute helper procedures.
 The functions described in this file are contained in SYSUTIL.LIB. 

 These procedures make it easier to deal with filesystem attributes.
 They also support list attributes; variable attributes which maintains 
 lists of 4-byte quanitities.
****************************************************************************/
#ifndef FSUTIL_INCLUDED
#define FSUTIL_INCLUDED

#ifndef FS_INCLUDED
#include <fs.h>
#endif

/****************************************************************************
 GetNodeName	returns STATUS
	Gets the name attribute of a given filesystem node. 

 Use this function to easily get the name of a node.
*/
STATUS EXPORTED GetNodeName(
	OBJECT					handle,	  	// File or dir handle.
	P_STRING				pName);		// Out: name.


/****************************************************************************
 GetAttr	returns STATUS
	Gets a single FIX32 attribute from a filesystem handle. 

 This is only for FIX32 attributes when you have a handle 
 onto the node; see GetSingleAttr for a	more general function.
*/
STATUS EXPORTED GetAttr(
	FS_ATTR_LABEL			attrLabel,	// Attribute label.
	OBJECT					handle,	  	// File or dir handle.
	P_U32					pValue);	// Out: attribute value.


/****************************************************************************
 GetSingleAttr	returns STATUS
	Gets a single FIX32, FIX64, or known-size STRING attribute.

*/
STATUS EXPORTED GetSingleAttr(
	FS_ATTR_LABEL			attrLabel,	// In: Attribute label.
	OBJECT					handle,		// In: handle of node.
	P_STRING				pPath,		// In: path of node.
	P_UNKNOWN				pValue);	// Out: attribute value.


/****************************************************************************
 SetAttr	returns STATUS
	Sets a single FIX32 attribute on a filesystem handle.

 This is only for FIX32 attributes when you have a handle 
 onto the node; see SetSingleAttr for a	more general function.
*/
STATUS EXPORTED SetAttr(
	FS_ATTR_LABEL			attrLabel,	// Attribute label.
	OBJECT					handle,		// File or dir handle.
	U32						value);		// Attribute value.


/****************************************************************************
 SetSingleAttr	returns STATUS
	Sets a single FIX32, FIX64, or STRING attribute.

*/
STATUS EXPORTED SetSingleAttr(
	FS_ATTR_LABEL			attrLabel,	// In: Attribute label.
	OBJECT					handle,		// In: handle of node.
	P_STRING				pPath,		// In: path of node.
	P_UNKNOWN				pValue);	// In: attribute value.


/****************************************************************************
 GetListX		returns STATUS
	Gets a VAR attribute that is organized as a list of values.

 Allocates ppList from the process local stack. Caller must HeapBlockFree 
 ppList when done adding, removing, and putting the list.
*/
STATUS EXPORTED GetListX(
	OBJECT					handle,		// File or dir handle.
	P_STRING				pPath,		// Path relative to handle.
	FS_ATTR_LABEL			attrLabel,	// Attribute label.
	PP_UNKNOWN				ppList,		// Out: list.
	P_U16					pSize);		// Out: size (in bytes) of list.


/****************************************************************************
 PutListX		returns STATUS
	Updates a list attribute with a new list.

*/
STATUS EXPORTED PutListX(
	OBJECT					handle,		// File or dir handle.
	P_STRING				pPath,		// Path relative to handle.
	FS_ATTR_LABEL			attrLabel,	// Attribute label.
	P_UNKNOWN				pList,		// List.
	U16						size);		// Size (in bytes) of list.


/****************************************************************************
 FindListItemX	returns STATUS
	Finds an element in a list.

 The list must first be gotten via GetList. pOffset is 0 based. The list
 array can be indexed with pOffset to get the actual data.  The comparison
 is done via a memcmp, so things must be EXACTLY the same.

 Return Value
	stsNoMatch:		Item not found.
*/
STATUS EXPORTED FindListItemX(
	P_UNKNOWN				pItem,	  	// Data to search for.
	U16						itemSize,	// Size of data to search for.
	P_UNKNOWN				pList,		// List.
	U16						listSize,	// Size of list.
	P_U16					pOffset);	// Out: offset of found item.


/****************************************************************************
 AddListItemX		returns STATUS
	Adds an item to the end of a list. 

 The list must first be gotten via GetList. The heap that the list uses is
 resized. pSize is updated to reflect the new list size.
*/
STATUS EXPORTED AddListItemX(
	P_UNKNOWN				pItem,  	// Item to add.
	U16						itemSize,	// Size of item in bytes.
	PP_UNKNOWN				ppList,		// In:Out List.
	P_U16					pSize);		// In:Out size of list in bytes.


/****************************************************************************
 RemoveListItemX		returns STATUS
	Removes an item from a list, given an offset.

 The list must first be gotten via GetList. The heap that the list uses is
 resized. If pSize == 1 (only 1 item left) then *pSize is set to 0, but
 the list heap is not resized. offset is 0-based.
*/
STATUS EXPORTED RemoveListItemX(
	U16						offset,		// Offset of item to remove.
	U16						size,		// Size of item to remove.
	PP_UNKNOWN				ppList,	    // In:Out List.
	P_U16					pSize);		// In:Out Size of list.


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Private																   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* 
 Below are the "old" attribute list functions. These are here for
 backwards compatability only!
*/

/****************************************************************************
 GetList		returns STATUS
	Gets a VAR attribute that is organized as a list of 4 byte values.

 Allocates ppList from the process local stack. Caller must HeapBlockFree 
 ppList when done adding, removing, and putting the list.
*/
STATUS EXPORTED GetList(
	OBJECT					handle,		// File or dir handle.
	P_STRING				pPath,		// Path relative to handle.
	FS_ATTR_LABEL			attrLabel,	// Attribute label.
	PP_OBJECT				ppList,		// Out: list.
	P_U16					pCount);	// Out: number of elements.


/****************************************************************************
 PutList		returns STATUS
	Updates a list attribute with a new list.

*/
STATUS EXPORTED PutList(
	OBJECT					handle,		// File or dir handle.
	P_STRING				pPath,		// Path relative to handle.
	FS_ATTR_LABEL			attrLabel,	// Attribute label.
	P_OBJECT				pList,		// List.
	U16						count);		// Number of elements.


/****************************************************************************
 FindListItem	returns STATUS
	Finds an element in a list.

 The list must first be gotten via GetList. pIndex is 0 based. The list
 array can be indexed with pIndex to get the actual data.

 Return Value
	stsNoMatch:		Item not found.
*/
STATUS EXPORTED FindListItem(
	OBJECT					item,	  	// Data to search for.
	P_OBJECT				pList,		// List.
	U16						count,		// Number of elements in list.
	P_U16					pIndex);	// Out: index of found item.


/****************************************************************************
 AddListItem		returns STATUS
	Adds an item to the end of a list. 

 The list must first be gotten via GetList. The heap that the list uses is
 resized. pCount is updated to reflect the new list size.
*/
STATUS EXPORTED AddListItem(
	OBJECT					item,  		// Item to add.
	PP_OBJECT				ppList,		// In:Out List.
	P_U16					pCount);	// In:Out number of elements in list.


/****************************************************************************
 RemoveListItem		returns STATUS
	Removes an item from a list, given an index.

 The list must first be gotten via GetList. The heap that the list uses is
 resized. If pCount == 1 (only 1 item left) then *pCount is set to 0, but
 the list heap is not resized. index is 0-based.
*/
STATUS EXPORTED RemoveListItem(
	U16						index,		// Index of item to remove.
	PP_OBJECT				ppList,	    // In:Out List.
	P_U16					pCount);	// In:Out Number of elements in list.

#endif	// FSUTIL_INCLUDED

