/****************************************************************************
 field.h

 Copyright 1992, GO Corporation, All Rights Reserved

 $Revision:   1.13  $
   $Author:   kcatlin  $
	 $Date:   12 Mar 1992 16:14:36  $

 This file contains the API definition for clsField
 clsField inherits from clsLabel.
 Implements the UI component to edit, validate and display string data.

 Fields implement the basic UI component to edit simple strings of text.
 The user-interface for fields has been optimized for simple short 
 one row strings of text, although they will function for multiple lines.
 All display information for translated fields is handled in clsLabel.
 Typically the label layout is fixed, and shrink wrap will
 be turned off in the label.  Otherwise the field size will change
 as the value of the string changes, and lead to strange results and
 behavior.  There are three basic User-Interfaces supported through the
 API to edit fields.  These are defined in field.style.editType.  

 Fields with editType of fstInline support direct writing, appending,
 and a number of gestural editing operations, including bringing up
 an IP.  Fields with editType of fstPopUp will only allow editing
 through an IP. Fields with editType of fstOverWrite make the field 
 combed and allow over-writing on individual characters.
 These fields have very precise stroke targetting due to the 
 character box constraints.  This, in combination with only allowing
 three editing gestures (insert space, delete range, and delete character)
 allows for highly accurate handwriting and gesture recognition and
 for quick correction of mistakes.  The down side of this style of field
 is that a specific UI look is implied.

 To further increase recognition accuracy, fields require a 
 translator for both inline editing and in the IP.  Translators
 have a rich API to provide various types of contextual information.
 This greatly increases translation accuracy.  See msgNew, 
 msgFieldGetXlate, msgFieldSetXlate, msgFieldCreateTranslator.

 Fields can also be run in delayed mode.  Delayed fields allow the
 user to write into an empty field, and not translate the strokes on 
 pen out of proximity.  Delayed fields are translated when 
 msgFieldTranslateDelayed is sent to the field. See msgFieldTranslateDelayed,
 msgFieldSetDelayScribble, and msgFieldGetDelayScribble for more information.

 Fields will replace newLines with spaces, and will strip trailing spaces
 when their value is retrieved.  The value should be set via
 msgLabelSetString and retrieved via msgLabelGetString.

 Messages from clsInput, messages from clsGWin (other then msgGWinGesture), 
 messages from clsWin, messages from clsLabel, messages from
 clsSelection, messages from clsXfer, messages from xlate, and messages 
 from clsTracker should NOT be overridden by subclasses of clsField.  

 Finally, fields provide simple hooks to allow clients or subclasses to
 perform various validation according to a common protocol.  See 
 msgFieldValidate for details.
****************************************************************************/

#ifndef FIELD_INCLUDED
#define FIELD_INCLUDED

												#ifndef GO_INCLUDED
#include <go.h>
												#endif
												#ifndef LABEL_INCLUDED
#include <label.h>
												#endif
												#ifndef XTEMPLT_INCLUDED
#include <xtemplt.h>
												#endif

// Next Up:	31	Recycled: 28

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT FIELD;

/****  Field Editing Types  ****/
/*
 * These define the types of edit User-Interface the field provides,
 * defining the behavior of the field. These are used for style.editType.
 */
#define fstInline         1		// Direct editing on field, or through IP
#define fstPopUp		  2		// Editing only through an IP
#define fstOverWrite	  3		// Editing in combed overwrite field

/****  Insertion Pad Types  ****/
/*
 * These define the type of Insertion Pad that will be
 * created in msgFieldCreatePopUp when the type parameter
 * is fipReplaceAll.  Note: A call to msgFieldCreatePopUp when
 * the type parameter is fipInsert will look at the system 
 * preferences to determine the type of IP. These are used
 * for style.popUpType.
 */
//#define fstEditBox		  1	// Obsolete
#define fstCharBox		  2		// The pop-up is an ipsCharBox IP
#define fstCharBoxButtons 2		// Obsolete

/****  Character Box Memory  ****/
/*
 * For fstOverWrite fields, this defines the number of characters
 * that should be used sent to the translator via 
 * msgXlateCharMemorySet.  This causes the translator to cycle
 * through choices and not return the same character from a translation.
 * These are use for style.boxMemory.
 */
#define fstBoxMemoryZero	0	// Box memory is zero characters
#define fstBoxMemoryOne		1	// Box memory is one character
#define fstBoxMemoryFour	2	// Box Memory is four characters

/****  Selection/Input Target  ****/
/*
 * These define the interaction the field should have with 
 * both the selection manager and the input target when:
 *	- msgFieldKeyboardActivate is called 
 *	- the pen is interacting with the field
 *	- msgFieldTranslateDelayed is called
 *	- the field is the recipient of a move/copy operation 
 * These are used for style.focusStyle.
 */
#define fstInputSelection 1		// Field takes selection and input target
#define fstInput		  2		// Field takes input target only
#define fstNone			  3		// Field takes neither selection nor target

/****  Upper Case Writer Rules  ****/
/*
 * These define the capitalization heuristic rules used by the field
 * translator. These rules do not apply when the translator is provided 
 * by the client of the field, or the writer is not an all-caps writer.
 * These are used for style.capOutput.
 */
#define fstCapAsIs			1	
#define fstCapFirstWord		2	
#define fstCapAllWords		3	
#define fstCapAll		  	4

/****  Translator Type  ****/
/*
 * These define the type of translator given to and maintained by 
 * the field, and affects the parameters to msgFieldGetTranslator 
 * and msgFieldSetTranslator, the interaction with msgFieldCreateTranslator, 
 * and msgNew.  See these messages for more information.  These
 * are used for style.xlateType.
 */
#define fstXlateObject		0
#define fstXlateTemplate	1

/****  Field Style Structure  ****/
/*
 * The field style structure defines the overall behavior of the field.
 * Information on the various flags can be found elsewhere.  For information
 * on focusStyle, capOutPut, popUpType, editType, xlateType, delayed and
 * boxMemory, see above.  
 *
 * For information on noSpace and veto, see msgFieldCreateTranslator.
 */
typedef struct FIELD_STYLE {
	U16 focusStyle:		2,			// How field does selection and target
		capOutput:		3,			// Upper case writer cap rules for xlate
		popUpType:	    3,  	 	// Insertion pad style for fipReplaceAll
		editType:		2,  	 	// Type of editing in field
		xlateType:	    1,  	 	// 0=xlate object, 1=xtemplate
		clientValidate: 1,  	 	// client performs validation
	    clientPreValidate:    1, 	// Notify client before validation
		clientPostValidate:	  1, 	// Notify client after successful valid
	    clientNotifyInvalid:  1, 	// Notify client when invalid
		clientNotifyReadOnly: 1; 	// Notify client when attempt to modify
									// readonly field
	U16	clientNotifyModified: 1, 	// Notify client when field modified
	    validatePending:	  1, 	// Field not valid since last modification
		delayed:			  1, 	// Delayed translation field.  Capture
									// strokes till msgFieldTranslateDelayed
		upperCase:			  1, 	// Field and IP forced to upper case
		noSpace:			  1, 	// Turn on no space in fld created xlate
		privateData1:		  1, 	// Internal use only
		veto:				  1, 	// Turn on veto in fld created xlate
		privateData2:		  1, 	// Internal use only
		boxMemory:			  2, 	// Enable box memory in field and IP
		dataMoveable:		  1,
		dataCopyable:		  1,
		reserved:			  5; 	// Reserved for future use
} FIELD_STYLE, *P_FIELD_STYLE;

/****  Popup Editing Types  ****/
/*
 * These defines are parameters in msgFieldCreatePopUp and 
 * msgFieldActivatePopUp.  They specify what type of edit operation should
 * be performed by this pop-up.  Internally, an edit gesture (circle) in
 * an fstInline field or pen input into fstPopUp field will call 
 * these messages with fipReplaceAll.  An insert caret in an fstInline
 * field will call with fipInsert. 
 */
#define fipReplaceAll 	0		// The IP displays/edits the field value
#define fipInsert     	1		// The IP inserts new text at the insertion pt
#define fipReplaceRange 2 		// Unimplemented

/****  Validation data structure  ****/
/*
 * This data structure is used as a parameter to msgFieldValidateEdit,
 * and msgFieldNotifyInvalid to capture all validation information.
 */
typedef struct {
	MESSAGE         failureMessage; // Reason validation failed
	OBJECT			field;			// Field to validate
} FIELD_NOTIFY, *P_FIELD_NOTIFY;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_FIELD_NEW, returns STATUS
	category: class message
	Creates and initializes a new instance of clsField.  

 Will force the label.style to lsBoxTicks for fields of editType
 fstOverWrite.  Overwrite fields must have label style of lsBoxTicks.
 Will force gWin.style.gestureEnable to TRUE.  Extreme
 care should be taken if changing either of these. The xlate parameter in
 conjunction with style.xlateType specifies the type of translator
 the field uses.  If xlateType is 0, and pNew->field.xlate.translator
 does not equal objNull, the translator will be used for all translations
 in the field and in the IP, and destroyed when the field is destroyed.
 If xlateType is 1, pNew->field.xlate.pTemplate is used to create, allocate,
 and compile a template.  It will also be freed when the field is destroyed.
 A translator will be created and destroyed as needed via 
 msgFieldCreateTranslator from this compiled template.  msgFieldCreateTrans
 will also be used when xlateType is 0 and pNew->field.xlate.translator is
 objNull.

 See Also
	msgFieldSetXlate
	msgFieldGetXlate
	msgFieldCreateTranslator
*/

typedef union FIELD_XLATE {
	OBJECT		translator;      
	P_XTM_ARGS	pTemplate;    
} FIELD_XLATE, *P_FIELD_XLATE;

typedef struct FIELD_NEW_ONLY {
	FIELD_STYLE style;	    // field style, see  above
	FIELD_XLATE xlate;	    // xlate object or template
	U16			maxLen;		// maximum field string length.  0 means no limit
	U32		    reserved;   // reserved for future use, must be  0
} FIELD_NEW_ONLY, *P_FIELD_NEW_ONLY;

#define fieldNewFields			  \
	labelNewFields				  \
	FIELD_NEW_ONLY	  field;

typedef struct FIELD_NEW {
   fieldNewFields
} FIELD_NEW, *P_FIELD_NEW;

/****************************************************************************
 msgNewDefaults		takes P_FIELD_NEW, returns STATUS
	category: class message
	Initializes the FIELD_NEW structure to default values.

 Initializes the default values.  Care should be taken when changing
 the default values of parent classes.  Examples are win.flags.input,
 or gwin.style.

 Zeros out pNew->field and sets
//{
 	fld.field.style.dataMoveable = true;
	fld.field.style.dataCopyable = true;
	fld.field.style.focusStyle = fstInputSelection;
	fld.field.style.capOutput = fstCapAsIs;
	fld.field.style.editType = fstInline;
	fld.field.style.popUpType = fstCharBoxButtons;
	fld.field.style.xlateType = fstXlateObject;
	fld.field.style.boxMemory = fstBoxMemoryFour;
	fld.field.maxLen = 64;
	fld.border.style.edge = bsEdgeBottom;
	fld.gwin.style.firstEnter = TRUE;
	fld.gwin.style.askOtherWin = TRUE;
	fld.gwin.style.otherWinSaysYes = TRUE;
	fld.win.flags.input = inputTip | inputStroke | 
			inputOutProx | inputInk | inputEnter | 
			inputHoldTimeout | inputLRContinue | 
			inputAutoTerm | inputTimeout | inputHWTimeout;
//}
*/

/****************************************************************************
 msgFieldGetStyle	takes P_FIELD_STYLE, returns STATUS
	Passes back the style value held by the field.
*/
#define msgFieldGetStyle			MakeMsg(clsField, 1)

/****************************************************************************
 msgFieldSetStyle	takes P_FIELD_STYLE, returns STATUS
	Sets the style of the field.  
 
 If the field is active, will return stsFailed.  Setting or clearing
 the delayed flag will cause changes in wm.flags necessary to implement 
 delayed fields.  Setting the editType to fstOverWrite will 
 set label.style.displayType to lsBoxTicks.   Will cancel any current
 delayed translation taking place and remove the scribbles in the field.

 Return Value
	stsFailed:		The field is currently being edited. This is either
					through the pen, or a pop up IP.
*/
#define msgFieldSetStyle			MakeMsg(clsField, 2)

/****************************************************************************
 msgFieldGetXlate	takes P_UNKNOWN, returns STATUS
	Passes back the translator information for the field.
 
 If xlateType is 0, the parameter is assumed to be a P_OBJECT and 
 the translator object id is returned.  Otherwise the parameter is 
 assumed to be a P_UNKNOWN and the COMPILED template is returned.

 See Also
	xtemplate.h
	xlate.h
*/
#define msgFieldGetXlate			MakeMsg(clsField, 3)

/****************************************************************************
 msgFieldSetXlate	takes P_UNKNOWN, returns STATUS
	Specifies the translator information for the field.
 
 If xlateType is 0, the argument is assumed to be P_OBJECT being
 a translator.  The old translator is not destroyed.  If xlateType is 1, 
 the argument is assumed to be an uncompiled template (P_XTM_ARGS).
 The field code will compile the template and use it to create a translator.  
 Any old compiled template will not be freed, and must be done so by a call 
 to XTemplateFree() by the client.  Calling on a delayed field will cancel 
 the delayed field, destroying any scribbles captured by the field.

 Return Value
	stsFailed:		The field is currently being edited with the pen, or
					through an IP.

 See Also
	msgFieldCreateTranslator
	msgFieldNew
	xtemplate.h
	xlate.h
*/
#define msgFieldSetXlate			MakeMsg(clsField, 4)

/****************************************************************************
 msgFieldGetMaxLen	takes P_U16, returns STATUS
	Passes back the maximum length allowed for input in the field.
*/
#define msgFieldGetMaxLen 	  		MakeMsg(clsField, 5)

/****************************************************************************
 msgFieldSetMaxLen	takes P_U16, returns STATUS
	Sets the Maximum length for input in the field.  

 Sets the limit for the number of characters that are allowed in a field.
 If maxLen is 0, the maxLen is assumed to be a maxU16.  However,
 it is not recommended that fields of that size be created.
 If the value is less than the old value, the value displayed in the
 field will be truncated to the new value during the next edit.
*/
#define msgFieldSetMaxLen			MakeMsg(clsField, 6)

/****************************************************************************
 msgFieldSetCursorPosition  takes P_U16, returns STATUS
	Sets the cursor position of the keyboard insertion point in the field.

 The cursor position will not be displayed unless the field has the 
 input target. As a performance optimization, this message
 is not self-sent to set the cursor position.
*/
#define msgFieldSetCursorPosition	MakeMsg(clsField, 7)

/****************************************************************************
 msgFieldGetCursorPosition	takes P_U16, returns STATUS
	Passes the current keyboard insertion cursor position in the field.

 If no cursor position has been set, 0 is returned.  As a performance
 optimization, this message is not self-sent to inquire
 cursor position.
*/
#define msgFieldGetCursorPosition	MakeMsg(clsField, 8)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Insertion Pad Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgFieldActivatePopUp	takes P_FIELD_ACTIVATE_POPUP, returns STATUS
	Called to cause an Insertion pad to be brought up for the field.

 If msgFieldActivate has not been called (due to pen input into the field)
 it will be called. Will bring the up the IP at the passed in pRect
 location. If NULL, the IP will be centered over the field.  The type of IP
 will be passed to msgFieldCreatePopUP.  Will return stsFailed if the
 pop-up is not valid given the type and state of the field.  For example,
 an fipInsert on a filled to maxLen field will return stsFailed.

 Return Value
	stsFailed:		A popup up could not be created given the state of the 
					field.
*/
#define msgFieldActivatePopUp			MakeMsg(clsField, 18)

typedef struct {
	U16			type;
	P_RECT32	pRect;
	U32         reserved;
} FIELD_ACTIVATE_POPUP, * P_FIELD_ACTIVATE_POPUP;

/****************************************************************************
 msgFieldAcceptPopUp	takes void, returns STATUS
	Causes the Insertion pad to be accepted.  
	
 Called when the user collapses the insertion pad by hitting the 
 OK button or accepts the IP.  Can be called programatically as well.
*/
#define msgFieldAcceptPopUp				MakeMsg(clsField, 19)

/****************************************************************************
 msgFieldCancelPopUp	takes void, returns STATUS
	Cancels the edit in the pop-up insertion pad.  
	
 Causes the old value to be preserved unchanged.  Called when the
 user hits the cancel button or cancels the IP.  Can be called
 programatically as well.
*/
#define msgFieldCancelPopUp				MakeMsg(clsField, 20)


/****************************************************************************
 msgFieldCreatePopUp	takes P_FIELD_CREATE_POPUP, returns STATUS
	Creates and passes back the insertion pad when the pop up is invoked.  

 Will create the insertion pad for use in the field.  If type is
 fipReplaceAll, will look at style.popUpType to determine the type
 of IP to create.  If type is fipInsert, will look at the system
 preferences for writing style and create the appropriate type of 
 Insertion pad.  Will return stsFailed if the type is fipInsert
 and the field data length is equal to maxLen.

 Return Value
	stsFailed:		The pop-up could not be created for the field.
*/
#define msgFieldCreatePopUp					MakeMsg(clsField, 27)

typedef struct {
	U16		type;
	OBJECT  ip;
	U32 reserved;
} FIELD_CREATE_POPUP, * P_FIELD_CREATE_POPUP;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Delayed Field Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgFieldTranslateDelayed	takes NULL, returns STATUS
	Translates a field with delayed captured strokes.

 Causes translation to occur for a field that has style.delayed and
 has captured strokes pending translation.  Returns stsMessageIgnored
 if style.delayed is not set, or if there is no pending translation.

 Return Value
	stsMessageIgnore:		The field did not have a delayed scribble to 
							translate.
*/
#define msgFieldTranslateDelayed		MakeMsg(clsField, 25)

/****************************************************************************
 msgFieldGetDelayScribble	takes P_OBJECT, returns STATUS
	Returns the delayed scribble for delayed fields.

 Return Value
	stsMessageIgnore:		The field did not have a delayed scribble to 
							translate.  Either not a delayed field or
							no scribbles in the field.
*/
#define msgFieldGetDelayScribble			MakeMsg(clsField, 26)

/****************************************************************************
 msgFieldSetDelayScribble	takes P_OBJECT, returns STATUS
	Puts the field in delayed mode with the given scribble.  

 Return Value
	stsFailed:		The field is currently being edited. This is either
					through the pen, an IP, or the field contains
					delayed strokes in delayed mode. Undefined behavior
					if called on a field with delayed scribbles.
*/
#define msgFieldSetDelayScribble	MakeMsg(clsField, 30)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Miscellaneous Messages										   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgFieldClear	takes NULL, returns STATUS
	Clears the value of the field.

 Clears the delay scribble if one exists, otherwise clears the value
 of the field.
*/
#define msgFieldClear				MakeMsg(clsField, 29)

/****************************************************************************
 msgFieldReadOnly	takes self, returns STATUS
	Self called when an attempt is made to modify a read only field.

 Will send msgFieldReadOnly to control.client if clientNotifyReadOnly is set.
 it exists.
*/
#define msgFieldReadOnly				MakeMsg(clsField, 21)

/****************************************************************************
 msgFieldModified	takes self, returns STATUS 
	Self called when a a field is modified.  

 If the control.dirty bit is clear and the clientNotifyModified bit is set, 
 will send msgFieldModified to control.client.
 Will set the control.dirty bit.  It is the clients responsibility to clear
 this bit.  Will also set the validatePending bit.  This bit
 is cleared after successful validation.
*/
#define msgFieldModified				MakeMsg(clsField, 22)

/****************************************************************************
 msgFieldKeyboardActivate	takes void, returns STATUS
	Activates field for keyboard use.

 Called by client whenever the field is activated for use with the
 keyboard.  Primarily useful for item managers that are dealing with
 keyboard navigation between fields.
*/
#define msgFieldKeyboardActivate		MakeMsg(clsField, 23)

/****************************************************************************
 msgFieldCreateTranslator	takes P_OBJECT, returns STATUS
	Self called to create a translator.  Passes back the translator.

 Used to create the translator based on the compiled template.  
 Called when xlate.xlateType = 1 or when xlate.xlateType = 0 and
 xlate.translator = NULL to create the translator.  Will create
 the translator and respect the style.noSpace, style.veto, and
 style.capOutput settings (for all caps writers).  This translator
 will be destroyed when msgFieldDeactivate is called.
*/
#define msgFieldCreateTranslator		MakeMsg(clsField, 15)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Validation Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgFieldValidate	takes void, returns STATUS
	Peforms the validation protocall for a field.

 Forces validation of a field.  Called when the field loses the input target
 and validatePending is TRUE.  Also called when translation is completed in
 a previouisly empty field.  Returns non-error status for failed
 validation, or stsOK for a valid field.

    -:	calls msgFieldPreValidate on client if field.style.clientPreValidate 
    -:	calls msgFieldValidateEdit on client or on self, depending 
		on style.clientValidate
	-:	calls msgFieldNotifyInvalid if msgFieldValidateEdit returns > stsOK
	-:	calls msgFieldPostValidate on client if 
		field.style.clientPostValidate and msgFieldValidateEdit returns stsOK
    -:	calls msgFieldFormat to format the field if msgFieldValidateEdit
		returns stsOK.
	-:	sets the validatePending bit to 0

 See Also
	msgFieldValidateEdit
	msgFieldPreValidate
	clientPreValidate
	clientValidateEdit
	clientPostValidate
	clientNotifyInvalid
*/
#define msgFieldValidate			MakeMsg(clsField, 9)

/****************************************************************************
 msgFieldPreValidate	takes self, returns STATUS
	Called on client if the field.style.clientPreValidate is set before 
	validation.
	
 Called on the control.client if clientPreValidate is set before
 validation.  Allows clients to pre-process the value of a field
 before validation occurs.  
*/
#define msgFieldPreValidate			MakeMsg(clsField, 10)

/****************************************************************************
 msgFieldValidateEdit	takes P_FIELD_NOTIFY, returns STATUS
	Self call to perform validation on the field.

 Called on self if clientValidate is false, or on the client
 if clientValidate is set.
 Returns stsOK when successful.  Puts a failure message in the 
 failureMessage field of P_FIELD_NOTIFY if not successful, and
 returns a non-error return code.  Default returns stsOK.
*/
#define msgFieldValidateEdit   		MakeMsg(clsField, 11)

/****************************************************************************
 msgFieldNotifyInvalid	takes P_FIELD_NOTIFY, returns STATUS
	Called to notify a field was invalid.

 Called on client if fld.field.style.notifyInvalid bit is set
 and the msgFieldValidateEdit returns a > stsOK
 return code.  Allows clients to post a failure message for 
 validation.
*/
#define msgFieldNotifyInvalid		MakeMsg(clsField, 12)

/****************************************************************************
 msgFieldPostValidate	takes self, returns STATUS
	Self call to perform post-validation processing.

 Called on client if field.style.clientPostValidate is set. 
 Only called if msgFieldValidateEdit returns stsOK.  Allows client
 to perform post validation processing.
*/
#define msgFieldPostValidate   		MakeMsg(clsField, 13)

/****************************************************************************
 msgFieldFormat		takes void, returns STATUS
	Self call to perform formatting.

 Self called after validation to perform any formatted the field
 requires to display itself correctly. Intended to be overridden
 by clients to support field formatting.  Only called when 
 msgFieldValidateEdit returns stsOK.
*/
#define msgFieldFormat				MakeMsg(clsField, 14)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Private Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgFieldActivate	takes void, returns STATUS
	Forces activation of the field.  
 
 Pen input into the field, or a call to msgFieldActivatePopUp, 
 will cause a self send of this message.  Causes temporary memory 
 needed for editing to be allocated.  This method should not be 
 overridden, and should not be called
 
 Return Value
	stsFailed:		The field is readOnly and can't be activated.
*/
#define msgFieldActivate				MakeMsg(clsField, 16)

/****************************************************************************
 msgFieldDeactivate		takes void, returns STATUS
	Forces deactivation of the field.  

 A completed edit operation causes a self send of this message.  Causes
 temporary memory to be deallocated.  This message should not be 
 overridden, and should not be called.
*/
#define msgFieldDeactivate				MakeMsg(clsField, 17)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFree	takes OBJ_KEY, returns STATUS
	Defined in object.h.

 Deactivates the field if necessary.  Will free the translator if xlateType
 is 0 and a translator was handed to the field.  Will free the compiled
 template if xlateType is 1.  Inherits ancestor behavior.

 See Also
	object.h
*/

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Defined in object.h.

 Inherits ancestor behavior first and then
 stores in the resource file all information about the current state of 
 the field, including the translator or template information or the
 delayed strokes the field contains. Fields will not save any information
 about a current editing operation (through a pop-up, keyboard, or pen)
 in effect.

 See Also
	object.h
*/

/****************************************************************************
 msgRestore		takes P_OBJ_RESTORE, returns STATUS
	Defined in object.h.

 Inherits ancestor information and restores all information about the field
 including translator information or the delayed strokes the field
 contains.

 See Also
	msgSave
	object.h
*/

/****************************************************************************
 msgIPDataAvailable		takes OBJECT, returns STATUS
	Defined in insert.h.

 Sent to the field from an insertion pad when there is data to
 retrieve from the pop-up pad.  Depending on the operation that
 brought up the pad (an insert or edit gesture), will either
 insert the text from the pad at the current insertion point, or
 replace the value of the field with the IP value.  Will destroy
 the pop-up pad created.

 See Also
	insert.h
*/

/****************************************************************************
 msgIPCancelled		takes OBJECT, returns STATUS
	Defined in insert.h.

 Sent to the field when the insertion pad has been canceled.  Will
 destroy the pad and any changes to the text in the pad are ignored.

 See Also
	insert.h
*/

/****************************************************************************
 msgControlSetDirty		takes BOOLEAN, returns STATUS
	Defined in control.h.

 Inherits behavior from superclass.  Will clear all character box memory
 stored for an overwrite field, allowing characters to be returned
 immediatly from the translator.

 See Also
	control.h
*/

#endif


