/****************************************************************************
 dvkeybd.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the keyboard device interface.

 $Revision:   1.8  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:10:54  $
****************************************************************************/
#ifndef DVKEYBD_INCLUDED
#define DVKEYBD_INCLUDED

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the keyboard device. 
 The definitions contained here are data structures that are necessary
 for the MIL keyboard specific functions along with standard
 MIL device data structures with additional fields specific to the
 keyboard device.
*/

/**** Keyboard Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the keyboard device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milKeybdGetDevParameters	milLastCommonFunction+1
#define milKeybdReadKeyCont			milLastCommonFunction+2
#define milKeybdSetLEDStatus		milLastCommonFunction+3
#define milKeybdSetAutoRepeat		milLastCommonFunction+4

#define fnCountKeybd				fnCountBase+4

/**** Keyboard Device Function Transfer Table ****/
/*
 This structure is the declaration of the Keyboard Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Keyboard device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_KEYBD {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountKeybd];
} MIL_FTT_KEYBD, *P_MIL_FTT_KEYBD;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Keyboard Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the keyboard device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/




/****************************************************************************
 milKeybdGetDevParameters		returns STATUS
 	Provides device specific information about the keyboard device.

 This function is a single stage function that provides device
 specific information about the keyboard device.
*/

/**** Function Specific Definitions ****/
/*
 The following definitions are used to specify information in the
 fields returned by milKeybdGetDevParameters.
*/

/*
 keybdEncodingType

 The following definitions are used to specify how the keyboard input
 is encoded.

	kbEncodingPcXt:
		The keyboard input is encoded using the PC-XT format.

	kbEncodingPcAt:
		The keyboard input is encoded using the PC-AT format.

	kbEncodingASCII:
		The keyboard input is encoded in ASCII format.

	kbEncodingUnicode:
		The keyboard input is encoded in Unicode format.

*/

#define kbEncodingPcXt		0
#define kbEncodingPcAt		1
#define kbEncodingASCII		2
#define kbEncodingUnicode	3

/*
 keybdType

 The following definitions are used to specify the keyboard type.

	kbTypeUnknown:
		The keyboard type is unknown.

	kbTypePc83Key:
		The keyboard is an 83 key keyboard.

	kbTypePc101Key:
		The keyboard is a 101 key keyboard.

*/

#define kbTypeUnknown	0
#define kbTypePc83Key	1
#define kbTypePc101Key	2

/*
 supportedLEDs:

 The following definitions are used to specify the keyboard LEDs 
 that are supported.

	kbScrollLockLED:
		The keyboard supports a Scroll Lock LED.

	kbNumLockLED:
		The keyboard supports a Num Lock LED.

	kbCapsLockLED:
		The keyboard supports a Caps Lock LED.

	kbShiftLockLED:
		The keyboard supports a Shift Lock LED.

*/
	
#define kbScrollLockLED	1
#define kbNumLockLED	2
#define kbCapsLockLED	4
#define kbShiftLockLED	8

/*
 keybdFlags

 The following definitions are used to define the modes of operation
 supported by the keyboard.

	milMakeBreakReported:
		The keyboard sends both a make and a break key for each key
		stroke.

	milAutoRepeatSetSupported:
		The keyboard supports the capability to set the auto repeat
		rate.

	milSetLEDSupported:
		The keyboard supports the capability to set keyboard LEDs.

*/

#define milMakeBreakReported       flag0
#define milAutoRepeatSetSupported  flag1
#define milSetLEDSupported         flag2


/**** Device Specific Request Block Fields ****/
/*

 The milKeybdGetDevParameters funtion request block contains the
 following fields.

	keybdFlags:
		This field defines the capabilities supported by the keyboard. 
		See the definitions above.

	keybdEncodingType:
		This field defines how the keyboard input data is encoded.

	keybdType:
		This field defines the physical keyboard type or layout.

	supportedLEDs:
		This field defines the keyboard LEDs that are supported by
		the keyboard.

	autoRepeatRate:
		This field specifies the current automatic key repeat rate.
		This defines the rate key strokes are generated when a key
 		is held down.

	autoRepeatRateLowLimit:
		This field defines the slowest repeat rate supported.

	autoRepeatRateHighLimit:
		This field defines the fastest repeat rate supported.

	autoRepeatDelay:
		This field is the current time interval that must pass after
		a key is pressed before auto repeat begins.

	autoRepeatDelayLowLimit:
		This field specifies the shortest delay supported.

	autoRepeatDelayHighLimit:
		This field specifies the longest delay supported.

*/
typedef struct MIL_KEYBD_DEVICE_PARAMETERS {
	U16 keybdFlags;
	U16 keybdEncodingType;
	U16 keybdType;
	U16 supportedLEDs;
	U16 autoRepeatRate;
	U16 autoRepeatRateLowLimit;
	U16 autoRepeatRateHighLimit;
	U16 autoRepeatDelay;
	U16 autoRepeatDelayLowLimit;
	U16 autoRepeatDelayHighLimit;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_KEYBD_DEVICE_PARAMETERS, *P_MIL_KEYBD_DEVICE_PARAMETERS;

/*
 rbPublicFieldsKeybdGetDevParameters

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsKeybdGetDevParameters \
	MIL_KEYBD_DEVICE_PARAMETERS keybdDevParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milKeybdGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_KEYBD_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsKeybdGetDevParameters
} MIL_RB_PUB_KEYBD_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_KEYBD_GET_DEV_PARAMETERS;


/****************************************************************************
 milKeybdReadKeyCont		returns STATUS
 	Keyboard interrupt handler.

 This function is a stage on interrupt function that processes keyboard
 interrupts.
*/

/**** Function Specific Definitions ****/
/*
 The following definitions are used to specify information in the
 fields returned by milKeybdReadKeyCont.
*/

/*
 keybdEvent

 The following keyboard events are defined.

	milKeyBreak:
		The event is a break event. This means a key has been released.

	milKeyMake:
		The event is a make event. This means a key has been pressed.

	milKeyRepeat:
		The event is an auto repeat event. This means a key is being
		held down.
	
	milKeyCharacter:

*/		

Enum16(KEYBD_EVENT) {
	milKeyBreak,
	milKeyMake,
	milKeyRepeat,
	milKeyCharacter
};

/**** Device Specific Request Block Fields ****/
/*
 The following fields are defined in the milKeybdReadKeyCont request
 block.

	keyCode:
		This field contains the keycode input.

	shiftFlags:
		This field contains the current state of the shift flags.

	scanCode:
		This field contains the current key position.

	keybdEvent:
		This field indicates the type of event that has occurred.

	pAsyncEventFunc:
		This field is a call back function. This function is called
		for each event.

*/

#define rbPublicFieldsKeybdReadKeyCont \
	U16	keyCode; \
	U16	shiftFlags; \
	U16 scanCode; \
	KEYBD_EVENT keybdEvent; \
	P_MIL_ASYNC_EVENT_FUNC pAsyncEventFunc; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milKeybdReadKeyCont Request Block

*/

typedef struct MIL_RB_PUB_KEYBD_READ_KEY_CONT {
	milRequestBlockCommonFields
	rbPublicFieldsKeybdReadKeyCont
} MIL_RB_PUB_KEYBD_READ_KEY_CONT, *P_MIL_RB_PUB_KEYBD_READ_KEY_CONT;


/****************************************************************************
 milKeybdSetLEDStatus		returns STATUS
 	Sets the keyboard LEDs.

 This function is a single stage function that sets the indicated LED
 state.

*/

/**** Device Specific Request Block Fields ****/
/*
 The following fields are defined in the milKeybdSetLEDStatus request
 block.

	newLEDStatus:
		This field contains bits that identify the current state of the
		keyboard LEDs.

*/

#define rbPublicFieldsKeybdSetLEDStatus \
	U16 newLEDStatus; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milKeybdSetLEDStatus Request Block

*/

typedef struct MIL_RB_PUB_KEYBD_SET_LED_STATUS {
	milRequestBlockCommonFields
	rbPublicFieldsKeybdSetLEDStatus
} MIL_RB_PUB_KEYBD_SET_LED_STATUS, *P_MIL_RB_PUB_KEYBD_SET_LED_STATUS;

/****************************************************************************
 milKeybdSetAutoRepeat		returns STATUS
 	Sets the keyboard auto repeat rate.

 This function is a single stage function that sets the indicated keyboard
 auto-repeat rate.

*/

/**** Device Specific Request Block Fields ****/
/*
 The following fields are defined in the milKeybdSetAutoRepeat request
 block.

	autoRepeatRate:
		This field contains the requested auto repeat rate.

	autoRepeatDelay:
		This field contains the requested pre-auto-repeat delay.

*/


#define rbPublicFieldsKeybdSetAutoRepeat \
	U16 autoRepeatRate; \
	U16 autoRepeatDelay; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milKeybdSetAutoRepeat Request Block

*/

typedef struct MIL_RB_PUB_KEYBD_SET_AUTO_REPEAT {
	milRequestBlockCommonFields
	rbPublicFieldsKeybdSetAutoRepeat
} MIL_RB_PUB_KEYBD_SET_AUTO_REPEAT, *P_MIL_RB_PUB_KEYBD_SET_AUTO_REPEAT;


#endif
