/****************************************************************************
 dvinit.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The definition of the public MIL initialization device interface.

 $Revision:   1.11  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:11:46  $
****************************************************************************/
#ifndef DVINIT_INCLUDED
#define DVINIT_INCLUDED

/*
 The sole purpose of the initialization device is to provide PenPoint with 
 the information needed to initialize the rest of the MIL. This requires
 some special characteristics for this device. First, an initial common
 data structure is provided by the MIL at system boot that only defines
 the initialization device. Second, the initialization device does not
 need to be initialized. Third, all of the functions are single-staged. 
 Finally, the initialization device type is special in that its logical
 ID is always known to be zero.

 The initialization process is built around the fact that the MIL has
 no control of and no ability to allocate memory. Instead, PenPoint
 queries the MIL's needs through a series of requests, allocates the 
 needed blocks, and gives the MIL the opportunity to fill-in these blocks.

 At boot-time, the MIL is entered and immediately calls PenPoint's
 initial entry point. The MIL provides PenPoint with an initial
 common data structure that describes only one device-the initialization
 device. PenPoint then calls the initialization device using the function
 milInitGetDevParameters. 

 The MIL then provides a map of the memory and returns to PenPoint. Next,
 PenPoint allocates a device information array based on numInfoEntries
 and the size of MIL_DEVICE_INFO and calls the initialization device waih
 the function milInitGetDeviceInfo. The variables pRamExtensions, 
 ramExtensionsLength and numInfoEntries must again be passed to the MIL
 as it has no RAM in which to store them during the 
 milInitGetDevParameters function.

 The MIL fills in the provided device information array and returns to
 PenPoint. Now PenPoint starts to build the actual common data area and
 its associated FTTs and device blocks. For each device listed in the 
 information array, PenPoint allocats a FTT. PenPoint allocates one or
 more device blocks, using the number of devices specified by
 numLogicalDevices, and copies the common fields to them. Pointers to
 the new FTT and device block are now stored in the next free entry
 in the devicePtrs array in the common data structure.

 Finally, the function milInitSetCommonData is requested that allows the
 MIL to fill in missing fields in the newly created common data structure
 such as machine type.

 The field initDevBlkRoutine of MIL_DEVICE_INFO structure fills-in the
 devices public data structures such as the FTT and device data block
 and returns. It is called at an early point in the initialization
 process before PenPoint is fully functional.

*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the initialization device. 
 The definitions contained here are data structures that are necessary
 for the MIL initialization specific functions along with standard
 MIL device data structures with additional fields specific to the
 initialization device.
*/

/*
 The initialization device is special in that its logical Id is well known.
*/

#define milIdInit 0

/**** Initialization Device Specific Function Numbers ****/
/*
 The following definitions are the device specific functions supported
 by the initialization device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milInitGetDevParameters	milLastCommonFunction+1
#define milInitGetDeviceInfo	milLastCommonFunction+2
#define milInitSetCommonData	milLastCommonFunction+3

#define fnCountInit				fnCountBase+3

/**** Initialization Device Function Transfer Table ****/
/*
 This structure is the declaration of the Initialization Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Initialization device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_INIT {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountInit];
} MIL_FTT_INIT, *P_MIL_FTT_INIT;

#define milMinExtraLogicalIds 20

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 		   	Initialization Device Specific Functions	   				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the initialization device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milInitGetDevParameters		returns STATUS
 	Provides device specific information about the init device.

 This function is a single stage function that provides device
 specific information about the init device.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milInitGetDevParameters function request block contains the following
 fields.
	
	initFlags:
		This field contains flags that define capabilities, status, and
 		modes of operation of the init device. Currently there are no
		bits defined for this field.

	pRamExtensions:
		This field is an input field that contains a pointer to the
		location of the RAM extensions.

	ramExtensionsLength:
		This field specifies the length of the RAM extensions.

	pRomExpensions:
		This field is an input field that contains a pointer to the
		location of the ROM extensions.

	romExtensionsLength:
		This field specifies the length of the ROM extensions.

	numInfoEntries:
		This field identifies the number of Info entries. Note that
		this field is not the same as the number of devices. It is
		possible to allocate multiple device blocks that share the 
		same code.


*/
		

typedef struct MIL_INIT_DEVICE_PARAMETERS {
	U16 initFlags;
	P_MIL_EXTENSION pRamExtensions;
	U32 ramExtensionsLength;
	P_MIL_EXTENSION pRomExtensions;
	U32 romExtensionsLength;
	U16 numInfoEntries;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_INIT_DEVICE_PARAMETERS, *P_MIL_INIT_DEVICE_PARAMETERS;

/*
 rbPublicFieldsInitGetDevParameters

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsInitGetDevParameters \
	MIL_INIT_DEVICE_PARAMETERS initDevParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milInitGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_INIT_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsInitGetDevParameters
} MIL_RB_PUB_INIT_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_INIT_GET_DEV_PARAMETERS;




/****************************************************************************
 milInitGetDeviceInfo		returns STATUS
 	Provides device specific information about the MIL devices.

 This function is a single stage function that provides device
 specific information about the MIL devices.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milInitGetDeviceInfo function request block contains the following
 fields.
	
	ramExtensions:
		This field is an input field that contains a pointer to the
		location of the RAM extensions.

	ramExtensionsLength:
		This field specifies the length of the RAM extensions.

	numInfoEntries:
		This field identifies the number of Info entries. Note that
		this field is not the same as the number of devices. It is
		possible to allocate multiple device blocks that share the 
		same code. This fields is the number of entries in pInfo[].

	numMemoryRegions:
		This field is an output field that specified the total number
		of memory regions.

	pInfo:
		This field is a pointer to an array of device information.

	privateDataSize:
		This field is the size of the space to be added to the end
		of the common data area

*/

#define rbPublicFieldsInitGetDeviceInfo \
	U32 ramExtensions; \
	U32 ramExtensionsLength; \
	U16 numInfoEntries; \
	U16 numMemoryRegions; \
	P_MIL_DEVICE_INFO pInfo; \
	U32 privateDataSize; \
	U16 extraLogicalIds; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milInitGetDeviceInfo Request Block

*/

typedef struct MIL_RB_PUB_INIT_GET_DEVICE_INFO {
	milRequestBlockCommonFields
	rbPublicFieldsInitGetDeviceInfo
} MIL_RB_PUB_INIT_GET_DEVICE_INFO, *P_MIL_RB_PUB_INIT_GET_DEVICE_INFO;


/****************************************************************************
 milInitSetCommonData		returns STATUS
 	Completes the initialization of the common data area.

 This function is a single stage function that completes the initialization
 of the common data area.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milInitSetCommonData function request block does not contain any device
 or function specific fields.

*/

#define rbPublicFieldsInitSetCommonData \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milInitSetCommonData Request Block

*/

typedef struct MIL_RB_PUB_INIT_SET_COMMON_DATA {
	milRequestBlockCommonFields
	rbPublicFieldsInitSetCommonData
} MIL_RB_PUB_INIT_SET_COMMON_DATA, *P_MIL_RB_PUB_INIT_SET_COMMON_DATA;

extern MIL_COMMON_DATA initialCommonData;

#endif

