/****************************************************************************
 dvhspkt.h

 Copyright 1992 GO Corporation. All Rights Reserved.
 Copyright 1991 Travelling Software, Inc. All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  
	 
 THIS SAMPLE CODE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 AND GO CORPORATION EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING 
 BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 FOR A PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 TRAVELLING SOFTWARE HEREBY DISCLAIMS ALL WARRANTIES OF ANY KIND, WHETHER
 EXPRESS, IMPLIED, ARISING BY USAGE OF TRADE OR OTHERWISE, INCLUDING ANY 
 AND ALL IMPLIED WARRANTIES OF MERCHANTABILITY, TITLE OR FITNESS FOR ANY
 PARTICULAR PURPOSE OR USE. IN NO EVENT SHALL TRAVELLING SOFTWARE BE LIABLE
 FOR DAMAGES OF ANY KIND RESULTING FROM OR RELATING IN ANY WAY TO THIS
 CODE, INCLUDING WITHOUT LIMITATION ANY CONSEQUENTIAL, INCIDENTAL, SPECIAL,
 DIRECT, GENERAL PUNITIVE OR EXEMPLARY DAMAGES, WHETHER BASED ON CONTRACT
 OR TORT (INCLUDING BUT ONT LIMITED TO STRICT LIABILITY AND NEGLIGENCE), 
 AND IN ANY EVENT NO DAMAGES IMPOSED ON TRAVELLING SOFTWARE SHALL EXCEED
 THE VALUE OF THE CODE PROVIDED HEREUNDER.

 The public definition of the high speed communication device interface.

 $Revision:   1.16  $
   $Author:   unknown  $
     $Date:   25 Mar 1992 16:39:44  $
		 
****************************************************************************/
#ifndef DVHSPKT_INCLUDED
#define DVHSPKT_INCLUDED

#ifndef MIL_INCLUDED
#include "mil.h"
#endif
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the hspkt device. 
 The definitions contained here are data structures that are necessary
 for the MIL hspkt specific functions along with standard
 MIL device data structures with additional fields specific to the
 hspkt device.
*/

#define	milDefaultAsyncSIOBaud		115200	// default baud rate (async mode)
#define	milParByteBaud				500000	// approximate speed (4-bit par mode)
#define	milParNibbleBaud			400000	// approximate speed (8-bit par mode)


/**** Hspkt Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the hspkt device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milHSPacketGetDevParameters		milLastCommonFunction+1
#define milHSPacketSetDevParameters		milLastCommonFunction+2
#define milHSPacketSend					milLastCommonFunction+3
#define milHSPacketTestConnect			milLastCommonFunction+4
#define milHSPacketRxCont				milLastCommonFunction+5

#define fnCountHSPacket					fnCountBase+5

/**** Hspkt Device Function Transfer Table ****/
/*
 This structure is the declaration of the Hspkt Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL hspkt device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_HS_PACKET
{
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountHSPacket];
} MIL_FTT_HS_PACKET, *P_MIL_FTT_HS_PACKET;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Hspkt Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the hspkt device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milHSPacketGetDevParameters		returns STATUS
 	Provides device specific information about the HSPkt device.

 These single stage functions sets/returns specific information of the
 High Speed Communication device.

 PROTOCOL DESCRIPTION
	 
 This device implements a built-in RTS/CTS type protocol by sending a lead in
 character and expecting a data acknowledge character in return before 
 actually sending a packet of data.
 
 In addition, a connection detection protocol is implemented when running
 in parallel mode.  In asynchronous mode, DVHSPKT returns connected 
 whenever DSR is high, regardless of what the serial port is connected to.

 ASYNCHRONOUS SERIAL BAUD RATE NEGOTIATION
	 
 Dynamic baud rate negotiations is possible in asynchronous serial mode, if
 a device of type milHSPktAsyncSIO (default type for an async serial port)
 is changed to milHSPktAsyncSIONoBaud through the use of the
 milHSPacketSetDevParameters function.  

 In case of send errors, baud rates are renegotiated with remote station 
 automatically.
	 
 PARALLEL CABLE CONNECTION DETECTION
	 
 The parallel connection protocol involves sending a parallel connect
 character and expecting to receive a connect acknowledge character back.
	
 PROTOCOL VARIATIONS
	 
 By setting leadInChar to 0, no lead in character is sent (and of course
 no dataAckChar is expected).
		 
 By setting dataAckChar to 0, no acknowledge character is sent upon 
 receiving the lead in character, i.e. DVHSPKT goes on expecting data to
 arrive right after the lead in character.

 By setting parConnectChar to 0, DVHSPKT reports always connected, leaving
 it up to users at a higher level to determine actual connection.
	 
 By setting parConnectAckChar to 0, no connect acknowledge character is 
 sent upon receiving the connect character.

 NOTES
	 
 As data is transmitted/received in parallel mode, DVHSPKT remains
 synchronized with the other side during the entire data transfer.  As
 a result, both transmissions and receptions are subject to failure, making
 the use of a lead in/ack protocol not always a necessity when communicating
 through the parallel port.  The use of at least a lead in character may
 however improve performance as fake interrupts would be noticed early (by
 DVHSPKT itself) saving the upper layer code the trouble of validating
 the beginning of a packet.

 Parallel transfer speeds depend on the speed of the machines transmitting/
 receiving data. 
	 
 In asynchronous serial mode, DVHSPKT synchronizes itself with the other
 side only upon receiving the first byte (lead in) by sending a data ack
 character to inform the other side.  The absence of a lead in/ack protocol
 might then cause overruns on slower machines.  

 See MIL specifications for an explanation of the protocols used by this
 device.
	 

*/

/**** Function Specific Definitions ****/
/*
 Packet Device Parameters

 This data structure contains the device specific information about
 the HSPacket device. This data structure contains the following 
 fields.

	hsPktDevFlags:
		This field contains flags that define capabilities, status,
		and mode of operation for this device. Currently there are
		no flags defined for thsi field.

*/
typedef struct MIL_HS_PACKET_DEVICE_PARAMETERS
{
	U16 hsPktDevFlags;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_HS_PACKET_DEVICE_PARAMETERS, *P_MIL_HS_PACKET_DEVICE_PARAMETERS;

/*
 Packet Device Types

 The following definition defines all the HSpacket device types.
	
	milHSPktParUnknown:
		This value is used when the type of device is a parallel
		port but the bit width is not known.

	milHSPktNibblePIO:
		This value is used when the device is a parallel port that
		is passing information 4 bits at a time.

	milHSPktBytePIO:
		This value is used when the device is a parallel port that
		is passing information 8 bits at a time.

	milHSPktAsyncSIO:
		This value is used when the device is a serial port.

	milHSPktAsyncSIONoBaud:
		This value is used when the device is a serial port.
	
*/	

Enum16(MIL_HS_PACKET_DEVICE_TYPE)
{
	milHSPktParUnknown		= 0,
	milHSPktNibblePIO		= 1,
	milHSPktBytePIO			= 2,
	milHSPktAsyncSIO		= 3,
	milHSPktAsyncSIONoBaud	= 4
};

/* 
 Packet Status Flags

 The following definitions define the different packet status values.

	hsPktStatusTimeOut:
		A timeout has occured in passing a packet.

	hsPktStatusIdle:
		The HSPacket device is idle.

*/

#define hsPktStatusTimeOut		flag0
#define hsPktStatusIdle			flag1

/*
 Special Protocol Characters

 The following definitions define the special byte values used in the
 HSPacket protocol.

*/

#define	milDefaultLeadInChar		0x01
#define	milDefaultDataAckChar		0x40
#define	milDefaultParConnectChar	0x08
#define	milDefaultParConnectAckChar 0x40

/*
 Packet Device Unit Parameters

 The following data structure contains the unit specific information.
 The data strucure contains the following fields.

	hsPktUnitFlags:
		This field conatins the unit flags. The values for this field
		are defined above.

	hsPktStatus:
		This field contains the current unit status.

	hsPktBaud:
		This field contains the baud rate.

	parConnectChar:
		This field contains the hand shaking character that is used
		to determine the parallel mode.

	leadInChar:
		This field is the current packet lead in character.
	
	dataAckChar:
		This field is the current data acknowledge character.

	hsPktOrgBaud:
		This field keeps track of the last baud rate connected.

*/
 
typedef struct MIL_HS_PACKET_UNIT_PARAMETERS
{
	MIL_HS_PACKET_DEVICE_TYPE hsPktUnitFlags;  // out: device type.
									// BUT in: only if milHSPktAsyncSIO or
									// or milHSPktAsyncSIONoBaud:
									// set to milHSPktAsyncSIONoBaud to
									// activate baud rate negotiation
									// functionality
	U16 hsPktStatus;				// out: current device status
	U32	hsPktBaud;					// Parallel mode - out: approx. baud rate 
									// Serial mode - in/out: baud rate
	U16	parConnectChar;				// char sent to determine parallel mode
									// connection state
	U16	parConnectAckChar;			// char returned after receiving
									// parConnectChar
	U16	leadInChar;					// char sent to prime transmission
	U16	dataAckChar;				// char returned to sender after 
									// leadInChar is received (parallel mode)
	U32	hsPktOrgBaud;				// disconnected/original baud rate
	U32 reservedU32[2];
} MIL_HS_PACKET_UNIT_PARAMETERS, *P_MIL_HS_PACKET_UNIT_PARAMETERS;

/**** Device Specific Request Block Fields ****/
/*
 The milHSPacketGetDevParameters function request block contains the
 following fields.

	hsPktDevParms:
		This field contains the information concerning the device. This
		data structure is defined above.

	hsPktUnitParms:
		This field contains the information concerning the unit. This
		data structure is defined above.
	
*/

#define rbPublicFieldsHSPacketDeviceParameters			\
	MIL_HS_PACKET_DEVICE_PARAMETERS	hsPktDevParms;		\
	MIL_HS_PACKET_UNIT_PARAMETERS	hsPktUnitParms;		\
	U16 funcReservedU16[2];								\
	U32 funcReservedU32[2];

/*
 
 milHSPacketGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_HS_PACKET_DEVICE_PARAMETERS
{
	milRequestBlockCommonFields
	rbPublicFieldsHSPacketDeviceParameters
} MIL_RB_PUB_HS_PACKET_DEVICE_PARAMETERS, *P_MIL_RB_PUB_HS_PACKET_DEVICE_PARAMETERS;

/****************************************************************************
 milHSPacketSetDevParameters		returns STATUS
 	Sets device specific information about the HSPkt device.

 These single stage functions sets specific information of the
 High Speed Communication device.

 This function uses the same request block as milHSPacketSetDevParameters.
 Please see the definitions above.

*/


/****************************************************************************
 milHSPacketSend		returns STATUS
 	Sends the specified packet.

 This single stage function is used to transmit packets over the
 high speed communcations port.  Data is formatted by the caller.
 
 For performance reasons, the entire data transfer takes place
 with interrupts OFF.  So as to not impair foreground system performance
 unreasonably, users should be sure to keep packet sizes reasonably small.
 (1K packets have shown to not affect significantly the general perception
 users have of the system performance).

 Errors are returned in hsPktStatus. (When successful, hsPktStatusIdle
 is returned.)
	 
 If the unit parameters leadInChar is zero, the contents of firstByte 
 are used for the lead in character.  If both are zero, the first byte
 in the send buffer (at pPacket) is used.

 NOTE
	 
 PACKETS MUST NOT START WITH the connectChar character, used in the cable
 detection protocol.
	 

*/
/**** Device Specific Request Block Fields ****/
/*
 The milHSPacketSend function request block contains the
 following fields.

	pPacket:
		This field is a pointer to the packet to be sent.

	packetSize:
		This field contains the size of the packet.

	hsPktStatus:
		This field is the final status of the operation.

	firstByte:
		This field contains the first byte to be transmitted.
	
*/

#define rbPublicFieldsHSPacketSend					\
	P_UNKNOWN				pPacket;				\
	U32						packetSize;				\
	U16						hsPktStatus;			\
	U16						firstByte;				\
	U16						funcReservedU16[2];		\
	U32						funcReservedU32[2];
/*
 
 milHSPacketSend Request Block

*/

typedef struct MIL_RB_PUB_HS_PACKET_SEND
{
	milRequestBlockCommonFields
	rbPublicFieldsHSPacketSend
} MIL_RB_PUB_HS_PACKET_SEND, *P_MIL_RB_PUB_HS_PACKET_SEND;

/****************************************************************************
 milHSPacketTestConnect		returns STATUS
 	Tests the for connection.

 In async mode, this function does nothing more than test for DSR.  If
 DSR is set, this function returns hsPktMaybe.
	 
 This single stage function is used to test for explicit connection.
 DVHSPKT sends a Connect character and waits for an acknowledge character.
	 
 If the data exchange is successful, this function returns hsPktYes.  
 Otherwise it returns hsPktNo.
	 
 Standard notification still takes place as per the attachment continuous
 function specification.  This call is strictly informational.

*/
/**** Function Specific Definitions ****/
/*
 Packet Connection Values

 The following definition specifies the various connect states.

*/

Enum16(MIL_HS_PACKET_CONNECT)
{
	hsPktYes		  = 0,
	hsPktMaybe		  = 1,
	hsPktNo			  = 2
};

/**** Device Specific Request Block Fields ****/
/*
 The milHSPacketTestConnect function request block contains the
 following fields.

	hsPktConnectState:
		This field returns the current connection status.

*/

#define rbPublicFieldsHSPacketTestConnect			\
	MIL_HS_PACKET_CONNECT	hsPktConnectState;		\
	U16						funcReservedU16[2];		\
	U32						funcReservedU32[2];

/*
 
 milHSPacketConnectState Request Block

*/
typedef struct MIL_RB_PUB_HS_PACKET_TEST_CONNECT
{
	milRequestBlockCommonFields
	rbPublicFieldsHSPacketTestConnect
} MIL_RB_PUB_HS_PACKET_TEST_CONNECT, *P_MIL_RB_PUB_HS_PACKET_TEST_CONNECT;



/****************************************************************************
 milHSPacketRxCont		returns STATUS
 	Receives packet data.

 This function is a continuous function responsible for receiving 
 data.  Data is made available to the user-provided receive handler
 one character at a time as it is received.

 This function expects to read data until the receive handler reports
 that the packet is complete, which the receive handler does by setting
 packetIsComplete to true.
	 
 For performance reasons, the packet is read with interrupts off.

 Receive error conditions are reported in hsPktStatus.
	 
 firstByte is the first byte received in a packet, set ONLY if the unit
 parameters' leadInChar is zero.

 IMPORTANT NOTE:
	 
 This function must be called BEFORE the attachment continuous function
 is called.
	 
*/

/**** Device Specific Request Block Fields ****/
/*
 The milHSPacketRxCont function request block contains the
 following fields.

	pRxHandlerFunc:
		This field contains a pointer to a function that is called 
		when a data byte is received.
	
	firstByte:
		This field specifies the first byte to be received if any.

	packetIsComplete:
		This field indicates a complete packet has been received.

	hsPktStatus:
		This field indicates the status of the received packet.

*/

#define rbPublicFieldsHSPacketRxCont				\
	P_MIL_ASYNC_EVENT_FUNC	pRxHandlerFunc;			\
	U16						receivedByte;			\
	U16						firstByte;				\
 	BOOLEAN					packetIsComplete;		\
	U16						hsPktStatus;			\
	U16						funcReservedU16[2];		\
	U32						funcReservedU32[2];

/*
 
 milHSPacketRxCont Request Block

*/

typedef struct MIL_RB_PUB_HS_PACKET_RX_CONT
{
	milRequestBlockCommonFields
	rbPublicFieldsHSPacketRxCont
} MIL_RB_PUB_HS_PACKET_RX_CONT, *P_MIL_RB_PUB_HS_PACKET_RX_CONT;

#endif
