/****************************************************************************
 debug.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.33  $
  $Author:   kcatlin  $
	$Date:   30 Mar 1992 19:00:08  $

 This file contains the definitions of some of PenPoint's debugging
 support.

 The functions described in this file are contained in PENPOINT.LIB. 
****************************************************************************/

/**** Introduction. ****/
/*
 This file contains the definitions of some of PenPoint's debugging
 support.

 One of the most important characteristics of this package is that many of
 the macros compile into nothing unless the pre-processor variable DEBUG is
 defined during compilation.
*/

/**** Debugging Flags. ****/
/*
 As part of its debugging support, PenPoint includes a collection of
 debugging flags which allow developers to control the runtime behavior of
 their programs.

 For convenience, the debugging flags are broken into "sets"  of 32 one bit
 flags.  In PenPoint 1.0, there are 255 sets;  future versions of PenPoint
 may have more sets.  Some sets are reserved for use by PenPoint itself; 
 all other sets are available for use by other developers.  The allocation
 of sets is documented elsewhere in this file.
*/

/**** Setting and Examining Debug Flags. ****/
/*
 The debugging flags can be set via the DebugSet environment variable in
 PenPoint's environ.ini file.  The debugging flags can also be set with the
 "fs" command in the MiniDebugger and DB.  (The debugging flags can be
 examined with the "fl" command.)  Both the environ.ini file and the
 PenPoint debuggers allow the flag sets to be identified with either a or
 an 8 bit hexadecimal number.  See the PenPoint developer's documentation
 for more information.
*/

/**** Example. ****/
/*
 The debugging output in the following fragment appears only if the code
 was compiled with DEBUG defined and the debug flag is on.

 As illustrated in this example, most debugging code should surrounded by
 some sort of conditional compilation that causes the debugging code to
 "disappear" when compiled appropriately.
//{
	if (someCondition) {
 		DbgFlag(0x80, 0x1, Debugf("someCondition is TRUE");)
		...
	} else {
 		DbgFlag(0x80, 0x1, Debugf("someCondition is FALSE");)
		...
	}
//}

 Here's an example of setting debugging flags in PenPoint's environ.ini
 file:
//{
	...
	DebugSet=/DD8000 /DB800
	...
//}
*/

#ifndef DEBUG_INCLUDED
#define DEBUG_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Exported Macros 						   	   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 DbgFlag	returns void.
	Executes an expression under control of a debug flag IF the source is
	compiled with DEBUG defined.

 The DbgFlag() macro is used to execute an expression if (1) the source
 module was compiled with DEBUG defined and (2) if the appropriate
 debugging flag is turned on at runtime.
*/
#ifdef DEBUG
#define DbgFlag(f,v,e) if (DbgFlagGet(f, v)) e
#else
#define DbgFlag(f,v,e)
#endif


/****************************************************************************
 Dbg	returns void.
	Used to control the compile-time inclusion of debugging code.

 The Dbg() macro is used to comment out code when the DEBUG flags is
 undefined.  For example, the following code is present if the source
 file is compiled with DEBUG defined but "disappears" if DEBUG is not
 defined.
//{
	Dbg(Debugf("Only shows up in DEBUG version");)
//}
*/
#ifdef DEBUG
#define Dbg(x) x
#else
#define Dbg(x)
#endif


/****************************************************************************
 ASSERT		returns void.
	Used to verify that some runtime condition is true.

 The ASSERT() macro is used to test for conditions and print out a warning
 if the condition is violated.  The code "disappears" if the module is 
 compiled without DEBUG being defined.

 See Also	
	assert.h
*/
#ifdef DEBUG
#define ASSERT(cond, str)	((void)(!(cond) ? \
		(Debugf("==> ERROR, File: %s, Line: %d ==> %s\n", \
				__FILE__, __LINE__, str)),1: 0))
#else
#define ASSERT(cond, str)
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Exported Functions						   	   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 Debugf		returns void
	Prints a formatted string on the debug output device, followed by a 
	newline.
 
 Debugf is very similar to the standard C runtime library function printf()
 except that (1) Debugf directs it output to PenPoint's debug output device
 and (2) Debugf prints a newline at the end of its output.

 Unless surrounded by something Dbg() or DbgFlag(), Debugf does not
 disappear, even if compiled without DEBUG defined.

 Use DPrintf to avoid having the trailing newline printed.

 See Also
	DPrintf
*/
void CDECL 
Debugf(char* str, ...);


/****************************************************************************
 DPrintf	returns void
	Prints a formatted string on the debug output device.

 DPrintf is very similar to the standard C runtime library function
 printf() except that DPrintf directs it output to PenPoint's debug output
 device.

 Unless surrounded by something Dbg() or DbgFlag(), DPrintf does not
 disappear, even if compiled without DEBUG defined.

 See Also
	Debugf
*/
void CDECL 
DPrintf(char* str, ...);


/****************************************************************************
 DbgFlagSet		returns void
	Sets the specified flag set to the value of the new flags.

	set:    flag set selector in the range 0..255, inclusive.  (Defined
	        as a U16 to allow for possible future expansion.)
	flags:  new values for the flag set.

 It is unusual for a program to call this function;  most developers should
 set the value of debugging flags using the techniques described in the
 introduction of this file rather than executing this function.

 Unless surrounded by something Dbg() or DbgFlag(), DbgFlagSet does not
 disappear, even if compiled without DEBUG defined.
*/
void EXPORTED 
DbgFlagSet(
	U16 set, 
	U32 flags);


/****************************************************************************
 DbgFlagGet		returns void
	Returns the state of the indicated flag set ANDed with the flags mask.

	set:   flag set selector in the range 0..255, inclusive.  (Defined
	       as a U16 to allow for possible future expansion.)
	flags: flags mask

 Unless surrounded by something Dbg() or DbgFlag(), DbgFlagGet does not
 disappear, even if compiled without DEBUG defined.
*/
U32 EXPORTED 
DbgFlagGet(
	U16 set, 
	U32 flags);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Debugging Flag Set Allocations					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#if 0
/****************************************************************************
//{
 Not to be used by anyone (interferes with parsing process):
	0x00
	0x09
	0x0A
	0x0D
	0x1A
	0x20

 Reserved for use outside of GO:
	Lower case alphabet, except f, h, i, s, and z.
	0x30 .. 0x39  digits
	0x80 .. 0xBF  half of the upper range

 Reserved for use by GO
	'f'
	'h'
	'i'
	'q'
	's'
	'z'
	everything else

 Here are the allocations within GO's range.  See other header files for
 more information on the interpretation of these flags.  Most flags only
 have effect if you load the debug versions of DLLs.

	'f':  GO Application Developer's Course

	'h':  Hwxtool and Insertion Pads

	'q':  Quick Help

	's':  Hwxtool

	'z':  Xlate

	'A':  Misc. system use.

		A0001:	Print loader information while loading

	'B':  System

		B0001:	Turns uuid cache tracing on
        B0002:	Enables OEM app/service installation after warm-boot This
                should only be turned on for tablet hardware; never on the
                SDK!
        B0800:	Enables theSelectedVolume disk viewing in Connections

	'C':  ClsMgr

	'D':  Debug system

        D0001:		disables all DebugStr output
        D0002:		disables StringPrint output
        D0004:		disables System Log output
        D0008:		disables System Log Non Error output
        D0010:		disables System Log App Error output
        D0020:		disables System Log System Error output
        D8000:		writes output to PENPOINT.LOG, file flushed every n chars
                	based on the environment variable DebugLogFlushCount.
        D10000:		disables mini-debugger in production version of Penpoint
        D20000:		disables memory statistics gestures (M,N,T) on Bookshelf
        D40000:		disables ^C entering the mini-debugger
        D80000000:	allows logging to log file even if in file system code
					(This may cause deadlocks and is for internal use only).

	'E':  Environment flags

	'F':  Application Developer's Course

	'G':  Kernel

	'H':  Service and Service Manager

        H0001:	turns on message tracing in clsService
        H0002:	turns on message tracing in clsServiceMgr
        H8000:	run sanity test in service.dll

	'I':  Installers (see instlmgr.h)

	'J':  Notebook

	'K':  UI Toolkit

	'L':  PicSegs and TIFF images

        L0001:	dumps the TIFF image tags.

	'M':  misc.lib

        M0001:	tracing in OrderedSetDelete
        M0002:	tracing in OrderedSetFindMinMax & MaxMin
        M0004:	tracing in OrderedSetInsertn
        M0008:	tracing in OrderedSetSearch
        M0100:	write/read debug header&trailer when filing ByteArray

	'N':  MiniText

	'O':  Outbox (obxserv and oboxsect)

        O0001:	enable automatic activate of outbox Notebook

	'P':  Printing

	'Q':  text.dll

	'R':  Application Framework

	'S':  Spelling, Proof, and XTemplate systems

        S0001:	low-level Spell/Proof debugs
        S0002:	medium-level Spell/Proof debugs
        S0004:	high-level Spell/Proof debugs
        S0010:	XTemplate display inputs
        S0020:	XTemplate display outputs

	'T':  text.dll

	'U':  undo.dll

	'V':  text.dll

	'W':  Window system

	'X':  xfer.lib

	'Y':  TOPS

	'Z':  Handwriting

	'@':  Bookshelf

	'=':  MiniNote/NotePaper

	'#':  GWin

	'!':  Test Manager

	'$':  File System

	'%':  UI Toolkit

    '*':  Heap Manager

	0xC0:  Fax Project

	0xC1:  Input

	0xC2:  VKey

    0xC3:  System Log trace flag

    0xC4:  2.0 tools

	0xF0:  Memory Tests // Internal use only

	0xF1:  Memory Tests // Internal use only
	
	0xFF:  C Runtime Library
//}
****************************************************************************/
#endif

#endif  // DEBUG_INCLUDED
