/****************************************************************************
 File: cnctions.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.21  $
   $Author:   rfoster  $
     $Date:   27 Feb 1992 23:15:08  $

 This file contains the API definition for the interface between the
 connections notebook and a generic service.

 The connections notebook is, effectively, an option sheet.  Because of this
 implementation choice, it is important to understand the option sheet
 protocol and messages, as defined in OPTION.H.  The terminology chosen
 herein reflects the close association between the connections notebook
 and an option sheet.

 The two default views that one gets, for disks and printers, in the
 connections notebook are each option sheets added as cards of the
 connections notebook option sheet.  Other sheets or windows can be added to
 the connections notebook.
	 
 The connections notebook observes the well-known list theConnections.  If an
 item is added to the list, the connections notebook calls that item with
 msgConnectionsAddSheet, with the P_ARGS being the main option sheet in the
 connections notebook.  By using msgOptionAddCard to the object passed in
 the aforementioned call, a service can add a sheet or just a single window
 to the connections notebook.  Once these items have been added, all
 responsibility for the user interface and functionality rests solely on the
 service.
	 
 Network disks and printers, however, are handled differently.  There are
 already predefined windows for these two items.  A network file-sharing
 system, for example, would add itself to the well-known list
 theVolumeServices.  The connections notebook, which observes this list, would
 send the object on the list a msgConnectionsStartConversation and a
 msgConnectionsSetConnectionsApp to pass along the application context of
 the connections notebook from this time.
	 
 If the network file-sharing service were to remove itself from
 theVolumeServices, the connections notebook would send
 msgConnectionsEndConveration to the object.
	 
 The object on the list is expected to be able to respond to the various
 connections messages.  If it has specified that it provides a UI, it will
 be asked for its network view when appropriate.

****************************************************************************/
#ifndef CNCTIONS_INCLUDED
#define CNCTIONS_INCLUDED

#ifndef INSTLMGR_INCLUDED
#include <instlmgr.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Warnings  ****/
#define	stsConnectionsAlreadyConnected			MakeWarning(clsConnections, 1)


/****  Statuses  ****/
#define	stsConnectionsPasswordFailed			MakeStatus(clsConnections, 1)
#define	stsConnectionsServiceDeinstalling		MakeStatus(clsConnections, 2)
#define	stsConnectionsNotConnected				MakeStatus(clsConnections, 3)


/****  Typedefs  ****/
typedef struct CONNECTIONS_MENU_ITEM {
	P_CHAR			pName;
	OBJECT			netService;
	P_UNKNOWN		netIdentifier;
	U32				reserved[2];
} CONNECTIONS_MENU_ITEM, * P_CONNECTIONS_MENU_ITEM;


typedef struct CONNECTIONS_ITEM {
	struct CONNECTIONS_ITEM		*pNextConnectionsItem;	// Next item
	P_UNKNOWN					pItemID;		// Service defined identifer
												// for this item
	TAG							itemIconTag;	// Item's icon tag
	TAG							itemTag;		// Item tag
	P_CHAR						name;			// Item name
	P_CHAR						serverName;		// Item's server's name
	P_CHAR						location;		// Item's location
	P_CHAR						type;			// Item's type
	BOOLEAN						connected;		// Connected?
	BOOLEAN						autoConnect;	// Auto-connect enabled?
	BOOLEAN						remember;		// Remember (menu) enabled?
	// fill in some more information here
	P_UNKNOWN					itemSpecificData;	// volume or printer stuff
	U32							filler[4];			// reserved
} CONNECTIONS_ITEM, * P_CONNECTIONS_ITEM, * * PP_CONNECTIONS_ITEM;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages											   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgConnectionsSetState:	takes P_CONNECTIONS_STATE, returns STATUS
 	Sets the specified states in the service.

*/
Enum16 ( CONNECTIONS_CONNECT_STATE ) {
	cnctManualConnections,					// Connect only when asked to
	cnctAutoConnections,					// Connect auto-connect items
	cnctPromiscuousConnections				// Connect to everything
};

Enum16 ( CONNECTIONS_WARNINGS ) {
	cnctWarningNone						= 0,		// No warnings
	cnctWarningPermissionsFailure		= flag0,	// On permissions failure
	cnctWarningOnConnection				= flag1,	// On connection
	cnctWarningOnUnconnection			= flag2		// On loss of connection
};

Enum16 ( CONNECTIONS_PASSWORDS ) {
	cnctPasswordNone					= 0,		// Do not save passwords
	cnctPasswordServer					= flag0,	// Save server passwords
	cnctPasswordItem					= flag1,	// Save item passwords
	cnctPasswordServerAndItem			= flag2		// Save server and item
													// passwords
};

Enum16 ( CONNECTIONS_PERMISSIONS ) {
	cnctPermissionsReadWrite,						// Connect Read/Write
	cnctPermissionsReadOnly							// Connect Read only
};

typedef struct CONNECTIONS_STATE {
	BOOLEAN							attached;			// Attached
	CONNECTIONS_CONNECT_STATE		connectMores;		// How to attach
	CONNECTIONS_WARNINGS			connectWarning;		// Level of warnings
	CONNECTIONS_PASSWORDS			connectPasswords;	// What passwords
	CONNECTIONS_PERMISSIONS			connectPermissions;	// What permissions
	U32								reserved[4];
} CONNECTIONS_STATE, * P_CONNECTIONS_STATE;

#define msgConnectionsSetState				MakeMsg ( clsConnections, 1 )


/****************************************************************************
 msgConnectionsGetState:	takes P_CONNECTIONS_STATE, returns STATUS
 	Gets the specified states in the service.

*/
#define msgConnectionsGetState				MakeMsg ( clsConnections, 2 )


/****************************************************************************
 msgConnectionsEnumerateItems:	takes P_CONNECTIONS_ENUMERATE, returns STATUS
 	Gets a list of the network items, per restrictions.

*/

#define cnctAttribMatchLocation		flag0		// Match on location
#define cnctAttribMatchServer		flag1		// Match on server
#define cnctAttribMatchConnect		flag2		// Match on connected state
#define cnctAttribMatchAutoConnect	flag3		// Match on auto-connect state
#define cnctAttribMatchMenu			flag4		// Match on menu
												// (remember) state

typedef struct ATTRIB {
	U32				flags;			// various meanings -- complete match
									//    match at beginning, match at end
									// connected, auto connect, remember
	P_CHAR			restrictName;	// match this string
	// other possible characteristics -- type, characteristics, etc.
	P_UNKNOWN		matchID;		// restrict enumeration to this file
									// server
	TAG				tag;			// Tag to match against
} ATTRIB, * P_ATTRIB;


#define cnctFlagLocationsOnly		flag0		// Look only at locations
#define cnctFlagServersOnly			flag1		// Look only at servers
#define cnctFlagOKFreeCIFields		flag14		// Free the CI fields
#define cnctFlagOKFreeCI			flag15		// Free the CI

typedef struct CONNECTIONS_ENUMERATE {
	ATTRIB				attributes;
	U16					count;	// in	= # of entries to return in list.
                                // out	= # of valid entries in list.
	U16					next;	// in	= 0 to start at beginning 
                                //        OR previous out value to pick up
                                //        where we left off.
	P_CONNECTIONS_ITEM	pEntry;	// in	= pNull.
                                // out	= Link list of connections items.
	U16                 flags;  // in   = state flags to filter on.
								// out	= free state
} CONNECTIONS_ENUMERATE, * P_CONNECTIONS_ENUMERATE;

#define msgConnectionsEnumerateItems			MakeMsg ( clsConnections, 3 )


/****************************************************************************
 msgConnectionsEnumerateServers: takes P_CONNECTIONS_ENUMERATE, returns STATUS
 	Gets a list of the network servers, per restrictions.

	Use CONNECTIONS_ITEM with restriction of cnctFlagServersOnly.

*/


#define msgConnectionsEnumerateServers			MakeMsg ( clsConnections, 4 )


/****************************************************************************
 msgConnectionsEnumerateTags:	takes P_CONNECTIONS_ENUMERATE, returns STATUS
 	Gets a list of the known tags, per restrictions.

*/

typedef struct CONNECTIONS_TAG {
	TAG			tag;
} CONNECTIONS_TAG, * P_CONNECTIONS_TAG;

#define msgConnectionsEnumerateTags				MakeMsg ( clsConnections, 5 )


/****************************************************************************
 msgConnectionsGetNetworkView:	takes P_WIN, returns STATUS
	Each service is required to provide a window, which will be a client
	of a scrollwin, which will be set as the current (active) window
	when the network view is invoked.  This window will be able to make use
	of msgConnections calls to manipulate attachments, et al.

*/

#define msgConnectionsGetNetworkView			MakeMsg ( clsConnections, 6 )


/****************************************************************************
 msgConnectionsCompareItems:	takes P_CONNECTIONS_COMPARE, returns STATUS
 	Compares two pItemID values to see if they refer to the same item.

*/

typedef struct CONNECTIONS_COMPARE {
	P_UNKNOWN		item1;			// First item
	P_UNKNOWN		item2;			// Second item
	BOOLEAN			same;			// Out:  Are they the same?
	U32				forPublicUse;	// if any one needs this
} CONNECTIONS_COMPARE, * P_CONNECTIONS_COMPARE;

#define msgConnectionsCompareItems			MakeMsg ( clsConnections, 10 )


/****************************************************************************
 msgConnectionsTagItem:	takes P_CONNECTIONS_TAG_ITEM, returns STATUS
 	Tags the indicated item.

*/

typedef struct CONNECTIONS_TAG_ITEM {
	TAG				tag;				// Tag to set
	U32				flags;				// Type
	P_UNKNOWN		netAddress;			// Item's address
	U32				userInformation;
} CONNECTIONS_TAG_ITEM, * P_CONNECTIONS_TAG_ITEM;

#define msgConnectionsTagItem					MakeMsg ( clsConnections, 11 )


/****************************************************************************
 msgConnectionsGetServiceInfo:	takes P_CONNECTIONS_SERVICE_INFO, returns STATUS
 	Gets the service name and other information.

*/

typedef struct CONNECTIONS_SERVICE_INFO {
	CHAR			serviceName[nameBufLength];		// Service name
	U16				reserved:15,
					uiProvided:1;					// User interface provided
	U32				filler[2];
} CONNECTIONS_SERVICE_INFO, * P_CONNECTIONS_SERVICE_INFO;


#define msgConnectionsGetServiceInfo			MakeMsg ( clsConnections, 12 )


/****************************************************************************
 msgConnectionsGetItemInfo:					takes P_UNKNOWN, returns STATUS
 	Gets information for the specified item, specific to the service.

*/

#define msgConnectionsGetItemInfo			MakeMsg ( clsConnections, 13 )


/****************************************************************************
 msgConnectionsSetConnectionsApp:				takes OBJECT, returns STATUS
 	Passes the connections notebook app object to the service.

*/

#define msgConnectionsSetConnectionsApp		MakeMsg ( clsConnections, 14 )


/****************************************************************************
 msgConnectionsUpdate:				takes nothing, returns STATUS
	 Requests an update of the current network state.

*/

#define msgConnectionsUpdate		MakeMsg ( clsConnections, 15 )


/****************************************************************************
 msgConnectionsExpandCollapse:				takes BOOLEAN, returns STATUS
	 Requests an expand/collapse (depending on the argument) of the current
	 view of the network.

*/

#define msgConnectionsExpandCollapse	MakeMsg ( clsConnections, 16 )


/****************************************************************************
 msgConnectionsConnectItem:		takes P_CONNECTIONS_REQUEST, returns STATUS
 	Connect the specified item.

*/

typedef struct CONNECTIONS_REQUEST {
	P_UNKNOWN			pItemID;			// Item to connect
	U32					response;
} CONNECTIONS_REQUEST, * P_CONNECTIONS_REQUEST;

#define msgConnectionsConnectItem				MakeMsg ( clsConnections, 17 )


/****************************************************************************
 msgConnectionsUnconnectItem:	takes P_CONNECTIONS_REQUEST, returns STATUS
 	Unconnect the specified item.

*/

#define msgConnectionsUnconnectItem				MakeMsg ( clsConnections, 18 )


/****************************************************************************
 msgConnectionsRememberItem:  takes P_CONNECTIONS_REQUEST, returns STATUS
 	Remember the specified item.

*/

#define msgConnectionsRememberItem				MakeMsg ( clsConnections, 19 )


/****************************************************************************
 msgConnectionsForgetItem:  takes P_CONNECTIONS_REQUEST, returns STATUS
 	Forget the specified item.

*/

#define msgConnectionsForgetItem				MakeMsg ( clsConnections, 20 )


/****************************************************************************
 msgConnectionsAutoConnectItem:	takes P_CONNECTIONS_REQUEST, returns STATUS
 	Sets the auto connect state on for the specified item.

*/

#define msgConnectionsAutoConnectItem			MakeMsg ( clsConnections, 21 )


/****************************************************************************
 msgConnectionsUnAutoConnectItem: takes P_CONNECTIONS_REQUEST, returns STATUS
 	Sets the auto connect state off for the specified item.

*/

#define msgConnectionsUnAutoConnectItem			MakeMsg ( clsConnections, 22 )


/****************************************************************************
 msgConnectionsAddSheet:						takes OBJECT, returns STATUS
 	Permits items on the connections to add items to the contents.

*/

#define msgConnectionsAddSheet					MakeMsg ( clsConnections, 23 )


/****************************************************************************
 msgConnectionsAddCards:					takes P_OPTION_TAG, returns STATUS
 	Sent to network views, when they are not the foremost view, to run
	the option protocol.

*/

#define msgConnectionsAddCards					MakeMsg ( clsConnections, 24 )


/****************************************************************************
 msgConnectionsSetSelection:				takes P_UNKNOWN, returns STATUS
	Sent by the connections notebook to the appropriate service, informing
	the service what the currently selected item is.

*/

#define msgConnectionsSetSelection				MakeMsg ( clsConnections, 25 )


/****************************************************************************
 msgConnectionsGetTopCard:						takes P_TAG, returns STATUS
	Sent by the connections notebook to the appropriate service, inquiring
	of that service what the appropriate top card is to be.

*/

#define msgConnectionsGetTopCard				MakeMsg ( clsConnections, 26 )


/****************************************************************************
 msgConnectionsStartConversation:				takes nothing, returns STATUS
	Sent by the Connections Notebook to the appropriate service, informing
	that service that the Connections Notebook is planning on conversing
	with it.  This message will be sent at first page turn and at restore (of
	the Connections Notebook) time.

*/

#define msgConnectionsStartConversation			MakeMsg ( clsConnections, 27 )


/****************************************************************************
 msgConnectionsEndConversation:					takes nothing, returns STATUS
	Sent by the Connections Notebook to the appropriate service, informing
	that service that the Connections Notebook is stopping conversing
	with it.  This message will be sent at save (of	the Connections Notebook)
	time.

*/

#define msgConnectionsEndConversation			MakeMsg ( clsConnections, 28 )


/****************************************************************************
 msgConnectionsIsParent:	takes P_CONNECTIONS_COMPARE, returns STATUS
 	Compares two pItemID values to see if item1 is a parent of item2.

*/

#define msgConnectionsIsParent					MakeMsg ( clsConnections, 31 )


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 						Notification Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgConnectionsConnectedChanged: takes P_CONNECTIONS_NOTIFY, returns STATUS
	 Sent by the appropriate service, indicating when an item has been
	 connected to or unconnected from.

*/

#define msgConnectionsConnectedChanged			MakeMsg ( clsConnections, 7 )

typedef struct CONNECTIONS_NOTIFY {
	OBJECT				manager;		// manager that sent notification
	IM_HANDLE			handle;			// handle to service
	OBJECT				service;		// service that sent notification
	P_UNKNOWN			pItemID;		// pointer to affected item
	U16					reserved:13,
						server:1,		// Unused
						uiProvided:1,	// Unused
						state:1;		// connected or unconnected
	U16					notifyLength;	// Length of notify info which follows
} CONNECTIONS_NOTIFY, * P_CONNECTIONS_NOTIFY;



/****************************************************************************
 msgConnectionsAutoConnectChanged: takes P_CONNECTIONS_NOTIFY, returns STATUS
	 Sent by the appropriate service, indicating when an item has had
		 the auto connect state set or turned off for it.

*/

#define msgConnectionsAutoConnectChanged		MakeMsg ( clsConnections, 8 )


/****************************************************************************
 msgConnectionsRememberChanged:		takes P_CONNECTIONS_NOTIFY, returns STATUS
	 Sent by the appropriate service, indicating when an item has had
		 the remember state set or turned off for it.

*/

#define msgConnectionsRememberChanged			MakeMsg ( clsConnections, 9 )

/****************************************************************************
 msgConnectionsItemChanged:	takes P_CONNECTIONS_NOTIFY, returns STATUS
	 Sent by the appropriate service, indicating when an item has been
	 noticed or lost.

*/

#define msgConnectionsItemChanged				MakeMsg ( clsConnections, 30 )

/****************************************************************************
 msgConnectionsServiceChanged:	takes P_CONNECTIONS_NOTIFY, returns STATUS
	 Sent by the appropriate service, indicating when it is available for
	 use or unavailable.

*/

#define msgConnectionsServiceChanged				MakeMsg ( clsConnections, 32 )



//REFGEN BEGINIGNORE
// Next up:  33

//REFGEN ENDIGNORE
#endif
