/****************************************************************************

 File: browser.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.34  $
 $Author:   DYEE  $
 $Date:   30 Mar 1992 16:32:00  $

 This file contains the API definition for clsBrowser.

 clsBrowser inherits from clsScrollWin.
 clsBrowser provides the UI for viewing and manipulating notebooks and disks.
 
 clsBrowser provides both the Table Of Contents view of "live" data in the
 notebook and the Disk Viewer view of "dead" data on disk.  clsBrowser 
 functions include displaying notebook and disk items,
 navigating the notebook or file system hierarchy, move/copy of documents, 
 export of notebook documents to disk, import of files from disks 
 into the notebook, deleting notebook and disk items, and creating notebook 
 and disk items.

 clsBrowser is useful to applications that need to allow users to select 
 sections or documents in the notebook, or items from disk.    
 
 Some messages apply only to the TOC view or to the disk view.  Disk View 
 only messages are labeled DskView only,  TOC view only messages are 
 labeled TOC only.   

 Many browser messages are sent to self allowing subclasses to modify browser
 behavior.

****************************************************************************/


/****  Move/Copy Conventions  ****/
/*

 See embedwin.h for move/copy protocol.
 
 When the source of a move/copy, the browser responds to msgXferGetList
 with:
 
    XferName:           can xfer the name of the selection
    XferFullPathName:   can xfer the full path name of the selection
    XferFlatLocator:    can xfer the flat locator of the selection
    clsFileSystem:      can xfer as a file or directory
    clsEmbeddedWin:     can xfer as "live" data notebook, section, or
                        document
    clsExport:          If source is TOC and export mode is in effect
                        then do export instead of copy.
                        (see export.h for details)

 If the destination is the disk and the xferList contains clsExport then
 do export instead of move/copy.

 If not an export, and the xferList contains clsEmbeddedWin then let the 
 embedded win superclass will handle the move/copy.

 If the destination is the TOC and source is not a clsEmbeddedWin then 
 invoke the import code.

 Otherwise, if the source is clsFileSystem do a file system move or copy.
 
*/

#ifndef BROWSER_INCLUDED
#define BROWSER_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef FRAME_INCLUDED
#include <frame.h>
#endif 

#ifndef FS_INCLUDED
#include <fs.h>
#endif

#ifndef RESFILE_INCLUDED
#include <resfile.h>
#endif

#ifndef SWIN_INCLUDED
#include <swin.h>
#endif



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                    Common #defines and typedefs                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****  Sort Types  ****/
/*
 * Defines the order the browser will sort display items by.
 */
Enum16 ( SORT_BY ) {
    browserSortByName = 1,
    browserSortBySize = 2,
    browserSortByDate = 3,
    browserSortByPage = 4,
    browserSortByType = 5
};

/*
 * These are tags for the icons used by clsBrowser
 */

#define tagBrowserSmallFileIcon         MakeTag(clsBrowser,1)
#define tagBrowserBigFileIcon           MakeTag(clsBrowser,2)
#define tagBrowserSmallClosedDirIcon    MakeTag(clsBrowser,3)
#define tagBrowserBigClosedDirIcon      MakeTag(clsBrowser,4)
#define tagBrowserSmallOpenDirIcon      MakeTag(clsBrowser,5)
#define tagBrowserBigOpenDirIcon        MakeTag(clsBrowser,6)
#define tagBrowserSmallClosedSectIcon   MakeTag(clsBrowser,7)
#define tagBrowserBigClosedSectIcon     MakeTag(clsBrowser,8)
#define tagBrowserSmallOpenSectIcon     MakeTag(clsBrowser,9)
#define tagBrowserBigOpenSectIcon       MakeTag(clsBrowser,10)
#define tagBrowserSmallDefaultDocIcon   MakeTag(clsBrowser,11)
#define tagBrowserBigDefaultDocIcon     MakeTag(clsBrowser,12)

/*
 * These are the help ID's used for the various browser items.
 */

#define hlpBrowser              MakeTag(clsBrowser,170) // Generic TOC
#define hlpBrowIcon             MakeTag(clsBrowser,169) // TOC
#define hlpBrowName             MakeTag(clsBrowser,171) // TOC
#define hlpBrowPage             MakeTag(clsBrowser,172) // TOC
#define hlpBrowType             MakeTag(clsBrowser,173) // TOC
#define hlpBrowDate             MakeTag(clsBrowser,174) // TOC
#define hlpBrowTime             MakeTag(clsBrowser,175) // TOC
#define hlpBrowSize             MakeTag(clsBrowser,176) // TOC
#define hlpBrowBookmark         MakeTag(clsBrowser,177) // TOC
#define hlpBrowColumn           MakeTag(clsBrowser,178) // TOC


/* DskViewer help tags  */

#define hlpBrowserDV            MakeTag(clsBrowser,180) // Generic DSKVIEW                  

#define hlpBrowNameDV           MakeTag(clsBrowser,181) // DSKVIEW
#define hlpBrowTypeDV           MakeTag(clsBrowser,183) // DSKVIEW
#define hlpBrowDateDV           MakeTag(clsBrowser,184) // DSKVIEW
#define hlpBrowTimeDV           MakeTag(clsBrowser,185) // DSKVIEW
#define hlpBrowSizeDV           MakeTag(clsBrowser,186) // DSKVIEW


/* Column Tag - identify columns for msgBrowserGesture */

#define tagBrowNameColumn       MakeTag(clsBrowser,191) 
#define tagBrowPageColumn       MakeTag(clsBrowser,192) 
#define tagBrowTypeColumn       MakeTag(clsBrowser,193) 
#define tagBrowDateColumn       MakeTag(clsBrowser,194) 
#define tagBrowTimeColumn       MakeTag(clsBrowser,195) 
#define tagBrowSizeColumn       MakeTag(clsBrowser,196) 
#define tagBrowBookmarkColumn   MakeTag(clsBrowser,197) 
#define tagBrowUserColumn0      MakeTag(clsBrowser,198) 
#define tagBrowUserColumn1      MakeTag(clsBrowser,199) 
#define tagBrowUserColumn2      MakeTag(clsBrowser,200) 
#define tagBrowUserColumn3      MakeTag(clsBrowser,201) 


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                    Messages                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgNewDefaults:    takes P_BROWSER_NEW, returns STATUS
    category: class message
    Initializes the BROWSER_NEW structure to default values.

 Zeros out pNew->browser.
*/

/****************************************************************************
 msgNew:    takes P_BROWSER_NEW, returns STATUS
    category: class message
    Creates a new browser object.

 Creates a browser which will display the file system within the specified
 base directory.  If the browser will be looking at "live" notebook sections
 and documents set tocView to true; If the browser will be looking at "dead"
 directories, files, or documents and sections on disk then set tocView to 
 false.
*/

typedef struct BROWSER_NEW_ONLY {
    FS_LOCATOR  base;       // Points to where the browser will display.
                            // Note: This UID must not be an absolute path!
    OBJECT      client;     // UID of client.
    U16         tocView;    // TRUE for TOC view, FALSE for disk view.
    U8          spare[8];
} BROWSER_NEW_ONLY, *P_BROWSER_NEW_ONLY;


#define    browserNewFields        \
    scrollWinNewFields          \
    BROWSER_NEW_ONLY    browser;


typedef struct BROWSER_NEW {
    browserNewFields
} BROWSER_NEW, *P_BROWSER_NEW;


/****************************************************************************
 msgBrowserCreateDir    takes nothing, returns STATUS
    Creates a directory at the selection.

 If nothing is selected, this message creates a directory at the top 
 level of the disk.  DskView message only.  Usually sent from menu.
*/
#define msgBrowserCreateDir            MakeMsg(clsBrowser, 1)


/****************************************************************************
 msgBrowserByName       takes nothing, returns STATUS
    Sorts by name order.  

 Displays all displayed items sorted by name order.  Usually sent from menu. 
*/
#define msgBrowserByName            MakeMsg(clsBrowser, 2)


/****************************************************************************
 msgBrowserByType       takes nothing, returns STATUS
    Sorts by type order.

 Displays all displayed items sorted by type order.  Usually sent from menu. 
*/
#define msgBrowserByType            MakeMsg(clsBrowser, 40)


/****************************************************************************
 msgBrowserBySize       takes nothing, returns STATUS
    Sorts by size order.

 Displays all displayed items sorted by size order.  Usually sent from menu. 
*/
#define msgBrowserBySize            MakeMsg(clsBrowser, 3)


/****************************************************************************
 msgBrowserByDate       takes nothing, returns STATUS
    Sorts by date order.

 Displays all displayed items sorted by date order.  Usually sent from menu. 
*/
#define msgBrowserByDate            MakeMsg(clsBrowser, 4)


/****************************************************************************
 msgBrowserExpand       takes nothing or P_FS_FLAT_LOCATOR, returns STATUS
    Expands sections or directories.

 If pArgs is P_FS_FLAT_LOCATOR, expands P_FS_FLAT_LOCATOR otherwise
 if pArgs is pNull and the browser has the selection, the selection is 
 expanded.  Otherwise, every displayed closed selection is expanded.
*/
#define msgBrowserExpand            MakeMsg(clsBrowser, 5)


/****************************************************************************
 msgBrowserCollapse     takes nothing or P_FS_FLAT_LOCATOR, returns STATUS
    Collapses sections or directories.

 If pArgs is P_FS_FLAT_LOCATOR, collapses P_FS_FLAT_LOCATOR otherwise
 if pArgs is pNull and the browser has the selection, the selection is 
 collapsed; otherwise, every open selection is collapsed.
*/
#define msgBrowserCollapse            MakeMsg(clsBrowser, 6)


/****************************************************************************
 msgBrowserRefresh       takes nothing, returns STATUS
    Refreshes the disk image the browser is displaying.
*/
#define msgBrowserRefresh           MakeMsg(clsBrowser, 15)


/****************************************************************************
 msgBrowserDelete        takes nothing or P_FS_FLAT_LOCATOR, returns STATUS
    Deletes selection if pNull or P_FS_FLAT_LOCATOR otherwise.

 Sent to self to allow subclass to override.
*/
#define msgBrowserDelete            MakeMsg(clsBrowser, 22)

/****************************************************************************
 msgBrowserRename        takes nothing or P_FS_FLAT_LOCATOR, returns STATUS
    Renames browser items.

 Pops up rename dialog box for the selection if pNull; otherwise the item
 pointed to by P_FS_FLAT_LOCATOR is renamed.
 Sent to self to allow subclass to override.
*/
#define msgBrowserRename            MakeMsg(clsBrowser, 23)

/****************************************************************************
 msgBrowserConfirmDelete        takes BOOLEAN, returns STATUS
    Sets a flag whether to confirm deletions within a browser.
*/
#define msgBrowserConfirmDelete        MakeMsg(clsBrowser, 24)

/****************************************************************************
 msgBrowserExport        takes nothing, returns STATUS
    Puts the selection into export mode.

 After this message is received by TOC the selected item is highlighted with
 the copy box.  Then if notebook item is dragged to the DiskViewer, it will 
 be exported, not copied.  The export mode is cancelled when the selection is
 cancelled or the export completes.  TOC only.
*/
#define msgBrowserExport            MakeMsg(clsBrowser,118)


/****************************************************************************
 msgBrowserByPage        takes nothing, returns STATUS
    Sorts by page number.

 TOC only.
*/
#define msgBrowserByPage            MakeMsg(clsBrowser, 25)


/****************************************************************************
 msgBrowserWriteState    takes nothing, returns STATUS
    Writes the current browser expanded/collapsed state to a file.

 This message saves the name of each expanded section or 
 directory to a disk file.  By using msgBrowserSetSaveFile clients or 
 subclasses can set which file this information is stored in.
 By default the state file ends up in the OSThisApp's directory in a file 
 named BROWSTAT.    
*/
#define msgBrowserWriteState        MakeMsg(clsBrowser, 26)


/****************************************************************************
 msgBrowserReadState     takes nothing, returns STATUS
    Reads the browser expanded/collapsed state from a disk file.

 This message restores the state of the browser view of the notebook or
 file system.  By using msgBrowserSetSaveFile clients or 
 subclasses can set which file this information is stored in.
 By default the state file ends up in the OSThisApp's dir in a file named
 browstate.    
*/
#define msgBrowserReadState            MakeMsg(clsBrowser, 27)


/****************************************************************************
 msgBrowserSetSaveFile  takes P_FS_LOCATOR, returns STATUS
    Sets the file that the browser will save open/close state to.
*/
#define msgBrowserSetSaveFile   MakeMsg(clsBrowser,148)


/****************************************************************************
 msgBrowserGetMetrics    takes P_BROWSER_METRICS, returns STATUS
    Gets browser metrics. 
*/
#define msgBrowserGetMetrics        MakeMsg(clsBrowser, 28)


/****  SubClass-definable Column Type  ****/
/*
 * Defines attributes of the subclass definable browser columns.
 * Subclasses can control up to browUserColumns (4) columns.
 *
 * User Columns are columns of checkboxes or text, that subclasses of 
 * clsBrowser can control.  The subclass can supply the header above
 * the column and whether or not the boxes appear next to sections or 
 * documents or both. 
 * 
 * User columns are enabled by setting pMetrics->userColumn.showUserColumn.
 * 
 * The browser sends msgBrowserUserColumnQueryState to subclasses to 
 * determine the initial state of the columns.  
 *
 * When a column is tapped, msgBrowserUserColumnChanged 
 * notifies subclasses that the checkbox has toggled.
 * 
 *
 */

#define browDefaultColumns          7   // Number of default columns.
#define browUserColumns             4   // Maximum number of user columns.

/*
 * Display justifications
 */
Enum16 ( BROW_JUSTIFY ) {

    browserLeftJustify = 0,             // Left justification.
    browserRightJustify = 1,            // Right justification.
    browserCenterJustify = 2,           // Center justification.
    browserUserJustify = 3              // Miscellaneous justification.
}; 

/*
 * User column type
 */
Enum16 ( USER_COLUMN_TYPE ) {

    browserButtonType = 0,              // Button user column.
    browserTextType = 1,                // Text user column.
    browserUserType = 2                 // User defined user column.
};

typedef struct {
                                                    
    BROW_JUSTIFY    headerJustify;               // Justification of header.
    BROW_JUSTIFY    columnJustify;               // Justification of column.
    CHAR            columnHeader[nameBufLength]; // Text for column.

} BROWSER_DEF_COLUMN, *P_BROWSER_DEF_COLUMN;

typedef struct {
    U16                 showUserColumn : 1;         // Must be set to TRUE for the
                                                    // following fields to apply.
    U16                 userColumnOnSections : 1;   // Show userColumn next to sections.
    U16                 userColumnOnDocs : 1;       // Show userColumn next to documents.
    USER_COLUMN_TYPE    userColumnType;             // Type of field if user column.
    CHAR                userColumnHeader[nameBufLength]; // Text of column header.
    TAG                 helpTag;                    // Tag for quick help
    CHAR                checkedChar;                // Character to show when checked.
    CHAR                uncheckedChar;              // Character to show when unchecked.
    BROW_JUSTIFY        headerJustify;              // Justification of header.
    BROW_JUSTIFY        columnJustify;              // Justification of column.
    U8                  spare[4];                   // Spare: reserved.
} BROWSER_COLUMN, *P_BROWSER_COLUMN;


typedef struct BROWSER_METRICS {
    U16                 showIcon : 1;               // Show icons.
    U16                 showType : 1;               // Show type field.
    U16                 showSize : 1;               // Show size field.
    U16                 showDate : 1;               // Show date field.
    U16                 showBookmark : 1;           // Show bookmark field. (TOC only)
    U16                 showHeader : 1;             // Show column header.
    U16                 computeRecursiveSize : 1;   // Computes recursive size
                                                    // for directories.
                                                    // TOC does this by default.
    U16                 showIconButton : 1;         // Show page turn buttons 
                                                    // instead of icons. (TOC only)
    SORT_BY             sortBy;                     // Field by which to sort items.
    BROWSER_COLUMN      userColumn[browUserColumns]; // Subclass-definable columns  
    BROWSER_DEF_COLUMN  defaultColumn[browDefaultColumns]; // Default columns
    U8                  spare[40];                  // Spare: reserved.

} BROWSER_METRICS, *P_BROWSER_METRICS;


/****************************************************************************
 msgBrowserSetMetrics    takes P_BROWSER_METRICS, returns STATUS
    Sets browser metrics.

 This message will cause a refresh if userColumn or recursive size 
 become turned on.
*/
#define msgBrowserSetMetrics            MakeMsg(clsBrowser, 29)


/****************************************************************************
 msgBrowserUserColumnGetState     takes P_BROWSER_USER_COLUMN, returns STATUS
    Does nothing.
*/
#define msgBrowserUserColumnGetState    MakeMsg(clsBrowser, 62)

typedef struct {
    BOOLEAN     changed;                // TRUE if this column has changed.
    BOOLEAN     state;                  // State of item check box.
    CHAR        text[nameBufLength];    // Text of field for item.
    BOOLEAN     shown;                  // TRUE if this column is shown.
                                        // Same as showUserColumn of METRICS.
    BOOLEAN     active;                 // TRUE if this column is active 
                                        // for this browser item.
} BROWSER_COLUMN_STATE;

typedef struct {
    FS_FLAT_LOCATOR         flat;                   // Locator of browser item.
    BROWSER_COLUMN_STATE    column[browUserColumns]; // Column information.
    U8                      spare[12];              // Spare: reserved.
} BROWSER_USER_COLUMN, *P_BROWSER_USER_COLUMN;


/****************************************************************************
 msgBrowserUserColumnSetState     takes P_BROWSER_USER_COLUMN, returns STATUS
    Sets the user column states in the browser for columns that are marked changed.

 If the changed BOOLEAN is set, the user column state will be set.
 Does not generate a msgBrowserUserColumnStateChanged.
 The entire BROWSER_USER_COLUMN structure must be cleared before 
 setting the fields that are changing.
*/
#define msgBrowserUserColumnSetState    MakeMsg(clsBrowser, 63)


/****************************************************************************
 msgBrowserUserColumnStateChanged  takes P_BROWSER_USER_COLUMN, returns STATUS
    Notifies subclass when user checks a user column checkbox.
    
 The changed field is true for the column that was tapped.
*/
#define msgBrowserUserColumnStateChanged     MakeMsg(clsBrowser, 68)


/****************************************************************************
 msgBrowserUserColumnQueryState  takes P_BROWSER_USER_COLUMN, returns STATUS
    Gets the user column state from subclass.    

 This message is sent to self when the browser needs to know the user
 column states for a notebook item.  The FS_FLAT_LOCATOR points to the file
 system item the browser needs to know the state of.  The subclass should
 pass back the state or the text of each user column for the file system 
 item.
*/
#define msgBrowserUserColumnQueryState    MakeMsg(clsBrowser, 69)


/****************************************************************************
 msgBrowserShowIcon     takes BOOLEAN, returns STATUS
    Controls icon field display.
*/
#define msgBrowserShowIcon            MakeMsg(clsBrowser, 100)


/****************************************************************************
 msgBrowserShowButton   takes BOOLEAN, returns STATUS
    Controls button field display.
*/
#define msgBrowserShowButton        MakeMsg(clsBrowser, 99)


/****************************************************************************
 msgBrowserShowSize     takes BOOLEAN, returns STATUS
    Controls size field display.
*/
#define msgBrowserShowSize            MakeMsg(clsBrowser, 102)


/****************************************************************************
 msgBrowserShowDate     takes BOOLEAN, returns STATUS
    Controls date field display.
*/
#define msgBrowserShowDate            MakeMsg(clsBrowser, 103)


/****************************************************************************
 msgBrowserShowType     takes BOOLEAN, returns STATUS
    Controls type field display.
*/
#define msgBrowserShowType            MakeMsg(clsBrowser, 33)


/****************************************************************************
 msgBrowserShowBookmark     takes BOOLEAN, returns STATUS
    Controls bookmark field display.

 TOC only.
*/
#define msgBrowserShowBookmark        MakeMsg(clsBrowser, 104)


/****************************************************************************
 msgBrowserShowHeader       takes BOOLEAN, returns STATUS
    Controls column header display.
*/
#define msgBrowserShowHeader        MakeMsg(clsBrowser, 39)


/****************************************************************************
 msgBrowserGoto             takes BOOLEAN, returns STATUS
    Takes true to goto, false to bringto the selection.

 TOC only.  Used by menu.
*/
#define msgBrowserGoto               MakeMsg(clsBrowser, 105)


/****************************************************************************
 msgBrowserGotoBringto      takes P_BROWSER_GOTO, returns STATUS
    Takes P_BROWSER_GOTO.  If pFlat is pNull, applies to selection.

 Sent to self to allow subclass to override. TOC only.
*/
#define msgBrowserGotoBringto        MakeMsg(clsBrowser, 134)

typedef struct {
    BOOLEAN         doGoto;     // TRUE - Goto document.
                                // FALSE - Bringto document.
                                // (Goto if bringto is disabled.)
    FS_FLAT_LOCATOR flat;       // Document to goto-bringto .
} BROWSER_GOTO, *P_BROWSER_GOTO;


/****************************************************************************
 msgBrowserUndo             takes nothing, returns STATUS
    Does nothing yet...
*/
#define msgBrowserUndo                MakeMsg(clsBrowser, 106)


/****************************************************************************
 msgBrowserSetSelection     takes P_FS_FLAT_LOCATOR, returns STATUS
    Causes browser/TOC to select and display the given file system item.

 As long as the locator points to an item within the browser's base directory
 subtree, the browser will open directories and scroll the display as 
 necessary to display the selected item.
*/
#define msgBrowserSetSelection      MakeMsg(clsBrowser,32)


/****************************************************************************
 msgBrowserSetClient        takes OBJECT, returns STATUS
    Sets the target of the browser client messages.

 This message controls who gets the various browser client messages.
*/
#define msgBrowserSetClient         MakeMsg(clsBrowser, 108)


/****************************************************************************
 msgBrowserGetClient        takes P_OBJECT, returns STATUS
    Passes back the target of the browser client messages.
*/
#define msgBrowserGetClient         MakeMsg(clsBrowser, 64)


/****************************************************************************
 msgBrowserGetBaseFlatLocator  takes P_FS_FLAT_LOCATOR, returns STATUS
    Passes back the directory the browser is looking at.

 Passes back the root directory within which the browser is looking.
*/
#define msgBrowserGetBaseFlatLocator    MakeMsg(clsBrowser, 65)


/****************************************************************************
 msgBrowserSelectionPath    takes P_BROWSER_PATH, returns STATUS
    Passes back the full path of the selection.

 Also responds to msgXferGet with id XferFullPathName to get the selections
 path.  Note: If possible use msgBrowserSelection with flat locators to 
 avoid duplicate volume name confusion.
*/
#define msgBrowserSelectionPath     MakeMsg(clsBrowser, 109)

typedef struct {
    CHAR    path[fsMaxPathLength];
} BROWSER_PATH, *P_BROWSER_PATH;


/****************************************************************************
 msgBrowserSelection        takes P_FS_FLAT_LOCATOR, returns STATUS
    Passes back the flat locator of the selection.

 Also responds to msgXferGet with id XferFlatLocator to get the selections
 path.
*/
#define msgBrowserSelection     MakeMsg(clsBrowser, 79)


/****************************************************************************
 msgBrowserSelectionUUID    takes P_UUID, returns STATUS
    Passes back the UUID of the selection.
*/
#define msgBrowserSelectionUUID     MakeMsg(clsBrowser,117)


/****************************************************************************
 msgBrowserSelectionDir    takes P_FS_FLAT_LOCATOR, returns STATUS
    Passes back the flat locator of the directory the selection is in.
*/
#define msgBrowserSelectionDir      MakeMsg(clsBrowser, 110)


/****************************************************************************
 msgBrowserSelectionName    takes P_CHAR, returns STATUS
    Returns the name of the selection.

 Also responds to msgXferGet with id XferName to get the selections name
*/
#define msgBrowserSelectionName MakeMsg(clsBrowser, 111)


/****************************************************************************
 msgBrowserSelectionOn      takes nothing, returns STATUS
    Notifies client when a selection is made inside the browser.    
*/
#define msgBrowserSelectionOn   MakeMsg(clsBrowser,112)


/****************************************************************************
 msgBrowserSelectionOff     takes nothing, returns STATUS
    Notifies client when selection is yielded by the browser.
*/
#define msgBrowserSelectionOff  MakeMsg(clsBrowser,113)


/****************************************************************************
 msgBrowserBookmark         takes P_BROWSER_BOOKMARK, returns STATUS
    Notifies client that the bookmark specified by locator has toggled.
*/
#define msgBrowserBookmark      MakeMsg(clsBrowser,107)


typedef struct {
    FS_LOCATOR  loc;
} BROWSER_BOOKMARK, *P_BROWSER_BOOKMARK;


/****************************************************************************
 msgBrowserCreateDoc    takes P_BROWSER_CREATE_DOC, returns STATUS
    Creates a directory.

 The directory is created at the selection if there is one.  If not, the
 directory is created at the top level shown.  DiskView only.
*/
#define msgBrowserCreateDoc     MakeMsg(clsBrowser,152)

typedef struct {
    CLASS       docClass;
    P_CHAR      pName;
    BOOLEAN     atSelection;
    XY32        xy;
} BROWSER_CREATE_DOC, *P_BROWSER_CREATE_DOC;


/****************************************************************************
 msgBrowserGetBrowWin   takes pObject, returns STATUS
    Passes back the browser's internal display window.

 The browser's internal display window is the selected object for any
 selection based operations.
*/
#define msgBrowserGetBrowWin    MakeMsg(clsBrowser,149)


/****************************************************************************
 msgBrowserGesture      takes P_BROWSER_GESTURE, returns STATUS
    Sends to self gesture and which file it landed on.     

 Allows subclasses to respond to gestures targeted at browser items.
 If the status returned by the subclass is >= stsOK the gesture will NOT be
 sent to browser superclass.  So subclasses should ignore this message or
 return stsOK to signify it has been handled.    
*/
#define msgBrowserGesture       MakeMsg(clsBrowser,59)

typedef struct {
    MESSAGE             gesture;    // Gesture that occurred.
    P_FS_FLAT_LOCATOR   pFlat;      // Item on which to apply the gesture.
    P_GWIN_GESTURE      pGest;      // Original gesture struct.
    TAG                 columnTag;  // Tag of column on which to apply the gesture.
                                    // 0  if not on a column.
    U32                 info;       // Internal browser information.
    U32                 spare[2];   // Spare: reserved.
} BROWSER_GESTURE, *P_BROWSER_GESTURE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *    Private                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
    msgBrowserGetThisApp       takes P_OBJECT, returns STATUS
    Returns the application associated with this instance of clsBrowser.
*/

#define msgBrowserGetThisApp            MakeMsg(clsBrowser, 160)

/****************************************************************************
    msgBrowserSetThisApp       takes OBJECT, returns STATUS
    Sets the application associated with this instance of clsBrowser.
*/

#define msgBrowserSetThisApp            MakeMsg(clsBrowser, 161)

/****************************************************************************
 msgBrowserSetViewClass  takes P_BROWSER_SET_VIEW_CLASS returns STATUS
    Sets the class of the kind of thing that the browser should be displaying
*/
#define msgBrowserSetViewClass          MakeMsg(clsBrowser, 162)

typedef struct BROWSER_SET_VIEW_CLASS {
    UID                 viewCls;        // Browser view class.
    U32                 spare1;
    U32                 spare2;
} BROWSER_SET_VIEW_CLASS, *P_BROWSER_SET_VIEW_CLASS;

#endif // BROWSER_INCLUDED


