/****************************************************************************
 File: bitmap.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.8  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 11:57:14  $

 This file contains the API for clsBitmap.

 clsBitmap inherits from clsObject.
 Support class for clsIcon (see icon.h). Serves as data object for 
 the Bitmap Editor. Based on cached images (see sysgraf.h).

 clsBitmap takes a sampled image description, and optionally a mask,
 and a hotspot. It will file this description. It also provides 
 messages to modify the bitmap appearance. The Bitmap Editor treats 
 bitmaps as data objects. It creates a bitmap, files it, and will export 
 it as resource file. This resource file can be processed further by
 SDK utility programs (see resappnd).

 A bitmap will prepare an argument structure for use by msgDcCacheImage
 so that the sampled image data in the bitmap can be converted to a cached
 image for quick rendering. See msgBitmapCacheImageDefaults.

****************************************************************************/

#ifndef BITMAP_INCLUDED
#define BITMAP_INCLUDED

#ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Typedefs, #defines, and Status Values                                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define bitmapResId             MakeTag(clsBitmap, 1)

#define bmEncodeNone            0               // no data
#define bmEncodeRunLength       1               // run length encoded
#define bmEncode1BPS            2               // 1 bit  per sample
#define bmEncode2BPS            3               // 2 bits per sample
#define bmEncode4BPS            4               // 4 bits per sample
#define bmEncode8BPS            5               // 8 bits per sample
#define bmEncode16BPS           6               // unused (reserved)
#define bmEncode24BPS           7               // unused (reserved)

#define bmMono                  0               // default
#define bmColorMap              1               // Not Working (reserved)
#define bmDirectColor           2               // Not Working (reserved)


typedef struct BITMAP_STYLE 
{
  U16  pixEncoding     : 4,
       maskEncoding    : 4,
       colorEncoding   : 3,
       version         : 5;                     // currently 0

} BITMAP_STYLE, *P_BITMAP_STYLE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Messages                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew takes P_BITMAP_NEW, returns STATUS
    category: class message
    Creates a bitmap.
*/

typedef struct BITMAP_NEW_ONLY 
{
  BITMAP_STYLE  style;          // overall style
  SIZE16        size;           // # of source samples
  P_U8          pPixels;        // actual samples
  P_U8          pMask;          // mask (must be bmEncode1BPS) or pNull
  XY16          hotSpot;        // lower-left corner relative hot spot
  U32           spare1;
  U32           spare2;

} BITMAP_NEW_ONLY, *P_BITMAP_NEW_ONLY,
  BITMAP_METRICS, *P_BITMAP_METRICS;

#define bitMapNewFields         \
        objectNewFields         \
        BITMAP_NEW_ONLY bitmap;

typedef struct BITMAP_NEW 
{
  bitMapNewFields

} BITMAP_NEW, *P_BITMAP_NEW;

/****************************************************************************
 msgNewDefaults takes P_BITMAP_NEW, returns STATUS
    category: class message
    Initializes the BITMAP_NEW structure to default values.
//{
 bitmap.style.pixEncoding   = bmEncode8BPS;
 bitmap.style.maskEncoding  = bmEncode1BPS;
 bitmap.style.colorEncoding = bmMono;
 bitmap.style.version       = 0;
 bitmap.size.w              = 0;
 bitmap.size.h              = 0;
 bitmap.pPixels             = pNull;
 bitmap.pMask               = pNull;
 bitmap.hotSpot.x           = 0;
 bitmap.hotSpot.y           = 0;
//}

*/

/****************************************************************************
 msgBitmapGetMetrics takes P_BITMAP_GET_METRICS, returns STATUS
    Gets bitmap metrics.
*/
#define msgBitmapGetMetrics             MakeMsg(clsBitmap, 0)

/****************************************************************************
 msgBitmapSetMetrics takes P_BITMAP_METRICS, returns STATUS
    Sets bitmap metrics.
*/
#define msgBitmapSetMetrics             MakeMsg(clsBitmap, 1)

/****************************************************************************
 msgBitmapSetSize takes P_SIZE16, returns STATUS
    Sets bitmap size, resizing heap block if necessary.
*/
#define msgBitmapSetSize                MakeMsg(clsBitmap, 2)

/****************************************************************************
 msgBitmapInvert takes nothing, returns STATUS
    Inverts the colors of the bitmap.
*/
#define msgBitmapInvert                 MakeMsg(clsBitmap, 3)

/****************************************************************************
 msgBitmapLighten takes nothing, returns STATUS
    Lightens the colors of the bitmap by 1/4.
*/
#define msgBitmapLighten                MakeMsg(clsBitmap, 4)

/****************************************************************************
 msgBitmapFill takes RGB value, returns STATUS
    Fills bitmap pixels with RGB value leaving mask alone.
*/
#define msgBitmapFill                   MakeMsg(clsBitmap, 6)

/****************************************************************************
 msgBitmapCacheImageDefaults takes P_SYSDC_CACHE_IMAGE, returns STATUS
    Prepares argument structure for msgDcCacheImage.

 After sending this message to the bitmap, pArgs is ready to be
 sent to a DC via using msgDcCacheImage (see sysgraf.h).
*/
#define msgBitmapCacheImageDefaults     MakeMsg(clsBitmap, 43)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Messages sent to observers                                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgBitmapPixChange takes P_BITMAP_PIX_CHANGE, returns STATUS
    category: observer notification
    Sent to observing objects if a pixel is dirty.
*/
#define msgBitmapPixChange              MsgNoError(MakeMsg(clsBitmap, 5))

typedef struct BITMAP_PIX_CHANGE 
{
  XY16              pix;
  OBJECT            sourceObject;
  P_BITMAP_METRICS  pBitmap;

} BITMAP_PIX_CHANGE, *P_BITMAP_PIX_CHANGE;

/****************************************************************************
 msgBitmapChange takes nothing, returns STATUS
    category: observer notification
    Sent to observing objects if bitmap has changed.
*/
#define msgBitmapChange                 MsgNoError(MakeMsg(clsBitmap, 10))

/****************************************************************************
 msgBitmapMaskChange takes nothing, returns STATUS
    category: observer notification
    Sent to observing objects if bitmap's mask has changed.
*/
#define msgBitmapMaskChange             MsgNoError(MakeMsg(clsBitmap, 11))

#endif  // BITMAP_INCLUDED



