/****************************************************************************
 File: abmgr.h    

 (C) Copyright 1992 by Go Corporation, All Rights Reserved.

 $Revision:   1.6  $
   $Author:   pma  $
     $Date:   31 Mar 1992 17:44:12  $

 This file contains the API definition for theAddressBookMgr.
	 
 theAddressBookMgr is an instance of a private class.  It is the only
 instance of that class in the system.

 theAddressBookManager is a well known object that handles registration
 of and access to "system" address books. Registered address books are
 primarily responsible for managing the storage and retrieval of service
 specific addressing information. 

 Registered address books adhere to the protocol defined in addrbook.h. 
 Information about its functionality and use can be found there.

 theAddressBookMgr provides the facility to help other applications to provide
 a UI for picking the system address book.  When an application wants to 
 provide this pick list as an option card, it just needs to pass on 
 msgOptionAddCards before it calls its ancestor to theAddressBookMgr.  
 TheAddressBookMgr will do the rest.
****************************************************************************/

#ifndef ABMGR_INCLUDED
#define ABMGR_INCLUDED

#include <uuid.h>
#include <go.h>

#define tagABMgrABList				MakeTag(theAddressBookMgr, 1)
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							  Status Codes								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define stsABMgrAddrBookNotActive		MakeStatus(theAddressBookMgr, 1)
#define stsABMgrAddrBookOpen			MakeStatus(theAddressBookMgr, 2)
#define stsABMgrNoneActive				MakeStatus(theAddressBookMgr, 3)
#define stsABMgrAddrBookNotRegistered	MakeStatus(theAddressBookMgr, 4)
#define stsABMgrNoOpenAddrBook			MakeStatus(theAddressBookMgr, 5)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Common #defines and typedefs                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

Enum16(AB_MGR_ID_TYPE) {
	abMgrApplication	= 0,	    // Client is an application
	abMgrObject			= 1,	    // Client is a service/data object
	abMgrNone			= 2,	    // abmgr internal use only
};

typedef struct AB_MGR_ID {
    CHAR            name[nameBufLength]; // Name of the address book
	AB_MGR_ID_TYPE	type;	             // Address book object type
	union {
		OBJECT	    uid;				// UID of the service/object
		UUID		uuid;				// UUID of the application working dir
	} value;
} AB_MGR_ID, *P_AB_MGR_ID;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						 Messages					                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgABMgrRegister   takes P_AB_MGR_ID, returns STATUS
    Registers an application or a service as an address book instance.

 When an instance of an address book is registered with theAddressBookMgr,
 it can later be selected as "the system address book".

 Address books send this message to register themselves with 
 theAddressBookMgr.  Each instance of each address book should be registered
 with theAddressBookMgr.  If an address book application is a subclass of
 clsAddrBookApplication(see addrbook.h), then theAddressBookMgr automatically
 registers a newly created instance of this class.
	 
 If an address book is an application, theAddressBookMgr will automatically
 re-registers the app on warm boot.  If an address book is a service,
 however, it would have to re-register itself after a warm boot.
*/
#define msgABMgrRegister			MakeMsg(theAddressBookMgr, 1)


/****************************************************************************
 msgABMgrUnregister takes P_AB_MGR_ID, returns STATUS
    Unregisters an application or a service as an address book instance.
 
 Address book send this message to theAddressBookMgr to unregister 
 themselves.  This is usually done when an application instance is deleted,
 or when a service is de-installed.  If an address book application is a
 subclass of clsAddrBookApplication(see addrbook.h), then theAddressBookMgr 
 automatically unregisters a deleted instance of this class.
*/
#define msgABMgrUnregister			MakeMsg(theAddressBookMgr, 2)


/****************************************************************************
 msgABMgrOpen   takes nothing, returns STATUS
    Used by address book clients to begin access to address books.

 Address book clients send msgABMgrOpen to theAddressBookMgr.
 If the system address book is an application, then theAddressBookMgr
 activates the application.  If the system address book is a service, then
 theAddressBookMgr binds to the service(msgSMBind)

 Clients must call msgABMgrClose when they're finished with the 
 address book.
	 
 On warm boots, theAddressBookMgr requires that clients reopen the
 system address book.
*/
#define msgABMgrOpen				MakeMsg(theAddressBookMgr, 3)


/****************************************************************************
 msgABMgrClose	takes nothing, returns STATUS
    Used by address book clients to end access to address books.

 If the system address book is an application, then theAddressBookMgr
 deactivates the application.  If the system address book is a service, then
 theAddressBookMgr binds to the service(msgSMUnbind).

 The address book is reference counted, so all msgABMgrOpen calls
 must be followed by an msgABMgrClose. 
*/
#define	msgABMgrClose				MakeMsg(theAddressBookMgr, 4)



typedef struct {
    BOOLEAN             activated;
    AB_MGR_ID           addressBook;
} AB_MGR_LIST, *P_AB_MGR_LIST;


/****************************************************************************
 msgABMgrList   takes P_LIST, returns STATUS
    Creates a list of currently registered address book in pArgs.  

 Every time msgABMgrList is called, a new list object is created.  
 It is up to the client to call msgListFree(not msgDestroy) to destroy the
 list and the items in the list.  Set the free mode to listFreeItemsAsData.

 Each element of the list is a P_AB_MGR_LIST.  
*/
#define msgABMgrList                MakeMsg(theAddressBookMgr,5)


/****************************************************************************
 msgABMgrActivate	    takes P_AB_MGR_ID, returns STATUS
	Make a registered address book the system address book.

 In the current implementation only one address book can be the system
 address book at a time.  If there is currently a system address book, that
 address book is deactivated first.

 Clients that are applications set the type field to 'application'
 and set the value field to the UUID of their application working directory.
 Clients that are services or data objects set the type field to
 'object' and set the value field to their object UID. 

 Return Value
    stsABMgrAddrBookOpen:  The current system address book is currently
						   open, therefore it can not be deactivated
	stsABMgrAddrBookNotRegistered:  The address book identified by pArgs is
									not a registered address book.
*/
#define	msgABMgrActivate				MakeMsg(theAddressBookMgr, 6)


/****************************************************************************
 msgABMgrDeactivate	    takes P_AB_MGR_ID, returns STATUS
    Deactivates the current system address book.

 Return Value
	stsABMgrAddrBookOpen:  The current system address book is currently
						   open, therefore it can not be deactivated
*/
#define	msgABMgrDeactivate				MakeMsg(theAddressBookMgr, 7)


/****************************************************************************
 msgABMgrIsActive       takes P_AB_MGR_ID, returns STATUS  
	Indicates if the specified AB_MGR_ID is currently set.  


 Return Value
	stsOK:					    Specified id is activated.
	stsABMgrNotActive:          Specified id is not activated, but
                                something is active.
    stsABMgrNoneActive:         No address book is currently active.
*/
#define	msgABMgrIsActive				MakeMsg(theAddressBookMgr, 8)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Observer Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgABMgrChanged	takes P_AB_MGR_NOTIFY, returns STATUS
	Sent to observers of theAddressBookMgr when the system address book
	changes.

 pArgs->activated is set to TRUE if pArgs->addressBook is made
 the system address book, and to FALSE if pArgs->addressBook has been
 deactivated as the system address book.
*/
#define msgABMgrChanged			MakeMsg(clsAddressBook, 9)


Enum16(AB_MGR_CHANGE_TYPE) {
	abMgrRegister		= 0,			// an ab has been registered
	abMgrUnregister		= 1,
	abMgrActivated		= 2,
	abMgrDeactivated	= 3,
	abMgrOpened			= 4,
	abMgrClosed			= 5,
};

typedef struct {
    AB_MGR_CHANGE_TYPE  type;
    AB_MGR_ID           addressBook;
} AB_MGR_NOTIFY, *P_AB_MGR_NOTIFY;

#endif

