/****************************************************************************
 File: adderapp.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.9  $
   $Author:   kcatlin  $
     $Date:   18 Mar 1992 08:37:56  $

 This file contains the implementation for clsAdderApp.
****************************************************************************/

#ifndef DEBUG_INCLUDED
#include <debug.h>
#endif

#ifndef _STRING_H_INCLUDED
#include <string.h>
#endif

#ifndef _CTYPE_H_INCLUDED
#include <ctype.h>
#endif

#ifndef RESFILE_INCLUDED
#include <resfile.h>
#endif

#ifndef WIN_INCLUDED
#include <win.h>
#endif

#ifndef XGESTURE_INCLUDED
#include <xgesture.h>
#endif

#ifndef XLATE_INCLUDED
#include <xlate.h>
#endif

#ifndef XLFILTER_INCLUDED
#include <xlfilter.h>           
#endif

#ifndef XTEMPLT_INCLUDED
#include <xtemplt.h>
#endif

#ifndef INSERT_INCLUDED
#include <insert.h>
#endif

#ifndef GWIN_INCLUDED
#include <gwin.h>
#endif

#ifndef APP_INCLUDED
#include <app.h>
#endif

#ifndef APPMGR_INCLUDED
#include <appmgr.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

#ifndef TK_INCLUDED
#include <tk.h>
#endif

#ifndef _STDIO_H_INCLUDED
#include <stdio.h>
#endif

#ifndef OS_INCLUDED
#include <os.h>         // for allocating memory
#endif

#ifndef ADDEREVL_INCLUDED
#include <adderevl.h>
#endif

#include <methods.h>


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Defines, Types, Globals, Etc	 			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

const S16 topMargin			  = 5;	// absolute logical units
const S16 sideMargin  		  = 10;	// absolute logical units

#define clsAdderApp				MakeWKN(3511,1,wknGlobal)

#define MAX_DISP 200			// maximum number of characters to display


// instance data for clsAdderApp
typedef struct ADDER_APP_INST {
	OBJECT	   	evalObj;	// expression evaluator
	OBJECT	   	iPad;		// insertion pad
	OBJECT	   	adderWin;	// our window
	OBJECT		label;		// display label

} ADDER_APP_INST, *P_ADDER_APP_INST;

typedef P_ADDER_APP_INST	 *PP_ADDER_APP_INST; 

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Utility Routines				 			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	RemoveNewLines

	This removes newline characters from the expression. Returns stsOK.
****************************************************************************/
STATUS LOCAL 
RemoveNewLines(
	P_STRING	expression)
{
	P_STRING	index;

	index = expression;
	while (*expression) {
		if (*expression != '\n') {
			*index++ = *expression;
		}
		expression++;
	} 
	*index++ = '\0';

	return stsOK;

} /* RemoveNewLines */

/****************************************************************************
	CreateLabel

	Create a label object.
****************************************************************************/
STATUS LOCAL 
CreateLabel (
	P_ADDER_APP_INST 	pInst)
{
	STATUS		s;
	LABEL_NEW	ln;

	//
	// Create a label object, and initialize it to be centered
	//
	ObjCallRet(msgNewDefaults, clsLabel, &ln, s);
	ln.border.style.edge = bsEdgeAll;
	ln.label.style.xAlignment	= lsAlignCenter;
	ln.label.style.yAlignment	= lsAlignCenter;
	ln.label.style.scaleUnits	= bsUnitsLayout;
	ln.label.pString			= "";
	ObjCallRet(msgNew, clsLabel, &ln, s);

	//
	// Insert the label into our adderWin window
	//
	pInst->label = ln.object.uid;
	ln.win.parent = pInst->adderWin;
	ObjCallRet(msgWinInsert, ln.object.uid, &ln.win, s);

	return stsOK;

} /* CreateLabel */


/****************************************************************************
	CreateInsertWindow

	Create the insertion pad.
****************************************************************************/
STATUS LOCAL
CreateInsertWindow (
	P_ADDER_APP_INST 	pInst, 
	OBJECT 				self)
{
	STATUS			s;
	WIN_METRICS		wm;
	IP_NEW			ipNew;
	P_UNKNOWN		pNewTemplate;
	XLATE_NEW		xNewTrans;
	U16				xlateFlags;
	XTM_ARGS		xtmArgs;

	//
	// Create an insertion pad, which is the standard mechanism
	// for accepting handwritten input. 
	//
	ObjectCall(msgNewDefaults, clsIP, &ipNew);

	// Do not show the resize button.
	ipNew.border.style.resize 	= bsResizeNone; 

	// Translate when the user lifts the pen out of proximity
	ipNew.ip.style.buttonType 	= ipsProxButton;
	ipNew.ip.style.embeddedLook = true;
	// set the listener field for notifications
	ipNew.ip.client 			= self;

	//
	// Create a translator for the insertion pad
	//
	ObjectCall(msgNewDefaults, clsXText, &xNewTrans);

	//
	// Create a template for the insertion pad
	//
	xtmArgs.xtmType 	= xtmTypeCharList;
	xtmArgs.xtmMode 	= 0; 					// no special modes
	xtmArgs.pXtmData 	= "0123456789+-.";		// ascii template
	StsRet(XTemplateCompile(&xtmArgs, osProcessHeapId, &pNewTemplate), s);
	xNewTrans.xlate.pTemplate = pNewTemplate;

	//
	// The handwriting engine is geared primarily towards text
	// translation. We can improve numeric translation by 
	// disabling these context assumptions:
	//
	// xltAlphaNumericEnable - enables character recognition that is
	//		geared towards text-processing (e.g., most characters
	//		are assumed to be letters, and numbers are separated from
	//		letters by spaces.)
	// xltPunctuationEnable - enables punctuation rules that are geared
	//		towards text-processing (e.g., periods are always followed
	//		by a space).
	// xltVerticalEnable - enables rules that help recognize a character
	//		based on its vertical orientation (e.g., "t" versus "+").
	// 
	// By disabling these flags, we get far fewer unrecognized characters
	// in our numeric expressions.
	// 
	xNewTrans.xlate.hwxFlags &= 
			~(xltAlphaNumericEnable | xltPunctuationEnable | xltVerticalEnable);

	ObjCallRet(msgNew, clsXText, &xNewTrans, s);

	ipNew.ip.xlate.translator = xNewTrans.object.uid;

	// give our template veto power	during the translation
	ObjCallRet(msgXlateGetFlags, xNewTrans.object.uid, &xlateFlags, s);
	xlateFlags |= xTemplateVeto | xltSpaceDisable;
	ObjCallRet(msgXlateSetFlags, xNewTrans.object.uid, (P_ARGS)xlateFlags, s);
  
	ObjCallRet(msgNew, clsIP, &ipNew, s); 
	pInst->iPad = ipNew.object.uid;
	
	//
	// Insert the insertion pad into our adderWin window
	//
	wm.parent 	= pInst->adderWin;
	wm.options 	= wsPosTop;
	ObjCallRet(msgWinInsert, ipNew.object.uid, &wm, s);

	return stsOK;

} /* CreateInsertWindow */

/****************************************************************************
	LayoutWindows

	Layout the label and the insertion pad in a custom layout.
****************************************************************************/
STATUS LOCAL 
LayoutWindows (
	P_ADDER_APP_INST pInst)
{
	CSTM_LAYOUT_CHILD_SPEC 	clcs;
	STATUS					s;

	//	Layout the label first.
	CstmLayoutSpecInit(&clcs.metrics);

	clcs.child					= pInst->label;
	clcs.metrics.x.constraint	= ClAlign(clCenterEdge, clSameAs, clCenterEdge);
	clcs.metrics.y.constraint	= ClAlign(clMaxEdge, clSameAs, clMaxEdge);
	clcs.metrics.y.value		= -topMargin;
	clcs.metrics.w.constraint	= clSameAs;
	clcs.metrics.w.value 		= -(2*sideMargin);
	clcs.metrics.h.constraint	= clAsIs;

	ObjCallRet(msgCstmLayoutSetChildSpec, pInst->adderWin, &clcs, s);
	
	// Layout the insertion pad window
	CstmLayoutSpecInit(&clcs.metrics);

	clcs.child					= pInst->iPad;
	clcs.metrics.x.relWin		= pInst->label;
	clcs.metrics.x.constraint	= clSameAs;
	clcs.metrics.y.constraint	= ClAlign(clMinEdge, clSameAs, clMinEdge);
	clcs.metrics.y.value		= topMargin;
	clcs.metrics.w.relWin		= pInst->label;
	clcs.metrics.w.constraint	= clSameAs;
	clcs.metrics.h.relWin		= pInst->label;
	clcs.metrics.h.constraint	= ClExtend(clSameAs, clMinEdge);
	clcs.metrics.h.value		= -5; // space between inner windows
	ObjCallRet(msgCstmLayoutSetChildSpec, pInst->adderWin, &clcs, s);

	return stsOK;

} /* LayoutWindows */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Message Handlers  							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	AdderAppFree
	
	Destroy the calculator instance data and expression calculator object.
****************************************************************************/

MSG_HANDLER AdderAppFree (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_ARGS				pArgs,
	const CONTEXT				ctx,
	const PP_ADDER_APP_INST		pData)

{
	P_ADDER_APP_INST			pInst;

	pInst = *pData;

	// Free the expr object
	ObjCallWarn(msgFree, pInst->evalObj, Nil(P_ARGS));

	// Free the instance data
	StsWarn(OSHeapBlockFree(pInst));
                                                     
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* AdderAppFree */


/****************************************************************************
	AdderAppInit
	
	Respond to msgInit. Create the expression evaluator object.
****************************************************************************/

MSG_HANDLER AdderAppInit (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_APP_NEW				pArgs,
	const CONTEXT				ctx,
	const PP_ADDER_APP_INST		pData)

{
	OBJECT_NEW			on;
	STATUS	   			s;
	P_ADDER_APP_INST	pInst;

	pInst = *pData;

	// Allocate the application's instance data
	OSHeapBlockAlloc(osProcessHeapId, SizeOf(ADDER_APP_INST), &pInst);

	//
	// Create the expression evaluator object
	//
	ObjCallRet(msgNewDefaults, clsAdderEvaluator, &on, s);
	ObjCallRet(msgNew, clsAdderEvaluator, &on, s);

	pInst->evalObj = on.uid;

	StsJmp( ObjectWrite(self, ctx, &pInst), s, error);

	return stsOK;

error:
	(void)OSHeapBlockFree(pInst);

	return s;
	MsgHandlerParametersNoWarning;
}  /* AdderAppInit */

	
/****************************************************************************
	AdderAppOpen
	
	Respond to msgAppOpen.
****************************************************************************/

MSG_HANDLER AdderAppOpen (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_APP_OPEN			pArgs,
	const CONTEXT				ctx,
	const PP_ADDER_APP_INST	pData)

{
	FRAME_METRICS	fm;
	APP_METRICS		am;
	STATUS			s;
	P_ADDER_APP_INST	pInst;
	CSTM_LAYOUT_NEW	cln;

	// Create the adder window (a custom layout with two children:
	// an insertion pad and a label)

	ObjCallJmp(msgNewDefaults, clsCustomLayout, &cln, s, error1);

	cln.win.flags.style				   &= ~wsClipChildren;
	cln.border.style.edge				= bsEdgeNone;
	cln.border.style.backgroundInk		= bsInkGray33;
	cln.border.style.leftMargin			= bsMarginSmall;
	cln.border.style.rightMargin  		= bsMarginSmall;
	cln.border.style.topMargin			= bsMarginSmall;
	cln.border.style.bottomMargin		= bsMarginSmall;

	ObjCallJmp(msgNew, clsCustomLayout, &cln, s, error1);

	pInst = *pData;
	pInst->adderWin = cln.object.uid;
	
	StsJmp( CreateLabel(pInst), s, error2);
	StsJmp( CreateInsertWindow(pInst, self), s, error2);
	StsJmp( LayoutWindows(pInst), s, error2);

	// Get the main application window.
	ObjCallRet(msgAppGetMetrics, self, &am, s);

	// Insert the custom layout window and set the frame style.
	ObjCallRet(msgFrameGetMetrics, am.mainWin, &fm, s);
	fm.clientWin = pInst->adderWin;
	ObjCallRet(msgFrameSetMetrics, am.mainWin, &fm, s);

	// Pass this message to our ancestor.
	pArgs->childAppParentWin = pInst->adderWin;

	ObjCallAncestorRet(msgAppOpen, self, pArgs, ctx, s);

	return stsOK;

error2:
	Dbg(Debugf("Cannot create adders's subwindows");)
	(void)ObjectCall(msgFree, pInst->adderWin, Nil(P_ARGS));

	return s;

error1:
	Dbg(Debugf("Cannot create the main expression calculator window");)

	return s;
	MsgHandlerParametersNoWarning;
}  /* AdderAppOpen */


/****************************************************************************
	AdderAppClose
	
	Respond to msgAppClose.
****************************************************************************/

MSG_HANDLER AdderAppClose (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_ARGS				pArgs,
	const CONTEXT				ctx,
	const PP_ADDER_APP_INST		pData)

{
	FRAME_METRICS		fm;
	APP_METRICS	   		am;
	STATUS				s;
	P_ADDER_APP_INST	pInst;

	pInst = *pData;

	// Get the application metrics
	ObjCallRet(msgAppGetMetrics, self, &am, s);

	// Deinstall the client window (view)
	ObjCallRet(msgFrameGetMetrics, am.mainWin, &fm, s);

	fm.clientWin = Nil(WIN);

	ObjCallRet(msgFrameSetMetrics, am.mainWin, &fm, s);

	// Destroy our window, label, and insertion pad
	ObjCallWarn(msgFree, pInst->label, Nil(P_ARGS));
	ObjCallWarn(msgFree, pInst->iPad, Nil(P_ARGS));
	ObjCallWarn(msgFree, pInst->adderWin, Nil(P_ARGS));

	return stsOK;
	MsgHandlerParametersNoWarning;
} /* AdderAppClose */


/****************************************************************************
  GetInsertPadData

  Method for processing the translated data reported by an Insertion Pad.
****************************************************************************/
MSG_HANDLER GetInsertPadData (
	const MESSAGE				msg,
	const OBJECT				self,
	const OBJECT				pArgs,
	const CONTEXT				ctx,
	const PP_ADDER_APP_INST		pData)

{
	IP_STRING			ipStr;
	STATUS				s;
	char				display[MAX_DISP];
	EVAL_FRAME			ex;
	P_ADDER_APP_INST	pInst;

	pInst = *pData;

	//
	// This message is sent to our application whenever an insertion pad
	// has data available, and the insertion pad's client is the app.
	// If the insertion pad passed in is not our instance's expression
	// insertion pad (which happens when the user circles the name of 
	// the app, and tries to change it), we need to pass this message on
	// to our ancestor.
	//
	if (pArgs != pInst->iPad) {
		return ObjectCallAncestor(msg, self, pArgs, ctx);
	}

	//
	// Get the text from our instance's insertion pad into ex.expresssion...
	//
	ipStr.len = SizeOf(ex.expression);
	ipStr.pString = ex.expression;
	ObjCallRet(msgIPGetXlateString, pInst->iPad, &ipStr, s);
	StsRet(RemoveNewLines (ex.expression), s);
   
	//
	// ...and evaluate the expression.
	//
	ObjCallRet(msgAdderEvaluatorEval, pInst->evalObj, &ex, s);

	Dbg(Debugf("Returned from msgAdderEvaluatorEval: ex.badExpr = %s,"
			"  value = %g  valueStr = <%s>", ex.badExpr? "true":"false",
			ex.value, ex.valueStr);)

	//
	// Construct the string to display
	//
	sprintf(display,"%s = %s", ex.expression, ex.valueStr);

	//
	// Display the string
	//
	ObjCallRet(msgLabelSetString, pInst->label, display, s);

	return stsOK;
	MsgHandlerParametersNoWarning;
}  /* GetInsertPadData */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          Installation	   					   		   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	ClsAdderAppInit
	
	Install the application.
****************************************************************************/

STATUS  ClsAdderAppInit (void)
{
	APP_MGR_NEW	new;
	STATUS		s;

	// Install the application class.
	ObjCallRet(msgNewDefaults, clsAppMgr, &new, s);

	new.object.uid	 	   	= clsAdderApp;
	new.object.key 		  	= (OBJ_KEY)clsAdderAppTable;
	new.cls.pMsg		   	= clsAdderAppTable;
	new.cls.ancestor	   	= clsApp;
	new.cls.size			= SizeOf(P_ADDER_APP_INST);
	new.cls.newArgsSize		= SizeOf(APP_NEW);

	strcpy(new.appMgr.defaultDocName, "Adder");
	strcpy(new.appMgr.company, "GO Corporation");
	// 213 (octal) is the "circle-c" copyright symbol
	new.appMgr.copyright 		= "\213 1991-1992 GO Corporation, All Rights Reserved.";
	new.appMgr.flags.stationery	= false;
	new.appMgr.flags.accessory	= true;

	ObjCallRet(msgNew, clsAppMgr, &new, s);

	return stsOK;

}  /* ClsAdderAppInit */


/****************************************************************************
	main
	
	Main application entry point.
****************************************************************************/

void CDECL
main(
	int			argc,
	char *		argv[],
	U16			processCount)
{
	if (processCount == 0) {
		
		StsWarn(ClsAdderAppInit());
		StsWarn(ClsAdderEvaluatorInit());

		AppMonitorMain(clsAdderApp, objNull);

	} else {

		// Start the application.
		AppMain();
	}

	Unused(argv); Unused(argc);
} /* main */
