/****************************************************************************
 File: xfer.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

  $Revision:   1.10  $
	  $Date:   25 Feb 1992 15:11:32  $

 This file contains the API definition for clsXfer and clsXferList.

 clsXfer inherits from clsStream.
 clsXfer defines the mechanisms used for transferring data between objects.

 clsXferList inherits from clsList.
 clsXferList is used by the transfer mechanism.

 Most clients of PenPoint's data transfer mechanism should use the
 procedural interfaces defined in this file.

 The functions described in this file are contained in XFER.LIB.
****************************************************************************/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Introduction								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Key Concepts ****/
/*
 This file describes some of PenPoint's support for transferring data.

 There are a few central concepts that underlie PenPoint's data transfer
 mechanism:

    -:	Sender and Receiver.  There are two sides to any data transfer.
        "Sender" refers to the object providing the data and "Receiver"
        refers to the object receiving the data.  These two objects can be
        in different processes, or in the same process.  They can even be
        the same object!

    -:	Two Stages.  Each PenPoint data transfer has two major stages.  In
        the first stage the Sender and Receiver engage in a simple protocol
        to determine if the data can be transferred, and if so what "type"
        the data has. In the second stage, the data is actually transferred
        using a protocol that is specific to the type agreed to during
        Stage 1.

    -:	Data Transfer Types.  A Sender and Receiver need to agree on a data
        transfer type that they both understand.  PenPoint defines several
        data transfer types and clients can define additional types.  See
        the section "Determining a Data Transfer Type" for more information.

    -:	Data Transfer Protocol.  Each data transfer type has an associated
        data transfer protocol.  Once a transfer type has been agreed upon,
        the Sender and Receiver engage in the type-specific protocol to
        actually move the data.  Note the same Data Transfer Protocol
        can be employed for multiple Data Transfer Types, but that each
        Data Transfer Type uses one and only one protocol.
*/


/**** Roadmap ****/
/*
 Typical Receivers use the following to determine the desired data transfer
 type.
	-:	XferMatch()

 Typical Senders respond to or use the following to provide a list of data
 transfer types.
	-:	msgXferList
	-:	XferAddIds()

 Typical Senders and Receivers who use data transfer types that use
 one-shot protocols use the following:
	-:	msgXferGet

 Senders and Receivers who use data transfer types that use stream-based
 protocols use the following:
	-:	msgXferStreamConnect
	-:	msgXferStreamWrite
	-:	msgXferStreamFreed
	-:	XferStreamConnect()
	-:	XferStreamAccept()
*/


/**** Relationship between Data Transfer and PenPoint's UI ****/
/*
 PenPoint's data transfer mechanism is intentionally independent of the user
 interface that might trigger a data transfer.  None of the interfaces
 defined in this file depend or define any part of a PenPoint application's
 user interface.

 However, the examples given in the commentary often use PenPoint's UI as
 an example of how a data transfer might be started.  The file sel.h
 describes PenPoint's Move and Copy operations in detail.

 During a Move or Copy operation, the Sender object is the owner of the
 selection.  The Receiver is the object upon which the move/copy icon was
 dropped and which receives msgSelMoveSelection or msgSelCopySelection as a
 result.  The Receiver sends msgSelOwner to theSelectionManager to get the
 Sender object and then engages in a data transfer with that object.
*/


/**** A Typical Scenario ****/
/*
 A typical data transfer session goes something like this:

    -:  The Receiver decides that it is the receiving end of a data
        transfer operation.  (For instance, the receiver might receive
        msgSelMoveSelection or msgSelCopySelection;  see sel.h.)

    -:	The Receiver figures out the UID of the Sender object.  (For
        instance, in the case of msgSelCopySelection or
        msgSelMoveSelection, the Sender object is the current selection
        owner, which can retrieved by sending msgSelOwner to
        theSelectionManager.)

    -:	The Receiver determines a mutually agreeable data transfer type
        using the utility routine XferMatch.  (See section "Determining a
        Common Data Transfer Type" for more detailed information about
        XferMatch and alternatives.)

    -:	The Sender and Receiver use the Data Transfer Protocol associated
        with the agreed-upon type to actually transfer the data.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Data Transfer Types								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 A data transfer type is represented by a TAG.

 Below is a list of PenPoint's predefined data transfer types and the data
 transfer protocol associated with each.  (The protocols are described in
 the next section.)
//{
	-:	xferString:			one-shot using XFER_FIXED_BUF
	-:	xferLongString:		one-shot using XFER_BUF
	-:	xferName:			one-shot using XFER_FIXED_BUF
	-:	xferFullPathName:	one-shot using XFER_FIXED_BUF
	-:	xferRTF:			stream
	-:	xferFlatLocator:	one-shot using XFER_FIXED_BUF
	-:	xferASCIIMetrics:	one-shot using XFER_ASCII_METRICS
	-:	xferScribbleObject:	one-shot using XFER_OBJECT
	-:	xferPicSegObject:	one-shot using XFER_OBJECT
//}

 In addition export.h and embedwin.h each define an additional data
 transfer type;  see these files for more information.
*/


/**** Determining a Common Data Transfer Type ****/
/*
 The Sender and Receiver must agree on a data transfer type.

 For instance, a note taking application might be willing to provide either
 xferScribbleObject or xferLongString data.  A text editor might be willing
 to consume xferString, xferLongString or xferRTF data.  Somehow the
 common data type (xferLongString) must be found and used.

 In PenPoint's data transfer mechanism, the Receiver is ultimately
 responsible for determining the mutually agreeable data transfer type.

 Typical Receivers can use a simply utility function, XferMatch, to compute
 the data transfer type.  Typical Senders must respond to msgXferList and
 add data transfer types to the provided list with the utility function
 XferAddIds.

 (Most clients don't need to know about the inner workings of XferMatch,
 but they are documented in the section "Details of XferMatch" for
 sophisticated clients or the merely curious.)
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Data Transfer Protocols							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Each data transfer type uses a specific data transfer protocol.

 There are three types of protocols:
	-:	one-shot protocols
	-:	stream-based protocols
	-:	client-defined protocols
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						One Shot Protocols								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Several data transfer types use a "One-Shot" protocol to transfer data.
 The protocols are called "one-shot" because all of the data can be
 transferred via a single message send.

 In all one-shot transfers, the Receiver uses ObjectSendUpdate to send
 msgXferGet to the Sender.  (ObjectSendUpdate must be used because the
 Sender and Receiver might be in different processes.)

 The type of the pArgs to msgXferGet depends on the data transfer type --
 the specific types are described in the section "Data Transfer Types."
 However, all legal pArgs to msgXferGet have one thing in common -- their
 first field is a data transfer type.  The Receiver must fill in at least
 this field before sending msgXferGet so that the Sender can tell which
 data transfer type is being used.

 The Sender responds to msgXferGet by filling in pArgs as necessary.  Some
 one-shot protocols require the Sender to allocate memory.  (For instance,
 the xferLongString data transfer type requires that the sender allocate
 memory for pArgs->pBuf field of an XFER_BUF.)

 Some one-shot protocols require that Sender allocate memory.  Any
 Sender-allocated memory must be allocated using OSHeapBlockAlloc and
 osProcessSharedHeapId.  The Receiver must free this memory with
 OSHeapBlockFree.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Stream-Based Protocols							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Stream-based protocols make use of a specialized stream that is
 implemented by clsXferStream.  clsXferStream adds the ability for two
 streams to be linked through an internal "pipe."

 Once a Receiver has decided to engage in a stream-based transfer (as
 described in the Section "A Typical Scenario" earlier), the steps in
 stream-based protocol are as follows:

	-:	The Receiver calls XferStreamConnect.

    -:	XferStreamConnect creates the Receiver's stream and then sends
        msgXferStreamConnect to the Sender.

    -:	In response to msgXferStreamConnect, the Sender calls
        XferStreamAccept.  Essentially all Senders of stream-based
        protocols should pass self as the "Producer" parameter when they
        call XferStreamAccept -- motivation and exceptions are described
        below.

    -:	XferStreamAccept properly creates the Sender's stream.

    -:	When control returns to it, the Receiver sends msgStreamReadData
        to its stream.

    -:	As a result of the Receiver's msgStreamReadData, the Sender receives
        msgXferStreamWrite.

    -:	In response to msgXferStreamWrite, the Sender writes data using
        msgStreamWriteData.  IMPORTANT NOTE: In order to avoid overflowing
        internal buffers, Senders should not write huge chunks of data in a
        single call.  Chunks than 64K won't work at all.  Memory is used
        more efficiently if chunk sizes don't exceed 10K, although things
        will work at any size up to 64K.

    -:	The last two steps can be repeated any number of times.  Eventually
        the Receiver gets stsEndOfData returned when sending
        msgStreamReadData.

    -:	The Receiver sends msgDestroy to its stream.

    -:	As a result of the Receiver's msgDestroy, the Sender receives
        msgXferStreamFree.

    -:	In response to msgXferStreamFree, the Sender sends msgDestroy to
        its stream.

 The Sender must be prepared to handle msgXferStreamFreed at any time.
 (In addition to normal termination, msgXferStreamFreed can indicate that
 the Receiver has died or otherwise has prematurely destroyed its side of
 the pipe.)
*/

/**** An Available Simplification ****/
/*
 Some Senders may know that they can contain only a limited amount of
 data.  Or they may find the obligation to respond to msgXferStreamWrite
 multiple times and record how much data was actually written each time to
 be unduly burdensome.

 These Senders can pass objNull as the "Producer" parameter in their call
 of XferStreamConnect.  As a result of doing this, msgXferStreamWrite will
 only be sent once, and in response these Senders should write all of their
 data in a single chunk.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Client-Defined Protocols						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Clients can define their own data transfer types.  There is a wide range
 of possibilities.  Clients can use msgXferGet that use a new pArgs
 type.  They can use streams but define structure on the data being
 streamed.  Or they define an entirely new transfer protocol.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Other Information								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Details of XferMatch ****/
/*
 Most clients can simply use XferMatch without understanding how it works,
 but it's described here for specialized clients or the curious.

	-:	XferMatch creates an instance of clsXferList

    -:	It then sends msgXferList to the passed-in Sender.

    -:	The Sender responds to msgXferList by adding items to the xfer list
        by calling XferAddIds.

    -:	XferMatch then scans the two lists (one passed in by the Receiver
        and one filled in by the Sender) using the utility function
		XferListSearch.

    -:	If no mutually acceptable data transfer type is found, XferMatch
        returns stsNoMatch.  Otherwise XferMatch returns stsOK and
		passes back the data transfer type in *pId.

    -:	Just before returning, XferMatch destroys the xferList.

 As an alternative to calling XferMatch, the Receiver could create the
 list, send msgXferList to the Sender, and then search the list for the
 best match (perhaps by using XferListSearch).

 Also, a sophisticated Sender can use msgListAddItem (rather than
 XferAddIds) to add the types to the list.
*/


/**** Creating Instances of clsXfer and clsXferList ****/
/*
 Normal clients of PenPoint's data transfer mechanism have no need to
 create instances of clsXfer and clsXferList.  Instances are created
 internally when using the data transfer functions.
*/

#ifndef XFER_INCLUDED
#define XFER_INCLUDED

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef STREAM_INCLUDED
#include <stream.h>
#endif

#ifndef STREAM_INCLUDED
#include <list.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Predefined Data Transfer Types  ****/
#define xferString			MakeTag(clsXfer, 1)		// XferGet (FixedBuf)
#define xferLongString		MakeTag(clsXfer, 2)		// XferGet (Buf)
#define xferName			MakeTag(clsXfer, 3)		// XferGet (FixedBuf)
#define xferFullPathName	MakeTag(clsXfer, 4)		// XferGet (FixedBuf)
#define xferRTF				MakeTag(clsXfer, 5)		// Stream
#define xferGoRTF			MakeTag(clsXfer, 6)		// Obsolete
#define xferFlatLocator	    MakeTag(clsXfer, 7)		// XferGet (FixedBuf)
#define xferASCIIMetrics	MakeTag(clsXfer, 10)	// XferGet (AsciiMetrics)
#define xferScribbleObject	MakeTag(clsXfer, 11)	// XferGet (Object)
#define xferPicSegObject	MakeTag(clsXfer, 12)	// XferGet (Object)


/****  XferList  ****/
/*
 * Normal clients need not create xferLists since the functions create and
 * destroy xferLists as needed.
 *
 * An xferList is a subclass of clsList that always allocates globally
 * accessible memory for the list.
*/
#define XFER_LIST_NEW 	LIST_NEW
#define P_XFER_LIST_NEW P_LIST_NEW

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								   Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgXferList	takes OBJECT, returns STATUS
	Ask Sender for its list of data transfer types.

 This message is sent to the Sender to have the Sender provide the list of
 data transfer types it can provide.

 The Sender can add types to the passed-in list using either msgListAddItem
 or XferListAddIds.

 If the Sender has a preferred data transfer type, it should put this type
 at the beginning of the list.  The Sender can use clsList messages to
 change the ordering of the list (see list.h).

 See Also
	msgListAddItems
	XferAddIds
*/
#define	msgXferList	MakeMsg(clsXfer, 1)


/****************************************************************************
 msgXferGet			takes lots-of-things, returns STATUS
	Sent by a Receiver to get "one-shot" data transfer information.

 msgXferGet is sent by the Receiver to the stream to retrieve the data
 being transferred.

 The type of this message's pArgs depends on the data transfer type being
 used.  In all cases, the first field of pArgs must be a data transfer type
 so that the Sender (when it receives this message) knows what type of data
 to supply and what the true type of pArgs really is.

 Return Value
	stsNoMatch:		specified data transfer type is inappropriate
*/
#define	msgXferGet	MakeMsg(clsXfer, 8)


/****  Variable Size Buffer  ****/
/*
 * This type is used as the pArgs of msgXferGet when the data transfer type
 * is xferLongString.  This type might also be used for client-defined
 * data transfers.
 *
 * [The rest of this description is complicated by the reversal of names.
 * The Receiver side of the data transfer operation sends msgXferGet and the
 * the Sender side of the data transfer operation receives msgXferGet.]
 *
 * The Receiver (which sends msgXferGet) must set the "id" field to
 * xferLongString.  The Sender receives msgXferGet and fills in the rest of
 * the structure.
 *
 * The Sender allocates the memory for pArgs->pBuf using OSHeapBlockAlloc
 * from osProcessSharedHeapId.  The Receiver must free this data using
 * OSHeapBlockFree.
 *
 * When used for xferLongString, the "pBuf" field is a null-terminated string
 * and the "len" field includes the terminating null character.  (In other
 * words, upon return, pArgs->len must equal (strlen(pArgs->pBuf) + 1).)
*/
typedef struct XFER_BUF {

	TAG			id;				// In: Data transfer type
	U32			data;			// Unused: future use
	U32			len;			// Out: Length of data in pBuf
	P_UNKNOWN	pBuf;			// Out: Buffer containing data
								// Must be SHARED and freed by caller
} XFER_BUF, *P_XFER_BUF;


/****  Fixed Size Buffer  ****/
/*
 * This type is used as the pArgs of msgXferGet when the data transfer type
 * is
 *	-:  xferString
 *	-:  xferName
 *	-:  xferFullPathName
 *	-:  xferFlatLocator
 *
 * [The rest of this description is complicated by the reversal of names.
 * The Receiver side of the data transfer operation sends msgXferGet and the
 * the Sender side of the data transfer operation receives msgXferGet.]
 *
 * The Receiver (which sends msgXferGet) must set the "id" field to one of
 * the data transfer types listed above.  The Sender receives msgXferGet and
 * fills in the rest of the structure.
*/
typedef struct XFER_FIXED_BUF {

	TAG			id;				// In: Data transfer type
	U32			data;			// Unused.  Reserved for future use
	U32			len;			// Out: Length of data in buf
	U8			buf[300];		// Out: Buffer containing data

} XFER_FIXED_BUF, *P_XFER_FIXED_BUF;


/****  Object Transfer  ****/
/*
 * This type is used as the pArgs of msgXferGet when the data transfer type
 * is:
 *	-:  xferScribbleObject
 *	-:  xferPicSegObject.
 *
 * [The rest of this description is complicated by the reversal of names.
 * The Receiver side of the data transfer operation sends msgXferGet and the
 * the Sender side of the data transfer operation receives msgXferGet.]
 *
 * The Receiver (which sends msgXferGet) must set the "id" field to one of
 * the data transfer types listed above, and must set the "receiver" field to
 * self (or some other object in the Receiver's task).  The Sender receives
 * msgXferGet and fills in the rest of the structure.
 *
 * The Sender makes a copy of the object using msgCopy and returns the uid of
 * the object in pArgs->uid.  When the Sender sends msgCopy, it should use
 * pArgs->receiver as the value of msgCopy's pArgs->requestor.
*/
typedef struct XFER_OBJECT {

	TAG			id;		 		// In: Data transfer type
	OBJECT		receiver;		// In: Receiver
	OBJECT		uid;	  		// Out: Uid of object
	CLASS		objClass; 		// Out: Class of object
	U32			reserved[4];	// Reserved for future use

} XFER_OBJECT, * P_XFER_OBJECT;


/****  ASCII Metrics  ****/
/*
 * This type is used as the pArgs of msgXferGet when the data transfer type
 * is xferASCIIMetrics.
 *
 * [The rest of this description is complicated by the reversal of names.
 * The Receiver side of the data transfer operation sends msgXferGet and the
 * the Sender side of the data transfer operation receives msgXferGet.]
 *
 * The Receiver (which sends msgXferGet) must set the "id" field to
 * xferASCIIMetrics.  The Sender receives msgXferGet and fills in the rest of
 * the structure.
 *
 * "ASCII Metrics" include information about the character data that can be
 * transferred from the Sender.  In some cases (e.g. PenPoint's text
 * component) it describes the selected text.
 *
 * (Essentially any Sender that can provide xferASCIIMetrics can also provide
 * some type of character data -- typically xferString, xferLongString or
 * xferRTF.)
 *
 * The "spare" field is always set to 0.  The "first" field is offset of the
 * first selected character.  The "length" field is the number of characters
 * in the selection.  The "level" field describes which lexical unit the
 * selection "contains."
*/
typedef struct XFER_ASCII_METRICS {

	TAG			id;				// In: data transfer type.
	U32			spare;			// Out: always 0
	U32			first;			// Out: character offset w.r.t. entire text
								//		maxU32 implies a bad request
	U32			length;			// Out: number of chars available to transfer
	U16			level;			// Out: 0: undefined or unknown, 1: chars,
								//		2: words, 3: sentences, 4: paragraphs

} XFER_ASCII_METRICS, *P_XFER_ASCII_METRICS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						   Stream Specific Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgXferStreamConnect	takes XFER_CONNECT, returns STATUS
	Sent to the Sender to ask it to link the Sender's and Receiver's pipe.

 The Sender responds by calling XferStreamAccept to complete the
 connection.

 In its call to XferStreamAccept, the Sender identifies the object that
 will generate the actual data, known as the Producer.  Essentially all
 Senders should pass self as the value of Producer.

 See the section "Stream-Based Protocols" for more information.
*/
#define	msgXferStreamConnect	MakeMsg(clsXfer, 2)

typedef struct XFER_CONNECT {

	TAG				id;			// In: Id Receiver sent to XferStreamConnect
	OBJECT			stream;		// In: Stream created by Receiver
	P_UNKNOWN		clientData;	// In: clientData Receiver sent to
								// XferStreamConnect

} XFER_CONNECT, *P_XFER_CONNECT;


/****************************************************************************
 msgXferStreamAuxData	takes PP_UNKNOWN, returns STATUS
	Passes back auxiliary information associated with the pipe.

 The Sender or Receiver can store auxiliary information with the pipe.
 using msgXferStreamSetAuxData and retrieve that information with
 msgXferStreamAuxData.

 This information can be used by either the Sender or Receiver to store
 private information or to or to pass information across the pipe.

 Warning:  There is only one auxiliary data slot in the pipe.  Only one of
 the Sender or Receiver should write the data, although both can read it.
 Subclasses must be aware of their ancestor's behavior in this regard.

 See Also
	msgXferStreamSetAuxData
*/
#define	msgXferStreamAuxData	MakeMsg(clsXfer, 4)


/****************************************************************************
 msgXferStreamSetAuxData	takes P_UNKNOWN, returns STATUS
	Stores arbitrary client data with the pipe.

 See Also
	msgXferStreamAuxData
*/
#define	msgXferStreamSetAuxData	MakeMsg(clsXfer, 5)


/****************************************************************************
 msgXferStreamWrite	takes STREAM, returns STATUS
	Asks the Sender to write more data to the stream.

 The Sender responds by writing to its stream using msgStreamWrite.

 The Sender may need access to its instance data to handle this message.
 The Sender can either implement its own facility for mapping from the
 stream to the necessary instance data (perhaps using properties; see
 clsmgr.h) or it can use msgXferStreamSetAuxData and msgXferStreamAuxData.

 See the section "Stream-Based Protocols" for more information.
*/
#define	msgXferStreamWrite	MakeMsg(clsXfer, 3)


/****************************************************************************
 msgXferStreamFreed		takes STREAM, returns STATUS
	Sent to the Sender when the Receiver's side of the stream has been
	freed.

 The Sender handles this message by sending msgDestroy to the stream
 passed in as a parameter.  This means that both streams (and hence both
 ends of the "pipe") have been freed.

 See the section "Stream-Based Protocols" for more information.
*/
#define	msgXferStreamFreed	MakeMsg(clsXfer, 6)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							  Public Functions							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 XferMatch		returns STATUS
	The Receiver calls XferMatch to find a mutually acceptable data transfer
	type.

 See the section "Determining a Common Data Transfer Type" for detailed.
 information.

 Return Value
	stsNoMatch:		No common data transfer type could be found.
	non-error:		The common data transfer type is passed back in *pId.

 See Also
	XferListSearch
	msgXferList
*/
STATUS EXPORTED XferMatch(
	OBJECT		Sender,		// In: Sender to find match with
	TAG			ids[],		// In: Array of types the Receiver understands
	SIZEOF		idsLen,		// In: Length of the ids[] array
	P_TAG		pId);		// Out: matching data type


/****************************************************************************
 XferListSearch	returns STATUS
	Searches two sets of data transfer types for a match.

 Most clients of the data transfer mechanism use XferMatch rather than
 calling this function.

 XferListSearch scans the two sets of transfer types (one in listObject and
 one in the passed-in array) to find the best match.

 XferListSearch checks each item in listObject against each item in the
 array in order from 0 to n-1.  Hence if the array contains [tagA, tagB]
 and the list contains [tabB, tagA], tagA is returned.  Objects should put
 data types into the listObject or the array in order of most desired to
 least desired.

 Return Value
	stsNoMatch:		No common data transfer type could be found.
	non-error:		The common data transfer type is passed back in *pId.

 See Also
	XferMatch
*/
STATUS EXPORTED XferListSearch(
	OBJECT		listObject,	// In: List object containing Sender types
	TAG			ids[],		// In: Array of types the Receiver understands
	SIZEOF		idsLen,		// In: Length of the ids[] array
	P_TAG		pId);		// Out: Matching data type


/****************************************************************************
 XferAddIds	returns STATUS
	Adds data transfer types to listObject.

 Typical Senders call this function while handling msgXferList.

 XferAddIds adds each item in the array of data transfer types to the list
 by sending msgListAddItem to listObject.
*/
STATUS EXPORTED XferAddIds(
	OBJECT		listObject,
	TAG			ids[],
	SIZEOF		idsLen);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						  Stream Specific Functions						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 XferStreamConnect	returns STATUS
	A Receiver calls this function to create a stream connection to a
	Sender.

 See the section "Stream-Based Protocols" for more information.
*/
STATUS EXPORTED XferStreamConnect(
	OBJECT		owner,		// In: Sender to connect stream to
	TAG			id,			// In: Desired data transfer type.  (This is
							// passed to Sender via msgXferStreamConnect.)
	P_UNKNOWN	clientData,	// In: clientData.  (This is passed to Sender
							// via msgXferStreamConnect.)
	P_OBJECT	pStream);	// Out: Stream to perform msgStreamRead on


/****************************************************************************
 XferStreamAccept	returns STATUS
	Called by Sender in response to msgXferStreamConnect.

 As part of the Sender's response to msgXferStreamConnect, the Sender calls
 XferStreamAccept to properly create the Sender's side of the stream.

 See the section "Stream-Based Protocols" for more information.
*/
STATUS EXPORTED XferStreamAccept(
	OBJECT		connect,	// In: pArgs->stream from msgXferStreamConnect
	U16			bufSize,	// In: Size of transfer buffer (up to 64k)
	OBJECT		Producer,	// In: Object to receive msgXferStreamWrite
	P_OBJECT	pStream);	// Out: Stream for Sender side of the "pipe"


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				  			Private 									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgXferStreamInit	takes pNull, returns STATUS
	Initializes a stream after it has been created.
*/
#define	msgXferStreamInit	MakeMsg(clsXfer, 7)


/****************************************************************************
 XferStreamNew	returns STATUS
	Create the receiver's end of a data transfer stream.

 Upon return, the stream is only partially complete and needs to be linked
 with the sender's end.
*/
STATUS EXPORTED XferStreamNew(
	P_OBJECT	pStream);


//REFGEN ENDIGNORE

#endif
