/****************************************************************************
 File: vol.h

 (C) Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.35  $
   $Author:   twiegman  $
	 $Date:   19 Mar 1992 09:55:32  $

 clsVolume inherits from clsObject.
 Provides volume support.

 Information in this file is useful if you are writing an installable volume.
 Also see volgodir.h for additional information.
****************************************************************************/
#ifndef VOL_INCLUDED
#define VOL_INCLUDED


/* Include file dependencies */


#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OS_INCLUDED
#include <os.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Defines ****/

#define fsDirPosFirst				(U32)0	
#define VOL_METHOD					STATUS EXPORTED

/* Flag to direct VNCreate to create short directory names (See VNCreate) */
#define fsShortDirName				fsNodeReadOnly

/* Error status codes */
#define stsNoMoreBuffers			MakeStatus(clsVolume, 1)

/* Informational status codes */
#define stsVolFormatIsTimeConsuming	MakeWarning(clsVolume, 1)

/* Resource ids for volume icons */
/* Defined with MakeWknResId (clsVolume, tag) */

/* Stored in groups of 10 values:	*/
/*	Base value defines large icon,	*/
/*	+1 value defines smaller icon,	*/
/*	+2 thru +9 reserved for future.	*/

#define tagVolHardDiskIcon			0	// 1-9	 define variants, see above
#define tagVolFloppyDiskIcon		10	// 11-19 define variants, see above
#define tagVolRemotePCIcon			20	// 21-29 define variants, see above
#define tagVolRemoteMacIcon			30	// 31-39 define variants, see above


/**** Types ****/

typedef OBJECT VOL;

typedef P_FS_ATTR_SIZE	*PP_FS_ATTR_SIZE;
typedef P_FS_ATTR_LABEL	*PP_FS_ATTR_LABEL;
typedef U32 FS_ATTR_VALUE, *P_FS_ATTR_VALUE, **PP_FS_ATTR_VALUE;

typedef U32 VOL_VNODE, *P_VOL_VNODE;

typedef struct DIR_ID_CACHE {
	P_MEM				pBuf;			// Sorted array of vol dir ids
	U32					used;			// Number used, of allocated space
	U32					free;			// Number free, of allocated space
} DIR_ID_CACHE;

typedef struct VOL_CACHE {
	VOL_VNODE			vnodeNotKnown;	// Used to fake volRAM
	P_MEM				pRoot;			// Cache dir elem for root vnode
	DIR_ID_CACHE		dirIds;			// Dir id cache
	OS_MILLISECONDS		lastAccess;		// Last access to cache layer
	OS_MILLISECONDS		lastVolAccess;	// Last access to volume
	OS_MILLISECONDS		lastVolWrite;	// Last write to volume
	OS_MILLISECONDS		refreshRate;	// Check with volume this often
										// to see if volume has changed
										// since last vol access
										// maxU32 implies unchangeable
	OS_MILLISECONDS		flushRate;		// Flush cached dirty files
										// after this much time has passed
										// 0 implies flush immediately
										// maxU32 implies no flushing
										// Default is 2000 (2 secs)
	U16					numDirs;		// Total num of dirs in the cache
										// Includes both open and closed
	U16					numFiles;		// Total num of files in the cache
										// Includes both open and closed
	U16					openDirs;		// Num of dirs in the cache
										// that are opened on the vol
	U16					openFiles;		// Num of files in the cache
										// that are opened on the vol
	U16					refdDirs;		// Num of opened dirs that have
										// non-zero reference counts
	U16					refdFiles;		// Num of opened files that have
										// non-zero reference counts
	U16					maxOpenDirs;	// Max dirs that can be left open
										// for caching purposes.
										// 0 implies no dirs
										// maxU16 implies as many as wanted
										// Default is maxU16
	U16					maxOpenFiles;	// Max files that can be left open
										// for caching purposes.
										// 0 implies no files
										// maxU16 implies as many as wanted
										// Default is maxU16
	P_MEM				pFirst;			// First cache entry
	P_MEM				pLast;			// Last cache entry

	P_MEM				pWrite;			// Write is to this cache entry
	U32					writePos;		// Write at this position
	U32					writeAmt;		// Write for this amount

	U8					readDirFullInProgress;
										// If non-zero then fully cached
										// dirs will not be "purged".

	U8					spareU8;
	U16					spareU16;
	U32					spares[5];
} VOL_CACHE;

Enum16(VOL_CMN_FLAGS) {
	vcVolIsOnBootDevice	= flag0,		// This volume is on the boot
										// device (as defined by the MIL)
										// but isn't necessarily THE boot
										// volume.
	vcVolIsDetachable  	= flag1,		// This volume is not removable
										// but may be detachable.
	vcVolIsSwapVolume  	= flag2			// This is the swap volume.
};

typedef struct VOL_COMMON {
	struct VOL_RTNS	   *pRtns;
	OS_SEMA_ID			fsSema;
	OS_SEMA_ID			volSema;
	VOL_CMN_FLAGS		flags;
	U16					vnodeCount;
	OS_HEAP_ID			vnodeHeap;
	U16					spare1;
	U16					dhCount;
	P_MEM				dhHead;
	U16					spare2;
	U16					fhCount;
	P_MEM				fhHead;
	VOL_CACHE			cache;
	OBJECT				dirIndexFile;
	BOOLEAN				dirIndexFileVerified;
	U16					spare;
	U32					spares[5];
} VOL_COMMON;

typedef struct VOL_INFO {
	struct VOL_INFO	   *pNext;
	FS_VOL_HEADER		hdr;
	VOL_COMMON			cmn;
	// Volume specific volInfo struct goes here...
} VOL_INFO, *P_VOL_INFO, **PP_VOL_INFO;


Enum16(VNODE_ACCESS) {
 // Delete node at handle free time?
	vnodeTemp			= flag0,
 // Read/write intentions for this handle
	vnodeReadOnly		= flag2,
 // Upper byte: exclusivity requirements
	vnodeNoExclusivity	= MakeU16(0, 0),
	vnodeDenyWriters	= MakeU16(0, 1),
	vnodeExclusiveOnly	= MakeU16(0, 2),
 // Uncompress file at VNGet time?
	vnodeUncompress		= flag14,
 // Default
	vnodeDefaultAccess	= 0 // perm, read/write, noExclusivity
};

#define vnodeIgnoreAccessInfo	0x8000

typedef struct VNODE_CMN_ATTRS {
	FS_NODE_FLAGS		nodeFlags;
	FS_DATE_TIME		nodeCreated;
	FS_DATE_TIME		nodeModified;
} VNODE_CMN_ATTRS, *P_VNODE_CMN_ATTRS;

/* Which attr buffers should be filled in? */
Enum16(VNODE_ATTR_FLAGS) {
	vnAttrNodeFlags		= flag0,
	vnAttrNodeCreated	= flag1,
	vnAttrNodeModified	= flag2,

	vnAttrLabelsBuffer	= flag8,
	vnAttrValuesBuffer	= flag9,
	vnAttrSizesBuffer	= flag10
};


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			Typedefs for functions supported by each volume class		   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Volume related functions follow... ****/

/****************************************************************************
 VolStatus	returns STATUS
	Has a volume check for readiness.

 Possible return status are stsOK, stsFSVolDisconnected, other errors.
 If status is okay, should indicate if volume has changed.
*/
typedef STATUS FunctionPtr(P_VOL_STATUS) (
	P_VOL_INFO			pVolInfo,
	P_BOOLEAN			pChanged		// In/Out: Has volume changed?
);
#define VolStatus(pVolInfo, pChanged) \
	((pVolInfo)->cmn.pRtns->pVolStatus) \
		(pVolInfo, pChanged)

/****************************************************************************
 VolSetVolName	returns STATUS
	Has a volume change its volume name.
*/
typedef STATUS FunctionPtr(P_VOL_SET_VOL_NAME) (
	P_VOL_INFO			pVolInfo,
	P_STRING			pName			// New volume name
);
#define VolSetVolName(pVolInfo, pName) \
	((pVolInfo)->cmn.pRtns->pVolSetVolName) \
		(pVolInfo, pName)

/****************************************************************************
 VolUpdateVolInfo	returns STATUS
	Requests that a volume updates its user accessable volume info.
*/
typedef STATUS FunctionPtr(P_VOL_UPDATE_VOL_INFO) (
	P_VOL_INFO			pVolInfo		// Vol Info
);
#define VolUpdateVolInfo(pVolInfo) \
	((pVolInfo)->cmn.pRtns->pVolUpdateVolInfo) \
		(pVolInfo)

/****************************************************************************
 VolSpecificMsg	returns STATUS
	Passes a volume specific message down to a volume.
*/
typedef STATUS FunctionPtr(P_VOL_SPECIFIC_MSG) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	MESSAGE				msg,			// Message
	P_UNKNOWN			pArgs			// In/Out: Arguments for message
);
#define VolSpecificMsg(pVolInfo, vnode, msg, pArgs) \
	((pVolInfo)->cmn.pRtns->pVolSpecificMsg) \
		(pVolInfo, vnode, msg, pArgs)


/**** Common vnode access/release functions follow... ****/

/****************************************************************************
 VNGet	returns STATUS
	Gets a vnode given pVolInfo, dirVNode and name of node in the directory.
*/
typedef STATUS FunctionPtr(P_VNODE_GET) (
	P_VOL_INFO			pVolInfo,		// Vol Info
	VOL_VNODE			dirVNode,		// VNode of parent directory
	P_STRING			pName,			// Name of node in directory
	VNODE_ACCESS		access,			// R/W access, exclusivity, etc
	P_UNKNOWN			pVolSpecific,	// Vol specific info
	P_VOL_VNODE			pVNode			// Out: Returned vnode handle
);
#define VNGet(pVolInfo, dirVNode, pName, access, pVolSpecific, pVNode) \
	((pVolInfo)->cmn.pRtns->pVNodeGet) \
		(pVolInfo, dirVNode, pName, access, pVolSpecific, pVNode)

/****************************************************************************
 VNNextChild	returns STATUS
	Gets a vnode given pVolInfo, dirVNode and dir position in a directory.
*/
typedef STATUS FunctionPtr(P_VNODE_NEXT_CHILD) (
	P_VOL_INFO			pVolInfo,		// Vol Info
	VOL_VNODE			dirVNode,		// VNode of parent directory
	P_U32				pDirPos,		// In/Out: directory position data
	VNODE_ACCESS		access,			// R/W access, exclusivity, etc
	P_STRING			pName,			// Out: Name of node
	P_VOL_VNODE			pVNode			// Out: VNode handle
);
#define VNNextChild(pVolInfo, dirVNode, pDirPos, access, pName, pVNode) \
	((pVolInfo)->cmn.pRtns->pVNodeNextChild) \
		(pVolInfo, dirVNode, pDirPos, access, pName, pVNode)

/****************************************************************************
 VNGetByDirId	returns STATUS
	Gets the vnode of a directory (and its name) given its directory id.
*/
typedef STATUS FunctionPtr(P_VNODE_GET_BY_DIR_ID) (
	P_VOL_INFO			pVolInfo,		// Vol Info
	VOL_VNODE			dirVNode,		// VNode of parent directory
	U32					dirId,			// Dir id of directory
	P_STRING			pName,			// Out: Name of node
	P_VOL_VNODE			pVNode			// Out: Returned dir vnode handle
);
#define VNGetByDirId(pVolInfo, dirVNode, dirId, pName, pVNode) \
	((pVolInfo)->cmn.pRtns->pVNodeGetByDirId) \
		(pVolInfo, dirVNode, dirId, pName, pVNode)

/****************************************************************************
 VNDup	returns STATUS
	Increments the reference count on a vnode.
*/
typedef STATUS FunctionPtr(P_VNODE_DUP) (
	P_VOL_INFO			pVolInfo,		// Vol Info
	VOL_VNODE			vnode,			// The vnode being dupped
	VNODE_ACCESS		access			// R/W, exclusivity, etc.
);
#define VNDup(pVolInfo, vnode, access) \
	((pVolInfo)->cmn.pRtns->pVNodeDup) \
		(pVolInfo, vnode, access)

/****************************************************************************
 VNRelease	returns STATUS
	Returns a vnode to the volume.
*/
typedef STATUS FunctionPtr(P_VNODE_RELEASE) (
	P_VOL_INFO			pVolInfo,		// Vol Info
	VOL_VNODE			vnode			// The vnode being released
);
#define VNRelease(pVolInfo, vnode) \
	((pVolInfo)->cmn.pRtns->pVNodeRelease) \
		(pVolInfo, vnode)


/**** Directory handle related functions follow... ****/

/****************************************************************************
 VNCreate	returns STATUS
	Creates a new file or directory node in the given (directory) node.

 Note: the parameter type only uses the flag fsNodeDir to distinguish
 between directories and files and the flag fsShortDirName to direct
 the volume to use a short name replacement for the directory name.
 Directories are only shortened if they reside in the PenPoint tree.
 The flag fsShortDirName overlaps fsNodeReadOnly, which is never used
 in conjunction with directories.
*/
typedef STATUS FunctionPtr(P_VNODE_CREATE) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			dirVNode,		// Handle of directory vnode
	P_STRING			pName,			// Name of the new file
	FS_NODE_FLAGS		type			// File or directory?
);
#define VNCreate(pVolInfo, dirVNode, pName, type) \
		((pVolInfo)->cmn.pRtns->pVNodeCreate) \
		(pVolInfo, dirVNode, pName, type)

/****************************************************************************
 VNDelete	returns STATUS
	Deletes the given node.
	
 VNode may be returned differently to mark it as a vnode that points to a
 deleted vnode.
*/
typedef STATUS FunctionPtr(P_VNODE_DELETE) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// VNode to delete
	BOOLEAN				visible			// At root of hierarchical delete?
);
#define VNDelete(pVolInfo, vnode, visible) \
	((pVolInfo)->cmn.pRtns->pVNodeDelete) \
		(pVolInfo, vnode, visible)

/****************************************************************************
 VNMove	returns STATUS
	Moves/renames a node (and any children) to a new node.
*/
typedef STATUS FunctionPtr(P_VNODE_MOVE) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			srcDirVNode,	// Handle of dir node of source
	VOL_VNODE			srcVNode,		// Handle of source vnode of move
	VOL_VNODE			dstDirVNode,	// Handle of dir node of dest
	P_STRING			pDstName		// New name to give the node
);
#define VNMove(pVolInfo, srcDirVNode, srcVNode, dstDirVNode, pDstName) \
	((pVolInfo)->cmn.pRtns->pVNodeMove) \
		(pVolInfo, srcDirVNode, srcVNode, dstDirVNode, pDstName)

/****************************************************************************
 VNDirPosDeleteAdjust	returns STATUS
	Makes any necessary adjustment to the dirPos after a node has been deleted.
*/
typedef STATUS FunctionPtr(P_VNODE_DIR_POS_DEL_ADJ) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			dirVNode,		// Handle of directory vnode
	VOL_VNODE			vnode,			// Handle of deleted vnode
	P_U32				pDirPos			// Dir position data before delete
);
#define VNDirPosDeleteAdjust(pVolInfo, dirVNode, vnode, pDirPos) \
	((pVolInfo)->cmn.pRtns->pVNodeDirPosDelAdj) \
		(pVolInfo, dirVNode, vnode, pDirPos)

/****************************************************************************
 VNGetDirId	returns STATUS
	Gets a directory node's dir id, given the vnode.
*/
typedef STATUS FunctionPtr(P_VNODE_GET_DIR_ID) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	P_U32				pDirId			// In/Out: dir id of dir node
);
#define VNGetDirId(pVolInfo, vnode, pDirId) \
	((pVolInfo)->cmn.pRtns->pVNodeGetDirId) \
		(pVolInfo, vnode, pDirId)


/**** File handle related functions follow... ****/

/****************************************************************************
 VNRead	returns STATUS
	Transfers n bytes from position m in a file to a buffer.
*/
typedef STATUS FunctionPtr(P_VNODE_READ) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	U32					filePos,		// Starting point of read
	U32					numBytes,		// Number of bytes to be read
	P_U8				pReadBuffer,	// Destination of bytes read
	P_U32				pCount			// In/Out: Actual bytes read
);
#define VNRead(pVolInfo, vnode, filePos, numBytes, pReadBuffer, pCount) \
	((pVolInfo)->cmn.pRtns->pVNodeRead) \
		(pVolInfo, vnode, filePos, numBytes, pReadBuffer, pCount)

/****************************************************************************
 VNWrite	returns STATUS
	Transfers n bytes from a buffer to position m in a file.
*/
typedef STATUS FunctionPtr(P_VNODE_WRITE) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	U32					filePos,		// Starting point of the write
	U32					numBytes,		// Number of bytes to write
	P_U8				pWriteBuffer,	// Destination of bytes to write
	P_U32				pCount			// In/Out: Actual bytes written
);
#define VNWrite(pVolInfo, vnode, filePos, numBytes, pWriteBuffer, pCount) \
	((pVolInfo)->cmn.pRtns->pVNodeWrite) \
		(pVolInfo, vnode, filePos, numBytes, pWriteBuffer, pCount)

/****************************************************************************
 VNGetSize	returns STATUS
	Gets a node's size given the vnode.
*/
typedef STATUS FunctionPtr(P_VNODE_GET_SIZE) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	P_FS_FILE_SIZE		pFileSize		// In/Out: Node's size
);
#define VNGetSize(pVolInfo, vnode, pFileSize) \
	((pVolInfo)->cmn.pRtns->pVNodeGetSize) \
		(pVolInfo, vnode, pFileSize)

/****************************************************************************
 VNSetSize	returns STATUS
	Sets a node's size given the vnode and the new size.

 This function could be used to either truncate or grow the file/resFile.
*/
typedef STATUS FunctionPtr(P_VNODE_SET_SIZE) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	FS_FILE_SIZE		fileSize		// Node's new size
);
#define VNSetSize(pVolInfo, vnode, fileSize) \
	((pVolInfo)->cmn.pRtns->pVNodeSetSize) \
		(pVolInfo, vnode, fileSize)


/**** Attribute related functions follow... ****/

/****************************************************************************
 VNGetName	returns STATUS
	Gets a node's name, given the vnode.
*/
typedef STATUS FunctionPtr(P_VNODE_GET_NAME) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	P_STRING			pName			// In/Out: name of node
);
#define VNGetName(pVolInfo, vnode, pName) \
	((pVolInfo)->cmn.pRtns->pVNodeGetName) \
		(pVolInfo, vnode, pName)

/****************************************************************************
 VNGetNumAttrs	returns STATUS
	Returns the number of non-standard attributes, given the vnode.
*/
typedef STATUS FunctionPtr(P_VNODE_GET_NUM_ATTRS) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	P_U16				pNumAttrs		// Out: num of attrs to get
);
#define VNGetNumAttrs(pVolInfo, vnode, pNumAttrs) \
	((pVolInfo)->cmn.pRtns->pVNodeGetNumAttrs) \
		(pVolInfo, vnode, pNumAttrs)

/****************************************************************************
 VNGetAttrInfo	returns STATUS
	Returns a node's attributes, given the vnode.

 Which common attributes and which arrays of the label/value/size arrays
 that need to be filled in are defined by the flgs field.  Which particular
 elements of each (label/value/size) array to be filled in is defined by the
 pWhich byte array.  If num is 0 or pWhich is null then no label/value/size
 array elements should be filled in.  If an element of pWhich is maxU8 then
 the corresponding label/value/size array element should be filled in.
 If the data is known and set then the pWhich array element should be set to
 1 after setting the values.
*/
typedef STATUS FunctionPtr(P_VNODE_GET_ATTR_INFO) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	U16					num,			// Num of attrs to get
	VNODE_ATTR_FLAGS	flgs,			// Get which attrs
	P_VNODE_CMN_ATTRS	pCmn,			// Common attrs
	P_U8				pWhich,			// Which user defined attrs
	P_FS_ATTR_LABEL		pLbls,			// In/Out: attribute labels
	P_FS_ATTR_VALUE		pVals,			// In/Out: attribute values
	P_FS_ATTR_SIZE		pSizs			// In/Out: attribute sizes
);
#define VNGetAttrInfo(pVolInfo, vnode, num, flgs, pCmn, pWhich, pLbls, pVals, pSizs) \
	((pVolInfo)->cmn.pRtns->pVNodeGetAttrInfo) \
		(pVolInfo, vnode, num, flgs, pCmn, pWhich, pLbls, pVals, pSizs)

/****************************************************************************
 VNSetAttrInfo	returns STATUS
	Sets a node's attributes, given the vnode.

 Which common attributes and which arrays of the label/value/size arrays
 that need to be stored are defined by the flgs field.  Which particular
 elements of each (label/value/size) array to be filled in is defined by the
 pWhich byte array.  If num is 0 or pWhich is null then no label/value/size
 array elements should be stored.  If an element of pWhich is maxU8 then
 the corresponding label/value/size array element should be stored.
 If the data is stored successfully then the pWhich array element should be
 set to 1.
*/
typedef STATUS FunctionPtr(P_VNODE_SET_ATTR_INFO) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	U16					num,			// Num of attrs to set
	VNODE_ATTR_FLAGS	flgs,			// Set which attrs
	P_VNODE_CMN_ATTRS	pCmn,			// Common attrs
	P_U8				pWhich,			// Which user defined attrs
	P_FS_ATTR_LABEL		pLbls,			// In/Out: attribute labels
	P_FS_ATTR_VALUE		pVals,			// In/Out: attribute values
	P_FS_ATTR_SIZE		pSizs			// In/Out: attribute sizes
);
#define VNSetAttrInfo(pVolInfo, vnode, num, flgs, pCmn, pWhich, pLbls, pVals, pSizs) \
	((pVolInfo)->cmn.pRtns->pVNodeSetAttrInfo) \
		(pVolInfo, vnode, num, flgs, pCmn, pWhich, pLbls, pVals, pSizs)

/****************************************************************************
 VNMakeNative	returns STATUS
	Gets rid of all concepts not native to a file system (ie res/info fields)
	and return the native form name of the file after being "stripped".
*/
typedef STATUS FunctionPtr(P_VNODE_MAKE_NATIVE) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode,			// Handle of vnode
	P_STRING			pName			// In/Out: Return buffer for native name
);
#define VNMakeNative(pVolInfo, vnode, pName) \
	((pVolInfo)->cmn.pRtns->pVNodeMakeNative) \
		(pVolInfo, vnode, pName)


/**** Misc functions follow... ****/

/****************************************************************************
 VNFlush	returns STATUS
	Flushes all buffers associated with this vnode.
*/
typedef STATUS FunctionPtr(P_VNODE_FLUSH) (
	P_VOL_INFO			pVolInfo,
	VOL_VNODE			vnode			// Handle of vnode
);
#define VNFlush(pVolInfo, vnode) \
	((pVolInfo)->cmn.pRtns->pVNodeFlush) \
		(pVolInfo, vnode)

/****************************************************************************
 DirIdGetParent	returns STATUS
	Gets the dir id of the parent of a node (also identified by dir id).
*/
typedef STATUS FunctionPtr(P_DIRID_GET_PARENT) (
	P_VOL_INFO			pVolInfo,
	U32					node,			// Node identified by dir id
	P_U32				pParent,		// In/Out: dir id of parent
	P_BOOLEAN			pParentIsRoot	// In/Out: parent is root
);
#define DirIdGetParent(pVolInfo, node, pParent, pParentIsRoot) \
	((pVolInfo)->cmn.pRtns->pDirIdGetParent) \
		(pVolInfo, node, pParent, pParentIsRoot)


/**** Debugging functions follow... ****/

/****************************************************************************
 VNRefCount	returns STATUS
	Gets the volume's ref count for a vnode.
*/
typedef STATUS FunctionPtr(P_VNODE_REF_COUNT) (
	P_VOL_INFO			pVolInfo,		// Vol Info
	VOL_VNODE			vnode,			// The vnode to get info about
	P_U16				pRefCount		// Out: Reference count on vnode
);
#define VNRefCount(pVolInfo, vnode, pRefCount) \
	((pVolInfo)->cmn.pRtns->pVNodeRefCount) \
		(pVolInfo, vnode, pRefCount)


/**** This is the definition for the table of volume routines: ****/

typedef struct	VOL_RTNS {
	// Vol General...
	P_VOL_STATUS			pVolStatus;
	P_VOL_SET_VOL_NAME		pVolSetVolName;
	P_VOL_UPDATE_VOL_INFO	pVolUpdateVolInfo;
	P_VOL_SPECIFIC_MSG		pVolSpecificMsg;
	// VNode Access...
	P_VNODE_GET				pVNodeGet;
	P_VNODE_NEXT_CHILD		pVNodeNextChild;
	P_VNODE_GET_BY_DIR_ID	pVNodeGetByDirId;
	P_VNODE_DUP				pVNodeDup;
	P_VNODE_RELEASE			pVNodeRelease;
	// Directory Handle Related...
	P_VNODE_CREATE			pVNodeCreate;
	P_VNODE_DELETE			pVNodeDelete;
	P_VNODE_MOVE			pVNodeMove;
	P_VNODE_DIR_POS_DEL_ADJ	pVNodeDirPosDelAdj;
	P_VNODE_GET_DIR_ID		pVNodeGetDirId;
	// File Handle Related...
	P_VNODE_READ			pVNodeRead;
	P_VNODE_WRITE			pVNodeWrite;
	P_VNODE_GET_SIZE		pVNodeGetSize;
	P_VNODE_SET_SIZE		pVNodeSetSize;
	// Attributes...
	P_VNODE_GET_NAME		pVNodeGetName;
	P_VNODE_GET_NUM_ATTRS	pVNodeGetNumAttrs;
	P_VNODE_GET_ATTR_INFO	pVNodeGetAttrInfo;
	P_VNODE_SET_ATTR_INFO	pVNodeSetAttrInfo;
	P_VNODE_MAKE_NATIVE		pVNodeMakeNative;
	// Misc...
	P_VNODE_FLUSH			pVNodeFlush;
	P_DIRID_GET_PARENT		pDirIdGetParent;
	// Debugging...
	P_VNODE_REF_COUNT		pVNodeRefCount;
	// Spares...
	P_UNKNOWN				pSpare1;
	P_UNKNOWN				pSpare2;
	P_UNKNOWN				pSpare3;
} VOL_RTNS, *P_VOL_RTNS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Class FileSystem Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** These messages are used by volume code ****/

/****************************************************************************
 msgFSRegisterVolClass		takes P_FS_REGISTER_VOL_CLASS, returns STATUS
	Registers a volume class with the file system.
*/
#define msgFSRegisterVolClass			MakeMsg(clsFileSystem, 0)

typedef struct FS_REGISTER_VOL_CLASS {
	CLASS				volClass;		// Vol class of volume
	FS_VOL_TYPE			volType;		// Type of volume
} FS_REGISTER_VOL_CLASS, *P_FS_REGISTER_VOL_CLASS;


/****************************************************************************
 msgFSInstallVol			takes P_FS_INSTALL_VOL, returns STATUS
	Creates a volume's root dir handle and register it with the file system.

 The volume should mark itself as connected and all observers of
 theFileSystem will be notified that a volume has been installed.
 (Note: The message is defined in fs.h so observers can use it.)
*/
/*#define msgFSInstallVol				MakeMsg(clsFileSystem, 1)  */

typedef struct FS_INSTALL_VOL {
	OBJ_KEY				key;			// Volume's key.
	CLASS				volClass;		// Class of the volume.
	VOL_VNODE			vnode;			// Root directory vnode.
	P_VOL_INFO			pVolInfo;		// In/Out: Volume info block.
} FS_INSTALL_VOL, *P_FS_INSTALL_VOL;


/****************************************************************************
 msgFSRemoveVol				takes P_FS_REMOVE_VOL, returns STATUS
	Removes a volume from the file system and destroy its root dir handle.

 Observers of theFileSystem will be notified of the change.
 (Note: The message is defined in fs.h so observers can use it.)
*/
/*#define msgFSRemoveVol				MakeMsg(clsFileSystem, 2)  */

typedef struct FS_REMOVE_VOL {
	OBJ_KEY				key;			// Volume's key.
	CLASS				volClass;		// Class of the volume.
	P_VOL_INFO			pVolInfo;		// Volume info block.
} FS_REMOVE_VOL, *P_FS_REMOVE_VOL;


/****************************************************************************
 msgFSConnectVol			takes P_FS_CONNECT_VOL, returns STATUS
	Marks a volume as connected and notify observers of theFileSystem.

 (Note: The message is defined in fs.h so observers can use it.)
*/
/*#define msgFSConnectVol				MakeMsg(clsFileSystem, 3)  */

typedef struct FS_CONNECT_VOL {
	P_VOL_INFO			pVolInfo;		// Volume info block.
} FS_CONNECT_VOL, *P_FS_CONNECT_VOL;


/****************************************************************************
 msgFSDisconnectVol			takes P_FS_DISCONNECT_VOL, returns STATUS
	Marks a volume as disconnected and notify observers of theFileSystem.

 (Note: The message is defined in fs.h so observers can use it.)
*/
/*#define msgFSDisconnectVol			MakeMsg(clsFileSystem, 4)  */

typedef struct FS_DISCONNECT_VOL {
	P_VOL_INFO			pVolInfo;		// Volume info block.
} FS_DISCONNECT_VOL, *P_FS_DISCONNECT_VOL;


/****************************************************************************
 msgFSVolList				takes P_FS_VOL_LIST, returns STATUS
	Returns device list for given class and count of volumes of that class.
*/
#define msgFSVolList					MakeMsg(clsFileSystem, 5)

Enum16(FS_VOL_LIST_ACCESS) {
	fsAccessVolList		= 0,			// Also returns head of list.
	fsReleaseVolList	= 1,
	fsGetHeadOfVolList	= 2
};

typedef struct FS_VOL_LIST {
	FS_VOL_LIST_ACCESS	access;			// See above.
	OBJECT				volClass;		// Class of the volumes.
	U16					volCount;		// Out: Number of volumes.
	P_VOL_INFO			pVolInfo;		// Out: First vol info block.
} FS_VOL_LIST, *P_FS_VOL_LIST;


/****************************************************************************
 msgFSUnRegisterVolClass	takes P_CLASS, returns STATUS
	UnRegisters a volume class from the file system.
*/
#define msgFSUnRegisterVolClass			MakeMsg(clsFileSystem, 6)


/****************************************************************************
 msgFSVolIsBusy			takes P_FS_VOL_INFO, returns STATUS
	Checks to see if a volume can be removed.

 If no user files/dirs are open and all caches have been written to
 the volume then the volume may be removed.  This method should only
 be called by the volume to be removed.

 If the volume can be removed then stsOK is returned.
 If the volume can not be removed then stsFSVolBusy is returned.
*/
#define msgFSVolIsBusy					MakeMsg(clsFileSystem, 7)


/****************************************************************************
 msgFSExclVolAccess			takes P_FS_EXCL_VOL_ACCESS, returns STATUS
 	Allows a volume class to obtain exclusive access to a volume and
	to release the exclusive access.
	
 This is used during the update volume list portions of volume classes.
 Volume classes should not try to update a volume if it is busy.

 If the volume was not busy and was acquired then stsOK is returned.
 If the volume was busy then a non stsOK is returned.
*/
#define msgFSExclVolAccess				MakeMsg(clsFileSystem, 8)

Enum16(EXCL_VOL_ACCESS) {
	xvaAcquireVolIfNotBusy	= 1,		// Acquire volume if not accessed
	xvaReleaseVol			= 2
};

typedef struct FS_EXCL_VOL_ACCESS {
	EXCL_VOL_ACCESS		mode;
	P_VOL_INFO			pVolInfo;
} FS_EXCL_VOL_ACCESS, *P_FS_EXCL_VOL_ACCESS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						 Class Volume Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Note: clsVolume messages 100 and above are defined in private header file.
*/

/****************************************************************************
 msgVolUpdateVolumes		takes P_VOL_UPDATE_VOLUMES, returns STATUS
	Has the volume class update its list of volumes.

 All volumes are sent this message every two seconds to give them a chance
 to do periodic volume updating.  If the user has requested a disk/volume
 that is not connected then volumes are sent this message with the
 fsUpdateSearchingForVolume flag set.  Volumes should not notify observers
 of volume connections, diconnections etc if a search is in progress.  The
 notification should be deferred until a later update request is sent.
 If the user has triple tapped on the connections notebook, asking to update
 all volumes, then volumes are sent this message with the fsUpdateAllDevices
 flag set.
*/
#define msgVolUpdateVolumes				MakeMsg(clsVolume, 0)

Enum16(FS_UPDATE_VOLS_MODE) {
 // An update should be done to all devices
 	fsUpdateAllDevices			= flag0,
 // The update request is in response to a power down notification
 	fsUpdatePoweringDown		= flag1,
 // The update request is in response to a power up notification
 	fsUpdatePoweringUp			= flag2,
 // Update searching for a volume?
	fsUpdateSearchingForVolume	= flag3
};

typedef struct VOL_UPDATE_VOLUMES {
	FS_UPDATE_VOLS_MODE	updateMode;		// See above.
	U32					spare1;			// For future use.
	U32					spare2;			// For future use.
} VOL_UPDATE_VOLUMES, *P_VOL_UPDATE_VOLUMES;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						 Volume Specific Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgVolEjectMedia			takes void, returns STATUS
	Has the volume eject its media.

 Passed as a volume specific msg by the file system.
*/
#define msgVolEjectMedia				MakeMsg(clsVolume, 10)


/****************************************************************************
 msgVolInvalidateCaches		takes void, returns STATUS
	Allows volumes to invalidate cache buffers at warm boot time.

 Passed as a volume specific msg by the file system at power up time.
*/
#define msgVolInvalidateCaches			MakeMsg(clsVolume, 11)


/****************************************************************************
 msgVolUpdateBootCode		takes void, returns STATUS
	Reads image of boot sector from mil.res and stores onto boot sector.

 Passed as a volume specific msg by the installation utility.
*/
#define msgVolUpdateBootCode			MakeMsg(clsVolume, 12)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				  Class Volume Messages - Formatting					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgVolFormatVolumeInit		takes P_VOL_FORMAT_MEDIA_INIT, returns STATUS
	This msg is sent to a volume to initiate a reformat of the volume.

 This initiates the format from the current owner of the block device.
 The volume object is destroyed (although there is a possibility that
 the destroy will fail) and then the block device of that volume, the
 volume offset on the block device and the volume size are returned.
 Call the volume class that is to format the volume with the message
 msgVolFormatMediaInit passing it this information.
 It will return a format id.

 Note that all other format related messages are sent to the class of
 the volume, because the volume will no longer exist.
*/
#define msgVolFormatVolumeInit			MakeMsg(clsVolume, 20)


/****************************************************************************
 msgVolFormatMediaInit		takes P_VOL_FORMAT_MEDIA_INIT, returns STATUS
 	Takes a block device object and returns a format id to be used with
	the other format messages.

 NOTE: volumeOffset should be zero and volumeSize should be zero if you
 wish to format the entire device (vs a partition of the device).
*/
#define msgVolFormatMediaInit			MakeMsg(clsVolume, 21)

typedef struct VOL_FORMAT_MEDIA_INIT {
	OBJECT				blockDevice;	// A block device
	U32					volumeOffset;	// Format device beginning here
	U32					volumeSize;		// Amount of device to be formatted
	P_UNKNOWN			formatId;		// Out: Format id
} VOL_FORMAT_MEDIA_INIT, *P_VOL_FORMAT_MEDIA_INIT;


/****************************************************************************
 msgVolMediaCapacities		takes P_VOL_MEDIA_CAPACITIES, returns STATUS
	Returns the possible format capacities for the device requesting format.

 This messages is sent to the class of the volume.
*/
#define msgVolMediaCapacities			MakeMsg(clsVolume, 22)

typedef struct VOL_MEDIA_CAPACITIES {
	P_UNKNOWN			formatId;		// Format id from format/reformat.
	U16					maxCapacities;	// Size of output capacities array.
	U16					numCapacities;	// Out: Actual number of capacities.
	P_U32				pCapacities;	// In/Out: Capacities.
} VOL_MEDIA_CAPACITIES, *P_VOL_MEDIA_CAPACITIES;


/****************************************************************************
 msgVolFormatMediaSetup		takes P_VOL_FORMAT_MEDIA, returns STATUS
	Has the vol class set the media to be ready for a format and determines
	if the block device will require format media (vs format track).

 This messages is sent to the class of the volume.
*/
#define msgVolFormatMediaSetup			MakeMsg(clsVolume, 23)

typedef struct VOL_FORMAT_MEDIA {
	P_UNKNOWN			formatId;		// Format id from format/reformat.
	U32					capacity;		// Desired capacity to format for.
	P_STRING			pName;			// Name of re/formatted volume.
	U16					percentDone;	// Out: Progress report.
} VOL_FORMAT_MEDIA, *P_VOL_FORMAT_MEDIA;


/****************************************************************************
 msgVolFormatMediaBegin		takes P_VOL_FORMAT_MEDIA, returns STATUS
	Has the vol class begin the format of its media.

 This step may do a format media if format track is not supported by
 the block device and may partition the media if it needs partitioning.

 This messages is sent to the class of the volume.
*/
#define msgVolFormatMediaBegin			MakeMsg(clsVolume, 24)


/****************************************************************************
 msgVolFormatMediaCont		takes P_VOL_FORMAT_MEDIA, returns STATUS
	Has the vol class do a format of its media.
	
 If format track is supported then this step will format the next track.
 If the media was formatted during msgVolFormatMediaBegin then this will
 only do verifying of format. If percentDone is not 100, then keep calling
 this until it is.

 This messages is sent to the class of the volume.
*/
#define msgVolFormatMediaCont			MakeMsg(clsVolume, 25)


/****************************************************************************
 msgVolCancelFormat			takes P_UNKNOWN, returns STATUS
	Has the vol class cancel the format.

 This messages is sent to the class of the volume.
*/
#define msgVolCancelFormat				MakeMsg(clsVolume, 26)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Class Volume Messages - Duplicating					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgVolDuplicateVolume		takes PP_UNKNOWN, returns STATUS
	This msg is sent to a volume to initiate a duplication of that volume.

 A duplicate block is then allocated and a duplicateId that can be used
 with the other duplicate messages is returned.	Note that the other
 messages are sent to the class of the volume.
*/
#define msgVolDuplicateVolume			MakeMsg(clsVolume, 30)


/****************************************************************************
 msgVolDuplicateMedia		takes P_VOL_DUPLICATE_MEDIA, returns STATUS
	Has the volume class duplicate more of the disk.
	
 If source is TRUE then data will be read from the source disk.
 If source is FALSE then data is written to the destination disk.
 The value percentDone is updated to reflect how much of the
 duplication has been completed.	 If percentDone is not 100,
 then keep calling this until it is.
*/
#define msgVolDuplicateMedia			MakeMsg(clsVolume, 31)

typedef struct VOL_DUPLICATE_MEDIA {
	P_UNKNOWN			duplicateId;	// Duplicate id from duplicate.
	BOOLEAN				sourceDisk;		// Is this source or destination?
	U16					percentDone;	// Out: Progress report.
} VOL_DUPLICATE_MEDIA, *P_VOL_DUPLICATE_MEDIA;


/****************************************************************************
 msgVolDuplicateReady		takes P_VOL_DUPLICATE_MEDIA, returns STATUS
	Checks to see if the source/dest disk of the duplicate is ready.
	
 The return percentDone is unused.
*/
#define msgVolDuplicateReady			MakeMsg(clsVolume, 32)


/****************************************************************************
 msgVolCancelDuplication	takes P_UNKNOWN, returns STATUS
	Have the vol class cancel the duplication.
*/
#define msgVolCancelDuplication			MakeMsg(clsVolume, 33)


//REFGEN BEGINIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Private Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Class FileSystem Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFSGetVolRoots			takes P_FS_HANDLE_LIST, returns STATUS
	Returns an allocated array of rootDH's to all of the installed volumes.

 pHandles must be freed using OSHeapBlockFree.
*/
#define msgFSGetVolRoots				MakeMsg(clsFileSystem, 10)

typedef struct FS_HANDLE_LIST {
	U16					count;			// Out: Count of volumes.
	P_OBJECT			pHandles;		// Out: Array of root dir handles.
} FS_HANDLE_LIST, *P_FS_HANDLE_LIST;


/****************************************************************************
 msgFSUpdateVolumes			takes P_FS_UPDATE_VOLUMES, returns STATUS
	Have the file system request volume classes to update their volumes.

 Have theFileSystem send update requests to the volume caches and
 send msgVolUpdateVolumes to each volume class.  The volume classes
 to send the message to may be filtered with volType and volClass if
 searching is true.  Volume cache updating is not done if searching is
 false.  If fullUpdate is false then full updates are determined by
 the msgFSUpdateVolumes method (Usually done if base station has recently
 been connected or disconnected).  If fullUpdate is true then volumes will
 be requested to do full updates.

 This message is sent to theFileSystem by clsVolSearcher (FSUI)
 whenever a volume is being requested or periodically to allow for
 asynchronous volume detection and/or formatting.
*/
#define msgFSUpdateVolumes				MakeMsg(clsFileSystem, 12)

typedef struct FS_UPDATE_VOLUMES {
	BOOLEAN				searching;		// A volume request is in progress
	BOOLEAN				fullUpdate;		// Force full update of volumes
	FS_VOL_TYPE			volType;		// Type of vol to find (if searching)
	CLASS				volClass;		// Class of vol to find	(if searching)
	U32					spare1;			// For future use.
	U32					spare2;			// For future use.
} FS_UPDATE_VOLUMES, *P_FS_UPDATE_VOLUMES;


/****************************************************************************
 msgFSEjectVolume			takes P_FS_EJECT_VOLUME, returns STATUS
	Ejects media from any drive that could accept the requested volume.
	
 The parm volType is used as a filter if it is non-zero.
 The parm volClass, used to identify the requesting volume, is
 also used as a filter if it is non null.
*/
#define msgFSEjectVolume				MakeMsg(clsFileSystem, 15)

typedef struct FS_EJECT_VOLUME {
	FS_VOL_TYPE			volType;		// Type of volumes to eject.
	CLASS				volClass;		// Class of volumes to eject.
} FS_EJECT_VOLUME, *P_FS_EJECT_VOLUME;


/****************************************************************************
 msgFSFlushCache			takes P_FS_FLUSH_CACHE, returns STATUS
	Flushes volume's cache.
*/
#define msgFSFlushCache					MakeMsg(clsFileSystem, 16)

Enum16(FS_FLUSH_CACHE_MODE) {
	fscFlushAll			= flag0,	// Flush all
	fscFlushOldest		= 0,		// Flush oldest
	fscFlushWrite		= flag1,	// Write dirty nodes to vol & update dir
	fscFlushClose		= flag2,	// Close unused nodes
	fscFlushPurge		= flag3,	// Purge open node entry's content
	fscFlushFreeClosed	= flag4,	// Free closed node entries
	fscFlushFreeOpened	= flag5		// Free opened node entries
};

typedef struct FS_FLUSH_CACHE {
	FS_FLUSH_CACHE_MODE	mode;			// Flush oldest, flush all.
} FS_FLUSH_CACHE, *P_FS_FLUSH_CACHE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Class FileHandle Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Following clsFileHandle messages used internally by loader only */

/****************************************************************************
 msgFSNoCacheSeekRead		takes P_FS_NO_CACHE_READ_WRITE, returns STATUS
	Reads data from a file.

 Reads data from a file into the buffer provided by the client.
 pReadBuffer points to the buffer which can hold a maximum of numBytes
 characters.  count returns the number of bytes read.

 This interface does not go through the data cache.

 See msgStreamRead for more info.
*/
#define msgFSNoCacheSeekRead			MakeMsg(clsFileHandle,1)

typedef struct {
	U32					position;	// Read from here, if not maxU32
	U32					numBytes;	// Size of read/write request
	P_UNKNOWN			pBuf;		// Destination of read/write
	U32					count;		// Out: Number of bytes transferred
} FS_NO_CACHE_READ_WRITE, * P_FS_NO_CACHE_READ_WRITE;


/****************************************************************************
 msgFSNoCacheSeekWrite		takes P_FS_NO_CACHE_READ_WRITE, returns STATUS
	Writes data to a file.

 Writes numBytes from pWriteBuffer into the file.
 Returns stsOK if all bytes were written.
 This interface does not go through the data cache.

 See msgStreamWrite for more info.
*/
#define msgFSNoCacheSeekWrite			MakeMsg(clsFileHandle,2)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Class VolRAM - Sent via VolSpecific					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgVolRAMFlashFile			takes void, returns STATUS
	Requests that the destination for this file is flash memory.
*/
#define msgVolRAMFlashFile				MakeMsg(clsVolRAM, 1)


/****************************************************************************
 msgVolRAMResetRefCount		takes void, returns STATUS
	Resets the reference count of this file to zero.

	ZZZ: This is only used during debugging to verify that scavenging worked.
*/
#define msgVolRAMResetRefCount			MakeMsg(clsVolRAM, 100)


/* Keys for the file system and volume classes and objects */

#define clsFileSystemKey	0x44494E4F
#define clsDirHandleKey		0x46524544
#define clsFileHandleKey	0x42524E59

#define clsVolRAMKey		0x574C4D41
#define objVolRAMKey		0x42545459

#define clsVolMSDiskKey		0x50424C53
#define objVolMSDiskKey		0x424D424D

#define clsVolTopsKey		0x46415354
#define objVolTopsKey		0x45444945

//REFGEN ENDIGNORE

#endif	// VOL_INCLUDED
