/****************************************************************************
 File: strlbox.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.25  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 16:58:50  $

 This file contains the API for clsStringListBox.

 clsStringListBox inherits from clsListBox.
 Implements a listbox that behaves as a choice or a group of toggles.

 As with clsListBox, the client supplies entry information on demand.
 With clsStringListBox, however, the client supplies strings, not windows.
 These strings are used to create instances of clsButton, and it is these
 buttons that are used as entry windows within the listBox.

 A stringListBox may behave in one of three manners: as a list of
 individual toggles (as in clsToggleTable), as choice that has zero or
 one of its buttons 'on' at a time, or as a choice that always has exactly
 one of its buttons 'on' at once.  When a stringListBox is behaving as a
 choice, its value is the 'data' field of the entry that is currently chosen.
****************************************************************************/

#ifndef STRLBOX_INCLUDED
#define STRLBOX_INCLUDED

												#ifndef LISTBOX_INCLUDED
#include <listbox.h>
												#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Common #defines and typedefs					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

// String ListBox behavior styles (roles)
#define slbRoleToggles		0	// Act like a toggle table.
#define slbRoleChoice01		1	// Act like a choice ( <=1 entries chosen)
#define slbRoleChoice1		2	// Act like a choice (always 1 entry chosen)

// String ListBox entry looks
#define slbLookInvert		0	// Chosen entries have inverted appearance.
#define slbLookDecorate		1	// Chosen entries have decorated appearance.

typedef struct {
	U16		role	: 4,		// Overall behavior.
			look	: 2,		// Controls looks of entries.
			dirty	: 1,		// Dirty status (ref. control.h)
			spare	: 9;		// reserved
} STRLB_STYLE, *P_STRLB_STYLE;

/*
 Default STRLB_STYLE:
//{
	role	= slbRoleToggles
	look	= slbLookInvert
	dirty	= false
//}
*/

/****************************************************************************
 msgNew		takes P_STRLB_NEW, returns STATUS
	category: class message
	Creates a string listbox window.

 The fields you commonly set are:
	pArgs->stringListBox.style.role:	overall behavior
	pArgs->stringListBox.style.look:	entry looks
	pArgs->stringListBox.value:			initial value
*/

typedef struct {
	STRLB_STYLE		style;		// overall style
	U32				value;		// initial value (if slbRoleChoice01
								// or slbRoleChoice1)
	U32				spare;		// reserved
} STRLB_NEW_ONLY, *P_STRLB_NEW_ONLY;

/*
 The value is the 'data' field of the entry that is currently chosen.
*/

#define stringListBoxNewFields	\
	listBoxNewFields			\
	STRLB_NEW_ONLY				stringListBox;

typedef struct {
	stringListBoxNewFields
} STRLB_NEW, *P_STRLB_NEW;

#define stsStrListBoxNoValue				MakeStatus(clsStringListBox, 1)


/****************************************************************************
 msgNewDefaults					takes P_STRLB_NEW, returns STATUS
	category: class message
	Initializes the STRLB_NEW structure to default values.
*/


/****************************************************************************
 msgStrListBoxGetStyle			takes P_STRLB_STYLE, returns STATUS
	Passes back the style of the receiver.
*/
#define msgStrListBoxGetStyle				MakeMsg(clsStringListBox, 1)


/****************************************************************************
 msgStrListBoxGetDirty			takes P_BOOLEAN, returns STATUS
	Passes back true if the listbox has been altered since dirty was set false.
*/
#define msgStrListBoxGetDirty				MakeMsg(clsStringListBox, 2)


/****************************************************************************
 msgStrListBoxSetDirty			takes BOOLEAN, returns STATUS
	Sets the dirty bit of a string listbox.

 The receiver will send msgControlSetDirty(pArgs) to every entry window.
*/
#define msgStrListBoxSetDirty				MakeMsg(clsStringListBox, 3)


/****************************************************************************
 msgStrListBoxGetValue			takes P_U32, returns STATUS
	Passes back the value of a string listbox.
    
 The value is the data field of the entry that is currently chosen.  This
 message may be used on instances whose role is either slbRoleChoice01
 or slbRoleChoice1.  For instances whose role is slbRoleToggles, use
 msgListBoxEnum with enum.flags set to lbSelected.

 Return Value
	stsFailed:				the role is set to slbRoleToggles.
	stsStrListBoxNoValue:	there's no entry selected.
*/
#define msgStrListBoxGetValue				MakeMsg(clsStringListBox, 4)


/****************************************************************************
 msgStrListBoxSetValue			takes U32, returns STATUS
	Sets the value of a string listbox whose role is one of slbRoleChoice*.

 Will deselect any selected entry if the arg is maxU32 and the role
 is set to slbRoleChoice1.  For instances whose role is slbRoleToggles, send
 as many msgListBoxSetEntry messages as required.

 Return Value
	stsFailed:	the role is set to slbRoleToggles.
*/
#define msgStrListBoxSetValue				MakeMsg(clsStringListBox, 5)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Client Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgStrListBoxProvideString			takes P_STRLB_PROVIDE, returns STATUS
	category: self-sent/client responsibility
	This message requests the client (or subclass) to provide a string.

 This message is sent when clsStringListBox receives msgListBoxProvideEntry.

 The string listbox is constructing an entry to be put into the listbox,
 and it needs the string and some data for the entry.  The client
 (or subclass) should copy the string bytes into the pString buffer,
 and set the data field as desired.

 msgStrListBoxProvideString is sent first to the string listbox itself.
 If the message reaches the standard clsStringListBox message handler,
 this message is forwarded on to the client of the listbox.

 A string listbox will send this message even when the position it's asking
 for is >= the number of entries specified for the listBox (same behavior
 as msgListBoxProvideEntry).  In this case, the client is free to return a
 non-zero status value, indicating to the string listbox that no entry should
 be created for that position.  Providing another string in this case allows
 A listBox will send this message even when the position it's asking for
 is >= the number of entries specified for the listBox.  In this case,
 the client is free to return a non-zero status value, indicating to
 the listBox that no entry should be created for that position.  Providing
 another entry window in this case allows the user to create new entries
 by merely scrolling past the end of the list.

 Subclasses of clsStringListBox may choose to respond to
 msgStrListBoxProvideString, or bypass this mechanism altogether and
 respond instead to msgListBoxProvideEntry.
*/
#define msgStrListBoxProvideString			MakeMsg(clsStringListBox, 6)

typedef struct {
	OBJECT    strListBox;  //  In: requestor
	U16       position;    //  In: position of requested entry
	P_CHAR    pString;     // Out: a 256 byte buffer for the string
	U32       data;        // Out: data for the entry
	U32       spare;       // reserved
} STRLB_PROVIDE, *P_STRLB_PROVIDE;


/****************************************************************************
 msgStrListBoxNotify			    takes U32, returns STATUS
	category: self-sent/client responsibility
	This message is sent out whenever the value of a string listbox changes.

 The pArgs will be undefined when role is set to slbRoleToggles (use
 msgListBoxEnum with enum.flags set to lbSelected).

 clsStringListBox responds by forwarding the message to the client of the
 listbox.
*/
#define msgStrListBoxNotify     			MakeMsg(clsStringListBox, 7)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 clsStringListBox responds by filing away its style and value.  Note that
 clsListBox will have filed its data first according to the value of
 LIST_BOX_STYLE.filing.
*/


/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 clsStringListBox responds by restoring its style and value.
*/


/****************************************************************************
 msgWinSend 	takes P_WIN_SEND, returns STATUS
    Sends a message up a window ancestry chain.

 clsStringListBox responds when pArgs->msg is msgButtonBeginPreview,
 msgButtonCancelPreview, or msgButtonDone.  If pArgs->msg is anything else,
 clsStringListBox just returns the result of calling its ancestor.

 For these three messages, clsStringListBox will make the set of entry
 windows act as a group (as does clsChoiceMgr) and return stsManagerContinue.

 Return Value
	stsManagerContinue: returned for one of the above three messages.
*/


/****************************************************************************
 msgListBoxProvideEntry			takes P_LIST_BOX_ENTRY, returns STATUS
	category: self-sent/client responsibility
	Self-sent when a listBox needs a window for display.

 clsStringListBox responds by self-sending msgStrListBoxProvideString,
 using the resulting information to create an instance of clsButton,
 and passing back the new button in pArgs->win.
*/


/****************************************************************************
 msgListBoxAppendEntry			takes P_LIST_BOX_ENTRY, returns STATUS
	Appends an entry to the list box after the specified position.

 clsStringListBox responds by keeping its state in synch--if the position
 that is currently on is greater than the new entry, it's incremented.
*/


/****************************************************************************
 msgListBoxInsertEntry			takes P_LIST_BOX_ENTRY, returns STATUS
	Insert an entry to the list box before the specified position.

 clsStringListBox responds by keeping its state in synch--if the position
 that is currently on is greater than the new entry, it's incremented.
*/


/****************************************************************************
 msgListBoxRemoveEntry			takes U16, returns STATUS
	Removes an entry from the list box.

 clsStringListBox responds by keeping its state in synch--if the position
 that is currently on is less than the new entry, it's decremented.

 If the entry being removed is the current 'on' button, the receiver sets
 its current value to zero (if the role is slbRoleChoice1) or to maxU32
 (if the role is slbRoleChoice1).  msgStrListBoxNotify will be sent.
*/


/****************************************************************************
 msgListBoxSetEntry				takes P_LIST_BOX_ENTRY, returns STATUS
	Sets an entry's information.

 clsStringListBox responds by setting the tag and data for any new replacement
 entry window.
*/


#endif

