/****************************************************************************
 File: sbar.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.57  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 14:46:52  $

 This file contains the API definition for clsScrollbar.

 clsScrollbar inherits from clsControl.
 Scrollbars provide scrolling visuals and define a protocol for handling
 various kinds of scrolling actions.
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsScrollbar debugging flag is 'K'.  Defined values are:

    flag2  (0x0004):    protocol messages
    flag6  (0x0040):    painting
    flag10 (0x0400):    input
    flag14 (0x4000):    general debug info
*/

#ifndef SBAR_INCLUDED
#define SBAR_INCLUDED

												#ifndef CONTROL_INCLUDED
#include <control.h>
												#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define hlpScrollbarVertical  	MakeTag(clsScrollbar, 1)
#define hlpScrollbarHorizontal 	MakeTag(clsScrollbar, 2)
#define hlpScrollbarGeneral	  	hlpScrollbarVertical

typedef OBJECT SCROLLBAR;

/****  Direction  ****/
#define	sbDirectionVertical		0	// vertical scrollbar
#define	sbDirectionHorizontal	1	// horizontal scrollbar

typedef struct SCROLLBAR_STYLE {
	U16	direction	: 1,
		wide		: 1,	// no longer implemented
		spare		: 14;	// unused (reserved)
} SCROLLBAR_STYLE, *P_SCROLLBAR_STYLE;

/*
 Default SCROLLBAR_STYLE:
//{
	direction	= sbDirectionVertical
//}
*/

Enum16(SCROLLBAR_ACTION) {
	// For vertical scrollbars:
	sbLineUp			= 0,
	sbLineDown			= 1,
	sbPageUp			= 2,
	sbPageDown			= 3,
	sbThumbUpDown		= 4,
	sbLineToTop			= 11,
	sbLineToBottom		= 12,
	sbToTop				= 15,
	sbToBottom			= 16,

	// For horizontal scrollbars:
	sbLineLeft			= 5,
	sbLineRight			= 6,
	sbPageLeft			= 7,
	sbPageRight			= 8,
	sbThumbLeftRight	= 9,
	sbColumnToLeft		= 13,
	sbColumnToRight		= 14,
	sbToLeft			= 17,
	sbToRight			= 18,

	// Terminating action:
	sbEndScroll			= 10
};

typedef struct SCROLLBAR_SCROLL {
	SCROLLBAR			sb;			// in: originating scrollbar 
	SCROLLBAR_ACTION	action;		// in: current action
	S32					offset;		// in/out: current or new offset
	S32					lineCoord;	// in: coordinate of line in root win space
	U32					spare1;		// unused (reserved)
	U32					spare2;		// unused (reserved)
} SCROLLBAR_SCROLL, *P_SCROLLBAR_SCROLL;

typedef struct SCROLLBAR_PROVIDE {
	SCROLLBAR	sb;			// in: originating scrollbar 
	S32			viewLength;	// out: client-provided view width or height
	S32			docLength;	// out: client-provided document width or height
	S32			offset;		// out: client-provided current offset
	U32			spare;		// unused (reserved)
} SCROLLBAR_PROVIDE, *P_SCROLLBAR_PROVIDE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_SCROLLBAR_NEW, returns STATUS
	category: class message
	Creates a scrollbar window.

 The fields you commonly set are:
	pArgs->scrollbar.style.direction:	whether horizontal or vertical
*/

typedef struct SCROLLBAR_NEW_ONLY {
	SCROLLBAR_STYLE	style;
	U32				spare1;	// unused (reserved)
	U32				spare2;	// unused (reserved)
} SCROLLBAR_NEW_ONLY, *P_SCROLLBAR_NEW_ONLY;

#define	scrollbarNewFields	\
	controlNewFields		\
	SCROLLBAR_NEW_ONLY		scrollbar;

typedef struct SCROLLBAR_NEW {
	scrollbarNewFields
} SCROLLBAR_NEW, *P_SCROLLBAR_NEW;


/****************************************************************************
 msgNewDefaults				takes P_SCROLLBAR_NEW, returns STATUS
	category: class message
	Initializes the SCROLLBAR_NEW structure to default values.

 Zeroes out pArgs->scrollbar and sets
//{
	pArgs->win.flags.style |= wsShrinkWrapWidth | wsShrinkWrapHeight;
	pArgs->win.flags.input = inputTip;

	pArgs->gWin.helpId = hlpScrollbarVertical;

	pArgs->control.style.previewGrab = true;
	pArgs->control.style.previewRepeat = true;
	pArgs->control.style.previewEnable = true;

	pArgs->scrollbar.style.direction = sbDirectionVertical;
//}
*/


/****************************************************************************
 msgScrollbarGetStyle		takes P_SCROLLBAR_STYLE, returns STATUS
	Passes back the current style values.
*/
#define msgScrollbarGetStyle	  	MakeMsg(clsScrollbar, 1)


/****************************************************************************
 msgScrollbarSetStyle		takes P_SCROLLBAR_STYLE, returns STATUS
	Sets the style values.
*/
#define msgScrollbarSetStyle	  	MakeMsg(clsScrollbar, 2)


/****************************************************************************
 msgScrollbarUpdate		takes nothing, returns STATUS
	Forces the scrollbar to repaint with the latest info.

 Causes msgScrollbarProvideVert/HorizInfo to be sent to client.
*/
#define msgScrollbarUpdate		  	MakeMsg(clsScrollbar, 14)


/****  Self-Sent/Client Messages  ****/
/****************************************************************************
 msgScrollbarVertScroll		takes P_SCROLLBAR_SCROLL, returns STATUS
	category: client responsibility
	Client should perform vertical scroll.

 The passed offset is an estimate computed by the scrollbar based on
 the information obtained from msgScrollbarProvideVertInfo.

 If the client is unwilling to scroll to this offset, the client may scroll
 to a different offset.  Be sure to set pArgs->offset to the new offset
 if it's different from the passed value.
*/
#define msgScrollbarVertScroll	   	MakeMsg(clsScrollbar, 5)


/****************************************************************************
 msgScrollbarHorizScroll		takes P_SCROLLBAR_SCROLL, returns STATUS
	category: client responsibility
	Client should perform horizontal scroll.

 The passed offset is an estimate computed by the scrollbar based on
 the information obtained from msgScrollbarProvideHorizInfo.

 If the client is unwilling to scroll to this offset, the client may scroll
 to a different offset.  Be sure to set pArgs->offset to the new offset
 if it's different from the passed value.
*/
#define msgScrollbarHorizScroll	   	MakeMsg(clsScrollbar, 6)


/****************************************************************************
 msgScrollbarProvideVertInfo		takes P_SCROLLBAR_PROVIDE, returns STATUS
	category: client responsibility
	Client should provide the document and view info.
*/
#define msgScrollbarProvideVertInfo  	MakeMsg(clsScrollbar, 9)


/****************************************************************************
 msgScrollbarProvideHorizInfo		takes P_SCROLLBAR_PROVIDE, returns STATUS
	category: client responsibility
	Client should provide the document and view info.
*/
#define msgScrollbarProvideHorizInfo  	MakeMsg(clsScrollbar, 10)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 clsScrollbar responds by filing away its instance data.
*/


/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 clsScrollbar responds by restoring its instance data.
*/


/****************************************************************************
 msgWinRepaint takes nothing, returns STATUS
    category: descendant responsibility
    Tells a window to repaint itself.

 clsScrollbar responds by painting itself appropriately.
*/


/****************************************************************************
 msgWinLayoutSelf takes P_WIN_METRICS, returns STATUS
    Tell a window to layout its children.

 clsScrollbar does nothing if pArgs->options does not have wsLayoutResize
 turned on.  Otherwise, it will set pArgs->bounds.size only for the
 dimensions for which shrinkwrapping is set.  It will set pArgs->size.w and h
 to a value derived from msgBorderInsetToInnerRect and an internal constant
 (currently 13 points).

 The visuals of the scrollbar are painted within the innerRect.

 See Also
	msgBorderInsetToInnerRect:	insets arbitrary rect to border rect.
*/


/****************************************************************************
 msgInputEvent 					takes P_INPUT_EVENT, returns STATUS
	Notification of an input event. 

 clsScrollbar responds to input events by maintaining the state necessary
 to behave in the appropriate fashion.  The types of input state a scrollbar
 can be in are:
	null
	pen up over an arrow
	pen down over an arrow
	pen up over the thumb
	pen down over the thumb
	dragging the thumb
	pen up over the shaft
	gesturing over the shaft

 In particular, the scrollbar allows the normal msgControl*Preview protocol
 to ensue only when the pen is interacting with the scroll arrows.
*/


/****************************************************************************
 msgGWinGesture		takes P_GWIN_GESTURE, returns STATUS
	Called to process the gesture.

 clsScrollbar responds by returning stsMessageIgnored if the gesture has
 no meaning (e.g. xgsFlickUp on a horizontal scrollbar).

 Otherwise, the scrollbar will fill out a SCROLLBAR_SCROLL struct and send
 either msgScrollbarVertScroll or msgScrollbarHorizScroll to the
 CONTROL_METRICS.client.  Actually, the client will receive the message
 twice--once for the appropriate action, and once for the sbEndScroll action
 (although the second message with sbEndScroll may be dropped in the future).
*/


/****************************************************************************
 msgGWinComplete	takes void, returns STATUS
	Causes the gesture to be completed.

 clsScrollbar responds by clearing its internal state data left over
 from processing a gesture in the scrollbar shaft.
*/


/****************************************************************************
 msgControlBeginPreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent when msgPenDown is received.

 clsScrollbar responds by returning stsControlCancelPreview if the penDown
 occurred in the shaft.

 Otherwise, the scrollbar self-sends msgControlRepeatPreview so that at
 least one arrow scroll is done.

 Note that the scrollbar won't receive this message if the penDown occurred
 in the thumb, because in that case clsScrollbar's response to msgInputEvent
 returned stsInputTerminate (after creating and starting an instance
 of clsTrack).
*/


/****************************************************************************
 msgControlAcceptPreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent when msgPenUp is received.

 clsScrollbar responds by notifying its CONTROL_METRICS.client with
 msgScrollbar[Vert/Horiz]Scroll and an action of sbEndScroll.

 Although one might expect this message to be sent when the pen is lifted
 from a scroll arrow, under normal circumstances a scrollbar will never
 receive this message.  This is because clsScrollbar sees the
 msgInputEvent(msgPenUp) and self-sends msgGWinAbort to cancel the
 gesture (because the user really wasn't gesturing over the repeating arrow).
 clsControl responds to msgGWinAbort by self-sending msgControlCancelPreview,
 which sends out the sbEndScroll.
*/


/****************************************************************************
 msgControlCancelPreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent when style.previewGrab is false and msgPenExitDown is received.

 clsScrollbar responds by notifying its CONTROL_METRICS.client with
 msgScrollbar[Vert/Horiz]Scroll and an action of sbEndScroll.
*/


/****************************************************************************
 msgControlRepeatPreview		takes P_INPUT_EVENT, returns STATUS
	category: self-sent
	Self-sent if style.repeatPreview is true.

 clsScrollbar responds by notifying its CONTROL_METRICS.client with
 msgScrollbar[Vert/Horiz]Scroll and an action of sbLine[Up/Down]
 (if the scrollbar is vertical) or sbLine[Left/Right] (if horizontal).

 If the client indicated that no scrolling took place (by not changing
 the SCROLLBAR_SCROLL.offset field), then the scrollbar will return
 stsControlCancelRepeat.
*/


/****************************************************************************
 msgTrackDone					takes P_TRACK_METRICS, returns STATUS
	category: client notification
	Sent by a tracker when it's done.

 A scrollBar will receive this message when the user has lifted the pen
 after dragging the thumb.

 If the scrollBar's style.direction is sbDirectionVertical, the scrollBar
 will notify its CONTROL_METRICS.client with msgScrollbarVertScroll and
 one of these actions: sbThumbUpDown, sbToTop, or sbToBottom.

 If the scrollBar's style.direction is sbDirectionHorizontal, the scrollBar
 will notify its CONTROL_METRICS.client with msgScrollbarHorizScroll and
 one of these actions: sbThumbLeftRight, sbToLeft, or sbToRight.
*/
#endif	// SBAR_INCLUDED

