/****************************************************************************
 ospriv.h
 Copyright 1992, GO Corporation, All Rights Reserved

 This include file describes the prototypes for supervisor privilege 
 PenPoint routines. The functions described in this file are contained
 in PENPOINT.LIB.

 $Revision:   1.5  $
   $Author:   kcatlin  $
	 $Date:   27 Mar 1992 10:07:56  $

****************************************************************************/
#ifndef OSPRIV_INCLUDED
#define OSPRIV_INCLUDED


#ifndef OS_INCLUDED
#include <os.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Common #defines and typedefs							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* The following are heap modes for supervisor level clients */

#define osHeapSupervisor flag5	// heap memory access is limited to supervisor
#define osHeapNoSwap flag6		// heap memory is never swapped
#define osHeapSystem flag10		// heap is owned by the system not a process

/* Special heap defines for supervisor level clients */

#define osGlobalHeapId ((OS_HEAP_ID)10)		// predefined heap for sys clients

/* Physical address types */
typedef U32					OS_PHYS_ADDR;		// physical mem address
typedef OS_PHYS_ADDR *		P_OS_PHYS_ADDR;	   

/* Program region information */
typedef struct OS_PROGRAM_REGION_INFO {
	P_MEM			base;
	SIZEOF			length;
} OS_PROGRAM_REGION_INFO, *P_OS_PROGRAM_REGION_INFO;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 OSIntMask	returns STATUS
	Sets the interrupt mask for a given interrupt.

 Note: OR in the flag osIntNumIsHardwareLevel if intNum is a hardware 
 interrupt level (vs a MIL logical device id).  The flag is defined
 in ostypes.h.

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSIntMask (
	OS_INTERRUPT_ID intNum,		// logical interrupt id 
	P_BOOLEAN pEnable			// In-Out: TRUE = enable, returns old mask 
);

/****************************************************************************
 OSIntEOI	returns STATUS
   Send an EOI request to the interrupt controller device.

 Note: OR in the flag osIntNumIsHardwareLevel if intNum is a hardware
 interrupt level (vs a MIL logical device id).  The flag is defined
 in ostypes.h.

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSIntEOI (
	OS_INTERRUPT_ID intNum		// MIL device id or hw interrupt level
);


/****************************************************************************
 OSProgramRegionInfo	returns STATUS
	Passes back region information for the debugger.

 Warning!!!	 Supervisor privilege only.
*/
STATUS EXPORTED OSProgramRegionInfo (
	OS_PROG_HANDLE				progHandle,		// program handle 
	P_U32						pNRegions,		// Out: number of regions
	P_OS_PROGRAM_REGION_INFO	pRI				// Out: region information
);

/****************************************************************************
 OSSysSemaRequest	returns STATUS
	Requests access to a system semaphore.

 System semaphores are regular semaphores with a little more protection. 
 If a task owns a system semaphore, then that task cannot be terminated or 
 suspended by another task until the system semaphore is relinquished.  With
 this feature, tasks can be sure that any system critical data structures will
 be completely updated.  

 If the task terminates itself while it owns a system semaphore, then the 
 next task that acquires the system semaphore will get the warning 
 stsOSSemaLockBroken.

 OSSysSemaClear should be used to relinquish the system semaphore.  The
 function OSSemaCreate is used to create the system semaphore.  Any
 semaphore can become a system semaphore simply by calling this routine.
 System semaphores are only used for critical section management.  Do NOT use 
 system semaphores for event handling.  

 Like regular semaphores, system semaphores are counting semaphores.

 Warning!!!  Supervisor privilege only.

 Return Value
	stsOSSemaLockBroken:			Previous locker of semaphore died without
									clearing the semaphore

 See Also
	OSSemaCreate
	OSSysSemaClear
*/
STATUS EXPORTED OSSysSemaRequest (
	OS_SEMA_ID sema		// the semaphore to lock
);

/****************************************************************************
 OSSysSemaClear	returns STATUS
	Releases access to the the system semaphore.

 System semaphores are regular semaphores with a little more protection. 
 If a task owns a system semaphore, then that task cannot be terminated or 
 suspended by another task until the system semaphore is relinquished.  With
 this feature, tasks can be sure that any system critical data structures will
 be completely updated.  

 If the task terminates itself while it owns a system semaphore, then the 
 next task that acquires the system semaphore will get the warning 
 stsOSSemaLockBroken.

 OSSysSemaClear should be used to relinquish the system semaphore.  The
 function OSSemaCreate is used to create the system semaphore.  Any
 semaphore can become a system semaphore simply by calling OSSysSemaRequest/
 OSSysSemaClear. System semaphores are only used for critical section 
 management.  Do NOT use system semaphores for event handling.  

 Like regular semaphores, system semaphores are counting semaphores.

 Warning!!!  Supervisor privilege only.

 See Also
	OSSysSemaRequest
	OSSemaCreate
*/
STATUS EXPORTED OSSysSemaClear (
	OS_SEMA_ID sema		// the semaphore to unlock
);

/****************************************************************************
 OSSupervisorCall	returns U32
 	Performs a privilege transition to supervisor privilege.  

 The function passed into the routine will be called by OSSupervisorCall in 
 supervisor privilege.  This function will check to verify that the
 routine passed in is actually a supervisor level routine.

 OSSupervisorCall will work correctly if called in supervisor level.
*/
#if defined(__WATCOMC__) && defined(__386__)
#pragma aux OSSupervisorCall parm [eax] [edx] [ecx] modify [gs];
U32 __far OSSupervisorCall(
	P_UNKNOWN	pFunction,
	P_UNKNOWN	pStackParms,
	U32			nStackParms
);
#endif

/****************************************************************************
 OSTaskAddressInfo	returns STATUS
	Passes back task and system memory information.

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSTaskAddressInfo (
	P_MEM				pAddr,				// memory address
	OS_TASK_ID			owner,				// owner of address
	SIZEOF				statBufSize,		// size of info buffer (in bytes)
	P_OS_ADDRESS_INFO	pAddrInfo			// Out: info buffer
);

/****  Data structures used by OSResourcesAvailable  ****/

Enum16(OS_RESOURCE_ZONE) {
	osResourceZoneNormal,				// Normal: plenty of resource
	osResourceZoneCaution,				// Caution: resource is getting low
	osResourceZoneWarning,				// Warning: resource is low
	osResourceZoneDanger,				// Danger: resource is really low
	osResourceZoneCritical				// Critical: PenPoint will reboot
};

#define	numResourceZones		5

typedef struct OS_RESOURCE_AVAILABLE {
	OS_RESOURCE_ZONE		currentZone;
	U32						resourceAvailable;
	U32						zoneLimits[numResourceZones];
} OS_RESOURCE_AVAILABLE, *P_OS_RESOURCE_AVAILABLE;

typedef struct OS_RESOURCES_INFO {
	OS_RESOURCE_AVAILABLE	swappableMemory;
	OS_RESOURCE_AVAILABLE	nonSwappableMemory;
	OS_RESOURCE_AVAILABLE	objects;
} OS_RESOURCES_INFO, *P_OS_RESOURCES_INFO;

/****************************************************************************
 OSResourcesAvailable	returns STATUS
	Returns info on the available resources in the system.
*/
STATUS EXPORTED0 OSResourcesAvailable (
	SIZEOF					bufSize,	// size of the info buffer (in bytes)
	P_OS_RESOURCES_INFO		pInfo		// Out: info buffer
);

/****************************************************************************
 OSMemMapAlloc	returns STATUS
	Allocates linear memory for memory mapped hardware

 Creates a guard page after the memory.	 The memory is created with
 the attributes: read/write data, system privilege, owned by systemTId.

 Note: the physical address passed in physAddr must be within the first
 16MB of physical memory.

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSMemMapAlloc (
	U32					physAddr,		// address of memory mapped area
	U32					length,			// length of memory to allocate
	PP_MEM				ppMem			// Out: return ptr to the memory
);

/****************************************************************************
 OSMemMapFree	returns STATUS
	Frees memory which was allocated by OSMemMapAlloc

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSMemMapFree (
	P_MEM				pMem			// ptr to memory to free
);

/****************************************************************************
 OSDMAMemAlloc	returns STATUS
	Allocates linear memory that is DMA-able

 Creates a guard page after the memory.	 The memory is created with
 the following attributes:
	read/write access
	supervisor privilege
	Not swappable (every page locked).

 All pages are mapped in and are physically contiguous in memory.  For 
 machines that have DMA boundary conditions (e.g. can't cross 64k physical
 boundary), the memory allocated in this region is guaranteed to honor those
 conditions.  Memory will be allocated on system page size boundaries and 
 all allocations will be a minimum of the processor page size.

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSDMAMemAlloc (
	U32					length,			// length of memory to allocate
	OS_TASK_ID			owner,			// owning task id.
	PP_MEM				ppMem			// Out: return ptr to the memory
);

/****************************************************************************
 OSDMAMemFree	returns STATUS
	Frees memory which was allocated by OSDMAMemAlloc

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSDMAMemFree (
	P_MEM				pMem,			// ptr to memory to free
	OS_TASK_ID			owner			// owning task id.
);

/****************************************************************************
 OSTaskMemInfo	returns STATUS
	Provides memory info for the system.

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSTaskMemInfo (
	OS_TASK_ID		taskId,				// info will be returned for task id
	SIZEOF			memBufSize,			// size of the info buffer (in bytes)
	P_OS_MEM_INFO	pMemInfo			// Out: info buffer
);

/****************************************************************************
 OSVirtToPhys	returns U32
	Translates a virtual address into a physical address.

 Warning!!!  Supervisor privilege only.
*/
U32 EXPORTED OSVirtToPhys (
	P_UNKNOWN pMem				// virtual address
);

/****************************************************************************
 OSMemLock	returns STATUS
	Locks pages in memory.

 Locked pages will not be paged out of the system.	If the page is 
 paged out before this call, then the page will be brought into memory and 
 then locked.

 A counter is maintained to keep track of multiple locks on a given page.

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSMemLock (
	P_MEM				pMem,		// pointer to memory
	SIZEOF				length		// length in bytes of memory to lock
);

/****************************************************************************
 OSMemUnlock	returns STATUS
	Unlocks pages in memory.

 When the page is unlocked, it may be paged out by the memory manager.

 A counter is maintained to keep track of multiple locks on a given page.
 When the counter goes to 0 then the page will be unlocked.

 Warning!!!  Supervisor privilege only.
*/
STATUS EXPORTED OSMemUnlock (
	P_MEM				pMem,		// pointer to memory
	SIZEOF				length		// length in bytes of memory to unlock
);

#endif // OSPRIV_INCLUDED
