/****************************************************************************
 File: os.h													

 (C) Copyright 1992, GO Corporation, All Rights Reserved.
													  
 $Revision:   1.50  $								  
   $Author:   twiegman  $								  
	 $Date:   24 Mar 1992 10:00:58  $				  

 This file contains the API for the PenPoint kernel.
 The functions described in this file are contained in PENPOINT.LIB.

 The PenPoint kernel provides support for tasking, memory management,
 inter-task communication and timer services.
****************************************************************************/
#ifndef OS_INCLUDED
#define OS_INCLUDED

/**** Debugging Flags ****/
/*
 PenPoint kernel flag is 'G', values are:

	0001:	User configuration (copy exes from boot to theSelectedVolume)
	0002:	Enter debugger on faults while scavenging
	0004:	Display memory sizes for each module loaded and run
	0008:	Display Stack grow/shrink messages
	0010:	Save page fault information in a memory buffer
	0020:	Run in the Ram only configuration
	0100:	Print various memmgr details

	1000:	see resfile.h
	2000:	see resfile.h
	4000:	see resfile.h
	8000:	see resfile.h

	10000:	Internal use only
	20000:	Call the MIL using the common entry point for full debugging
*/

#ifndef GO_INCLUDED
#include <go.h>
#endif
#ifndef OSTYPES_INCLUDED
#include <ostypes.h>
#endif
#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Common #defines and typedefs							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define osPageSize				(4*1024)

/* Defines for OS_ITMSG_INFO (mode field) */
// To generate the mode, OR in OS_TASK_MODE with the defines below.
#define osITMsgNoCopy			flag7	// vs copy buffer
#define osITMsgFrontOfQ			flag6	// vs end of queue
#define osITMsgDefaultMode		0		// Copy msg to end of msg queue

//REFGEN BEGINIGNORE
#define osITMsgPaint			flag5	// Obsolete
//REFGEN ENDIGNORE

/* Defines for setting priority */
#define osNumPriorities			51
#define osDefaultPriority		0

/* Defines for region information */
typedef U8					OS_REGION_ATTRS;
#define osRgnLocal			flag0
#define osRgnHasAliases		flag1
#define osRgnLocked			flag2	// Not yet implemented!!
#define osRgnNotSwappable	flag3
#define osRgnFrozen			flag4	// Not yet implemented!!
#define osRgnInSlowMem		flag5

Enum16(OS_REGION_TYPE) {
  osRgnData,							// data region
  osRgnHeap,							// heap	region
  osRgnStack,							// stack region
  osRgnMemMapFile,						// memory mapped file region
  osRgnCode								// code region
};

/* Subtask function type */
typedef void FunctionPtr(P_OS_SUBTASK_ENTRY) (U32 arg);	 

Enum16(OS_SET_GET) {
  osValuesSet = flag0,					// Set the value(s) passed in
  osValuesReturn = flag1,				// return the value(s)
  osValuesReturnAndSet = flag0 | flag1	// return and set the value(s)
};

/* Memory access attributes */
Enum16(OS_ACCESS) {				// access rights of a page
  osReadAccess,					// page allows read access only
  osReadWriteAccess,			// page allows read and write access
  osExecuteAccess,				// page allows execute access only
  osExecuteReadAccess			// page allows execute and read access
};

Enum16(OS_SET_TIME_MODE) {
	osSetTime = flag0,						// set the time
	osSetDate = flag1,						// set the date
	osSetTimeZone = flag2,					// set only the time zone
	osSetDateAndTime = osSetTime|osSetDate,	// set both the date and time
	// set date, time, and time zone
	osSetAll = osSetTime|osSetDate|osSetTimeZone
};

/* Display modes */
Enum16(OS_DISPLAY_MODE) {
  osConsole,		// display mode is console
  osGraphics		// display mode is graphics
};

/* Beep error tones */
Enum16(OS_ERROR_TYPE) {
  osWarning,
  osFatal
};

/* System wide memory information */
typedef struct OS_MEM_INFO {
  U32		taskMemAllocated;		 // amt of mem allocated by the task
  U32		localTaskMemAllocated;	 // amt of local mem allocated by the task
  U16		numAllocatedRgns;		 // # allocated regions by the task
  U16		numAllocatedLocalRgns;	 // # local regions allocated
  U32		taskMemResident;		 // amt of allocated mem in ram-this task
  U32		taskMemSwapped;			 // amt of allocated mem in swap file-this task

  // system wide statistics
  U32		systemRamSize;			 // total amt of memory in the system
  U32		amtInMemoryPool;		 // amt of memory in the memory pool
  U32		memFree;				 // amt of free ram
  U32		memAllocated;			 // total amt of mem allocated by all
  U16		numRgnsAllocated;		 // total # regions allocated by all
  U16		numSharedRgnsAllocated;	 // # shared regions used by all
  U32		pageSize;				 // system page size

  // swap file statistics
  U32		memNotSwappable;		 // amt of memory not swappable
  U32		swapFileSize;			 // size of the swap file
  U32		swapMediaFreePages;		 // number of pages free on the swap media

  // system wide allocated memory statistics (currently in ram)
  U32		dataAllocated;			 // amt of data allocated
  U32		heapsAllocated;			 // amt of heap space allocated
  U32		stacksAllocated;		 // amt of stack space allocated
  U32		memMapFilesAllocated;	 // amt of mem map file space allocated
  U32		codeAllocated;			 // amt of code space allocated
} OS_MEM_INFO, * P_OS_MEM_INFO;

/* Memory usage information */

// Region info, per type of region (code, data, etc)
typedef struct OS_REGTYPE_INFO {
	U32		   			allocated;		// Max size of the region
	U32					swappable;		// swappable pages in memory
	U32					nonSwappable;	// non-swappable pages in memory
	U32					committed;		// committed pages 
} OS_REGTYPE_INFO, *P_OS_REGTYPE_INFO;

// Region info, per scope of region (local, shared, etc)
typedef struct OS_REGSCOPE_INFO {
	OS_REGTYPE_INFO		code;			// Executable code
	OS_REGTYPE_INFO		data;			// Data
	OS_REGTYPE_INFO		heap;			// Data used as heaps
	OS_REGTYPE_INFO		stack;			// Stack space
	OS_REGTYPE_INFO		memMapFile;		// Memory-mapped files
} OS_REGSCOPE_INFO, *P_OS_REGSCOPE_INFO;

typedef struct OS_MEM_USE_INFO {
	OS_REGSCOPE_INFO 	local;			// Owned by this task only, in local memory
	OS_REGSCOPE_INFO	shared;			// Owned by this task only, in shared memory
	OS_REGSCOPE_INFO	multiOwner;		// Owned by this task and at least one other
	OS_REGSCOPE_INFO	total;			// System-wide totals
	U32					pageSize;		// System page size
	U32					systemRamSize;	// total amt of memory in the system
	U32					memFree;	 	// mem in the "free" list
	U32					memAllocated;	// mem not in the "free" list
	U32					swapFileSize;	// size of the swap file
} OS_MEM_USE_INFO, *P_OS_MEM_USE_INFO;

/* Address information */
typedef struct OS_ADDRESS_INFO {		// Info for a given memory address
	P_MEM			pRegionBase;		// base of region
	SIZEOF			regionLength;		// length of the region
	OS_ACCESS		access;				// access rights of the region
	OS_TASK_ID		owner;				// owning task for this region
	BOOLEAN			userPriv;			// TRUE - user region, FALSE - kernel
	OS_REGION_ATTRS flags;				// see defines above
	SIZEOF			residentSize;		// amount of region that is resident
	SIZEOF			committedSize;		// amount of region that is committed
	OS_REGION_TYPE	regionType;			// type of region
} OS_ADDRESS_INFO, * P_OS_ADDRESS_INFO;

/* System configuration information */
typedef struct OS_SYSTEM_INFO {			// system configuration information
	BOOLEAN			mathProcessorPresent; // TRUE = present
	OS_MILLISECONDS	millisecondsPerSystick;	// ms per clock tick
} OS_SYSTEM_INFO, * P_OS_SYSTEM_INFO;

/* Date and time information */
// The time zone string is a POSIX format string.  See the Watcom library
// reference for PenPoint, TZ environment variable set section for more info.
typedef struct OS_DATE_TIME {       
  U32               seconds;    // seconds after the minute -- [0,61]
  U32               minutes;    // minutes after the hour   -- [0,59]
  U32               hours;      // hours after midnight     -- [0,23]
  U32               day;        // day of the month         -- [1,31]
  U32               month;      // months since January     -- [0,11]
  U32               year;       // years since 1900
  U32               dayOfWeek;  // days since Sunday        -- [0,6]
  U32               dayOfYear;  // days since January 1     -- [0,365]
  P_CHAR            pTimeZone;	// time zone string (POSIX format)
} OS_DATE_TIME, * P_OS_DATE_TIME;

/* Loaded program information */
typedef struct OS_PROG_INFO {	
	OS_PROG_HANDLE progHandle;	// program identifying handle
	CHAR name[32+1];			// module name (without the .exe)
	U32 initHeapSize;			// .exe-header initial heap allocation
	U32 initStackSize;			// .exe-header initial stack allocation
	U16 initCS;					// initial CS (selector, not segment#)
	U32 initIP;					// initial IP
	U32 nRegions;				// # of regions
	U16 initDS;					// initial DS
	U16 isDLL		:1,			// 0 for .exes, 1 for DLLs
		isUser		:1,			// 1 for user priv, 0  for system priv
		rsvd		:14;		// reserved for future use.
	U32 fixedSize;				// read-only segments + initialization data
	U32 sharedSize;				// shared read/write segments
	U32 privateSize;			// private read/write segments
	U32 nRequiredModules;		// # modules this depends upon
} OS_PROG_INFO, * P_OS_PROG_INFO;

/* Interrupt information */
// Note: OR in the flag osIntNumIsHardwareLevel if intNum is a hardware
// interrupt level (vs a MIL logical device id).  The flag is defined
// in ostypes.h.
typedef struct OS_INTERRUPT_INFO {			// struct used to set interrupts
  OS_INTERRUPT_ID		intNum;				// logical interrupt id
  P_UNKNOWN				pCode;				// ptr to interrupt routine
} OS_INTERRUPT_INFO, * P_OS_INTERRUPT_INFO;

/* Module entrypoint types */
Enum16(OS_ENTRYPOINT_TYPE) {
  osEntryName,							// entrypoint is named
  osEntryOrdinal						// entrypoint is an ordinal
};

/* Message information */
typedef struct OS_ITMSG_INFO {		// inter-task message information
  OS_ITMSG_FILTER	filter;			// filter of the message
  P_MEM				pITMsg;			// pointer to inter-task message buffer
  SIZEOF			length;			// length of the message buffer
  U32				token;			// user defined info field
  OS_TASK_ID		taskId;			// dest or sending task Id
  U16				mode;			// see defines for OS_ITMSG_INFO
} OS_ITMSG_INFO;

/* Fast sema struct */
typedef struct OS_FAST_SEMA {
  U16				count;			// top bit for test and set
									// bits 0-14 for recursive counting
  U16				nWaits;		 	// number of waiters
  OS_TASK_ID		owner;
} OS_FAST_SEMA, *P_OS_FAST_SEMA;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 OSProgramInstall  returns STATUS
	Installs a program into the loader database. 

 If a dlc file is provided, all dlls in the file will also be loaded if not
 loaded already.

 OSProgramInstall will not return until instance 0 of all loaded dlls and 
 exe are completed. No message dispatching will occur during this time. If
 communication to the calling task is required, use IMProgramInstall 
 (install.h, install.lib).

 See Also
	OSProgramDeinstall
	OSProgramInstantiate
	OSModuleLoad

 Return Value
	stsOSBadDLCFormat:			DLC file is incorrectly formatted
	stsOSBadExeFormat:			A DLL or EXE is invalid in the dlc file
	stsOSProgInstallError:		Use debug version of PenPoint for more info
	stsOSModuleNotFound:		Module name specified in dlc file is invalid
	stsOSMissingDependency:		Import module in an exe or dll was not found
	stsOSMisingEntryName:		Import name in an exe or dll was not found
	stsOSMissingEntryOrdinal:	Import number in an exe or dll was not found
*/
STATUS EXPORTED0 OSProgramInstall(
	P_CHAR pCommandLine,			// dlc or exe name (and arguments)
	P_CHAR pWorkingDir,				// working dir of the program
	P_OS_PROG_HANDLE pProgHandle,	// Out: program handle
	P_CHAR pBadName,				// Out: If error, dll/exe that was bad
	P_CHAR pBadRef				// Out: If error, reference that was bad
);

/****************************************************************************
 OSProgramDeinstall	returns STATUS
	Deinstalls a program already loaded into the loader database.

 This routine will terminate any dll task wrappers before deinstalling
 the code.	If an exe is being deintalled, all tasks must be terminated
 before calling this routine.

 See Also
	OSProgramInstall
	OSProgramInstantiate

 Return Value
	stsOSInvalidProgramHandle:		Program handle is incorrect
	stsOSDependenciesExist:			Another program requires this dll or
									a task is using this module
*/
STATUS EXPORTED0 OSProgramDeinstall(
	OS_PROG_HANDLE progHandle		// program handle
);

/****************************************************************************
 OSProgramInstantiate	returns STATUS
	Creates an instance of a program.

 The newly created process will be set to the same priority as the caller.

 Return Value
	stsBadParam:			Program handle is invalid

*/
STATUS EXPORTED0 OSProgramInstantiate(
	OS_PROG_HANDLE progHandle,		// program handle from install
	P_CHAR pCommandLine,			// pathname + arguments
	P_OS_TASK_ID pTaskId			// Out: Task id of the new task
);

/****************************************************************************
 OSSubTaskCreate	returns STATUS
	Creates a new execution thread in this context.

 The entrypoint that starts the subtask must NOT return.  To terminate the
 task, use OSTaskTerminate (OSThisTask ()) as the last line in the routine.
 The newly created task will be set to the same priority as the caller.

 The initial stack size of the subtask will be set to 4096 bytes.  The
 stackSize parameter will be ignored. Stacks will automatically grow 
 to accomodate a program's stack requirements.
*/
STATUS EXPORTED0 OSSubTaskCreate(
	P_OS_SUBTASK_ENTRY pEntrypoint,		// Function entrypoint
	SIZEOF stackSize,					// ignored.
	U16 mustBeZero,						// reserved
	U32 arg,							// arg passed to function
	P_OS_TASK_ID pTaskId				// Out: new task id
);

/****************************************************************************
 OSTaskTerminate	returns STATUS
	Terminates a task.

 Callers to OSTaskTerminate will not return until the task has successfully
 terminated.  Task termination will cause the following events to occur:
	1):	if a process is terminated, all subtasks are first terminated
	2):	observers of theProcess will be notified (see clsmgr.h).  The error
		code is provided with the notification.
	3):	objects owned by the terminating task will be scavenged
	4):	a broadcast message will be sent to all tasks to notify them of the
		the task termination.  The message will be sent on the filter
		osTerminatedTaskFilter.	 This filter is by default off.
*/
STATUS EXPORTED0 OSTaskTerminate(
	OS_TASK_ID taskId,					// task to terminate
	OS_TASK_ERROR exitCode				// reason for terminating exit code
);

/****************************************************************************
 OSNextTerminatedTaskId returns the next task that has terminated
	Notifies the caller of the tasks that have terminated.

 The broadcast message for task termination does not include the task 
 identifier of the task that has terminated.  To find this out, 
 this routine should be called to get the list of terminated tasks. 
 When osNullTaskId is returned, the list ends.	
*/
OS_TASK_ID EXPORTED0 OSNextTerminatedTaskId(
	P_OS_TASK_ERROR pExitCode		// Out: exit code of terminating task
);

/****************************************************************************
 OSThisTask	returns OS_TASK_ID
	Passes back the task identifier of the current running task.
*/
OS_TASK_ID EXPORTED OSThisTask(void);

//REFGEN BEGINIGNORE
/****************************************************************************
 OSThisProcess	returns OS_TASK_ID
	Passes back the task id of this tasks process.
		
	NOTE: Defined in ostypes.h
*/
//OS_TASK_ID EXPORTED0 OSThisProcess(void);
//REFGEN ENDIGNORE

/****************************************************************************
 OSTaskPrioritySet	returns STATUS
	Sets the priority of a task or a set of tasks.

 The task mode can be used to set the priority of just one task or all
 tasks in the process family.

 See Also
	OSTaskPriorityGet
*/
STATUS EXPORTED0 OSTaskPrioritySet(
	OS_TASK_ID taskId,					// target task
	OS_TASK_MODE mode,					// task mode
	OS_PRIORITY_CLASS priorityClass,	// new priority class
	U8 priority							// new priority number
);

/****************************************************************************
 OSTaskPriorityGet	returns STATUS
	Passes back the priority of a task.

 Both the priority class and the priority within that class are returned.

 See Also
	OSTaskPrioritySet
*/
STATUS EXPORTED0 OSTaskPriorityGet(
	OS_TASK_ID taskId,					// target task
	P_OS_PRIORITY_CLASS pPriorityClass, // Out: task's priority class
	P_U8 pPriority						// Out: task's priority number
);

/****************************************************************************
 OSTaskDelay	returns STATUS
	Delays the current task for a specified period of time.

 When the machine is turned off, the delay time freezes until the system
 is turned back on again.  OSTaskDelay cannot be called from an interrupt
 subtask.
*/
void EXPORTED0 OSTaskDelay(
	OS_MILLISECONDS timeLimit	// milliseconds to delay
);

/****************************************************************************
 OSITMsgSend	returns STATUS
	Sends an inter-task message to a task or set of tasks.

 OSITMsgSend is used to send an inter-task message to 1) a single task, or 
 2) all tasks in a task family, or 3) all tasks in the system.	The 
 combination of the taskId and mode fields are used to accomplish this.	 If
 broadcasting to all tasks, the taskId field is ignored.

 An inter-task message is an array of bytes completely uninterpreted by
 the kernel stored in the pITMsg field.	 If the inter-task message is 
 short (up to U32), it can be stored in the token field for improved 
 performance. The length field is used to store the length of the 
 inter-task message in pITMsg.	If the length field is 0,  the pITMsg 
 field is ignored and can be used for more information passing.

 Inter-task messages are passed to the destination task in two ways:
 copy and alias.  In copy mode, the message is copied into a new buffer 
 allocated in the context of the destination task.	In alias mode, the
 message is aliased into the destination task.	Messages must be full
 regions when using alias mode.

 Messages are normally inserted into the end of the destination message 
 queue.	 However, it is possible to specify that a message be inserted into
 the front of the message queue.

 Inter-task messages will get delivered to tasks that have a filter mask set
 to allow messages of the sending messages filter.	If sending a message on 
 multiple filters, the message will be delivered if any one of the filters 
 are allowed by the receiving task.	 No error status is returned if the 
 receiving task does not receive the message due to its filter mask setting.

 See Also
	OSITMsgReceive
	OSITMsgFilterMask
	OSITMsgPeek
*/
STATUS EXPORTED0 OSITMsgSend(
	P_OS_ITMSG_INFO pITMsgInfo		// inter-task message info block
);

/****************************************************************************
 OSITMsgReceive returns STATUS
	Receives a message from the task's message queue.

 Messages are received by specifying a filter or set of filters in the 
 pITMsgInfo struct.	 Any message with a filter that is in that set will
 match the receive request.	 The filter in the pITMsgInfo struct must 
 always be set on entry.

 When a message is received that matches the input filter, the message is
 removed from the queue and provided to the client.

 See Also
	OSITMsgSend
	OSITMsgFilterMask
	OSITMsgPeek
*/
STATUS EXPORTED0 OSITMsgReceive(
	P_OS_ITMSG_INFO pITMsgInfo,		// In-Out: message info block
	OS_MILLISECONDS timeLimit		// amount of time to wait for message
);

/****************************************************************************
 OSITMsgPeek	returns STATUS
	Gets the next message from the message queue without removing it.

 *pITMsgId of null peeks from the front of the queue.  Use the previous
 message id to peek further into the queue.	 The filter in the
 pITMsgInfo struct must always be set on entry.	 

 See Also
	OSITMsgFromId
	OSITMsgSend
	OSITMsgReceive
*/
STATUS EXPORTED0 OSITMsgPeek(
	P_OS_ITMSG_INFO pITMsgInfo,		// In-Out: message info block
	OS_MILLISECONDS timeLimit,		// amount of time to wait for message
	P_OS_ITMSG_ID pITMsgId			// In-Out: id of message received
); 

/****************************************************************************
 OSITMsgFromId	returns STATUS
	Passes back the message associated with the message identifier.

 The message identifier should be obtained by calling OSITMsgPeek.

 See Also
	OSITMsgPeek
*/
STATUS EXPORTED0 OSITMsgFromId(
	P_OS_ITMSG_INFO pITMsgInfo,		// In-Out: message info block
	OS_ITMSG_ID itMsgId				// message id obtained from OSITMsgPeek
);


/****************************************************************************
 OSITMsgQFlush	returns STATUS
	Flushes the message queue of all messages matching the message filter.

 If a message has other filters set in addition to itMsgFilter, then the
 message will NOT be flushed.
*/
STATUS EXPORTED0 OSITMsgQFlush(
	OS_ITMSG_FILTER itMsgFilter		// message filter of messages to flush
);

/****************************************************************************
 OSITMsgFilterMask	returns the old filter mask
	Sets the filter mask for this task.

 Setting the mask bit to 1 indicates the message is allowed by this task;
 0 otherwise.  Any messages sent to this task whose filter bits are off in
 the filter mask will be discarded.

 If setNewFilter is FALSE, newITMsgFilter is ignored and only the old 
 filter mask is returned.

 See Also
	OSITMsgSend
	OSITMsgReceive
	OSITMsgPeek
	OSITMsgFromId
*/
OS_ITMSG_FILTER EXPORTED0 OSITMsgFilterMask(
	OS_ITMSG_FILTER newITMsgFilter,		// new filter mask for this task
	BOOLEAN setNewFilter		// if true, the new filter mask will be set
);

/****************************************************************************
 OSSemaCreate	returns STATUS
	Creates a semaphore.								  

 The semaphore will automatically be opened for the process.

 See Also
	OSSemaOpen
*/
STATUS EXPORTED0 OSSemaCreate(
	P_OS_SEMA_ID pSema		// Out: new open semaphore
);

/****************************************************************************
 OSSemaOpen returns STATUS
	Opens (accesses) an already existing semaphore.			  

 Tasks should always open someone else's semaphore to guarantee that the 
 semphore will be around even if the original owner of the semaphore 
 terminates.

 See Also
	OSSemaCreate
*/
STATUS EXPORTED0 OSSemaOpen(
	OS_SEMA_ID sema,	   		// semaphore
	OS_TASK_ID task			    // task wanting to share ownership of sema
);

/****************************************************************************
 OSSemaDelete	returns STATUS
	Deletes a semaphore.							  

 The semaphore will be removed from the system when all owners of the 
 semaphore have deleted it.

 See Also
	OSSemaCreate
	OSSemaOpen
*/
STATUS EXPORTED0 OSSemaDelete(
	OS_SEMA_ID sema			// the semaphore to delete
);

/****************************************************************************
 OSSemaRequest	returns STATUS
	Locks the counting semaphore (increments the count).

 OSSemaRequest should be used in conjunction with OSSemaClear when using 
 semaphores to protect critical sections of code.  OSSemaRequest/OSSemaClear
 implement a counting semaphore model which allows nesting of OSSemaRequest
 calls.	 Only after the same number of OSSemaClear calls will the next 
 waiting task enter the critical section.  Up to 64K nestings are allowed.

 If a task has obtained a semaphore via OSSemaRequest and subsequently dies,
 the semaphore will be given to the next requestor and that requestor will
 be given the status stsOSSemaLockBroken.

 Return Value
	stsOSSemaLockBroken:			Previous locker of semaphore died without
									clearing the semaphore
	stsOSTimeOut:					The timelimit expired before obtaining
									the semaphore

 See Also
	OSSemaClear
*/
STATUS EXPORTED0 OSSemaRequest(
	OS_SEMA_ID sema,			// the semaphore to lock
	OS_MILLISECONDS timeLimit	// max time to wait if already locked
);

/****************************************************************************
 OSSemaClear	returns STATUS
	Unlocks the counting semaphore (decrements the count).

 OSSemaClear should be used in conjunction with OSSemaRequest when using 
 semaphores to protect critical sections of code.  OSSemaRequest/OSSemaClear
 implement a counting semaphore model which allows nesting of OSSemaRequest
 calls.	 Only after the same number of OSSemaClear calls will the next 
 waiting task enter the critical section.  Up to 64K nestings are allowed.

 See Also
	OSSemaRequest
*/
STATUS EXPORTED0 OSSemaClear(
	OS_SEMA_ID sema			// the semaphore to unlock
);

/****************************************************************************
 OSSemaReset	returns STATUS
	Resets event semaphore (no matter what count). 

 OSSemaReset is used with OSSemaSet and OSSemaWait to support event handling.
 In this model, the client waiting on the event should use OSSemaSet to set
 the semaphore to 1, and OSSemaWait to wait until the semaphore has been 
 reset to 0.  OSSemaReset will reset the semaphore to 0, thereby 
 notifying all tasks waiting on the event.  OSSemaReset is normally used in
 interrupt tasks.  The task that is processing the event may actually have
 received more than one event and should process all events after resetting
 the semaphore to avoid losing any events.

 See Also
	OSSemaSet
	OSSemaWait
*/
STATUS EXPORTED0 OSSemaReset(
	OS_SEMA_ID sema			// the semaphore to reset
);

/****************************************************************************
 OSSemaSet	returns STATUS
	Sets the event semaphore to 1.

 OSSemaSet is used with OSSemaWait and OSSemaReset to support event handling.
 In this model, the client waiting on the event should use OSSemaSet to set
 the semaphore to 1, and OSSemaWait to wait until the semaphore has been 
 reset to 0.  OSSemaReset will reset the semaphore to 0, thereby 
 notifying the task waiting on the event.

 See Also
	OSSemaReset
	OSSemaWait
*/
STATUS EXPORTED0 OSSemaSet(
	OS_SEMA_ID sema			// the semaphore to set
);

/****************************************************************************
 OSSemaWait returns STATUS
	Waits for the event semaphore to be reset.		   

 OSSemaWait is used with OSSemaSet and OSSemaReset to support event handling.
 In this model, the client waiting on the event should use OSSemaSet to set
 the semaphore to 1, and OSSemaWait to wait until the semaphore has been 
 reset to 0.  OSSemaReset will reset the semaphore to 0, thereby 
 notifying the task waiting on the event.

 Return Value
	stsOSSemaLockBroken:			Previous locker of semaphore died without
									clearing the semaphore
	stsOSTimeOut:					The timelimit expired before obtaining
									the semaphore

 See Also
	OSSemaReset
	OSSemaSet
*/
STATUS EXPORTED0 OSSemaWait(
	OS_SEMA_ID sema,			// the semaphore to wait on
	OS_MILLISECONDS timeLimit	// max time to wait for the count to go to 0
);

/****************************************************************************
 OSFastSemaInit		returns nothing.
	 Initialize fast sema.

 Fast semaphores provide a fast but unprotected semaphore model.  Fast 
 semaphores are simply memory provided by the client as storage area for
 the state of the semaphore.  This storage area must initially be set to 0.

 See Also
	OSFastSemaRequest
	OSFastSemaClear
*/
#define OSFastSemaInit(_pSem) memset ( (_pSem), 0, sizeof(OS_FAST_SEMA) )

/****************************************************************************
 OSFastSemaRequest	returns STATUS
 	Fast version of sema request. 
	
 OSFastSemaRequest should be used in conjunction with OSFastSemaClear 
 when using semaphores to protect critical sections of code.  
 OSFastSemaRequest/OSFastSemaClear implement a counting semaphore model 
 which allows nesting of OSFastSemaRequest calls.  Only after the same 
 number of OSFastSemaClear calls will the next waiting task enter the 
 critical section.  Up to 64K nestings are allowed.

 Fast semaphores are fast by sacrificing protection.  The semaphore 
 structure passed into this routine is modified in the same privilege level
 as the caller.  Only if another task owns the semaphore will a privilege
 level transition occur.  

 There are a number of important limitations that a developer should 
 understand about fast semaphores.

 1) If a task owns a fast semaphore and then dies before releasing it, the
 	semaphore will not be released automatically by the system.

 2) The fast semaphores should not be copied from one location to another.
 	The routines rely on the address of the semaphore structure being 
	the same.

 See Also
	OSFastSemaClear
*/
STATUS EXPORTED OSFastSemaRequest (
   P_OS_FAST_SEMA pSema
);

/****************************************************************************
 OSFastSemaClear	returns STATUS
	Fast version of sema clear.

 OSFastSemaClear should be used in conjunction with OSFastSemaRequest
 when using semaphores to protect critical sections of code.  
 OSFastSemaRequest/OSFastSemaClear implement a counting semaphore model 
 which allows nesting of OSFastSemaRequest calls.  Only after the same 
 number of OSFastSemaClear calls will the next waiting task enter the 
 critical section.  Up to 64K nestings are allowed.

 Fast semaphores are fast by sacrificing protection.  The semaphore 
 structure passed into this routine is modified in the same privilege level
 as the caller.  Only if another task is waiting on the semaphore will a 
 privilege level transition occur.  

 There are a number of important limitations that a developer should 
 understand about fast semaphores.

 1) If a task owns a fast semaphore and then dies before releasing it, the
 	semaphore will not be released automatically by the system.

 2) The fast semaphores should not be copied from one location to another.
 	The routines rely on the address of the semaphore structure being 
	the same.


 See Also
	OSFastSemaRequest

*/
STATUS EXPORTED OSFastSemaClear (
	P_OS_FAST_SEMA pSema
);

/****************************************************************************
 OSGetTime	returns STATUS
	Returns local time.

 If an error is returned, the time returned will be Jan 1, 1900.
*/
STATUS EXPORTED0 OSGetTime(
	SIZEOF structLength,		// size of the date/time struct
	P_OS_DATE_TIME pDateTime	// Out: date, time and time zone information
);

/****************************************************************************
 OSSetTime	returns STATUS
	Sets the time or time zone.								 
*/
STATUS EXPORTED0 OSSetTime(
	OS_SET_TIME_MODE setMode,	// which attributes to set
	SIZEOF structLength,		// size of the date/time struct
	P_OS_DATE_TIME pDateTime	// date, time and time zone information
);

/****************************************************************************
 OSProgramInfo	returns STATUS
	Returns information on the program from the loader.	 

	OSProgramInfo will return information on the program handle passed in.
	If no valid handle exists for that number, then the routine will return
	information on the numerically smallest program handle just larger
	than the number passed in.  The program handle found will be put in the
	returned information buffer.  If no valid handle exists that is 
	numerically larger than progHandle, then Nil will be returned in the
	program handle field of the information structure with stsOK being
	returned from the function.

	To iterate over all program handles in the system, simply start by
	calling OSProgramInfo with a progHandle of 0.  This will return the
	numerically smallest program handle.  On the next call, use that
	program handle + 1, and on and on until the returned program handle 
	is 0.
*/
STATUS EXPORTED0 OSProgramInfo(
	OS_PROG_HANDLE progHandle,		// program handle given out by the loader
	P_OS_PROG_INFO pInfo			// Out: information buffer
);

/****************************************************************************
 OSPowerUpTime	returns OS_MILLISECONDS
	Passes back the number of milliseconds since the last reset.
*/
OS_MILLISECONDS EXPORTED0 OSPowerUpTime(void);

/****************************************************************************
 ScreenOnlyStringPrint	returns nothing
	Prints a string onto the console.  
	
 This routine will not log output through the debug log.  It will 
 only display characters on the screen.
*/
void EXPORTED0 ScreenOnlyStringPrint(
	P_STRING pString			// string to print
);

/****************************************************************************
 Debugger	returns nothing
	Enters the debugger.

 This macro will call the symbolic debugger (DB).  If the symbolic debugger
 is not available the low-level kernel debugger is called.  In production
 code (i.e., compiled without /DDEBUG) this macro does nothing.
*/
#ifdef DEBUG
  #define Debugger()		OSDebugger()
#else
  #define Debugger()
#endif


/****************************************************************************
 OSDebugger	returns nothing
	Enters the debugger, should only be called in special situations.

 Most clients should call Debugger NOT OSDebugger.  OSDebugger is used
 in special situations were a debugger needs to be called in production
 code.  When a call to the production version of OSDebugger is made, the 
 debug flag /DD10000 must be set to actually enter the debugger.
 If the debug flag is not set the call is a NOP.

 NOTE: OSDebugger should only be called in exceptional cases, such as,
 page fault handling.
*/
void EXPORTED OSDebugger(void);

/****************************************************************************
 KeyPressed returns BOOLEAN
	Determines if a key is available.

 This routine is provided for support of low level code below the input
 system.

 The high byte of the key is the scan code.

 Return Value
	TRUE:	if a key is available
	FALSE:	if no key is available

 See Also
	KeyIn
*/
BOOLEAN EXPORTED0 KeyPressed(
	P_U16 pCh			// Out: the char if true is returned
);

/****************************************************************************
 KeyIn	returns a keyboard character
	Passes back the next key and the scan code from the keyboard.

 The KeyIn routine is provided for support of low level code below the input
 system.

 The high byte of the key is the scan code.

 See Also
	KeyPressed
*/
U16 EXPORTED0 KeyIn(void);

/****************************************************************************
 OSDisplay	returns the old display mode
	Changes the display to the console or the graphics screen.

 This call is only valid on single headed development systems.	In all
 other configurations, the call does nothing.
*/
OS_DISPLAY_MODE EXPORTED0 OSDisplay(
	OS_DISPLAY_MODE newDisplayMode		  // set the display mode.
);

/****************************************************************************
 OSSetInterrupt returns STATUS
	Sets up an interrupt handler.

 The old interrupt info is also returned.  Callable only in ring 0.
*/
STATUS EXPORTED OSSetInterrupt(
	P_OS_INTERRUPT_INFO	 pIntInfo	// In-Out: interrupt info
);

/****************************************************************************
 OSTimerAsyncSema	returns STATUS
	Reset a semaphore after time milliseconds.

 The transaction handle can be used to stop the request if desired.
*/
STATUS EXPORTED0 OSTimerAsyncSema(
	OS_MILLISECONDS time,			// waiting period before sema reset
	OS_SEMA_ID sema,				// semaphore to reset
	P_OS_HANDLE pTransactionHandle	// Out: ptr to transaction handle
);

/****************************************************************************
 OSTimerIntervalSema	returns STATUS
	Resets a semaphore after each time interval has elapsed.

 The transaction handle can be used to stop the request if desired.
*/
STATUS EXPORTED0 OSTimerIntervalSema(
	OS_MILLISECONDS timeInterval,		// time interval in milliseconds
	OS_SEMA_ID sema,					// semaphore to reset
	P_OS_HANDLE pTransactionHandle		// Out: timer transaction handle
);

/****************************************************************************
 OSTimerStop	returns STATUS
	Stops a timer request given its transaction handle.
*/
STATUS EXPORTED0 OSTimerStop(
	OS_HANDLE transactionHandle			// transaction to stop
);

/****************************************************************************
 OSTimerTransactionValid	returns STATUS
	Checks to see if the timer transaction is valid.
*/
STATUS EXPORTED0 OSTimerTransactionValid(
  OS_HANDLE transactionHandle
);

/****************************************************************************
 OSModuleLoad	returns STATUS
	Loads a module into the loader's database. 

 If a dlc file is provided, all dlls in the file will also be loaded if not
 loaded already.

 OSModuleLoad will not return until instance 0 of all loaded dlls
 are completed. No message dispatching will occur during this time. If
 communication to the calling task is required, use IMModuleLoad
 (install.h, install.lib).

 See Also
	OSProgramInstall
	OSEntrypointFind
*/
STATUS EXPORTED0 OSModuleLoad(
	P_CHAR moduleName,				// Module name or dlc name
	P_CHAR pWorkingDir,				// Working dir of the app
	P_OS_PROG_HANDLE pProgHandle,	// Out: Program handle
	P_CHAR pBadMod,			  		// Out: If error, name of module that
									// failed, buffer must be
									// maxModNameLength+1 long
	P_CHAR pBadReference			// Out: If error, ref name not understood
									// buffer must be maxModNameLength+1 long
);

/****************************************************************************
 OSEntrypointFind	returns STATUS
	Finds an entrypoint in a loaded module either by name or by ordinal.

 See Also
	OSModuleLoad
*/
STATUS EXPORTED0 OSEntrypointFind(
	OS_ENTRYPOINT_TYPE entryType,		// name or ordinal
	P_STRING pName,						// name if entryType is name
	U16 ordinal,						// ordinal if entryType is ordinal
	OS_PROG_HANDLE progHandle,			// Program handle
	PP_MEM ppEntrypoint					// Out: ptr to entrypoint address
);

/****************************************************************************
 OSProcessProgHandle	returns the program instance number
	Passes back the program handle for the process.
*/
U16 EXPORTED0 OSProcessProgHandle(
	P_OS_PROG_HANDLE pProgHandle		// Out: ptr to program handle
);

/****************************************************************************
 OSEnvSearch	returns STATUS
	Searches the environment for the specified variable and returns its value.
*/
STATUS EXPORTED0 OSEnvSearch(
	P_STRING  pVariable,			// variable name
	P_STRING  outBuf,				// Out: Output buffer for variable value
	SIZEOF bufLen					// output buffer length
);

/****************************************************************************
 OSTaskNameSet	returns STATUS
	Sets a 4 character name for the given task.
*/
STATUS EXPORTED0 OSTaskNameSet(
  OS_TASK_ID		taskId,			// task to name
  P_CHAR			name			// name of task
);

/****************************************************************************
 OSThisApp	returns OBJECT
	Passes back the application object stored with the current process.
*/
OBJECT EXPORTED0 OSThisApp(void);

/****************************************************************************
 OSTaskApp	returns OBJECT
	Passes back the application object for a given process.
*/
OBJECT EXPORTED0 OSTaskApp(OS_TASK_ID task);

/****************************************************************************
 OSAppObjectPoke	returns nothing
	Stores the application object for the current process.
*/
void EXPORTED0 OSAppObjectPoke(
  OBJECT object					// current processes application object
);

/****************************************************************************
 OSPowerDown	returns nothing
	Powers down the machine.
*/
void EXPORTED0 OSPowerDown(void);

/****************************************************************************
 OSErrorBeep	returns nothing
	Outputs a tone based on the type of error encountered.
*/
void EXPORTED0 OSErrorBeep(
  OS_ERROR_TYPE	 errorType		// type of error
);

/****************************************************************************
   OSTone		returns STATUS
	Sends a tone for a given duration at the specified volume level.
*/
STATUS EXPORTED0 OSTone(
	U16					frequency,		// in Hertz
	U16					duration,		// in milliseconds
	U16					volumeLevel		// 0 for off; 1 for on
);

/****************************************************************************
 OSThisWinDev	returns OBJECT
	Passes back the windowing device for this application.
*/
OBJECT EXPORTED0 OSThisWinDev(void);

/****************************************************************************
 OSWinDevPoke	returns nothing
	Stores the windowing device for the specified process.
*/
void EXPORTED0 OSWinDevPoke(
	OS_TASK_ID	process,		// owner of application
	OBJECT		winDev			// Window device object
);

/****************************************************************************
 OSTaskProcess	returns OS_TASK_ID
	Returns the process id for the task specified.

 If the task parameter is invalid, the routine will return osNullTaskId.
*/
OS_TASK_ID EXPORTED0 OSTaskProcess(
	OS_TASK_ID task
);

/****************************************************************************
 OSTaskInstallTerminate	returns nothing.
	Notifies tasks waiting on OSProgramInstall that the instance is finished.

 If the parameter is set the TRUE, then the caller will go into an infinite
 wait state in order to keep the task and it's allocated resources alive.
*/
void EXPORTED0 OSTaskInstallTerminate(
	BOOLEAN				wait
);

/****************************************************************************
 OSMemInfo  returns STATUS
    Returns information on memory usage for a specified task.
*/
STATUS EXPORTED0 OSMemInfo (
    SIZEOF 			memBufSize,         // size of the info buffer (in bytes)
    P_OS_MEM_INFO 	pMemInfo      		// Out: info buffer
);

/****************************************************************************
 OSMemUseInfo  returns STATUS
    Returns information on memory usage for a specified task.
*/
STATUS EXPORTED0 OSMemUseInfo (
    SIZEOF 				memBufSize,     // size of the info buffer (in bytes)
    P_OS_MEM_USE_INFO 	pMemInfo      	// Out: info buffer
);

/****************************************************************************
 OSMemAvailable returns STATUS
 	Return amount of swappable memory available (to caution zone).
*/
STATUS EXPORTED0 OSMemAvailable (
	P_U32				pAvailable
);

/****************************************************************************
 OSSystemInfo	returns STATUS
	Passes back information on the system configuration.
*/
STATUS EXPORTED OSSystemInfo (
    SIZEOF 				bufSize,     		// size of the info buffer (in bytes)
    P_OS_SYSTEM_INFO 	pSystemInfo    		// Out: info buffer
);

/****************************************************************************
 osPrintBufferRoutine	returns nothing.
	Function variable print routine.
	
 All debug out (Debugf, DPrintf, printf, etc) flows through this function.
*/
extern void FunctionPtr(osPrintBufferRoutine)(P_CHAR pStr, SIZEOF len);

//REFGEN BEGINIGNORE

// NOTE: The following are not available for PenPoint 1.0:

/****************************************************************************
 OSStorageTaskCreate	returns STATUS
	Creates a storage task.						   
*/
STATUS EXPORTED0 OSStorageTaskCreate(
	U32 classTag,				// storage task class identifier
	U32 value,					// storage task value identifier
	P_OS_TASK_ID pTaskId		// Out: task id of the storage task
);
//REFGEN ENDIGNORE

#endif
