/****************************************************************************
 File: obxsvc.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.8  $
     $Date:   30 Mar 1992 10:59:54  $

 This file contains the API definition for clsOBXService.
 clsOBXService inherits from clsIOBXService.
 Provides default behavior for Outbox Services.
****************************************************************************/

#ifndef OBXSVC_INCLUDED
#define OBXSVC_INCLUDED

#ifndef IOBXSVC_INCLUDED
#include <iobxsvc.h>
#endif

/****  1. Introduction  ****/

/*
 In PenPoint, output operations are handled by a special class of
 services known as the "outbox services."  An outbox service
 implements the "deferred output" feature in PenPoint: This concept
 permits a user to specify output operations regardless of the
 readiness of output devices.  If the output device (e.g., a printer,
 a phone plug, a LAN connection, etc.) is not available or not
 connected, documents waiting for output will be placed into an
 "output queue" associated with the output service.  (This output
 queue is a special section in the system Outbox notebook.)  Thus, the
 actual output process is deferred until the output device becomes
 ready.
*/

/****  2. The Target of an Outbox Service  ****/

/*
 PenPoint expects that the PenPoint computer will not always be
 attached to most output devices.  Therefore, the output process for
 any PenPoint documents will be deferred until a connection is
 established.  The software controlling an input/output device is
 often implemented as an I/O service.  In most cases, an outbox
 service will make such an I/O service as its "target."  (See
 service.h for more information about target services in general.)
 Examples of I/O services include drivers for serial ports, parallel
 ports, data and/or fax modems, and LAN servers.  By making an I/O
 service its target, an outbox service is notified whenever the
 physical output device becomes connected or disconnected.  When an
 outbox service is not actively sending out a document, the connection
 status of the device is displayed in the "Status" column of the
 Outbox notebook Table of Contents.
*/

/****  3. Enabling and Disabling an Outbox Service  ****/

/*
 An outbox service must be "enabled" before its output process can
 begin.  This enabled state is represented by a checkbox in the
 "Enabled?"  column of the Outbox notebook TOC.  Typically, an output
 device permits only exclusive access.  If multiple outbox services
 are connected to the same output device, only one can be enabled at a
 time.  Enabling an outbox service causes it to become the "owner" of
 its target service.  The service remains "enabled" until either it is
 manually disabled by the user (i.e., by unchecking the "Enabled?"
 box); or until it willingly releases ownership of the device so that
 another service can become the new owner.  For more details on the
 notion of service ownership, see the service API in service.h.

 The concept of enabling or disabling an outbox service also provides
 a convenient mechanism for the user to manage an output device that
 can not automatically determine whether or not it becomes connected
 or disconnected.  Because the outbox service will not be informed
 when its target service is connected or disconnected, its status will
 always remain "Connected" regardless of the connection status of the
 physical device.  Such services can be explicitly disabled to prevent
 documents from being sent to a device that is not ready for output.
*/

/****  4. Managing the Output Process via the Outbox Service Protocol  ****/

/*
 Each instance of an outbox service has a corresponding section in the
 system Outbox notebook.  The name of the service and the name of the
 section are the same.  For example, the user may create two instances
 of an outbox service class named "DotMatrix," say "Engineering
 Printer" and "Upstairs."  Each instance will have its own output
 queue, implemented as a section called "Engineering Printer" and
 "Upstairs" in the outbox notebook.  The primary function of an outbox
 service is to manage the output queue for each service instance.
 This function is implemented by a standard outbox protocol consisting
 of 8 inter-related messages, as summarized below:

 The client of an outbox service first sends msgOBXSvcMoveInDoc or
 msgOBXSvcCopyInDoc to the outbox service instance, telling it to add
 an existing PenPoint document to its output queue.  Once a document
 is added to the outbox, msgOBXSvcPollDocuments informs an outbox
 service that it should check to see if conditions are right to start
 an output process.  Other events may also cause the outbox service to
 receive msgOBXSvcPollDocument.  For example, an outbox service will
 self-send this message when the service has just been enabled.  If
 the service is enabled and the output device is connected, the
 service sends msgOBXSvcNextDocument to self to locate the next
 document ready for output.  If a document exists in the output queue
 but is not ready for output, the service self-sends
 msgOBXSvcScheduleDocument to reschedule output at a later time.  If a
 document is ready for output, the service will lock the document with
 msgOBXSvcLockDocument, and kick off the output process with
 msgOBXSvcOutputStart.  At the end of the output process, the document
 being sent will send msgOBXDocOutputDone to the outbox service.
 Finally, if the output finished normally, the service self-sends
 msgOBXSvcPollDocuments again to see if anything else is ready for
 output.  If the output didn't finish normally, the service self-sends
 msgOBXSvcUnlockDocument to restore the document to its "pre-output"
 state.
*/

/****  5. Outbox Documents  ****/

/*
 The primary focus of an outbox service is to manage its output queue.
 An output queue is essentially a collection of documents located in
 an outbox section.  The primary focus of an outbox document is to
 manage a single output job.

 An outbox document can be any PenPoint document, i.e., an instance of
 an application inheriting from clsApp.  It can be created, opened,
 and closed just like a regular page in the notebook.  An example of
 an outbox document would be an "address envelope" for an electronic
 mail service.

 An outbox document is also responsible for interacting with the
 outbox service and controlling the output process, such as sending
 out an electronic mail message through a communication link.  Thus,
 in addition to responding to clsApp messages, an outbox document also
 understand the following clsOBXService messages:

    msgOBXDocOutputStartOK
    msgOBXDocOutputStart
    msgOBXDocOutputCancel
    msgOBXDocOutputDone
    msgOBXDocStatusChanged

 For details see the description for each message.
*/

/****  6. Writing Your Own Outbox Service  ****/

/*
 clsOBXService is an abstract class.  You should always create a
 subclass of it.  This is because clsOBXService only manages the
 output queue, it does not actually cause the output to happen.
 Typically, your outbox service will inherit the output queue
 management behavior from clsOBXService, and add any service-specific
 behaviors for the communication protocol or devices you need to
 handle.

 The default behavior of the outbox service does not support
 sophisticated scheduling algorithms that may be required by some
 services.  However, it is not difficult to replace some default
 behaviors with new ones.  The messages you may want to handle on your
 own include:

    msgOBXSvcMoveInDoc
    msgOBXSvcCopyInDoc
    msgOBXSvcNextDocument
    msgOBXSvcLockDocument
    msgOBXSvcUnlockDocument
    msgOBXSvcScheduleDocument
    msgOBXSvcOutputStart

 For example, the default behavior of msgOBXSvcNextDocument treats the
 output queue as a simple, Fist-In-First-Out queue.  If this is not
 sufficient for the service you wish to develop, you can provide your
 own behavior and pass back a document not on the top of the queue, or
 even a document not located in the Outbox notebook if it makes sense
 for the service.

 Another example would be msgOBXSvcLockDocument and
 msgOBXSvcUnlockDocument.  Their default behavior is to mark the
 document so that gestures over the document icon will not be
 recognized while output is in progress.  A msgOBXSvcUnlockDocument
 typically indicates that the output has been aborted for some reason.
 You may wish to add to the default behavior, such as notifying your
 observers that some error has just occurred.

 For details see the description for each message.
*/

/****  7. Working with Existing Outbox Services  ****/

/*
 As explained before, all output operations should be performed
 through an outbox service in order to take advantage of the "deferred
 output" feature of PenPoint.  An application or a service can
 "bypass" the standard outbox protocol only if the output device is
 always present or is rarely detached from the PenPoint computer.

 The key to working with an existing outbox service is to conceptually
 break up the output process into two distinct phases.  The first
 phase is either adding an existing PenPoint document to the output
 queue, or creating a special document of some sort in temporary
 storage and and then move it into the output queue.  The second phase
 is the actual output process, during which a device-specific data
 stream is sent out via some communication link.  clsOBXService
 provides a framework for managing the transition from one phase to
 another.

 The separation of these two phases of output operation has an
 additional benefit.  In many cases, an application developer can
 avoid writing a new outbox service in order to handle
 application-specific output functions.  It is often sufficient to
 handle only one of the two phases of the output operaton.  There are
 several options, as explained below:

 One inexpensive solution is to have the application export the data
 into a format that is easier to output under an existing outbox
 service.  For example, a database document can generate a report as
 an ASCII file or a word processor document and move it into a
 printer, fax or e-mail outbox section.  Similarly, a spreadsheet
 document can export its pie chart into a popular drawing program
 document and move it to the outbox for output.

 Another approach is to allow the database or spreadsheet document
 itself to be moved or copied into the output queue.  When the
 document receives msgOBXSvcOutputStart, it knows that the output
 device is ready.  It then proceeds to perform the output operation
 the old-fashioned way.  This alternative may be an attractive one if
 we wish to port an existing PC application to PenPoint.  Such
 applications already have sophisticated output capabilities, and we
 only need to ensure not to start the output process until the device
 is ready.  The obvious disadvantage of this approach is that it
 requires additional memory if we have to make a copy of the document
 in order to put it into the outbox.

 A third approach represents a compromise between the two.  During the
 first phase of the output operation, a "surrogate" document, rather
 than the real one, is copied into the output queue.  This surrogate
 document not only understands the outbox output protocol, but also
 knows how to communicate with the original document.  It is
 effectively a "pointer" back to the original document.  When the
 output process begins, the surrogate document communicates with the
 original one to cause the device-specific data stream to be sent to
 the correct output port. 
*/

/****  8. Services that Handle Input and/or Output  ****/

/*
 clsOBXService deals only with output operations.  For those services
 that want to handle input operations, a similar class clsINBXService
 is provided by PenPoint.  If a service (e.g., an electronic mail
 service) wants to handle both input and output, another abstract
 class, clsIOBXService, is provided.  clsIOBXService associates the
 service with both an input queue and an output queue.  (The input
 queue is a section in the system Inbox notebook.)  The service, the
 inbox section, and the outbox section all have the same name.  In
 fact, clsOBXService is implemented as a subclass (hence a subset) of
 clsIOBXService.
*/
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                              Class Messages                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNewDefaults             takes P_OBXSVC_NEW, returns STATUS
    category: class message
    Initializes the P_OBXSVC_NEW structure to default values.

    Zeroes out pArgs->obxsvc and sets...
    pArgs->iobxsvc.out.autoPoll         = true;
    pArgs->obxsvc.sectionClass          = clsNBToc;

*/

typedef struct OBXSVC_NEW_ONLY {
    OBJECT  sectionClass;   // class of the outbox section (for output queue)
                            // This must be clsNBToc or a subclass of it.
    U32     unused1;
    U32     unused2;
    U32     unused3;
} OBXSVC_NEW_ONLY, *P_OBXSVC_NEW_ONLY;

#define obxServiceNewFields \
    ioSvcNewFields        \
    OBXSVC_NEW_ONLY         obxsvc;

typedef struct OBXSVC_NEW {
    obxServiceNewFields
} OBXSVC_NEW, *P_OBXSVC_NEW;

/****************************************************************************
 msgNew                     takes P_OBXSVC_NEW, returns STATUS
    category: class message
    Creates a new outbox service object.

*/

/****************************************************************************
 msgOBXSvcSwitchIcon    takes nothing, returns STATUS
    category: class message
    Toggles the outbox icon (to empty or filled) if neccessary.

 Check the content of the outbox notebook.  Show the "filled" icon
 if any document is found.  Show the "emtpy" icon otherwise.
*/
#define msgOBXSvcSwitchIcon                     msgIOBXSvcSwitchIcon

/****************************************************************************
 msgOBXDocGetService    takes P_OBX_DOC_GET_SERVICE, returns STATUS
    category: class message
    Gets the service name.

 Get the name of the service associated with an outbox document.
 If the document has not been placed into an outbox section, stsFailed
 is returned.

 Note that the document must be at the top level of an outbox section.
 That is, if the document is embedded within another document which
 is in an outbox section, stsFailed will be returned because the document
 is not at the top level of an outbox section.
*/
#define msgOBXDocGetService                     msgIOBXDocGetService

typedef struct OBX_DOC_GET_SERVICE {

    OBJECT  document;               // In: document uid
    CHAR    svcName[nameBufLength]; // Out: service name

} OBX_DOC_GET_SERVICE, *P_OBX_DOC_GET_SERVICE;

/****************************************************************************
 msgOBXDocInOutbox  takes P_OBX_DOC_IN_OUTBOX, returns STATUS
    category: class message
    Checks if a document is in a section in the Outbox.

 This message can be sent to clsOBXService to check if a PenPoint
 document represented by pArgs->uuid is already in the output queue of
 an outbox service inheriting from pArgs->svcClass.  stsOK is returned
 if it is, stsFailed otherwise.  If pArgs->svcClass is objNull, stsOK
 is returned if the document is anywhere in the Outbox notebook.
*/
#define msgOBXDocInOutbox                       msgIOBXDocInIOBox

typedef struct OBX_DOC_IN_OUTBOX {

    UUID    uuid;       // In: document uuid
    CLASS   svcClass;   // In: service class to check for

} OBX_DOC_IN_OUTBOX, *P_OBX_DOC_IN_OUTBOX;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *              Messages Sent to an Outbox Service Instance                *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgOBXSvcMoveInDoc takes P_OBXSVC_MOVE_COPY_DOC, returns STATUS
    Moves a document into the outbox section.

 Superclass behavior is to move the document located at pArgs->source
 into the output queue associated with the outbox service.  For
 example, set pArgs->sequence to 1 to move the document to the top
 of the queue.  Set it to maxU16 to move the document to the bottom of
 the queue.

 After the document is moved (or copied) to the output queue, it is
 considered to be in a state ready for output, even though the service
 may not be connected at the time.  Client should not alter the document
 in any way once it has been moved to the output queue.

 Subclasses can provide their own behavior if they wish.  Remember to
 use the class message msgOBXSvcSwitchIcon to change the outbox icon.
*/
#define msgOBXSvcMoveInDoc                      msgIOBXSvcMoveInDoc

typedef struct OBXSVC_MOVE_COPY_DOC {
    FS_LOCATOR  source;     // In: Location of source document.
    U16         sequence;   // In: Sequence number to move/copy in
                                    //     front of.
} OBXSVC_MOVE_COPY_DOC, *P_OBXSVC_MOVE_COPY_DOC;


/****************************************************************************
 msgOBXSvcCopyInDoc takes P_OBXSVC_MOVE_COPY_DOC, returns STATUS
    Copies a document into the Outbox section.

 Same as msgOBXSvcMoveInDoc, except that the document is copied
 to the output queue.
*/
#define msgOBXSvcCopyInDoc                      msgIOBXSvcCopyInDoc

/****************************************************************************
 msgOBXSvcGetTempDir    takes P_OBJECT, returns STATUS
    Passes back a handle for a temporary directory.

 This message is provided for clients who may want ot prepare their
 output document before moving it into the output queue.  The handle
 of an "official" temporary directory is passed back and it can be
 used as temporary storage for documents, data, etc.  Clients are
 responsible for deleting temporary files when they are done.  The
 directory will be flushed after a warm boot.
*/
#define msgOBXSvcGetTempDir                     msgIOBXSvcGetTempDir

/****************************************************************************
 msgOBXSvcPollDocuments     takes nothing, returns STATUS
    Polls all documents in an output queue and output those who are ready.

 This message tells the outbox service to look through its output
 queue and send out the first document ready for output.  The service
 will first make sure that it is enabled and is connected to the
 designated output port.  If these conditions are met, it will then
 self-send msgOBXSvcNextDocument to locate the next document ready for
 output.

 If msgOBXSvcNextDocument returns stsOK, indicating that a
 document is ready for output, this message proceeds to self-send
 msgOBXSvcLockDocument to lock the document, and finally
 msgOBXSvcOutputStart to initiate the output process.

 If msgOBXSvcNextDocument returns stsOBXSvcDocReady, indicating
 that the section is not empty but none of the documents are ready
 for output, this message self-sends msgOBXSvcScheduleDocument
 to schedule the document passed back in pArgs at a later time.

 Subclasses normally do not process this message.

 See Also
    msgOBXSvcNextDocument
    msgOBXSvcLockDocument
    msgOBXSvcUnlockDocument
    msgOBXSvcOutputStart
    msgOBXSvcScheduleDocument
*/
#define msgOBXSvcPollDocuments                  msgIOBXSvcPollDocuments

/****************************************************************************
  msgOBXSvcNextDocument     takes P_OBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Passes back the next document ready for output.

 Superclass behavior is to start from the top of the output queue and
 locate the first document ready for output.  If one is found,
 information about the document is passed back in pArgs.  The same
 pArgs will be passed to messages msgOBXSvcLockDocument and
 msgOBXSvcOutputStart.  By default, a document is ready for output
 when it is closed.  If the document is open, it will receive
 msgOBXDocOutputStartOK and it should return stsOK to indicate that it
 is ready for output.

 Subclasses can provide their own behavior if they wish.  Return
 stsOBXSvcSectionEmpty to give the superclass an opportunity to
 change the outbox icon from filled to empty.

 Return Value
    stsOK:                  A document is ready for output.
    stsOBXSvcSectionEmpty:  The output queue is empty.
    stsOBXSvcDocNotReady:   No document in the output queue is ready.
    Service-Specific Error Returns.

 See Also
    msgOBXSvcPollDocuments
*/
#define msgOBXSvcNextDocument                   msgIOBXSvcNextDocument

typedef struct OBXSVC_DOCUMENT {

    OBJECT      uid;                    // uid of the doc
    OBJECT      dir;                    // app dir of the doc
    OBJECT      docClass;               // class of the doc
    U16         sequence;               // sequence of the doc
    CHAR        pName[nameBufLength];   // name of this doc
    P_UNKNOWN   pDocData;               // subclass's private data

} OBXSVC_DOCUMENT, *P_OBXSVC_DOCUMENT;

/****************************************************************************
  msgOBXSvcLockDocument     takes P_OBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Locks the document in preparation for output.

 This message is a place holder for subclasses that may require
 additional preparatory work to be performed on a document before it
 is ready for output.  For example, a document may have to be "locked"
 so that it can not be opened during the output process.  This message
 may be used for other purposes as well.  For example, an outbox
 service may decide to store a light-weight "shadow" document (e.g., a
 report designator for a database application) in the output queue
 until it is chosen for output.  The service then handles this message
 by converting the shadow document to a real one (e.g., the actual
 report).

 The superclass behavior for this message is to stamp the document
 directory with the filesystem attribute iobxsvcDocOutputInProgress.  This
 stamp will prevent any gestures over the document from being
 processed.  This means that once a document is locked for output it
 can not be deleted, renamed, etc. via gestures.

 See Also
    msgOBXSvcUnlockDocument
*/
#define msgOBXSvcLockDocument                   msgIOBXSvcLockDocument

/****************************************************************************
  msgOBXSvcUnlockDocument       takes P_OBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Unlocks a document that was previously locked.

 This message is a place holder for subclasses that may require
 additional "cleanup" work to be performed on a document before it
 is put back to the output queue.

 The superclass behavior for this message is to remove the
 iobxsvcDocOutputInProgress stamp on the document directory.

 See Also
    msgOBXSvcLockDocument
*/
#define msgOBXSvcUnlockDocument                 msgIOBXSvcUnlockDocument

/****************************************************************************
  msgOBXSvcScheduleDocument     takes P_OBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Schedules a document that is not ready for output

 This message is sent when msgOBXSvcNextDocument locates a document in
 the output queue but the document is not ready for output.

 Subclasses should provide their own behavior.  The default behavior is
 to release the ownership of the target service (i.e., become disabled),
 with the expectation that the user must manually schedule the document
 later on (by re-enabling the section.)

 See Also
    msgOBXSvcNextDocument
*/
#define msgOBXSvcScheduleDocument               msgIOBXSvcScheduleDocument

/****************************************************************************
  msgOBXSvcOutputStart      takes P_OBXSVC_DOCUMENT, returns STATUS
     category: self-sent
     Starts the output process for a document in the output queue.

 Superclass behavior is to activate the outbox document if it isn't
 already active, and then send msgOBXDocOutputStart to the
 document instance.

 Subclasses can provide their own behavior if they wish.
*/
#define msgOBXSvcOutputStart                    msgIOBXSvcIOStart

/****************************************************************************
  msgOBXSvcOutputCancel     takes nothing, returns STATUS
     Cancels the output process.

 This message is sent to the service when the caller wishes to
 cancel any output operation in progress.  The service responds
 to this message by sending msgOBXDocOutuptCancel to an active
 outbox document.  After the document is cancelled, the service
 will post an error note to the user if there are other documents
 waiting to be processed.  The user then decides whether or not
 the service should proceed to send the remaining documents.

 Subclasses do not normally process this message.
*/
#define msgOBXSvcOutputCancel                   msgIOBXSvcIOCancel

/****************************************************************************
  msgOBXSvcOutputCleanUp        takes P_OBX_DOC_OUTPUT_DONE, returns STATUS
     category: self-post.
     Cleans up after the current output is done.

 This message is posted to self as a result of the service receiving
 msgOBXDocOutputDone, which is sent by the outbox document when it
 finishes the output operation.  The outbox document will be either
 deleted or marked as specified in pArgs, and when everything is
 properly cleaned up the service will post msgOBXSvcPollDocuments to
 self to see if anything else is waiting for output.

 Subclasses do not normally process this message.

 See Also
    msgOBXDocOutputDone
*/
#define msgOBXSvcOutputCleanUp                  msgIOBXSvcIOCleanUp

Enum32(OBX_DOC_EXIT_BEHAVIOR) {

    obxDocExitDoNothing         = 0,
    obxDocExitDelete            = 1,
    obxDocExitMarkAsFailed      = 2,
    obxDocExitMarkAsCancelled   = 3

};

typedef struct OBX_DOC_OUTPUT_DONE {

    OBX_DOC_EXIT_BEHAVIOR   behavior;   // exit behavior
    P_UNKNOWN               pDocData;   // Unused: document specific data

} OBX_DOC_OUTPUT_DONE, *P_OBX_DOC_OUTPUT_DONE;

/****************************************************************************
 msgOBXSvcStateChanged      takes OBJECT, returns STATUS
     category: observer notification.
     Tells observers that the service state just changed.

 Informs observers that the state of a service has just changed.  pArgs is
 the UID of the service.
*/
#define msgOBXSvcStateChanged                   msgIOBXSvcStateChanged

/****************************************************************************
 msgOBXSvcQueryState        takes P_OBXSVC_QUERY_STATE, returns STATUS
     Passes back the state of the service.

 This message is typically used to query what state the service
 instance is in.
*/
#define msgOBXSvcQueryState                     msgIOBXSvcQueryState

typedef struct {
    BOOLEAN     enabled;                // true if the service is enabled.
    CHAR        status[nameBufLength];  // text describing the status of
                                        // the service.
    CHAR        docName[nameBufLength]; // document being processed
    P_UNKNOWN   pStateData;             // subclass's private data
} OBXSVC_QUERY_STATE, *P_OBXSVC_QUERY_STATE;

/****************************************************************************
 msgOBXSvcGetEnabled        takes P_BOOLEAN, returns STATUS
    Gets the enabled state of the service.

 Subclasses can override this message and redefine the notion of
 "enabled."  The default behavior of the superclass is to equate
 "enabled" with the ownership of the target service (i.e., output
 device).  That is, the service is "enabled" when it owns the target
 service.  By appending to or replacing the default behavior, a
 subclass can define additional conditions which must be met before a
 service is considered enabled.
*/
#define msgOBXSvcGetEnabled                     msgIOBXSvcGetEnabled

/****************************************************************************
 msgOBXSvcSetEnabled        takes BOOLEAN, returns STATUS
    Sets the enabled state of the service.

 This message is sent to the service in response to service
 notification messages msgSvcOwnerAcquired and msgSvcOwnerReleased.
 Subclasses can provide their own behavior and thereby redefine the
 notion of "enabled" for the service.  If they do, they must pass this
 message up to the ancestor so that observers of the outbox service
 will be properly notified.
*/
#define msgOBXSvcSetEnabled                     msgIOBXSvcSetEnabled

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                      Outbox Document Messages                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgOBXDocOutputStartOK         takes nothing, returns STATUS
    Asks the outbox document if it is OK to start the output process

 When an outbox service finds an opened document in the outbox
 section, it sends this message to the document instance, asking
 whether it's OK to start the output operation while the document
 remains open.  When the document receives this message, it should
 return stsOK to give the service permission to begin the output
 process.  An error status, including stsNotUnderstood, is taken to
 mean that the document instance vetos the request and the service
 will not start the output process.
*/
#define msgOBXDocOutputStartOK                  msgIOBXDocIOStartOK

/****************************************************************************
 msgOBXDocOutputStart           takes nothing, returns STATUS
    Tells an outbox document to start the output process.

 This message is sent by the outbox service to a document.  The
 document should respond to this message by starting the output
 process.
*/
#define msgOBXDocOutputStart                    msgIOBXDocIOStart

/****************************************************************************
 msgOBXDocOutputDone        takes P_OBX_DOC_OUTPUT_DONE, returns STATUS
    category: client responsibility
    Tells the outbox service that output is finished.

 When the output process is finished, the outbox document in charge of
 the output should send this message to the outbox service.  This
 message must be sent even if the output process has been aborted.
 The pArgs for this message tells the outbox service what to do with
 the outbox document.  If obxDocExitDelete is specified, the document
 will be removed from the outbox.  In all other cases the document
 will be unlocked and left in the outbox.  If either
 obxDocExitMarkAsCancelled or obxDocExitMarkAsFailed are specified,
 the name of the document will be altered to provide visual indication
 for the user that the output process has not completed successfully.

 See Also
    msgOBXDocGetService
    msgOBXDocOutputCancel
    msgOBXDocOutputCleanUp
*/
#define msgOBXDocOutputDone                     msgIOBXDocIODone

/****************************************************************************
 msgOBXDocOutputCancel          takes nothing, returns STATUS
    Tells an outbox document to cancel the output process.

 This message is used by the outbox service to inform a document that
 it should cancel the output process.  The document should handle this
 message by terminating its output operation and then sending
 msgOBXDocOutputDone to the service with pArgs->behavior set to
 obxDocExistMarkAsCancelled.

*/
#define msgOBXDocOutputCancel                   msgIOBXDocIOCancel

/****************************************************************************
 msgOBXDocStatusChanged         takes P_OBX_DOC_STATUS_CHANGED, returns STATUS
    category: client responsibility
    Tells the outbox service that the document status is changed.

 This message is sent by the outbox document to the service whenever
 its status has just changed.  This status is displayed on Status
 column for the outbox section, in the Outbox notebook.
*/
#define msgOBXDocStatusChanged                  msgIOBXDocStatusChanged

typedef struct OBX_DOC_STATUS_CHANGED {

    CHAR        status[nameBufLength];  // Text describing document state
    P_UNKNOWN   pDocData;               // Unused: document-specific data

} OBX_DOC_STATUS_CHANGED, *P_OBX_DOC_STATUS_CHANGED;

#endif  // OBXSVC_INCLUDED



