/****************************************************************************
 key.h

 Copyright 1992, GO Corporation, All Rights Reserved
 
 $Revision:   1.9  $
   $Author:   kbentley  $
	 $Date:   06 Mar 1992 16:22:18  $

 This file contains the API definition for the keyboard driver.

 clsKey inherits from clsObject.

 The functions described in this file are contained in INPUT.LIB. 

 This file defines the data sent with keyboard event.  Keyboard events
 are generated by both the real keyboard and the virtual keyboard.  
****************************************************************************/
 
/**** Keyboard Events ****/
/*
 When keyboard devices (physical or virtual) generate input events, the
 events are delivered via msgInputEvent.  The following are the value of
 pArgs->devCode for msgInputEvent.

    msgKeyDown:		sent when a key is depressed. 
 
    msgKeyUp:		sent when a key is released.  
 
    msgKeyChar:		contains an individual character code and is sent when a
                    key is depressed.

    msgKeyMulti:	contains multiple character codes that have accumulated
                    since the last msgKeyMulti event was sent.  This allows
                    processing of multiple keys without the overhead of a
                    separate message for each key.  For all of these
                    values, pArgs->eventData should be cast to P_KEY_DATA. 
                    (A msgKeyMulti event contains the same information as
                    several msgKeyChar events.)
*/


/**** Input Flags ****/
/*
 Keyboard events can be enabled or disabled using input flags.  See input.h
 for more information.  The relevant flags for keyboard events are:
//{
    input flag         enables
    ================   =======================
    inputChar          msgKeyChar
    inputMultiChar     msgKeyMulti
    inputMakeBreak     msgKeyUp and msgKeyDown
//}

 Clients should still verify that the devCode is the particular message they
 are interested in.
*/


/**** Sample Code ****/
/*
 You can verify that your msgInputEvent handler is handling a keyboard
 message by checking as follows:
//{
	if (ClsNum(pArgs->devCode) == ClsNum(clsKey)) {
		...
	}
//}

 You should further verify that the devCode is the particular message that
 you are interested in processing.

 Once you've decided that you're looking at a key event, you can cast
 pArgs->eventData as follows:
//{
	P_KEY_DATA	pKeyData;
	pKeyData = (P_KEY_DATA)(pArgs->eventData);
//}

 This example shows how you might handle msgInputEvent with a devCode of
 msgKeyUp, msgKeyDown or msgKeyChar:
//{
	for (i=0, i<pKeyData->repeatCount; i++) {
		HandleSingleKey(pKeyData->keyCode, pKeyData->shiftState);
	}
//}

 This example shows how you might handle msgInputEvent with a devCode of
 msgKeyMulti:
//{
	P_KEY_MULTI	pMulti = pKeyData->multi;
	for (i=0, i<pKeyData->repeatCount; i++) {
		for (j=0; j<pMulti[i].repeatCount; j++) {
			HandleSingleKey(pMulti[i].keyCode, pKeyMulti[i].shiftState);
		}
	}
//}
*/

#ifndef KEY_INCLUDED
#define KEY_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Keyboard Event Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define msgKeyUp			MakeMsg(clsKey, 0)
#define msgKeyDown			MakeMsg(clsKey, 1)
#define msgKeyChar			MakeMsg(clsKey, 12)
#define msgKeyMulti		 	MakeMsg(clsKey, 13)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Shift Flags ****/
/*
 * These are used in the shiftState field of KEY_MULTI and KEY_DATA.  They
 * indicate which modifier keys were down when the event was generated.
*/
#define keyScrollLock		flag0
#define keyNumLock			flag1
#define keyCapsLock			flag2
#define keyShift			flag3
#define keyCtrl				flag4
#define keyAlt				flag5
#define keyLeftShift		flag6
#define keyRightShift		flag7
#define keyLeftCtrl			flag8
#define keyRightCtrl		flag9
#define keyLeftAlt			flag10
#define keyRightAlt			flag11
#define keyShiftLock		flag12
#define keyCtrlLock			flag13
#define keyAltLock			flag14


//REFGEN BEGINIGNORE
/*
 * For internal use only.
*/
#define keyMShift1	(keyShift | keyLeftShift | keyRightShift )
#define keyMShift2 (keyCtrl  | keyLeftCtrl  | keyRightCtrl )
#define keyMShift3 (keyAlt   | keyLeftAlt   | keyRightAlt )
#define keyMomentaryShiftFlags	( keyMShift1 | keyMShift2 | keyMShift3 )
#define keyToggleShiftFlags	( keyNumLock | keyCapsLock | keyScrollLock )
#define keyModifierFlag		flag15
#define keyBreakFlag		flag15
//REFGEN ENDIGNORE


/**** Key Codes ****/

/* Special ASCII characters */
#define uKeyBackSpace	0x08
#define uKeyTab			0x09
#define uKeyLineFeed	0x0a
#define uKeyReturn		0x0d
#define uKeyEscape		0x1b

/* Keys with no ASCII values;  mapped into the user area of Unicode. */
#define uKeyF1			0xf001
#define uKeyF2			0xf002
#define uKeyF3			0xf003
#define uKeyF4			0xf004
#define uKeyF5			0xf005
#define uKeyF6			0xf006
#define uKeyF7			0xf007
#define uKeyF8			0xf008
#define uKeyF9			0xf009
#define uKeyF10			0xf00a
#define uKeyF11			0xf00b
#define uKeyF12			0xf00c
#define uKeyInsert		0xf020
#define uKeyDelete		0xf021
#define uKeyHome		0xf022
#define uKeyEnd			0xf023
#define uKeyPageUp		0xf024
#define uKeyPageDown	0xf025
#define uKeyUp			0xf026
#define uKeyDown		0xf027
#define uKeyLeft		0xf028
#define uKeyRight		0xf029
#define uKeyCenter		0xf02a
#define uKeyPrintScreen	0xf02b
#define uKeyPause		0xf02c
#define uKeySysRq		0xf02d
#define uKeyBreak		0xf02e
#define uKeyBackTab		0xf02f
				   

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					msgInputEvent Argument Types						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * KEY_MULTI holds the variable length data for msgInputEvent with a
 * devCode of msgKeyMulti.
*/
typedef struct KEY_MULTI {
	U16 keyCode;			// ASCII value
	U16 scanCode;			// keyboard scan code
	U16 shiftState;			// state of the shift, ctrl & alt keys
	U16 repeatCount;		// number of autorepeats to apply
	U8  reservedA[4];		// reserved for future use
} KEY_MULTI, *P_KEY_MULTI;


/*
 * KEY_DATA is the "true" type of msgInputEvent's pArgs->eventData for all
 * keyboard event messages.
 *
 * If msgInputEvent's pArgs->devCode is msgKeyMulti, the keyCode, scanCode
 * and shiftState fields of this struct are undefined.  Each of these
 * fields is defined in a KEY_MULTI struct.
*/
typedef struct KEY_DATA {
	U16			keyCode;		// ASCII key translation 
	U16 		scanCode;		// keyboard scan code
	U16 		shiftState;		// state of the shift, ctrl & alt keys
	U16 		repeatCount;	// if not msgKeyMulti, the no. of identical
								// keycodes received.  If msgKeyMulti, the
								// number of KEY_MULTI structs in multi.
	U8  		reserved[24];
	KEY_MULTI	multi[1];		// if msgKeyMulti, an array of KEY_MULTIs
} KEY_DATA, *P_KEY_DATA;


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                          	Private		                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void EXPORTED KeyNewCode(U16 scanCode);	//	obsolete
typedef void (EXPORTED * P_KEY_SETUP)(U16 type);

STATUS EXPORTED KeyInstallDriverRoutine(P_KEY_SETUP pRoutine);	//	obsolete
//REFGEN ENDIGNORE


#endif	// KEY_INCLUDED
